import { generateProductionOrdersFromPlan } from './server/services/production/production-order-generator';
import { pool } from './server/postgres';

async function testGeneration() {
  const planId = 14;
  
  console.log(`\n🔧 === SYMULACJA GENEROWANIA ZLP DLA PLANU ${planId} ===\n`);
  
  // Pokaż plan przed generowaniem
  const planQuery = await pool.query(`
    SELECT 
      pp.id,
      pp.plan_number,
      pp.name,
      COUNT(ppl.id) as ile_linii
    FROM production.production_plans pp
    LEFT JOIN production.production_plan_lines ppl ON ppl.plan_id = pp.id
    WHERE pp.id = $1
    GROUP BY pp.id, pp.plan_number, pp.name
  `, [planId]);
  
  console.log('📋 Plan:', planQuery.rows[0]);
  
  // Pokaż linie planu
  const linesQuery = await pool.query(`
    SELECT 
      ppl.id,
      ppl.product_id,
      p.name as product_name,
      ppl.quantity,
      ppl.color_code,
      ppl.source_reference as zamowienie
    FROM production.production_plan_lines ppl
    JOIN catalog.products p ON p.id = ppl.product_id
    WHERE ppl.plan_id = $1
    ORDER BY ppl.id
  `, [planId]);
  
  console.log('\n📦 Linie planu:');
  linesQuery.rows.forEach(line => {
    console.log(`  - Line ${line.id}: ${line.quantity}× ${line.product_name} (${line.color_code}) - Zamówienie: ${line.zamowienie}`);
  });
  
  console.log('\n⚙️ Generuję Production Orders (ZLP)...\n');
  
  // Wywołaj generator
  const result = await generateProductionOrdersFromPlan(planId);
  
  if (!result.success) {
    console.error('❌ BŁĄD:', result.errors);
    process.exit(1);
  }
  
  console.log('✅ Generacja zakończona!\n');
  console.log('📊 Podsumowanie:');
  console.log(`  - Wygenerowane ZLP: ${result.summary.totalOrders}`);
  console.log(`  - Całkowita liczba komponentów: ${result.summary.totalComponents}`);
  console.log(`  - Rozkład po kolorach:`, result.summary.colorBreakdown);
  
  // Sprawdź szczegóły wygenerowanych ZLP
  console.log('\n📝 Szczegóły wygenerowanych ZLP:\n');
  
  for (const order of result.generatedOrders) {
    console.log(`  ${order.orderNumber} (${order.colorCode}):`);
    console.log(`    - Komponentów: ${order.componentCount}`);
    console.log(`    - Suma quantity: ${order.totalQuantity}`);
    
    // Sprawdź ile jest indywidualnych rekordów z quantity=1
    const bomItemsQuery = await pool.query(`
      SELECT 
        component_name,
        quantity,
        source_product_id,
        source_furniture_reference
      FROM production.production_order_bom_items
      WHERE production_order_bom_id = $1
      ORDER BY id
      LIMIT 5
    `, [order.bomId]);
    
    console.log(`    - Pierwsze 5 rekordów BOM items:`);
    bomItemsQuery.rows.forEach(item => {
      console.log(`      • ${item.component_name}: qty=${item.quantity}, ref=${item.source_furniture_reference}`);
    });
    
    // Pokaż agregację - ile jest duplikatów tej samej formatki
    const aggregationQuery = await pool.query(`
      SELECT 
        component_name,
        COUNT(*) as ile_rekordow,
        SUM(quantity) as suma_qty,
        ARRAY_AGG(DISTINCT source_furniture_reference) as zamowienia
      FROM production.production_order_bom_items
      WHERE production_order_bom_id = $1
      GROUP BY component_name
      ORDER BY ile_rekordow DESC
      LIMIT 3
    `, [order.bomId]);
    
    console.log(`    - Top 3 komponenty (agregacja):`);
    aggregationQuery.rows.forEach(agg => {
      console.log(`      • ${agg.component_name}: ${agg.ile_rekordow} osobnych rekordów, suma qty=${agg.suma_qty}, zamówienia: ${agg.zamowienia.join(', ')}`);
    });
    console.log('');
  }
  
  // Weryfikacja finalowa
  console.log('\n🔍 WERYFIKACJA: Czy każda formatka ma quantity=1?');
  
  for (const order of result.generatedOrders) {
    const checkQuery = await pool.query(`
      SELECT 
        MIN(quantity) as min_qty,
        MAX(quantity) as max_qty,
        COUNT(*) as total_records
      FROM production.production_order_bom_items
      WHERE production_order_bom_id = $1
    `, [order.bomId]);
    
    const check = checkQuery.rows[0];
    if (check.min_qty === '1.00' && check.max_qty === '1.00') {
      console.log(`  ✅ ${order.orderNumber}: WSZYSTKIE ${check.total_records} rekordów mają quantity=1`);
    } else {
      console.log(`  ❌ ${order.orderNumber}: Znaleziono quantity różne od 1! Min=${check.min_qty}, Max=${check.max_qty}`);
    }
  }
  
  await pool.end();
  console.log('\n✨ Symulacja zakończona!\n');
}

testGeneration().catch(err => {
  console.error('❌ Błąd:', err);
  process.exit(1);
});
