import { pool } from './postgres';
import axios from 'axios';
import crypto from 'crypto';

export interface WebhookPayload {
  event: string;
  timestamp: string;
  data: any;
}

// Sign webhook payload with secret for verification
function signPayload(payload: string, secret: string): string {
  return crypto
    .createHmac('sha256', secret)
    .update(payload)
    .digest('hex');
}

// Send webhook with retry logic
export async function sendWebhook(
  webhookId: number,
  event: string,
  payload: any,
  attempt: number = 1
): Promise<boolean> {
  try {
    // Get webhook configuration
    const webhookResult = await pool.query(
      'SELECT * FROM webhook_configs WHERE id = $1 AND is_active = true',
      [webhookId]
    );

    if (webhookResult.rows.length === 0) {
      console.log(`Webhook ${webhookId} not found or inactive`);
      return false;
    }

    const webhook = webhookResult.rows[0];
    
    // Check if this event is configured for this webhook
    const events = webhook.events || [];
    if (!events.includes(event) && !events.includes('*')) {
      console.log(`Event ${event} not configured for webhook ${webhookId}`);
      return false;
    }

    // Prepare webhook payload
    const webhookPayload: WebhookPayload = {
      event,
      timestamp: new Date().toISOString(),
      data: payload,
    };

    const payloadString = JSON.stringify(webhookPayload);
    const signature = webhook.secret ? signPayload(payloadString, webhook.secret) : null;

    // Send HTTP request
    const startTime = Date.now();
    try {
      const response = await axios.post(webhook.url, webhookPayload, {
        headers: {
          'Content-Type': 'application/json',
          'X-Webhook-Signature': signature || '',
          'User-Agent': 'Alpma-OMS-Webhook/1.0',
        },
        timeout: 10000, // 10 second timeout
      });

      const responseTime = Date.now() - startTime;

      // Log successful delivery
      await pool.query(
        `INSERT INTO webhook_logs (
          webhook_id, event, payload, status_code, response_body, 
          attempt_number, success, created_at
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8)`,
        [
          webhookId,
          event,
          webhookPayload,
          response.status,
          JSON.stringify(response.data).substring(0, 1000), // Limit response body size
          attempt,
          true,
          new Date(),
        ]
      );

      // Update last_triggered_at
      await pool.query(
        'UPDATE webhook_configs SET last_triggered_at = $1 WHERE id = $2',
        [new Date(), webhookId]
      );

      console.log(`✅ Webhook ${webhookId} delivered successfully (attempt ${attempt})`);
      return true;

    } catch (error: any) {
      const responseTime = Date.now() - startTime;
      
      // Log failed delivery
      await pool.query(
        `INSERT INTO webhook_logs (
          webhook_id, event, payload, status_code, response_body, 
          attempt_number, success, created_at
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8)`,
        [
          webhookId,
          event,
          webhookPayload,
          error.response?.status || 0,
          error.message?.substring(0, 1000) || 'Unknown error',
          attempt,
          false,
          new Date(),
        ]
      );

      // Retry logic
      if (attempt < (webhook.retry_attempts || 3)) {
        const retryDelay = Math.pow(2, attempt) * 1000; // Exponential backoff: 2s, 4s, 8s
        console.log(`❌ Webhook ${webhookId} failed (attempt ${attempt}), retrying in ${retryDelay}ms...`);
        
        await new Promise(resolve => setTimeout(resolve, retryDelay));
        return sendWebhook(webhookId, event, payload, attempt + 1);
      } else {
        console.error(`❌ Webhook ${webhookId} failed after ${attempt} attempts:`, error.message);
        return false;
      }
    }
  } catch (error) {
    console.error(`Error sending webhook ${webhookId}:`, error);
    return false;
  }
}

// Generic webhook trigger for any event
export async function triggerWebhook(event: string, data: any): Promise<void> {
  try {
    // Get all active webhooks for this event
    const webhooksResult = await pool.query(
      `SELECT id, events FROM webhook_configs 
       WHERE is_active = true`
    );

    const webhooks = webhooksResult.rows.filter(webhook => {
      const events = webhook.events || [];
      return events.includes(event) || events.includes('*');
    });

    if (webhooks.length === 0) {
      return;
    }

    console.log(`📤 Triggering ${webhooks.length} webhooks for event: ${event}`);

    // Send webhooks in parallel (non-blocking)
    const promises = webhooks.map(webhook => 
      sendWebhook(webhook.id, event, data).catch(err => {
        console.error(`Webhook ${webhook.id} error:`, err);
      })
    );

    // Don't wait for webhooks to complete (fire and forget)
    Promise.all(promises).catch(err => {
      console.error('Error sending webhooks:', err);
    });

  } catch (error) {
    console.error('Error triggering webhooks:', error);
  }
}

// Trigger webhooks for order events (alias for backward compatibility)
export async function triggerOrderWebhooks(event: string, orderData: any): Promise<void> {
  return triggerWebhook(event, orderData);
}

// Trigger webhooks for production events
export async function triggerProductionWebhook(event: string, productionData: any): Promise<void> {
  return triggerWebhook(event, productionData);
}

// Trigger webhooks for inventory events
export async function triggerInventoryWebhook(event: string, inventoryData: any): Promise<void> {
  return triggerWebhook(event, inventoryData);
}

// Get webhook logs with pagination
export async function getWebhookLogs(webhookId?: number, limit: number = 100) {
  try {
    let query = `
      SELECT 
        l.id,
        l.webhook_id,
        l.event as event_type,
        l.payload,
        l.status_code as response_status,
        l.response_body,
        l.attempt_number,
        CASE WHEN l.success = true THEN 'success' ELSE 'failed' END as status,
        l.created_at,
        w.name as webhook_name
      FROM webhook_logs l
      LEFT JOIN webhook_configs w ON l.webhook_id = w.id
      ${webhookId ? 'WHERE l.webhook_id = $1' : ''}
      ORDER BY l.created_at DESC
      LIMIT ${webhookId ? '$2' : '$1'}
    `;

    const params = webhookId ? [webhookId, limit] : [limit];
    const result = await pool.query(query, params);
    return result.rows;
  } catch (error) {
    console.error('Error fetching webhook logs:', error);
    throw error;
  }
}
