import swaggerJsdoc from 'swagger-jsdoc';

const options: swaggerJsdoc.Options = {
  definition: {
    openapi: '3.0.0',
    info: {
      title: 'Alpma OMS - External API',
      version: '1.0.0',
      description: `# Alpma OMS External API

REST API for external integrations (Odoo, ERP systems) with Alpma OMS order management system.

## Hybrid Architecture: Pull + Push

This API uses a hybrid approach:
- **Pull (REST)**: Query orders and products on-demand using the endpoints below
- **Push (Webhooks)**: Receive real-time notifications when order events occur

## Webhooks Setup

### 1. Creating a Webhook
Webhooks must be configured by an administrator through the admin panel:
1. Login as admin
2. Navigate to Webhooks management (/webhooks)
3. Create new webhook with:
   - **URL**: Your callback endpoint (must be HTTPS)
   - **Events**: Select which events to receive:
     - **Orders**: order.created, order.updated, order.paid, order.shipped
     - **Production**: production.started, production.completed, production.paused, production.resumed
     - **Inventory**: inventory.created, inventory.finalized
   - **Secret**: Optional HMAC secret for signature verification
   - **Retry Attempts**: Number of retries on failure (0-10, default: 3)

### 2. Webhook Payload Format
All webhooks send POST requests with this structure:
\`\`\`json
{
  "event": "order.paid",
  "timestamp": "2025-10-09T10:30:00.000Z",
  "data": {
    // Full order object with items
  }
}
\`\`\`

### 3. Webhook Security
If you provide a secret when creating the webhook, all requests will include an **X-Webhook-Signature** header containing an HMAC-SHA256 signature.

**Verification (Python)**:
\`\`\`python
import hmac
import hashlib

def verify_signature(payload, signature, secret):
    expected = hmac.new(
        secret.encode('utf-8'),
        payload.encode('utf-8'),
        hashlib.sha256
    ).hexdigest()
    return hmac.compare_digest(signature, expected)
\`\`\`

**Verification (Node.js)**:
\`\`\`javascript
const crypto = require('crypto');

function verifySignature(payload, signature, secret) {
  const expected = crypto
    .createHmac('sha256', secret)
    .update(payload)
    .digest('hex');
  return crypto.timingSafeEqual(
    Buffer.from(signature),
    Buffer.from(expected)
  );
}
\`\`\`

### 4. Webhook Retry Logic
- Failed webhooks are automatically retried based on configured retry attempts
- Exponential backoff: 1min, 2min, 4min, 8min...
- HTTP 2xx responses are considered successful
- All other responses (4xx, 5xx, timeouts) trigger retries

### 5. Available Events

**Order Events:**
- **order.created**: New order received from Allegro/Shoper
- **order.updated**: Order details changed (address, items, etc.)
- **order.paid**: Payment status changed to PAID
- **order.shipped**: Order marked as shipped with tracking number

**Production Events:**
- **production.started**: Production order started
- **production.completed**: Production order completed
- **production.paused**: Production order paused
- **production.resumed**: Production order resumed

**Inventory Events:**
- **inventory.created**: New inventory count created
- **inventory.finalized**: Inventory count finalized and stock updated

### 6. Your Callback Endpoint
Your webhook endpoint should:
- Accept POST requests with JSON body
- Respond with HTTP 200-299 for successful processing
- Process webhooks asynchronously (respond quickly)
- Verify signature if secret is configured
- Return non-2xx to trigger retry`,
      contact: {
        name: 'Alpma OMS Support',
        email: 'biuro@alpmeb.pl'
      }
    },
    servers: [
      {
        url: '',
        description: 'Current server (relative URL)'
      },
      {
        url: '{protocol}://{host}',
        description: 'Dynamic server URL',
        variables: {
          protocol: {
            enum: ['http', 'https'],
            default: 'https'
          },
          host: {
            default: 'alp-oms.replit.app',
            description: 'Your Replit app domain'
          }
        }
      }
    ],
    components: {
      securitySchemes: {
        BearerAuth: {
          type: 'http',
          scheme: 'bearer',
          bearerFormat: 'API Token',
          description: 'API token created in admin panel. Format: `Bearer your_api_token_here`'
        }
      },
      schemas: {
        Address: {
          type: 'object',
          properties: {
            city: { type: 'string', example: 'Warszawa' },
            street: { type: 'string', example: 'Marszałkowska 1' },
            zipCode: { type: 'string', example: '00-001' },
            countryCode: { type: 'string', example: 'PL' }
          }
        },
        Order: {
          type: 'object',
          properties: {
            order_number: { type: 'string', example: '00065' },
            source_order_id: { type: 'string', example: '5ce2e000-a4df-11f0-961e-6d069f850ff8' },
            source: { type: 'string', enum: ['ALLEGRO', 'SHOPER'], example: 'ALLEGRO' },
            order_date: { type: 'string', format: 'date-time', example: '2025-10-09T07:13:54.111Z' },
            buyer_first_name: { type: 'string', example: 'Jan' },
            buyer_last_name: { type: 'string', example: 'Kowalski' },
            buyer_email: { type: 'string', example: 'jan.kowalski@example.com' },
            buyer_phone: { type: 'string', example: '+48 123 456 789' },
            buyer_login: { type: 'string', example: 'jankowalski' },
            delivery_method: { type: 'string', example: 'Allegro Kurier DPD' },
            delivery_amount: { type: 'string', example: '15.99' },
            delivery_address: { $ref: '#/components/schemas/Address' },
            invoice_required: { type: 'boolean', example: false },
            buyer_company_name: { type: 'string', nullable: true, example: null },
            tax_id: { type: 'string', nullable: true, example: null },
            billing_address: { $ref: '#/components/schemas/Address', nullable: true },
            total_amount: { type: 'string', example: '884.10' },
            currency: { type: 'string', example: 'PLN' },
            payment_status: { 
              type: 'string', 
              enum: ['PENDING', 'PAID', 'CANCELLED'],
              example: 'PAID' 
            },
            payment_type: { type: 'string', example: 'ONLINE' },
            payment_amount: { type: 'string', example: '884.10' },
            payment_date: { type: 'string', format: 'date-time', nullable: true, example: '2025-10-09T11:15:00.000Z' },
            tracking_numbers: { 
              type: 'array',
              items: { type: 'string' },
              example: ['123456789']
            },
            shipments: {
              type: 'array',
              items: {
                type: 'object',
                properties: {
                  carrier: { type: 'string', example: 'DPD' },
                  tracking_number: { type: 'string', example: '123456789' },
                  shipped_at: { type: 'string', format: 'date-time', example: '2025-10-09T14:00:00.000Z' }
                }
              },
              nullable: true
            },
            buyer_notes: { type: 'string', nullable: true, example: 'Proszę dzwonić przed dostawą' },
            has_returns: { type: 'boolean', example: false },
            refund_amount: { type: 'string', example: '0.00' },
            refund_date: { type: 'string', format: 'date-time', nullable: true, example: null },
            created_at: { type: 'string', format: 'date-time', example: '2025-10-09T08:02:38.940Z' },
            updated_at: { type: 'string', format: 'date-time', example: '2025-10-09T08:12:12.161Z' }
          }
        },
        OrderWithItems: {
          allOf: [
            { $ref: '#/components/schemas/Order' },
            {
              type: 'object',
              properties: {
                items: {
                  type: 'array',
                  items: { $ref: '#/components/schemas/OrderItem' }
                }
              }
            }
          ]
        },
        OrderItem: {
          type: 'object',
          properties: {
            id: { type: 'string', example: 'item-123' },
            product_id: { type: 'string', example: 'prod-456' },
            product_name: { type: 'string', example: 'Laptop Dell XPS 15' },
            product_ean: { type: 'string', example: '5901234567890' },
            product_sku: { type: 'string', example: 'DELL-XPS-15-2024' },
            quantity: { type: 'integer', example: 1 },
            unit_price: { type: 'string', example: '4999.00' },
            total_price: { type: 'string', example: '4999.00' },
            image_url: { type: 'string', example: 'https://example.com/images/laptop.jpg' }
          }
        },
        Product: {
          type: 'object',
          properties: {
            product_id: { type: 'string', example: 'prod-123' },
            name: { type: 'string', example: 'Laptop Dell XPS 15' },
            ean: { type: 'string', example: '5901234567890' },
            sku: { type: 'string', example: 'DELL-XPS-15-2024' },
            category: { type: 'string', example: 'Elektronika' },
            price: { type: 'string', example: '4999.00' },
            currency: { type: 'string', example: 'PLN' },
            stock_quantity: { type: 'integer', example: 5 },
            image_url: { type: 'string', example: 'https://example.com/images/laptop.jpg' },
            created_at: { type: 'string', format: 'date-time', example: '2025-10-09T08:00:00.000Z' },
            updated_at: { type: 'string', format: 'date-time', example: '2025-10-09T08:00:00.000Z' }
          }
        },
        Pagination: {
          type: 'object',
          properties: {
            page: { type: 'integer', example: 1 },
            limit: { type: 'integer', example: 50 },
            total: { type: 'integer', example: 1215 },
            totalPages: { type: 'integer', example: 25 }
          }
        },
        WebhookEvent: {
          type: 'object',
          properties: {
            event: { 
              type: 'string', 
              enum: [
                'order.created', 'order.updated', 'order.paid', 'order.shipped',
                'production.started', 'production.completed', 'production.paused', 'production.resumed',
                'inventory.created', 'inventory.finalized'
              ],
              example: 'order.paid'
            },
            timestamp: { type: 'string', format: 'date-time', example: '2025-10-09T10:30:00.000Z' },
            data: { 
              type: 'object',
              description: 'Event data - structure depends on event type (order, production, or inventory)'
            }
          }
        },
        InventoryCount: {
          type: 'object',
          description: 'Arkusz inwentaryzacji magazynowej',
          properties: {
            id: { type: 'integer', example: 15 },
            name: { type: 'string', example: 'Inwentaryzacja Q1 2026' },
            status: { type: 'string', enum: ['draft', 'in_progress', 'completed'], example: 'in_progress' },
            notes: { type: 'string', nullable: true, example: 'Inwentaryzacja półroczna magazynu głównego' },
            created_at: { type: 'string', example: '2026-01-08 10:30' },
            completed_at: { type: 'string', nullable: true, example: null },
            created_by_username: { type: 'string', example: 'marcin' },
            items_count: { type: 'integer', example: 145 }
          }
        },
        InventoryCountItem: {
          type: 'object',
          description: 'Pozycja w arkuszu inwentaryzacji',
          properties: {
            id: { type: 'integer', example: 1234 },
            material_id: { type: 'integer', nullable: true, example: 56 },
            stock_panel_id: { type: 'integer', nullable: true, example: null },
            packaging_material_id: { type: 'integer', nullable: true, example: null },
            packed_product_id: { type: 'integer', nullable: true, example: null },
            system_quantity: { type: 'number', example: 100 },
            counted_quantity: { type: 'number', nullable: true, example: 98 },
            difference: { type: 'number', nullable: true, example: -2 },
            notes: { type: 'string', nullable: true, example: '2 szt uszkodzone' },
            item_name: { type: 'string', example: 'Płyta MDF 18mm Biały' },
            item_sku: { type: 'string', example: 'MDF-18-BIALY' }
          }
        },
        CatalogProduct: {
          type: 'object',
          description: 'Produkt z katalogu wraz z BOM',
          properties: {
            id: { type: 'integer', example: 42 },
            sku: { type: 'string', example: '5905806234567' },
            name: { type: 'string', example: 'Szafka na buty V50x30D siedzisko 50cm BC' },
            description: { type: 'string', nullable: true, example: 'Szafka z siedziskiem i schowkiem' },
            category: { type: 'string', nullable: true, example: 'Szafki na buty' },
            base_price: { type: 'string', example: '259.00' },
            is_active: { type: 'boolean', example: true },
            created_at: { type: 'string', example: '2025-06-15 14:20' },
            updated_at: { type: 'string', example: '2026-01-05 09:15' },
            images: {
              type: 'array',
              items: { '$ref': '#/components/schemas/ProductImage' }
            },
            bom: { '$ref': '#/components/schemas/ProductBOM' }
          }
        },
        ProductImage: {
          type: 'object',
          description: 'Zdjęcie produktu',
          properties: {
            id: { type: 'integer', example: 156 },
            url: { type: 'string', example: 'https://files.alpsys.pl/OMS/products/images/5905806234567_1.jpg' },
            thumbnail_url: { type: 'string', nullable: true, example: 'https://files.alpsys.pl/OMS/products/thumbs/5905806234567_1.jpg' },
            medium_url: { type: 'string', nullable: true, example: 'https://files.alpsys.pl/OMS/products/medium/5905806234567_1.jpg' },
            is_primary: { type: 'boolean', example: true },
            alt_text: { type: 'string', nullable: true, example: 'Szafka na buty biały' },
            sort_order: { type: 'integer', example: 1 }
          }
        },
        ProductBOM: {
          type: 'object',
          description: 'Bill of Materials - lista materiałów do produkcji',
          properties: {
            id: { type: 'integer', example: 89 },
            name: { type: 'string', example: 'BOM Szafka V50' },
            version: { type: 'integer', example: 1 },
            status: { type: 'string', enum: ['draft', 'active', 'archived'], example: 'active' },
            bom_type: { type: 'string', example: 'final_product' },
            is_active: { type: 'boolean', example: true },
            is_approved: { type: 'boolean', example: true },
            estimated_production_time_minutes: { type: 'integer', nullable: true, example: 45 },
            components_count: { type: 'integer', example: 8 },
            components: {
              type: 'array',
              items: { '$ref': '#/components/schemas/BOMComponent' }
            }
          }
        },
        BOMComponent: {
          type: 'object',
          description: 'Komponent w BOM - formatka, akcesoria, itp.',
          properties: {
            id: { type: 'integer', example: 234 },
            name: { type: 'string', example: 'Bok prawy 400x500x18' },
            component_type: { type: 'string', enum: ['formatka', 'accessory', 'hardware', 'upholstery'], example: 'formatka' },
            quantity: { type: 'integer', example: 2 },
            unit_of_measure: { type: 'string', example: 'szt' },
            length: { type: 'number', nullable: true, example: 400.0 },
            width: { type: 'number', nullable: true, example: 500.0 },
            thickness: { type: 'number', nullable: true, example: 18.0 },
            color: { type: 'string', nullable: true, example: 'Biały połysk' },
            edging_pattern: { type: 'string', nullable: true, example: '1100' },
            is_mandatory: { type: 'boolean', example: true },
            position_in_bom: { type: 'integer', example: 1 }
          }
        },
        ProductionOrder: {
          type: 'object',
          description: 'Zlecenie produkcyjne (ZLP)',
          properties: {
            id: { type: 'integer', example: 156 },
            order_number: { type: 'string', example: 'ZLP-2026-0156' },
            product_id: { type: 'integer', nullable: true, example: 42 },
            quantity_planned: { type: 'integer', example: 5 },
            quantity_produced: { type: 'integer', example: 3 },
            status: { type: 'string', enum: ['draft', 'confirmed', 'planned', 'in_progress', 'paused', 'done', 'cancelled'], example: 'in_progress' },
            priority: { type: 'string', enum: ['low', 'normal', 'high', 'urgent'], example: 'normal' },
            planned_start_date: { type: 'string', nullable: true, example: '2026-01-10 08:00' },
            planned_end_date: { type: 'string', nullable: true, example: '2026-01-10 16:00' },
            actual_start_date: { type: 'string', nullable: true, example: '2026-01-10 08:15' },
            actual_end_date: { type: 'string', nullable: true, example: null },
            notes: { type: 'string', nullable: true, example: 'Zamówienie ekspresowe' },
            created_at: { type: 'string', example: '2026-01-08 14:30' },
            updated_at: { type: 'string', example: '2026-01-10 12:45' }
          }
        },
        Error: {
          type: 'object',
          properties: {
            success: { type: 'boolean', example: false },
            error: { type: 'string', example: 'Unauthorized - Invalid API token' }
          }
        }
      },
      parameters: {
        WebhookSignatureHeader: {
          name: 'X-Webhook-Signature',
          in: 'header',
          description: 'HMAC-SHA256 signature of the webhook payload using the configured secret. Used to verify webhook authenticity. Always verify this signature before processing the webhook.',
          required: true,
          schema: {
            type: 'string',
            example: 'a1b2c3d4e5f6g7h8i9j0k1l2m3n4o5p6q7r8s9t0u1v2w3x4y5z6'
          }
        }
      },
      examples: {
        WebhookOrderCreated: {
          summary: 'Order Created Event',
          value: {
            event: 'order.created',
            timestamp: '2025-10-09T10:30:00.000Z',
            data: {
              order_number: 1234,
              source: 'ALLEGRO',
              source_order_id: 'abc123-def456',
              buyer_email: 'customer@example.com',
              buyer_first_name: 'Jan',
              buyer_last_name: 'Kowalski',
              total_amount: '4999.00',
              payment_amount: '0.00',
              currency: 'PLN',
              payment_status: 'PENDING',
              payment_type: 'ONLINE',
              delivery_method: 'DPD Kurier',
              tracking_numbers: [],
              created_at: '2025-10-09T10:30:00.000Z',
              items: [
                {
                  product_name: 'Laptop Dell XPS 15',
                  quantity: 1,
                  unit_price: '4999.00',
                  total_price: '4999.00'
                }
              ]
            }
          }
        },
        WebhookOrderPaid: {
          summary: 'Order Paid Event',
          value: {
            event: 'order.paid',
            timestamp: '2025-10-09T11:15:00.000Z',
            data: {
              order_number: 1234,
              source: 'ALLEGRO',
              payment_status: 'PAID',
              payment_amount: '4999.00',
              payment_date: '2025-10-09T11:15:00.000Z'
            }
          }
        },
        WebhookOrderShipped: {
          summary: 'Order Shipped Event',
          value: {
            event: 'order.shipped',
            timestamp: '2025-10-09T14:00:00.000Z',
            data: {
              order_number: 1234,
              tracking_numbers: ['1234567890'],
              shipments: [{
                carrier: 'DPD',
                tracking_number: '1234567890',
                shipped_at: '2025-10-09T14:00:00.000Z'
              }]
            }
          }
        }
      }
    },
    security: [{ BearerAuth: [] }],
    tags: [
      {
        name: 'Orders',
        description: 'Order management endpoints'
      },
      {
        name: 'Products',
        description: 'Product catalog endpoints'
      },
      {
        name: 'Inventory',
        description: 'Inventory count management endpoints'
      },
      {
        name: 'Catalog',
        description: 'Catalog product endpoints with images'
      },
      {
        name: 'Production',
        description: 'Production order control endpoints'
      },
      {
        name: 'Webhooks',
        description: 'Webhook events documentation (Push notifications)'
      }
    ],
    paths: {
      '/api/external/orders': {
        get: {
          tags: ['Orders'],
          summary: 'List all orders',
          description: 'Retrieve a paginated list of orders with optional filtering by source, payment status, and date range.',
          parameters: [
            {
              name: 'page',
              in: 'query',
              description: 'Page number for pagination',
              required: false,
              schema: { type: 'integer', default: 1, minimum: 1 }
            },
            {
              name: 'limit',
              in: 'query',
              description: 'Number of items per page',
              required: false,
              schema: { type: 'integer', default: 50, minimum: 1, maximum: 100 }
            },
            {
              name: 'source',
              in: 'query',
              description: 'Filter by order source platform',
              required: false,
              schema: { type: 'string', enum: ['ALLEGRO', 'SHOPER'] }
            },
            {
              name: 'payment_status',
              in: 'query',
              description: 'Filter by payment status',
              required: false,
              schema: { type: 'string', enum: ['PENDING', 'PAID', 'CANCELLED'] }
            },
            {
              name: 'date_from',
              in: 'query',
              description: 'Data od (format: RRRR-MM-DD GG:MM, np. 2026-01-08 00:00)',
              required: false,
              schema: { type: 'string', example: '2026-01-08 00:00' }
            },
            {
              name: 'date_to',
              in: 'query',
              description: 'Data do (format: RRRR-MM-DD GG:MM, np. 2026-01-08 23:59)',
              required: false,
              schema: { type: 'string', example: '2026-01-08 00:00' }
            }
          ],
          responses: {
            '200': {
              description: 'Successful response with orders list',
              content: {
                'application/json': {
                  schema: {
                    type: 'object',
                    properties: {
                      success: { type: 'boolean', example: true },
                      data: {
                        type: 'array',
                        items: { $ref: '#/components/schemas/Order' }
                      },
                      pagination: { $ref: '#/components/schemas/Pagination' }
                    }
                  }
                }
              }
            },
            '401': {
              description: 'Unauthorized - Invalid or missing API token',
              content: {
                'application/json': {
                  schema: { $ref: '#/components/schemas/Error' }
                }
              }
            },
            '500': {
              description: 'Internal server error',
              content: {
                'application/json': {
                  schema: { $ref: '#/components/schemas/Error' }
                }
              }
            }
          }
        }
      },
      '/api/external/orders/{orderNumber}': {
        get: {
          tags: ['Orders'],
          summary: 'Get order details',
          description: 'Retrieve detailed information about a specific order by order number (e.g., "00065")',
          parameters: [
            {
              name: 'orderNumber',
              in: 'path',
              description: 'Order number (formatted with leading zeros like "00001", "00065", etc.)',
              required: true,
              schema: { type: 'string', example: '00065' }
            }
          ],
          responses: {
            '200': {
              description: 'Successful response with order details including items',
              content: {
                'application/json': {
                  schema: {
                    type: 'object',
                    properties: {
                      success: { type: 'boolean', example: true },
                      data: { $ref: '#/components/schemas/OrderWithItems' }
                    }
                  }
                }
              }
            },
            '404': {
              description: 'Order not found',
              content: {
                'application/json': {
                  schema: { $ref: '#/components/schemas/Error' }
                }
              }
            },
            '401': {
              description: 'Unauthorized - Invalid or missing API token',
              content: {
                'application/json': {
                  schema: { $ref: '#/components/schemas/Error' }
                }
              }
            }
          }
        }
      },
      '/api/external/products': {
        get: {
          tags: ['Products'],
          summary: 'List all products',
          description: 'Retrieve a paginated list of products from the catalog',
          parameters: [
            {
              name: 'page',
              in: 'query',
              description: 'Page number for pagination',
              required: false,
              schema: { type: 'integer', default: 1, minimum: 1 }
            },
            {
              name: 'limit',
              in: 'query',
              description: 'Number of items per page',
              required: false,
              schema: { type: 'integer', default: 50, minimum: 1, maximum: 100 }
            },
            {
              name: 'category',
              in: 'query',
              description: 'Filter by product category',
              required: false,
              schema: { type: 'string' }
            },
            {
              name: 'ean',
              in: 'query',
              description: 'Filter by EAN code',
              required: false,
              schema: { type: 'string' }
            },
            {
              name: 'sku',
              in: 'query',
              description: 'Filter by SKU',
              required: false,
              schema: { type: 'string' }
            }
          ],
          responses: {
            '200': {
              description: 'Successful response with products list',
              content: {
                'application/json': {
                  schema: {
                    type: 'object',
                    properties: {
                      success: { type: 'boolean', example: true },
                      data: {
                        type: 'array',
                        items: { $ref: '#/components/schemas/Product' }
                      },
                      pagination: { $ref: '#/components/schemas/Pagination' }
                    }
                  }
                }
              }
            },
            '401': {
              description: 'Unauthorized - Invalid or missing API token',
              content: {
                'application/json': {
                  schema: { $ref: '#/components/schemas/Error' }
                }
              }
            }
          }
        }
      },
      '/api/external/inventory-counts': {
        get: {
          tags: ['Inventory'],
          summary: 'List inventory counts',
          description: 'Retrieve a paginated list of inventory counts (spisy inwentaryzacyjne)',
          parameters: [
            { name: 'page', in: 'query', schema: { type: 'integer', default: 1 } },
            { name: 'limit', in: 'query', schema: { type: 'integer', default: 50, maximum: 100 } },
            { name: 'status', in: 'query', description: 'Filter by status', schema: { type: 'string', enum: ['draft', 'in_progress', 'completed'] } }
          ],
          responses: {
            '200': { description: 'List of inventory counts with pagination' },
            '401': { description: 'Unauthorized' }
          }
        }
      },
      '/api/external/inventory-counts/{id}': {
        get: {
          tags: ['Inventory'],
          summary: 'Get inventory count details',
          description: 'Retrieve inventory count with all items',
          parameters: [
            { name: 'id', in: 'path', required: true, schema: { type: 'integer' } }
          ],
          responses: {
            '200': { description: 'Inventory count with items' },
            '404': { description: 'Inventory count not found' },
            '401': { description: 'Unauthorized' }
          }
        }
      },
      '/api/external/inventory-counts/{countId}/items/{itemId}': {
        patch: {
          tags: ['Inventory'],
          summary: 'Update inventory count item',
          description: 'Update counted quantity for an inventory item',
          parameters: [
            { name: 'countId', in: 'path', required: true, schema: { type: 'integer' } },
            { name: 'itemId', in: 'path', required: true, schema: { type: 'integer' } }
          ],
          requestBody: {
            required: true,
            content: {
              'application/json': {
                schema: {
                  type: 'object',
                  required: ['counted_quantity'],
                  properties: {
                    counted_quantity: { type: 'number', description: 'Actual counted quantity' },
                    notes: { type: 'string', description: 'Optional notes' }
                  }
                }
              }
            }
          },
          responses: {
            '200': { description: 'Item updated successfully' },
            '400': { description: 'Invalid request' },
            '404': { description: 'Item not found' }
          }
        }
      },
      '/api/external/catalog/products': {
        get: {
          tags: ['Catalog'],
          summary: 'List catalog products',
          description: 'Pobierz listę produktów z katalogu wraz ze zdjęciami i podsumowaniem BOM',
          parameters: [
            { name: 'page', in: 'query', schema: { type: 'integer', default: 1 } },
            { name: 'limit', in: 'query', schema: { type: 'integer', default: 50, maximum: 100 } },
            { name: 'search', in: 'query', description: 'Search by name or SKU', schema: { type: 'string' } },
            { name: 'sku', in: 'query', description: 'Filter by exact SKU', schema: { type: 'string' } },
            { name: 'category', in: 'query', description: 'Filter by category', schema: { type: 'string' } }
          ],
          responses: {
            '200': { description: 'Lista produktów z obrazami i podsumowaniem BOM (id, name, version, status, components_count)' },
            '401': { description: 'Unauthorized' }
          }
        }
      },
      '/api/external/catalog/products/{id}': {
        get: {
          tags: ['Catalog'],
          summary: 'Get catalog product details',
          description: 'Pobierz produkt z pełną galerią zdjęć i komponentami BOM',
          parameters: [
            { name: 'id', in: 'path', required: true, schema: { type: 'integer' } }
          ],
          responses: {
            '200': { description: 'Produkt ze zdjęciami i pełnym BOM (komponenty z wymiarami, ilościami, kolorami)' },
            '404': { description: 'Product not found' },
            '401': { description: 'Unauthorized' }
          }
        }
      },
      '/api/external/production/orders': {
        get: {
          tags: ['Production'],
          summary: 'List production orders',
          description: 'Retrieve paginated list of production orders (ZLP)',
          parameters: [
            { name: 'page', in: 'query', schema: { type: 'integer', default: 1 } },
            { name: 'limit', in: 'query', schema: { type: 'integer', default: 50, maximum: 100 } },
            { name: 'status', in: 'query', description: 'Filter by status', schema: { type: 'string', enum: ['draft', 'confirmed', 'planned', 'in_progress', 'paused', 'done', 'cancelled'] } }
          ],
          responses: {
            '200': { description: 'List of production orders' },
            '401': { description: 'Unauthorized' }
          }
        }
      },
      '/api/external/production/orders/{id}/start': {
        post: {
          tags: ['Production'],
          summary: 'Start production order',
          description: 'Start a production order (changes status to in_progress)',
          parameters: [
            { name: 'id', in: 'path', required: true, schema: { type: 'integer' } }
          ],
          responses: {
            '200': { description: 'Order started successfully' },
            '400': { description: 'Cannot start order - invalid status' },
            '401': { description: 'Unauthorized' }
          }
        }
      },
      '/api/external/production/orders/{id}/complete': {
        post: {
          tags: ['Production'],
          summary: 'Complete production order',
          description: 'Complete a production order (changes status to done)',
          parameters: [
            { name: 'id', in: 'path', required: true, schema: { type: 'integer' } }
          ],
          responses: {
            '200': { description: 'Order completed successfully' },
            '400': { description: 'Cannot complete order - must be in_progress' },
            '401': { description: 'Unauthorized' }
          }
        }
      },
      '/api/external/production/orders/{id}/pause': {
        post: {
          tags: ['Production'],
          summary: 'Pause production order',
          description: 'Pause a production order in progress',
          parameters: [
            { name: 'id', in: 'path', required: true, schema: { type: 'integer' } }
          ],
          responses: {
            '200': { description: 'Order paused successfully' },
            '400': { description: 'Cannot pause order - must be in_progress' },
            '401': { description: 'Unauthorized' }
          }
        }
      },
      '/api/external/production/orders/{id}/resume': {
        post: {
          tags: ['Production'],
          summary: 'Resume production order',
          description: 'Resume a paused production order',
          parameters: [
            { name: 'id', in: 'path', required: true, schema: { type: 'integer' } }
          ],
          responses: {
            '200': { description: 'Order resumed successfully' },
            '400': { description: 'Cannot resume order - must be paused' },
            '401': { description: 'Unauthorized' }
          }
        }
      },
      '/webhooks/callback': {
        post: {
          tags: ['Webhooks'],
          summary: 'Webhook callback endpoint (Client Implementation)',
          description: `This is an **example** of the webhook callback endpoint that YOU (the client/Odoo) must implement on your server.

Alpma OMS will send POST requests to your configured URL when order events occur.

**Implementation Requirements:**
1. Accept POST requests with JSON body
2. Verify the X-Webhook-Signature header (HMAC-SHA256)
3. Process the event asynchronously
4. Return HTTP 200-299 to acknowledge receipt
5. Return non-2xx to trigger retry (up to configured retry attempts)

**Security:** Always verify the signature to ensure the webhook is authentic.`,
          parameters: [
            { $ref: '#/components/parameters/WebhookSignatureHeader' }
          ],
          requestBody: {
            required: true,
            content: {
              'application/json': {
                schema: { $ref: '#/components/schemas/WebhookEvent' },
                examples: {
                  'order.created': { $ref: '#/components/examples/WebhookOrderCreated' },
                  'order.paid': { $ref: '#/components/examples/WebhookOrderPaid' },
                  'order.shipped': { $ref: '#/components/examples/WebhookOrderShipped' }
                }
              }
            }
          },
          responses: {
            '200': {
              description: 'Webhook received and processed successfully',
              content: {
                'application/json': {
                  schema: {
                    type: 'object',
                    properties: {
                      success: { type: 'boolean', example: true },
                      message: { type: 'string', example: 'Event processed' }
                    }
                  }
                }
              }
            },
            '400': {
              description: 'Invalid signature or malformed payload - will trigger retry',
              content: {
                'application/json': {
                  schema: {
                    type: 'object',
                    properties: {
                      success: { type: 'boolean', example: false },
                      error: { type: 'string', example: 'Invalid signature' }
                    }
                  }
                }
              }
            },
            '500': {
              description: 'Internal error - will trigger retry',
              content: {
                'application/json': {
                  schema: {
                    type: 'object',
                    properties: {
                      success: { type: 'boolean', example: false },
                      error: { type: 'string', example: 'Internal server error' }
                    }
                  }
                }
              }
            }
          },
          security: []
        }
      }
    }
  },
  apis: []
};

export const swaggerSpec = swaggerJsdoc(options);
