// Mapowanie statusów zamówień na polskie nazwy

// Cache dla niestandardowych statusów Shoper
let shoperStatusCache: Record<string, string> | null = null;
let shoperStatusCacheTime: number | null = null;
const CACHE_DURATION = 3600000; // 1 godzina

export const SHOPER_STATUS_MAP_FALLBACK: Record<string, string> = {
  '1': 'złożone',
  '2': 'przyjęte do realizacji',
  '3': 'oczekiwanie na dostawę',
  '4': 'w trakcie kompletowania',
  '5': 'oczekiwanie na płatność',
  '6': 'gotowe do wysłania',
  '7': 'przesyłka wysłana',
  '8': 'anulowane',
  '9': 'zakończone'
};

export const ALLEGRO_STATUS_MAP: Record<string, string> = {
  'NEW': 'Nowe',
  'FILLED_IN': 'Wypełnione',
  'READY_FOR_PROCESSING': 'Gotowe do realizacji',
  'PROCESSING': 'W trakcie realizacji',
  'SENT': 'Wysłane',
  'CANCELLED': 'Anulowane'
};

export const ERLI_STATUS_MAP: Record<string, string> = {
  'new': 'Nowe',
  'pending': 'Oczekujące',
  'processing': 'W trakcie realizacji',
  'ready_to_ship': 'Gotowe do wysyłki',
  'shipped': 'Wysłane',
  'delivered': 'Dostarczone',
  'completed': 'Zakończone',
  'cancelled': 'Anulowane',
  'refunded': 'Zwrócone'
};

/**
 * Pobiera i cache'uje niestandardowe statusy ze Shopera
 */
export async function getShoperStatusMap(): Promise<Record<string, string>> {
  // Sprawdź cache
  if (shoperStatusCache && shoperStatusCacheTime) {
    if (Date.now() - shoperStatusCacheTime < CACHE_DURATION) {
      return shoperStatusCache;
    }
  }

  try {
    // Dynamiczny import aby uniknąć circular dependencies
    const { getShoperStatuses } = await import('./shoper-api.js');
    const statuses = await getShoperStatuses();
    
    const statusMap: Record<string, string> = {};
    statuses.forEach((status: any) => {
      const statusId = status.status_id;
      const statusName = status.translations?.pl_PL?.name;
      
      if (statusId && statusName) {
        statusMap[statusId] = statusName;
      }
    });
    
    shoperStatusCache = statusMap;
    shoperStatusCacheTime = Date.now();
    
    console.log('✅ Shoper status map cached:', statusMap);
    return statusMap;
  } catch (error) {
    console.error('❌ Failed to fetch Shoper statuses, using fallback:', error);
    return SHOPER_STATUS_MAP_FALLBACK;
  }
}

/**
 * Mapuje status zamówienia na polski odpowiednik
 */
export async function mapOrderStatus(status: string | null, source: string): Promise<string> {
  if (!status) return 'Nieznany';
  
  if (source === 'SHOPER') {
    const statusMap = await getShoperStatusMap();
    return statusMap[status] || status;
  } else if (source === 'ALLEGRO') {
    return ALLEGRO_STATUS_MAP[status] || status;
  } else if (source === 'ERLI') {
    return ERLI_STATUS_MAP[status.toLowerCase()] || status;
  }
  
  return status;
}

/**
 * Synchroniczna wersja mapowania (używa cache lub fallback)
 */
export function mapOrderStatusSync(status: string | null, source: string): string {
  if (!status) return 'Nieznany';
  
  if (source === 'SHOPER') {
    // Użyj cache jeśli dostępne, w przeciwnym razie fallback
    const statusMap = shoperStatusCache || SHOPER_STATUS_MAP_FALLBACK;
    return statusMap[status] || status;
  } else if (source === 'ALLEGRO') {
    return ALLEGRO_STATUS_MAP[status] || status;
  }
  
  return status;
}

/**
 * Mapuje status płatności na polski odpowiednik
 */
export function mapPaymentStatus(paymentStatus: string | null): string {
  if (!paymentStatus) return 'Nieznany';
  
  const PAYMENT_STATUS_MAP: Record<string, string> = {
    'PAID': 'Zapłacone',
    'PENDING': 'Oczekuje na płatność',
    'CANCELLED': 'Anulowane',
    'REFUNDED': 'Zwrócone'
  };
  
  return PAYMENT_STATUS_MAP[paymentStatus] || paymentStatus;
}
