/**
 * Pallet Flow Aggregator
 * 
 * Grupuje komponenty na palety na podstawie zbieżności ścieżek produkcyjnych.
 * 
 * Logika:
 * 1. Wszystkie formatki z jednego koloru startują razem (zbieżne cięcie)
 * 2. Po cięciu rozdzielamy na palety wg następnej operacji:
 *    - PALETA-OKLEJANIE: formatki idące na oklejanie (CO, COW, BOCZKI-TYL-SZUFLAD)
 *    - PALETA-WIERCENIE: formatki idące bezpośrednio na wiercenie (CW - np. DNO)
 *    - PALETA-GOTOWE: formatki gotowe po cięciu (C - np. HDF)
 * 3. Po oklejaniu znowu rozdzielamy:
 *    - CO → gotowe (brak dalszych operacji)
 *    - COW, BOCZKI-TYL-SZUFLAD → kontynuują na wiercenie
 */

import { pool } from '../../postgres';

/**
 * Definicje ścieżek operacji dla znanych wariantów marszrut.
 * Operacje są w kolejności wykonywania (bez magazynowania/transportu/pakowania).
 */
export const ROUTING_OPERATION_PATHS: Record<string, string[]> = {
  // Warianty z production_routing_variant_rules
  'CO': ['cutting', 'edging'],                    // Cięcie → Oklejanie (gotowe)
  'COW': ['cutting', 'edging', 'drilling'],       // Cięcie → Oklejanie → Wiercenie
  'CW': ['cutting', 'drilling'],                  // Cięcie → Wiercenie (bez oklejania)
  'C': ['cutting'],                               // Tylko cięcie (np. HDF)
  'BOCZKI-TYL-SZUFLAD': ['cutting', 'edging', 'drilling'], // Jak COW
  
  // Marszruty główne (fallback gdy brak wariantu)
  'marsz_formatki_cow': ['cutting', 'edging', 'drilling'],
  'marsz_formatki_co': ['cutting', 'edging'],
  'marsz_formatki_cw': ['cutting', 'drilling'],
  'marsz_ciecie': ['cutting'],
  'marsz_szuflady': ['cutting', 'edging', 'drilling'],
};

/**
 * Grupy ścieżek - określają które warianty mogą być na tej samej palecie
 * w danej fazie produkcji.
 */
export const FLOW_PATH_GROUPS = {
  // Faza: PO CIĘCIU
  POST_CUTTING: {
    // Grupa 1: Idą na oklejanie (wspólna ścieżka: cutting→edging)
    TO_EDGING: ['CO', 'COW', 'BOCZKI-TYL-SZUFLAD', 'marsz_formatki_cow', 'marsz_formatki_co', 'marsz_szuflady'],
    // Grupa 2: Idą bezpośrednio na wiercenie (cutting→drilling)
    TO_DRILLING: ['CW', 'marsz_formatki_cw'],
    // Grupa 3: Gotowe po cięciu
    DONE: ['C', 'marsz_ciecie'],
  },
  // Faza: PO OKLEJANIU
  POST_EDGING: {
    // Grupa 1: Gotowe po oklejaniu
    DONE: ['CO', 'marsz_formatki_co'],
    // Grupa 2: Kontynuują na wiercenie
    TO_DRILLING: ['COW', 'BOCZKI-TYL-SZUFLAD', 'marsz_formatki_cow', 'marsz_szuflady'],
  },
};

/**
 * Typy faz palet
 */
export type PalletPhase = 'CUTTING' | 'POST_CUTTING_EDGING' | 'POST_CUTTING_DRILLING' | 'POST_CUTTING_DONE' | 
                         'POST_EDGING_DONE' | 'POST_EDGING_DRILLING';

/**
 * Informacja o palecie przepływowej
 */
export interface FlowPallet {
  palletCode: string;           // np. "BIALY-OKLEJANIE-1"
  phase: PalletPhase;           // Faza palety
  splitAfterOperation: string;  // Operacja po której paleta się rozdziela
  variantCodes: string[];       // Kody wariantów na tej palecie
  routingIds: number[];         // ID marszrut
}

/**
 * Określa grupę ścieżki dla danego wariantu/marszruty w fazie PO CIĘCIU.
 */
export function getPostCuttingGroup(variantOrRoutingCode: string): 'TO_EDGING' | 'TO_DRILLING' | 'DONE' {
  const code = variantOrRoutingCode.toLowerCase();
  
  // Sprawdź czy idzie na oklejanie
  if (FLOW_PATH_GROUPS.POST_CUTTING.TO_EDGING.some(v => v.toLowerCase() === code)) {
    return 'TO_EDGING';
  }
  
  // Sprawdź czy idzie bezpośrednio na wiercenie (CW, marsz_formatki_cw)
  if (FLOW_PATH_GROUPS.POST_CUTTING.TO_DRILLING.some(v => v.toLowerCase() === code)) {
    return 'TO_DRILLING';
  }
  
  // Sprawdź czy gotowe po cięciu
  if (FLOW_PATH_GROUPS.POST_CUTTING.DONE.some(v => v.toLowerCase() === code)) {
    return 'DONE';
  }
  
  // Fallback: na podstawie ścieżki operacji (próbuj różne warianty klucza)
  const path = ROUTING_OPERATION_PATHS[variantOrRoutingCode] || 
               ROUTING_OPERATION_PATHS[variantOrRoutingCode.toUpperCase()] ||
               ROUTING_OPERATION_PATHS[variantOrRoutingCode.toLowerCase()];
  if (path) {
    if (path.includes('edging')) return 'TO_EDGING';
    if (path.includes('drilling') && !path.includes('edging')) return 'TO_DRILLING';
    if (path.length === 1 && path[0] === 'cutting') return 'DONE';
  }
  
  // Domyślnie: na oklejanie (najbardziej typowa ścieżka COW)
  return 'TO_EDGING';
}

/**
 * Określa grupę ścieżki dla danego wariantu/marszruty w fazie PO OKLEJANIU.
 */
export function getPostEdgingGroup(variantOrRoutingCode: string): 'DONE' | 'TO_DRILLING' | null {
  const code = variantOrRoutingCode.toLowerCase();
  
  // Najpierw sprawdź czy w ogóle przechodzi przez oklejanie
  const postCuttingGroup = getPostCuttingGroup(variantOrRoutingCode);
  if (postCuttingGroup !== 'TO_EDGING') {
    return null; // Nie przechodzi przez oklejanie
  }
  
  // Sprawdź czy gotowe po oklejaniu (CO, marsz_formatki_co)
  if (FLOW_PATH_GROUPS.POST_EDGING.DONE.some(v => v.toLowerCase() === code)) {
    return 'DONE';
  }
  
  // Sprawdź czy kontynuuje na wiercenie (COW, BOCZKI-TYL, marsz_formatki_cow, marsz_szuflady)
  if (FLOW_PATH_GROUPS.POST_EDGING.TO_DRILLING.some(v => v.toLowerCase() === code)) {
    return 'TO_DRILLING';
  }
  
  // Fallback: na podstawie ścieżki operacji (próbuj różne warianty klucza)
  const path = ROUTING_OPERATION_PATHS[variantOrRoutingCode] || 
               ROUTING_OPERATION_PATHS[variantOrRoutingCode.toUpperCase()] ||
               ROUTING_OPERATION_PATHS[variantOrRoutingCode.toLowerCase()];
  if (path) {
    if (path.includes('drilling')) return 'TO_DRILLING';
    if (!path.includes('drilling')) return 'DONE';
  }
  
  // Domyślnie: na wiercenie (najbardziej typowa ścieżka COW)
  return 'TO_DRILLING';
}

/**
 * Generuje kod palety przepływowej dla grupy komponentów.
 */
export function generateFlowPalletCode(
  colorCode: string,
  postCuttingGroup: 'TO_EDGING' | 'TO_DRILLING' | 'DONE',
  postEdgingGroup: 'DONE' | 'TO_DRILLING' | null,
  sequence: number
): string {
  let suffix: string;
  
  if (postCuttingGroup === 'TO_DRILLING') {
    suffix = 'WIERC'; // Bezpośrednio na wiercenie (CW)
  } else if (postCuttingGroup === 'DONE') {
    suffix = 'CIECIE'; // Tylko cięcie (C)
  } else {
    // TO_EDGING - dalej rozdzielamy po oklejaniu
    if (postEdgingGroup === 'DONE') {
      suffix = 'OKLEJANIE'; // Gotowe po oklejaniu (CO)
    } else {
      suffix = 'COW'; // Kontynuuje na wiercenie (COW)
    }
  }
  
  return `${colorCode}-${suffix}-${sequence}`;
}

/**
 * Interfejs komponentu do agregacji (uproszczony)
 */
export interface ComponentForPalletAggregation {
  componentId: number;
  generatedName: string;
  color: string;
  variantCode: string | null;
  routingCode: string | null;
  routingId: number | null;
}

/**
 * Wynik agregacji palet przepływowych
 */
export interface PalletAggregationResult {
  palletCode: string;
  phase: string;
  splitAfterOperation: string | null;
  nextPalletPhase: string | null;
  variantCodes: string[];
  routingIds: number[];
  componentIds: number[];
}

/**
 * Agreguje komponenty na palety przepływowe w ramach jednego koloru.
 * 
 * Logika:
 * 1. Grupuj po postCuttingGroup (TO_EDGING, TO_DRILLING, DONE)
 * 2. Dla TO_EDGING dodatkowo grupuj po postEdgingGroup (DONE, TO_DRILLING)
 */
export function aggregateComponentsToFlowPallets(
  colorCode: string,
  components: ComponentForPalletAggregation[]
): PalletAggregationResult[] {
  // Mapa: "postCuttingGroup|postEdgingGroup" -> komponenty
  const groups = new Map<string, {
    postCuttingGroup: 'TO_EDGING' | 'TO_DRILLING' | 'DONE';
    postEdgingGroup: 'DONE' | 'TO_DRILLING' | null;
    variantCodes: Set<string>;
    routingIds: Set<number>;
    componentIds: number[];
  }>();
  
  for (const comp of components) {
    // Określ kod wariantu/marszruty do analizy ścieżki
    const code = comp.variantCode || comp.routingCode || 'COW'; // Domyślnie COW
    
    const postCuttingGroup = getPostCuttingGroup(code);
    const postEdgingGroup = getPostEdgingGroup(code);
    
    const groupKey = `${postCuttingGroup}|${postEdgingGroup || 'N/A'}`;
    
    if (!groups.has(groupKey)) {
      groups.set(groupKey, {
        postCuttingGroup,
        postEdgingGroup,
        variantCodes: new Set(),
        routingIds: new Set(),
        componentIds: [],
      });
    }
    
    const group = groups.get(groupKey)!;
    if (comp.variantCode) group.variantCodes.add(comp.variantCode);
    if (comp.routingId) group.routingIds.add(comp.routingId);
    group.componentIds.push(comp.componentId);
  }
  
  // Konwertuj do wyników
  const results: PalletAggregationResult[] = [];
  let sequence = 1;
  
  for (const group of Array.from(groups.values())) {
    const palletCode = generateFlowPalletCode(
      colorCode,
      group.postCuttingGroup,
      group.postEdgingGroup,
      sequence
    );
    
    // Określ punkt rozdzielenia i następną fazę
    let splitAfterOperation: string | null = null;
    let nextPalletPhase: string | null = null;
    let phase: string;
    
    if (group.postCuttingGroup === 'TO_EDGING') {
      if (group.postEdgingGroup === 'DONE') {
        phase = 'POST_EDGING_DONE';
        splitAfterOperation = 'edging';
        nextPalletPhase = null; // Koniec
      } else {
        phase = 'POST_EDGING_DRILLING';
        splitAfterOperation = 'edging';
        nextPalletPhase = 'DRILLING';
      }
    } else if (group.postCuttingGroup === 'TO_DRILLING') {
      phase = 'POST_CUTTING_DRILLING';
      splitAfterOperation = 'cutting';
      nextPalletPhase = 'DRILLING';
    } else {
      phase = 'POST_CUTTING_DONE';
      splitAfterOperation = 'cutting';
      nextPalletPhase = null; // Koniec
    }
    
    results.push({
      palletCode,
      phase,
      splitAfterOperation,
      nextPalletPhase,
      variantCodes: Array.from(group.variantCodes),
      routingIds: Array.from(group.routingIds),
      componentIds: group.componentIds,
    });
    
    sequence++;
  }
  
  return results;
}

/**
 * Pobiera ścieżkę operacji dla marszruty z bazy danych (cache'owane).
 */
const routingPathCache = new Map<number, string[]>();

export async function getRoutingOperationPath(routingId: number, client?: any): Promise<string[]> {
  if (routingPathCache.has(routingId)) {
    return routingPathCache.get(routingId)!;
  }
  
  const db = client || pool;
  const result = await db.query(`
    SELECT code
    FROM production.production_routing_operations
    WHERE routing_id = $1
      AND code IN ('cutting', 'edging', 'drilling', 'wiercenie', 'wiercenie 2')
    ORDER BY sequence
  `, [routingId]);
  
  const path = result.rows.map((r: any) => {
    // Normalizuj nazwy operacji
    const code = r.code.toLowerCase();
    if (code === 'wiercenie' || code === 'wiercenie 2') return 'drilling';
    return code;
  });
  
  routingPathCache.set(routingId, path);
  return path;
}

/**
 * Czyści cache ścieżek operacji (np. po zmianie konfiguracji marszrut).
 */
export function clearRoutingPathCache(): void {
  routingPathCache.clear();
}

/**
 * Loguje strukturę palet przepływowych dla debugowania.
 */
export function logFlowPalletStructure(results: PalletAggregationResult[]): void {
  console.log('\n🎨 [Pallet Flow Aggregator] Struktura palet:');
  for (const pallet of results) {
    console.log(`  📦 ${pallet.palletCode}`);
    console.log(`     Faza: ${pallet.phase}`);
    console.log(`     Rozdzielenie po: ${pallet.splitAfterOperation || 'N/A'}`);
    console.log(`     Następna faza: ${pallet.nextPalletPhase || 'KONIEC'}`);
    console.log(`     Warianty: [${pallet.variantCodes.join(', ')}]`);
    console.log(`     Komponenty: ${pallet.componentIds.length} szt.`);
  }
  console.log('');
}
