import { Pool } from "pg";
import type { ProductionOrder } from "@shared/schema";

interface InsertProductionOrder {
  orderNumber: string;
  productId: number;
  bomId?: number | null;
  routingId?: number | null;
  status?: string;
  priority?: string;
  quantityPlanned: number;
  unitOfMeasure?: string;
  plannedStartDate?: Date | null;
  plannedEndDate?: Date | null;
  responsibleUserId?: number | null;
  sourceOrderNumber?: string | null;
  locationId?: number | null;
  notes?: string | null;
}

function mapRowToOrder(row: any): ProductionOrder {
  return {
    id: row.id,
    orderNumber: row.order_number,
    productId: row.product_id,
    bomId: row.bom_id,
    routingId: row.routing_id,
    status: row.status,
    workflowStage: row.workflow_stage,
    priority: row.priority,
    quantityPlanned: row.quantity_planned,
    quantityProduced: row.quantity_produced,
    quantityScrap: row.quantity_scrap,
    unitOfMeasure: row.unit_of_measure,
    plannedStartDate: row.planned_start_date,
    plannedEndDate: row.planned_end_date,
    actualStartDate: row.actual_start_date,
    actualEndDate: row.actual_end_date,
    workflowStageUpdatedAt: row.workflow_stage_updated_at,
    workflowStageUpdatedBy: row.workflow_stage_updated_by,
    responsibleUserId: row.responsible_user_id,
    sourceOrderNumber: row.source_order_number,
    carrierRequirements: row.carrier_requirements,
    locationId: row.location_id,
    parameters: row.parameters,
    notes: row.notes,
    colorCode: row.color_code,
    cuttingSequence: row.cutting_sequence,
    cuttingPriority: row.cutting_priority,
    requiresCutting: row.requires_cutting,
    transportBatchId: row.transport_batch_id,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

interface OrderFilters {
  status?: string;
  workflowStage?: string;
  priority?: string;
  productId?: number;
  responsibleUserId?: number;
  startDate?: Date;
  endDate?: Date;
  search?: string; // order number or product name
  limit?: number;
  offset?: number;
}

export async function getOrders(pool: Pool, filters?: OrderFilters): Promise<ProductionOrder[]> {
  let query = `SELECT * FROM production.production_orders WHERE 1=1`;
  const params: any[] = [];
  let paramIndex = 1;

  if (filters?.status) {
    query += ` AND status = $${paramIndex++}`;
    params.push(filters.status);
  }

  if (filters?.workflowStage) {
    query += ` AND workflow_stage = $${paramIndex++}`;
    params.push(filters.workflowStage);
  }

  if (filters?.priority) {
    query += ` AND priority = $${paramIndex++}`;
    params.push(filters.priority);
  }

  if (filters?.productId) {
    query += ` AND product_id = $${paramIndex++}`;
    params.push(filters.productId);
  }

  if (filters?.responsibleUserId) {
    query += ` AND responsible_user_id = $${paramIndex++}`;
    params.push(filters.responsibleUserId);
  }

  if (filters?.startDate) {
    query += ` AND planned_start_date >= $${paramIndex++}`;
    params.push(filters.startDate);
  }

  if (filters?.endDate) {
    query += ` AND planned_end_date <= $${paramIndex++}`;
    params.push(filters.endDate);
  }

  if (filters?.search) {
    query += ` AND order_number ILIKE $${paramIndex++}`;
    params.push(`%${filters.search}%`);
  }

  query += ` ORDER BY created_at DESC`;

  if (filters?.limit) {
    query += ` LIMIT $${paramIndex++}`;
    params.push(filters.limit);
  }

  if (filters?.offset) {
    query += ` OFFSET $${paramIndex++}`;
    params.push(filters.offset);
  }

  const result = await pool.query(query, params);
  
  return result.rows.map(mapRowToOrder);
}

export async function getOrderById(pool: Pool, id: number): Promise<any | null> {
  // Get order with routing and location names
  const result = await pool.query(`
    SELECT 
      po.*,
      r.code as routing_code,
      r.name as routing_name,
      l.code as location_code,
      l.name as location_name
    FROM production.production_orders po
    LEFT JOIN production.production_routings r ON r.id = po.routing_id
    LEFT JOIN production.production_locations l ON l.id = po.location_id
    WHERE po.id = $1
  `, [id]);
  
  if (result.rows.length === 0) return null;
  
  const row = result.rows[0];
  const order = {
    ...mapRowToOrder(row),
    routingCode: row.routing_code,
    routingName: row.routing_name,
    locationCode: row.location_code,
    locationName: row.location_name,
  };

  // Fetch work orders (operations) with work center names, routing info, operators, and buffer info
  const workOrdersResult = await pool.query(`
    SELECT 
      wo.id,
      wo.work_order_number,
      wo.sequence,
      wo.status,
      wo.quantity_planned,
      wo.quantity_produced,
      wo.quantity_scrap,
      wo.quality_check_passed,
      wo.quality_check_notes,
      wo.notes,
      wo.routing_operation_id,
      ro.code as operation_code,
      ro.name as operation_name,
      ro.icon as operation_icon,
      ro.color as operation_color,
      ro.routing_id as routing_id,
      r.code as routing_code,
      r.name as routing_name,
      wo.work_center_id,
      wc.code as work_center_code,
      wc.name as work_center_name,
      wo.scheduled_start_time,
      wo.scheduled_end_time,
      wo.actual_start_time,
      wo.actual_end_time,
      wo.operator_id,
      wo.operator_name,
      op.full_name as primary_operator_name,
      op.short_code as primary_operator_code,
      ro.buffer_before_id,
      ro.buffer_after_id,
      lb_before.code as buffer_before_code,
      lb_before.name as buffer_before_name,
      lb_after.code as buffer_after_code,
      lb_after.name as buffer_after_name,
      wo.location_id as warehouse_location_id,
      wl.code as warehouse_location_code,
      wl.name as warehouse_location_name,
      wo.estimated_duration,
      wo.carrier_id,
      woc.code as carrier_code,
      woc.name as carrier_name
    FROM production.production_work_orders wo
    LEFT JOIN production.production_routing_operations ro ON ro.id = wo.routing_operation_id
    LEFT JOIN production.production_routings r ON r.id = ro.routing_id
    LEFT JOIN production.production_work_centers wc ON wc.id = wo.work_center_id
    LEFT JOIN production.production_operators op ON op.id = wo.operator_id
    LEFT JOIN production.production_locations lb_before ON lb_before.id = ro.buffer_before_id
    LEFT JOIN production.production_locations lb_after ON lb_after.id = ro.buffer_after_id
    LEFT JOIN production.production_locations wl ON wl.id = wo.location_id
    LEFT JOIN production.production_carriers woc ON woc.id = wo.carrier_id
    WHERE wo.production_order_id = $1
    ORDER BY wo.sequence
  `, [id]);

  // Fetch all assigned operators for work orders
  const operatorsResult = await pool.query(`
    SELECT 
      woo.work_order_id,
      woo.operator_id,
      woo.is_primary,
      o.full_name,
      o.short_code
    FROM production.work_order_operators woo
    JOIN production.production_operators o ON o.id = woo.operator_id
    WHERE woo.work_order_id IN (
      SELECT id FROM production.production_work_orders WHERE production_order_id = $1
    )
    ORDER BY woo.is_primary DESC, o.full_name
  `, [id]);

  // Group operators by work order
  const operatorsByWorkOrder: Record<number, Array<{operatorId: number; operatorName: string; operatorCode: string | null; isPrimary: boolean}>> = {};
  for (const row of operatorsResult.rows) {
    if (!operatorsByWorkOrder[row.work_order_id]) {
      operatorsByWorkOrder[row.work_order_id] = [];
    }
    operatorsByWorkOrder[row.work_order_id].push({
      operatorId: row.operator_id,
      operatorName: row.full_name,
      operatorCode: row.short_code,
      isPrimary: row.is_primary,
    });
  }

  // Count damaged items per operation code (from BOM items with required_operations)
  const damagedCountsResult = await pool.query(`
    SELECT 
      jsonb_array_elements_text(pobi.required_operations) as operation_code,
      COUNT(*) as damaged_count
    FROM production.production_order_bom_items pobi
    JOIN production.production_order_boms pob ON pob.id = pobi.production_order_bom_id
    WHERE pob.production_order_id = $1
      AND pobi.is_damaged = true
      AND pobi.required_operations IS NOT NULL
    GROUP BY jsonb_array_elements_text(pobi.required_operations)
  `, [id]);

  // Build a map of operation code -> damaged count
  const damagedByOperation: Record<string, number> = {};
  for (const row of damagedCountsResult.rows) {
    damagedByOperation[row.operation_code?.toLowerCase() || ''] = parseInt(row.damaged_count) || 0;
  }

  // Fetch pallets (production order pallets) with their current operation
  const palletsResult = await pool.query(`
    SELECT 
      pop.id,
      pop.pallet_label,
      pop.sequence,
      pop.color_code,
      pop.flow_code,
      pop.carrier_id,
      pop.carrier_code,
      pop.status,
      pop.is_filled,
      pop.current_load,
      pop.max_capacity,
      pop.current_operation_id,
      pop.current_operation_code,
      pop.current_location_id,
      pc.code as carrier_name,
      pl.name as location_name
    FROM production.production_order_pallets pop
    LEFT JOIN production.production_carriers pc ON pc.id = pop.carrier_id
    LEFT JOIN production.production_locations pl ON pl.id = pop.current_location_id
    WHERE pop.production_order_id = $1
    ORDER BY pop.sequence
  `, [id]);

  // Group pallets by operation code
  const palletsByOperation: Record<string, Array<{
    id: number;
    palletLabel: string;
    flowCode: string;
    carrierCode: string | null;
    carrierName: string | null;
    status: string;
    currentLoad: number;
    maxCapacity: number;
  }>> = {};
  
  for (const pallet of palletsResult.rows) {
    const opCode = pallet.current_operation_code?.toLowerCase() || 'unassigned';
    if (!palletsByOperation[opCode]) {
      palletsByOperation[opCode] = [];
    }
    palletsByOperation[opCode].push({
      id: pallet.id,
      palletLabel: pallet.pallet_label,
      flowCode: pallet.flow_code,
      carrierCode: pallet.carrier_code,
      carrierName: pallet.carrier_name,
      status: pallet.status,
      currentLoad: parseInt(pallet.current_load) || 0,
      maxCapacity: parseInt(pallet.max_capacity) || 100,
    });
  }

  // Build array of work orders first to compute inherited routing for warehouse ops
  const rawWorkOrders = workOrdersResult.rows.map((row: any) => ({
    id: row.id,
    workOrderNumber: row.work_order_number,
    sequence: row.sequence,
    status: row.status,
    quantityPlanned: row.quantity_planned,
    quantityProduced: row.quantity_produced,
    quantityScrap: row.quantity_scrap,
    qualityCheckPassed: row.quality_check_passed,
    qualityCheckNotes: row.quality_check_notes,
    notes: row.notes,
    routingOperationId: row.routing_operation_id,
    operationCode: row.operation_code,
    operationName: row.operation_name,
    operationIcon: row.operation_icon,
    operationColor: row.operation_color,
    routingId: row.routing_id,
    routingCode: row.routing_code,
    routingName: row.routing_name,
    workCenterId: row.work_center_id,
    workCenterCode: row.work_center_code,
    workCenterName: row.work_center_name,
    scheduledStartTime: row.scheduled_start_time,
    scheduledEndTime: row.scheduled_end_time,
    actualStartTime: row.actual_start_time,
    actualEndTime: row.actual_end_time,
    operatorId: row.operator_id,
    operatorName: row.primary_operator_name || row.operator_name,
    operatorCode: row.primary_operator_code,
    assignedOperators: operatorsByWorkOrder[row.id] || [],
    bufferBeforeId: row.buffer_before_id,
    bufferBeforeCode: row.buffer_before_code,
    bufferBeforeName: row.buffer_before_name,
    bufferAfterId: row.buffer_after_id,
    bufferAfterCode: row.buffer_after_code,
    bufferAfterName: row.buffer_after_name,
    warehouseLocationId: row.warehouse_location_id,
    warehouseLocationCode: row.warehouse_location_code,
    warehouseLocationName: row.warehouse_location_name,
    estimatedDuration: row.estimated_duration,
    isWarehouseOperation: Boolean(!row.routing_operation_id && row.warehouse_location_id),
    damagedCount: damagedByOperation[row.operation_code?.toLowerCase() || ''] || 0,
    pallets: palletsByOperation[row.operation_code?.toLowerCase() || ''] || [],
    carrierId: row.carrier_id,
    carrierCode: row.carrier_code,
    carrierName: row.carrier_name,
  }));
  
  // For warehouse operations, inherit routing from previous operation in sequence
  const workOrders = rawWorkOrders.map((wo, idx) => {
    if (wo.isWarehouseOperation && !wo.routingId) {
      // Find the previous non-warehouse operation to inherit routing from
      for (let i = idx - 1; i >= 0; i--) {
        if (rawWorkOrders[i].routingId) {
          return {
            ...wo,
            routingId: rawWorkOrders[i].routingId,
            routingCode: rawWorkOrders[i].routingCode,
            routingName: rawWorkOrders[i].routingName,
            inheritedRouting: true,
          };
        }
      }
    }
    return wo;
  });
  
  // Fetch BOM items from production order BOM (not from product template BOM)
  // This gives us the actual components generated for this specific production order
  const bomResult = await pool.query(`
    SELECT id, source_plan_id FROM production.production_order_boms
    WHERE production_order_id = $1
    ORDER BY created_at DESC
    LIMIT 1
  `, [id]);
  
  if (bomResult.rows.length === 0) {
    return { ...order, workOrders, components: [], sourcePlanId: null };
  }
  
  const bomId = bomResult.rows[0].id;
  const sourcePlanId = bomResult.rows[0].source_plan_id;
  
  // Fetch BOM items - these are the actual components for this production order
  const componentsResult = await pool.query(`
    SELECT 
      pobi.id,
      pobi.component_name as generated_name,
      pobi.component_type,
      pobi.length as calculated_length,
      pobi.width as calculated_width,
      pobi.thickness,
      pobi.color_code as color,
      pobi.quantity,
      pobi.unit_of_measure,
      pobi.is_damaged,
      pobi.damage_type,
      pobi.damage_notes,
      pobi.damaged_at,
      pobi.damaged_by_user_id as damaged_by,
      pobi.notes as production_notes,
      NULL as board_type,
      NULL as drilling_pattern,
      NULL as edging_pattern,
      NULL as edging_material,
      NULL as visualization_url
    FROM production.production_order_bom_items pobi
    WHERE pobi.production_order_bom_id = $1
    ORDER BY pobi.id
  `, [bomId]);
  
  return {
    ...order,
    workOrders,
    sourcePlanId,
    components: componentsResult.rows.map((row: any) => ({
      id: row.id,
      generatedName: row.generated_name,
      componentType: row.component_type,
      calculatedLength: row.calculated_length,
      calculatedWidth: row.calculated_width,
      thickness: row.thickness,
      color: row.color,
      boardType: row.board_type,
      drillingPattern: row.drilling_pattern,
      edgingPattern: row.edging_pattern,
      edgingMaterial: row.edging_material,
      quantity: row.quantity,
      unitOfMeasure: row.unit_of_measure,
      visualizationUrl: row.visualization_url,
      isDamaged: row.is_damaged,
      damageType: row.damage_type,
      damageNotes: row.damage_notes,
      damagedAt: row.damaged_at,
      damagedBy: row.damaged_by,
      productionNotes: row.production_notes,
    })),
  };
}

// Get BOM items (formatki/komponenty) for a specific production order (ZLP)
export async function getOrderBomItems(pool: Pool, orderId: number): Promise<any[]> {
  // First get the production order BOM for this order
  const bomResult = await pool.query(`
    SELECT id FROM production.production_order_boms
    WHERE production_order_id = $1
    ORDER BY created_at DESC
    LIMIT 1
  `, [orderId]);
  
  if (bomResult.rows.length === 0) {
    return [];
  }
  
  const bomId = bomResult.rows[0].id;
  
  // Fetch all BOM items for this production order with routing variant info
  const itemsResult = await pool.query(`
    SELECT 
      pobi.id,
      pobi.component_name,
      pobi.component_type,
      pobi.quantity,
      pobi.unit_of_measure,
      pobi.color_code,
      pobi.length,
      pobi.width,
      pobi.thickness,
      pobi.routing_variant_id,
      pobi.required_operations,
      pobi.source_plan_line_id,
      pobi.source_product_id,
      pobi.source_furniture_reference,
      pobi.is_damaged,
      pobi.damage_type,
      pobi.damage_notes,
      pobi.item_status,
      pobi.quantity_ordered,
      pobi.quantity_produced,
      pobi.quantity_damaged,
      pobi.quantity_scrapped,
      pobi.notes,
      pobi.created_at,
      pobi.is_drilled,
      pobi.drilled_at,
      pobi.drilled_by_user_id,
      pobi.last_warehouse_doc_id,
      u.username as drilled_by_username,
      prv.variant_code as routing_code,
      wd.doc_number as warehouse_doc_number
    FROM production.production_order_bom_items pobi
    LEFT JOIN warehouse.documents wd ON wd.id = pobi.last_warehouse_doc_id
    LEFT JOIN production.production_routing_variants prv ON prv.id = pobi.routing_variant_id
    LEFT JOIN users u ON u.id = pobi.drilled_by_user_id
    WHERE pobi.production_order_bom_id = $1
    ORDER BY pobi.component_type, pobi.component_name
  `, [bomId]);
  
  return itemsResult.rows.map((row: any) => {
    // Parse requiredOperations from JSON if it's a string
    let parsedOperations: string[] = [];
    if (row.required_operations) {
      try {
        parsedOperations = typeof row.required_operations === 'string' 
          ? JSON.parse(row.required_operations) 
          : row.required_operations;
      } catch (e) {
        parsedOperations = [];
      }
    }
    
    // Derive routing code from required operations if not set
    let routingCode = row.routing_code;
    if (!routingCode && parsedOperations.length > 0) {
      const hasOklejanie = parsedOperations.some(op => 
        op.toLowerCase().includes('oklej') || op.toLowerCase() === 'edging'
      );
      const hasWiercenie = parsedOperations.some(op => 
        op.toLowerCase().includes('wierc') || op.toLowerCase() === 'drilling'
      );
      
      if (hasOklejanie && hasWiercenie) routingCode = 'COW';
      else if (hasOklejanie) routingCode = 'CO';
      else if (hasWiercenie) routingCode = 'CW';
      else routingCode = 'C';
    }
    
    return {
      id: row.id,
      formatkaName: row.component_name,
      formatkaCode: row.component_name?.split('-').slice(0, -1).join('-') || row.component_name,
      componentType: row.component_type,
      quantity: parseFloat(row.quantity) || 0,
      unitOfMeasure: row.unit_of_measure,
      colorCode: row.color_code,
      length: row.length ? parseFloat(row.length) : null,
      width: row.width ? parseFloat(row.width) : null,
      thickness: row.thickness ? parseFloat(row.thickness) : null,
      routingVariantId: row.routing_variant_id,
      routingCode,
      requiredOperations: parsedOperations,
      sourcePlanLineId: row.source_plan_line_id,
      sourceProductId: row.source_product_id,
      sourceFurnitureReference: row.source_furniture_reference,
      isDamaged: row.is_damaged,
      damageType: row.damage_type,
      damageNotes: row.damage_notes,
      isDrilled: row.is_drilled || false,
      drilledAt: row.drilled_at,
      drilledByUserId: row.drilled_by_user_id,
      drilledByUsername: row.drilled_by_username,
      lastWarehouseDocId: row.last_warehouse_doc_id,
      warehouseDocNumber: row.warehouse_doc_number,
      itemStatus: row.item_status,
      quantityOrdered: row.quantity_ordered ? parseFloat(row.quantity_ordered) : null,
      quantityProduced: row.quantity_produced ? parseFloat(row.quantity_produced) : 0,
      quantityDamaged: row.quantity_damaged ? parseFloat(row.quantity_damaged) : 0,
      quantityScrapped: row.quantity_scrapped ? parseFloat(row.quantity_scrapped) : 0,
      notes: row.notes,
      createdAt: row.created_at,
    };
  });
}

export async function createOrder(pool: Pool, data: InsertProductionOrder): Promise<ProductionOrder> {
  const result = await pool.query(`
    INSERT INTO production.production_orders 
    (order_number, product_id, bom_id, routing_id, status, priority, 
     quantity_planned, unit_of_measure, planned_start_date, planned_end_date,
     responsible_user_id, source_order_number, location_id, notes)
    VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14)
    RETURNING *
  `, [
    data.orderNumber,
    data.productId,
    data.bomId ?? null,
    data.routingId ?? null,
    data.status ?? 'draft',
    data.priority ?? 'normal',
    data.quantityPlanned,
    data.unitOfMeasure ?? 'szt',
    data.plannedStartDate ?? null,
    data.plannedEndDate ?? null,
    data.responsibleUserId ?? null,
    data.sourceOrderNumber ?? null,
    data.locationId ?? null,
    data.notes ?? null,
  ]);
  
  return mapRowToOrder(result.rows[0]);
}

export async function updateOrder(pool: Pool, id: number, data: Partial<InsertProductionOrder>): Promise<ProductionOrder | null> {
  const updates: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  if (data.orderNumber !== undefined) {
    updates.push(`order_number = $${paramIndex++}`);
    values.push(data.orderNumber);
  }
  if (data.productId !== undefined) {
    updates.push(`product_id = $${paramIndex++}`);
    values.push(data.productId);
  }
  if (data.bomId !== undefined) {
    updates.push(`bom_id = $${paramIndex++}`);
    values.push(data.bomId);
  }
  if (data.routingId !== undefined) {
    updates.push(`routing_id = $${paramIndex++}`);
    values.push(data.routingId);
  }
  if (data.status !== undefined) {
    updates.push(`status = $${paramIndex++}`);
    values.push(data.status);
  }
  if (data.priority !== undefined) {
    updates.push(`priority = $${paramIndex++}`);
    values.push(data.priority);
  }
  if (data.quantityPlanned !== undefined) {
    updates.push(`quantity_planned = $${paramIndex++}`);
    values.push(data.quantityPlanned);
  }
  if (data.unitOfMeasure !== undefined) {
    updates.push(`unit_of_measure = $${paramIndex++}`);
    values.push(data.unitOfMeasure);
  }
  if (data.plannedStartDate !== undefined) {
    updates.push(`planned_start_date = $${paramIndex++}`);
    values.push(data.plannedStartDate);
  }
  if (data.plannedEndDate !== undefined) {
    updates.push(`planned_end_date = $${paramIndex++}`);
    values.push(data.plannedEndDate);
  }
  if (data.responsibleUserId !== undefined) {
    updates.push(`responsible_user_id = $${paramIndex++}`);
    values.push(data.responsibleUserId);
  }
  if (data.sourceOrderNumber !== undefined) {
    updates.push(`source_order_number = $${paramIndex++}`);
    values.push(data.sourceOrderNumber);
  }
  if (data.locationId !== undefined) {
    updates.push(`location_id = $${paramIndex++}`);
    values.push(data.locationId);
  }
  if (data.notes !== undefined) {
    updates.push(`notes = $${paramIndex++}`);
    values.push(data.notes);
  }

  if (updates.length === 0) return null;

  updates.push(`updated_at = CURRENT_TIMESTAMP`);
  values.push(id);

  const result = await pool.query(`
    UPDATE production.production_orders 
    SET ${updates.join(', ')}
    WHERE id = $${paramIndex}
    RETURNING *
  `, values);

  if (result.rows.length === 0) return null;
  
  return mapRowToOrder(result.rows[0]);
}

export async function deleteOrder(pool: Pool, id: number): Promise<boolean> {
  const result = await pool.query(`
    DELETE FROM production.production_orders WHERE id = $1
  `, [id]);
  
  return result.rowCount ? result.rowCount > 0 : false;
}

// Status transition methods
export async function startOrder(pool: Pool, id: number): Promise<ProductionOrder | null> {
  const result = await pool.query(`
    UPDATE production.production_orders 
    SET status = 'in_progress',
        actual_start_date = CURRENT_TIMESTAMP,
        updated_at = CURRENT_TIMESTAMP
    WHERE id = $1 AND status IN ('draft', 'confirmed', 'planned')
    RETURNING *
  `, [id]);

  if (result.rows.length === 0) return null;
  
  return mapRowToOrder(result.rows[0]);
}

export async function completeOrder(pool: Pool, id: number): Promise<ProductionOrder | null> {
  const result = await pool.query(`
    UPDATE production.production_orders 
    SET status = 'done',
        actual_end_date = CURRENT_TIMESTAMP,
        updated_at = CURRENT_TIMESTAMP
    WHERE id = $1 AND status = 'in_progress'
    RETURNING *
  `, [id]);

  if (result.rows.length === 0) return null;
  
  return mapRowToOrder(result.rows[0]);
}

export async function cancelOrder(pool: Pool, id: number): Promise<ProductionOrder | null> {
  const result = await pool.query(`
    UPDATE production.production_orders 
    SET status = 'cancelled',
        updated_at = CURRENT_TIMESTAMP
    WHERE id = $1 AND status NOT IN ('done', 'cancelled')
    RETURNING *
  `, [id]);

  if (result.rows.length === 0) return null;
  
  return mapRowToOrder(result.rows[0]);
}

// Workflow stage transitions
const WORKFLOW_STAGES = [
  'magazine',      // Material in warehouse
  'cutting',       // Cutting operation
  'edging',        // Edging operation
  'drilling',      // Drilling operation
  'upholstering',  // Upholstering operation
  'picking',       // Picking components
  'packing',       // Packing finished product
  'strapping',     // Strapping for shipping
  'ready',         // Ready for dispatch
  'shipped'        // Shipped to customer
] as const;

export async function updateWorkflowStage(
  pool: Pool,
  orderId: number,
  stage: string,
  userId?: number
): Promise<ProductionOrder | null> {
  // Validate stage
  if (!WORKFLOW_STAGES.includes(stage as any)) {
    throw new Error(`Invalid workflow stage: ${stage}`);
  }

  // Update workflow_stage with audit trail
  const result = await pool.query(`
    UPDATE production.production_orders 
    SET workflow_stage = $1,
        workflow_stage_updated_at = CURRENT_TIMESTAMP,
        workflow_stage_updated_by = $2,
        updated_at = CURRENT_TIMESTAMP
    WHERE id = $3
    RETURNING *
  `, [stage, userId || null, orderId]);

  if (result.rows.length === 0) return null;
  
  return mapRowToOrder(result.rows[0]);
}

// Auto-generate order number
export async function generateOrderNumber(pool: Pool): Promise<string> {
  const year = new Date().getFullYear();
  const result = await pool.query(`
    SELECT order_number FROM production.production_orders 
    WHERE order_number LIKE 'ZLP-${year}-%'
    ORDER BY order_number DESC
    LIMIT 1
  `);

  if (result.rows.length === 0) {
    return `ZLP-${year}-001`;
  }

  const lastNumber = result.rows[0].order_number;
  const match = lastNumber.match(/ZLP-\d{4}-(\d+)$/);
  
  if (match) {
    const nextNum = (parseInt(match[1]) + 1).toString().padStart(3, '0');
    return `ZLP-${year}-${nextNum}`;
  }

  return `ZLP-${year}-001`;
}

// Update damage status for BOM item
interface UpdateBomItemDamageData {
  isDamaged: boolean;
  damageType?: string | null;
  damageNotes?: string | null;
  damagedBy?: number;
}

export async function updateBomItemDamage(
  pool: Pool,
  itemId: number,
  data: UpdateBomItemDamageData
): Promise<any | null> {
  const result = await pool.query(`
    UPDATE production.production_order_bom_items
    SET is_damaged = $1,
        damage_type = $2,
        damage_notes = $3,
        damaged_at = CASE WHEN $1 = true THEN CURRENT_TIMESTAMP ELSE NULL END,
        damaged_by_user_id = CASE WHEN $1 = true THEN $4::integer ELSE NULL END,
        item_status = CASE WHEN $1 = true THEN 'damaged' ELSE 'pending' END,
        updated_at = CURRENT_TIMESTAMP
    WHERE id = $5
    RETURNING 
      id,
      component_name as generated_name,
      component_type,
      length as calculated_length,
      width as calculated_width,
      thickness,
      color_code as color,
      quantity,
      unit_of_measure,
      is_damaged,
      damage_type,
      damage_notes,
      damaged_at,
      damaged_by_user_id as damaged_by,
      item_status
  `, [
    data.isDamaged,
    data.isDamaged ? (data.damageType || null) : null,
    data.isDamaged ? (data.damageNotes || null) : null,
    data.damagedBy || null,
    itemId
  ]);

  if (result.rows.length === 0) return null;
  
  return {
    id: result.rows[0].id,
    generatedName: result.rows[0].generated_name,
    componentType: result.rows[0].component_type,
    calculatedLength: result.rows[0].calculated_length,
    calculatedWidth: result.rows[0].calculated_width,
    thickness: result.rows[0].thickness,
    color: result.rows[0].color,
    quantity: result.rows[0].quantity,
    unitOfMeasure: result.rows[0].unit_of_measure,
    isDamaged: result.rows[0].is_damaged,
    damageType: result.rows[0].damage_type,
    damageNotes: result.rows[0].damage_notes,
    damagedAt: result.rows[0].damaged_at,
    damagedBy: result.rows[0].damaged_by,
  };
}

export async function startWorkOrder(pool: Pool, workOrderId: number): Promise<any> {
  const client = await pool.connect();
  try {
    await client.query('BEGIN');
    
    const woResult = await client.query(`
      SELECT wo.*, po.status as order_status
      FROM production.production_work_orders wo
      JOIN production.production_orders po ON po.id = wo.production_order_id
      WHERE wo.id = $1
      FOR UPDATE OF wo
    `, [workOrderId]);
    
    if (woResult.rows.length === 0) {
      await client.query('ROLLBACK');
      return { error: 'not_found', message: 'Work order not found' };
    }
    
    const wo = woResult.rows[0];
    
    if (wo.status !== 'pending' && wo.status !== 'ready') {
      await client.query('ROLLBACK');
      return { error: 'invalid_status', message: `Cannot start work order with status: ${wo.status}` };
    }
    
    if (wo.order_status !== 'in_progress' && wo.order_status !== 'confirmed') {
      await client.query('ROLLBACK');
      return { error: 'invalid_order_status', message: `Production order must be in_progress or confirmed to start work orders` };
    }
    
    // Get the routing_id for this work order
    const routingResult = await client.query(`
      SELECT ro.routing_id 
      FROM production.production_routing_operations ro
      WHERE ro.id = $1
    `, [wo.routing_operation_id]);
    const currentRoutingId = routingResult.rows[0]?.routing_id;
    
    // Check previous work order in the SAME routing (not across all routings)
    const previousWo = await client.query(`
      SELECT wo2.id, wo2.status 
      FROM production.production_work_orders wo2
      JOIN production.production_routing_operations ro2 ON ro2.id = wo2.routing_operation_id
      WHERE wo2.production_order_id = $1 
        AND wo2.sequence < $2
        AND ro2.routing_id = $3
      ORDER BY wo2.sequence DESC
      LIMIT 1
      FOR UPDATE OF wo2
    `, [wo.production_order_id, wo.sequence, currentRoutingId]);
    
    if (previousWo.rows.length > 0 && previousWo.rows[0].status !== 'done') {
      await client.query('ROLLBACK');
      return { error: 'sequence_error', message: 'Previous work order must be completed first' };
    }
    
    const result = await client.query(`
      UPDATE production.production_work_orders
      SET 
        status = 'in_progress',
        actual_start_time = NOW(),
        updated_at = NOW()
      WHERE id = $1 AND status IN ('pending', 'ready')
      RETURNING *
    `, [workOrderId]);
    
    if (result.rows.length === 0) {
      await client.query('ROLLBACK');
      return { error: 'concurrent_update', message: 'Work order status was changed by another process' };
    }
    
    await client.query(`
      INSERT INTO production.production_order_logs 
      (production_order_id, work_order_id, event_type, event_data, message, created_at)
      VALUES ($1, $2, 'work_order_started', $3, $4, NOW())
    `, [
      wo.production_order_id,
      workOrderId,
      JSON.stringify({ oldStatus: wo.status, newStatus: 'in_progress' }),
      `Work order ${wo.work_order_number} started`
    ]);
    
    // Auto-lock the production plan when any work order starts
    // Find the plan ID through production_order_boms
    const planResult = await client.query(`
      SELECT DISTINCT pob.source_plan_id
      FROM production.production_order_boms pob
      WHERE pob.production_order_id = $1 AND pob.source_plan_id IS NOT NULL
      LIMIT 1
    `, [wo.production_order_id]);
    
    if (planResult.rows.length > 0 && planResult.rows[0].source_plan_id) {
      const planId = planResult.rows[0].source_plan_id;
      
      // Lock the plan if not already locked
      await client.query(`
        UPDATE production.production_plans
        SET zlp_locked = true, zlp_locked_at = NOW(), updated_at = NOW()
        WHERE id = $1 AND zlp_locked = false
      `, [planId]);
      
      console.log(`🔒 Auto-locked plan ${planId} because work order ${wo.work_order_number} was started`);
    }
    
    await client.query('COMMIT');
    
    return {
      id: result.rows[0].id,
      workOrderNumber: result.rows[0].work_order_number,
      status: result.rows[0].status,
      actualStartTime: result.rows[0].actual_start_time,
    };
  } catch (error) {
    await client.query('ROLLBACK');
    throw error;
  } finally {
    client.release();
  }
}

export interface CompleteWorkOrderOptions {
  quantityProduced?: number;
  quantityScrap?: number;
  qualityCheckPassed?: boolean;
  qualityCheckNotes?: string;
  notes?: string;
}

export async function completeWorkOrder(pool: Pool, workOrderId: number, options: CompleteWorkOrderOptions = {}): Promise<any> {
  const client = await pool.connect();
  try {
    await client.query('BEGIN');
    
    const woResult = await client.query(`
      SELECT wo.*, po.id as po_id, po.quantity_planned as po_qty
      FROM production.production_work_orders wo
      JOIN production.production_orders po ON po.id = wo.production_order_id
      WHERE wo.id = $1
      FOR UPDATE OF wo, po
    `, [workOrderId]);
    
    if (woResult.rows.length === 0) {
      await client.query('ROLLBACK');
      return { error: 'not_found', message: 'Work order not found' };
    }
    
    const wo = woResult.rows[0];
    
    if (wo.status !== 'in_progress') {
      await client.query('ROLLBACK');
      return { error: 'invalid_status', message: `Cannot complete work order with status: ${wo.status}. Must be in_progress.` };
    }
    
    const quantityPlanned = parseFloat(wo.quantity_planned);
    const quantityScrap = options.quantityScrap ?? 0;
    const quantityProduced = options.quantityProduced ?? (quantityPlanned - quantityScrap);
    const qualityCheckPassed = options.qualityCheckPassed ?? (quantityScrap === 0);
    
    const result = await client.query(`
      UPDATE production.production_work_orders
      SET 
        status = 'done',
        actual_end_time = NOW(),
        quantity_produced = $2,
        quantity_scrap = $3,
        quality_check_passed = $4,
        quality_check_notes = $5,
        notes = COALESCE($6, notes),
        updated_at = NOW()
      WHERE id = $1 AND status = 'in_progress'
      RETURNING *
    `, [workOrderId, quantityProduced, quantityScrap, qualityCheckPassed, options.qualityCheckNotes || null, options.notes || null]);
    
    if (result.rows.length === 0) {
      await client.query('ROLLBACK');
      return { error: 'concurrent_update', message: 'Work order status was changed by another process' };
    }
    
    const eventData: any = { 
      oldStatus: wo.status, 
      newStatus: 'done',
      quantityProduced,
      quantityScrap,
    };
    if (quantityScrap > 0) {
      eventData.qualityCheckPassed = qualityCheckPassed;
      eventData.qualityCheckNotes = options.qualityCheckNotes;
    }
    
    await client.query(`
      INSERT INTO production.production_order_logs 
      (production_order_id, work_order_id, event_type, event_data, message, created_at)
      VALUES ($1, $2, 'work_order_completed', $3, $4, NOW())
    `, [
      wo.production_order_id,
      workOrderId,
      JSON.stringify(eventData),
      quantityScrap > 0 
        ? `Work order ${wo.work_order_number} completed with ${quantityScrap} scrap items`
        : `Work order ${wo.work_order_number} completed`
    ]);
    
    // Auto-create pallets after cutting operation
    let autoCreatedPallets: any[] = [];
    if (wo.routing_operation_id) {
      const opResult = await client.query(`
        SELECT code FROM production.production_routing_operations WHERE id = $1
      `, [wo.routing_operation_id]);
      
      const operationCode = opResult.rows[0]?.code || '';
      
      // Check if this is a cutting operation using code heuristics
      if (isCuttingOperation(operationCode, '')) {
        autoCreatedPallets = await autoCreatePalletsAfterCutting(
          client, 
          wo.production_order_id, 
          operationCode, 
          wo.routing_operation_id
        );
        
        if (autoCreatedPallets.length > 0) {
          await client.query(`
            INSERT INTO production.production_order_logs 
            (production_order_id, work_order_id, event_type, event_data, message, created_at)
            VALUES ($1, $2, 'pallets_auto_created', $3, $4, NOW())
          `, [
            wo.production_order_id,
            workOrderId,
            JSON.stringify({ pallets: autoCreatedPallets.map(p => ({ id: p.id, label: p.palletLabel, flowCode: p.flowCode })) }),
            `Auto-created ${autoCreatedPallets.length} pallet(s) after ${operationCode} operation`
          ]);
        }
      }
      
      // Mark BOM items stage completion based on operation type
      // Update is_cut, is_edged, is_drilled for formatki that require this operation
      // Cutting: all formatki with cutting in required_operations (or with required_operations containing cięcie/cut/cutting)
      if (isCuttingOperation(operationCode, '')) {
        await client.query(`
          UPDATE production.production_order_bom_items pobi
          SET is_cut = true
          FROM production.production_order_boms pob
          WHERE pobi.production_order_bom_id = pob.id
            AND pob.production_order_id = $1
            AND pobi.is_cut = false
            AND pobi.is_damaged = false
            AND pobi.required_operations IS NOT NULL
            AND EXISTS (
              SELECT 1 FROM jsonb_array_elements_text(pobi.required_operations) op
              WHERE lower(op) LIKE '%cut%' OR lower(op) LIKE '%cieci%' OR lower(op) = 'cutting' OR lower(op) = 'cięcie'
            )
        `, [wo.production_order_id]);
        console.log(`✂️ Marked BOM items as cut for production order ${wo.production_order_id}`);
      }
      
      // Edging: formatki with edging/oklejanie in required_operations
      if (isEdgingOperation(operationCode, '')) {
        await client.query(`
          UPDATE production.production_order_bom_items pobi
          SET is_edged = true
          FROM production.production_order_boms pob
          WHERE pobi.production_order_bom_id = pob.id
            AND pob.production_order_id = $1
            AND pobi.is_edged = false
            AND pobi.is_damaged = false
            AND pobi.required_operations IS NOT NULL
            AND EXISTS (
              SELECT 1 FROM jsonb_array_elements_text(pobi.required_operations) op
              WHERE lower(op) LIKE '%oklej%' OR lower(op) LIKE '%edge%' OR lower(op) = 'edging' OR lower(op) = 'oklejanie'
            )
        `, [wo.production_order_id]);
        console.log(`📐 Marked BOM items as edged for production order ${wo.production_order_id}`);
      }
      
      // Drilling: formatki with drilling/wiercenie in required_operations
      if (isDrillingOperation(operationCode, '')) {
        await client.query(`
          UPDATE production.production_order_bom_items pobi
          SET is_drilled = true, drilled_at = NOW()
          FROM production.production_order_boms pob
          WHERE pobi.production_order_bom_id = pob.id
            AND pob.production_order_id = $1
            AND pobi.is_drilled = false
            AND pobi.is_damaged = false
            AND pobi.required_operations IS NOT NULL
            AND EXISTS (
              SELECT 1 FROM jsonb_array_elements_text(pobi.required_operations) op
              WHERE lower(op) LIKE '%wierc%' OR lower(op) LIKE '%drill%' OR lower(op) = 'drilling' OR lower(op) = 'wiercenie' OR lower(op) = 'cnc'
            )
        `, [wo.production_order_id]);
        console.log(`🔩 Marked BOM items as drilled for production order ${wo.production_order_id}`);
      }
    }
    
    const remainingWoResult = await client.query(`
      SELECT COUNT(*) as remaining
      FROM production.production_work_orders
      WHERE production_order_id = $1 AND status != 'done' AND status != 'cancelled'
    `, [wo.production_order_id]);
    
    const remainingCount = parseInt(remainingWoResult.rows[0].remaining);
    
    if (remainingCount === 0) {
      await client.query(`
        UPDATE production.production_orders
        SET 
          status = 'done',
          actual_end_date = NOW(),
          quantity_produced = quantity_planned,
          updated_at = NOW()
        WHERE id = $1
      `, [wo.production_order_id]);
      
      await client.query(`
        INSERT INTO production.production_order_logs 
        (production_order_id, event_type, event_data, message, created_at)
        VALUES ($1, 'order_auto_completed', $2, $3, NOW())
      `, [
        wo.production_order_id,
        JSON.stringify({ reason: 'all_work_orders_done' }),
        `Production order auto-completed - all work orders done`
      ]);
    }
    
    // Auto-start next warehouse operation if exists
    // Find the next pending work order after current sequence (not necessarily +1)
    const nextPendingOpResult = await client.query(`
      SELECT id, work_order_number, sequence, location_id, routing_operation_id
      FROM production.production_work_orders
      WHERE production_order_id = $1 
        AND sequence > $2
        AND status = 'pending'
      ORDER BY sequence ASC
      LIMIT 1
      FOR UPDATE
    `, [wo.production_order_id, wo.sequence]);
    
    // Check if the next pending operation is a warehouse operation
    const nextWarehouseOpResult = {
      rows: nextPendingOpResult.rows.filter(row => 
        row.routing_operation_id === null && row.location_id !== null
      )
    };
    
    let autoStartedWarehouseOp = null;
    
    if (nextWarehouseOpResult.rows.length > 0) {
      const nextWo = nextWarehouseOpResult.rows[0];
      
      await client.query(`
        UPDATE production.production_work_orders
        SET 
          status = 'in_progress',
          actual_start_time = NOW(),
          updated_at = NOW()
        WHERE id = $1
      `, [nextWo.id]);
      
      await client.query(`
        INSERT INTO production.production_order_logs 
        (production_order_id, work_order_id, event_type, event_data, message, created_at)
        VALUES ($1, $2, 'warehouse_operation_auto_started', $3, $4, NOW())
      `, [
        wo.production_order_id,
        nextWo.id,
        JSON.stringify({ previousWorkOrderId: workOrderId, locationId: nextWo.location_id }),
        `Warehouse operation ${nextWo.work_order_number} auto-started after completing ${wo.work_order_number}`
      ]);
      
      autoStartedWarehouseOp = {
        id: nextWo.id,
        workOrderNumber: nextWo.work_order_number,
        sequence: nextWo.sequence,
      };
      
      console.log(`🏪 Auto-started warehouse operation ${nextWo.work_order_number} after completing ${wo.work_order_number}`);
    }
    
    await client.query('COMMIT');
    
    return {
      id: result.rows[0].id,
      workOrderNumber: result.rows[0].work_order_number,
      status: result.rows[0].status,
      actualEndTime: result.rows[0].actual_end_time,
      quantityProduced: result.rows[0].quantity_produced,
      quantityScrap: result.rows[0].quantity_scrap,
      qualityCheckPassed: result.rows[0].quality_check_passed,
      qualityCheckNotes: result.rows[0].quality_check_notes,
      orderAutoCompleted: remainingCount === 0,
      autoStartedWarehouseOp,
      autoCreatedPallets,
    };
  } catch (error) {
    await client.query('ROLLBACK');
    throw error;
  } finally {
    client.release();
  }
}

/**
 * Get pallet flow data for a production order - includes buffers and pallets
 */
export async function getOrderPalletFlow(pool: Pool, orderId: number): Promise<any> {
  // Get work orders with buffer info from routing operations
  const workOrdersWithBuffers = await pool.query(`
    SELECT 
      wo.id,
      wo.work_order_number,
      wo.sequence,
      wo.status,
      ro.code as operation_code,
      ro.name as operation_name,
      ro.routing_id,
      r.code as routing_code,
      ro.buffer_before_id,
      ro.buffer_after_id,
      lb_before.code as buffer_before_code,
      lb_before.name as buffer_before_name,
      lb_after.code as buffer_after_code,
      lb_after.name as buffer_after_name
    FROM production.production_work_orders wo
    LEFT JOIN production.production_routing_operations ro ON ro.id = wo.routing_operation_id
    LEFT JOIN production.production_routings r ON r.id = ro.routing_id
    LEFT JOIN production.production_locations lb_before ON lb_before.id = ro.buffer_before_id
    LEFT JOIN production.production_locations lb_after ON lb_after.id = ro.buffer_after_id
    WHERE wo.production_order_id = $1
    ORDER BY wo.sequence
  `, [orderId]);

  // Get all buffer location IDs from this order
  const bufferLocationIds = new Set<number>();
  workOrdersWithBuffers.rows.forEach(row => {
    if (row.buffer_before_id) bufferLocationIds.add(row.buffer_before_id);
    if (row.buffer_after_id) bufferLocationIds.add(row.buffer_after_id);
  });

  // Get carriers (pallets) that are currently at buffer locations or available
  let carriersResult: any = { rows: [] };
  if (bufferLocationIds.size > 0) {
    carriersResult = await pool.query(`
      SELECT 
        c.id,
        c.code,
        c.name,
        c.barcode,
        c.status,
        c.capacity,
        c.capacity_unit,
        c.current_load,
        c.current_location_id,
        l.name as location_name,
        l.code as location_code,
        cg.name as group_name,
        cg.code as group_code
      FROM production.production_carriers c
      LEFT JOIN production.production_locations l ON l.id = c.current_location_id
      LEFT JOIN production.production_carrier_groups cg ON cg.id = c.carrier_group_id
      WHERE c.is_active = true
        AND (c.current_location_id = ANY($1) OR c.status = 'available')
      ORDER BY c.code
    `, [Array.from(bufferLocationIds)]);
  } else {
    // Get all available carriers
    carriersResult = await pool.query(`
      SELECT 
        c.id,
        c.code,
        c.name,
        c.barcode,
        c.status,
        c.capacity,
        c.capacity_unit,
        c.current_load,
        c.current_location_id,
        l.name as location_name,
        l.code as location_code,
        cg.name as group_name,
        cg.code as group_code
      FROM production.production_carriers c
      LEFT JOIN production.production_locations l ON l.id = c.current_location_id
      LEFT JOIN production.production_carrier_groups cg ON cg.id = c.carrier_group_id
      WHERE c.is_active = true AND c.status = 'available'
      ORDER BY c.code
      LIMIT 50
    `);
  }

  // Get formatki count by flow code (derived from routing_variant)
  // Maps variant_code to flow codes: BOCZKI-TYL-SZUFLAD→SZUFLADY, CW→CW, CO→CO, COW→COW
  let formatkiByPath = { rows: [] as any[] };
  try {
    formatkiByPath = await pool.query(`
      SELECT 
        CASE 
          WHEN UPPER(prv.variant_code) LIKE '%SZUFLAD%' THEN 'SZUFLADY'
          WHEN UPPER(prv.variant_code) = 'COW' THEN 'COW'
          WHEN UPPER(prv.variant_code) = 'CO' THEN 'CO'
          WHEN UPPER(prv.variant_code) = 'CW' THEN 'CW'
          WHEN UPPER(prv.variant_code) = 'C' THEN 'C'
          ELSE COALESCE(UPPER(prv.variant_code), 'C')
        END as path_code,
        COUNT(*) as formatki_count
      FROM production.production_order_bom_items pobi
      JOIN production.production_order_boms pob ON pob.id = pobi.production_order_bom_id
      LEFT JOIN production.production_routing_variants prv ON prv.id = pobi.routing_variant_id
      WHERE pob.production_order_id = $1
      GROUP BY 
        CASE 
          WHEN UPPER(prv.variant_code) LIKE '%SZUFLAD%' THEN 'SZUFLADY'
          WHEN UPPER(prv.variant_code) = 'COW' THEN 'COW'
          WHEN UPPER(prv.variant_code) = 'CO' THEN 'CO'
          WHEN UPPER(prv.variant_code) = 'CW' THEN 'CW'
          WHEN UPPER(prv.variant_code) = 'C' THEN 'C'
          ELSE COALESCE(UPPER(prv.variant_code), 'C')
        END
    `, [orderId]);
  } catch (e) {
    console.warn('Could not fetch formatki count by path:', e);
  }

  // Group carriers by location for easy lookup
  const carriersByLocation: { [locationId: number]: any[] } = {};
  carriersResult.rows.forEach((row: any) => {
    if (row.current_location_id) {
      if (!carriersByLocation[row.current_location_id]) {
        carriersByLocation[row.current_location_id] = [];
      }
      carriersByLocation[row.current_location_id].push({
        id: row.id,
        code: row.code,
        name: row.name,
        barcode: row.barcode,
        status: row.status,
        capacity: row.capacity,
        capacityUnit: row.capacity_unit,
        currentLoad: row.current_load,
        groupName: row.group_name,
        groupCode: row.group_code,
      });
    }
  });

  return {
    buffers: workOrdersWithBuffers.rows.map(row => ({
      workOrderId: row.id,
      workOrderNumber: row.work_order_number,
      sequence: row.sequence,
      status: row.status,
      operationCode: row.operation_code,
      operationName: row.operation_name,
      routingId: row.routing_id,
      routingCode: row.routing_code,
      bufferBeforeId: row.buffer_before_id,
      bufferBeforeCode: row.buffer_before_code,
      bufferBeforeName: row.buffer_before_name,
      bufferAfterId: row.buffer_after_id,
      bufferAfterCode: row.buffer_after_code,
      bufferAfterName: row.buffer_after_name,
      carriersInBufferBefore: row.buffer_before_id ? (carriersByLocation[row.buffer_before_id] || []) : [],
      carriersInBufferAfter: row.buffer_after_id ? (carriersByLocation[row.buffer_after_id] || []) : [],
    })),
    carriers: carriersResult.rows.map((row: any) => ({
      id: row.id,
      code: row.code,
      name: row.name,
      barcode: row.barcode,
      status: row.status,
      capacity: row.capacity,
      capacityUnit: row.capacity_unit,
      currentLoad: row.current_load,
      currentLocationId: row.current_location_id,
      locationName: row.location_name,
      locationCode: row.location_code,
      groupName: row.group_name,
      groupCode: row.group_code,
    })),
    availableCarriers: carriersResult.rows
      .filter((row: any) => row.status === 'available' && !row.current_location_id)
      .map((row: any) => ({
        id: row.id,
        code: row.code,
        name: row.name,
        barcode: row.barcode,
        groupName: row.group_name,
        groupCode: row.group_code,
      })),
    formatkiByPath: formatkiByPath.rows.reduce((acc: any, row: any) => {
      acc[row.path_code] = parseInt(row.formatki_count);
      return acc;
    }, {}),
  };
}

/**
 * Get formatki for a specific routing path within a production order
 * Note: zlp_items table doesn't exist, using BOM items as alternative
 */
export async function getOrderPathFormatki(pool: Pool, orderId: number, pathCode: string): Promise<any[]> {
  try {
    // Use BOM items instead of zlp_items which doesn't exist
    const result = await pool.query(`
      SELECT 
        pobi.id,
        pobi.component_name as formatka_name,
        pobi.length as width,
        pobi.width as length,
        pobi.thickness,
        pobi.color_code,
        pobi.quantity
      FROM production.production_order_bom_items pobi
      JOIN production.production_order_boms pob ON pob.id = pobi.production_order_bom_id
      WHERE pob.production_order_id = $1
        AND pobi.component_type = 'panel'
      ORDER BY pobi.component_name, pobi.id
    `, [orderId]);

    return result.rows.map(row => ({
      id: row.id,
      formatkaId: null,
      formatkaName: row.formatka_name,
      formatkaCode: null,
      width: row.width,
      length: row.length,
      thickness: row.thickness,
      colorCode: row.color_code,
      boardType: null,
      edgingTop: null,
      edgingBottom: null,
      edgingLeft: null,
      edgingRight: null,
      quantity: row.quantity,
      itemStatus: 'pending',
      routingPath: pathCode,
    }));
  } catch (e) {
    console.warn('Could not fetch path formatki:', e);
    return [];
  }
}

/**
 * Create a new pallet for a production order (uses production_order_pallets table)
 */
export async function createOrderPallet(pool: Pool, orderId: number, data: {
  flowCode: string;
  colorCode?: string;
  operationCode?: string;
  operationId?: number;
  routingId?: number;
  carrierId?: number;
  notes?: string;
}): Promise<any> {
  // Get order info for generating label
  const orderResult = await pool.query(`
    SELECT order_number, color_code FROM production.production_orders WHERE id = $1
  `, [orderId]);
  
  const orderNumber = orderResult.rows[0]?.order_number || `ZLP-${orderId}`;
  const colorCode = data.colorCode || orderResult.rows[0]?.color_code || '';
  
  // Get next sequence number for this order
  const seqResult = await pool.query(`
    SELECT COALESCE(MAX(sequence), 0) + 1 as next_seq
    FROM production.production_order_pallets
    WHERE production_order_id = $1
  `, [orderId]);
  
  const sequence = parseInt(seqResult.rows[0].next_seq);
  
  // Generate pallet label: e.g., "BIALY-CO-1" or "ZLP-0041-BIALY-CO-1"
  const colorPrefix = colorCode.toUpperCase().replace(/\s+/g, '-') || 'PALLET';
  const palletLabel = `${colorPrefix}-${data.flowCode}-${sequence}`;
  
  const result = await pool.query(`
    INSERT INTO production.production_order_pallets 
    (production_order_id, pallet_label, sequence, color_code, flow_code, 
     routing_id, carrier_id, status, current_operation_id, current_operation_code, 
     max_capacity, current_load, notes, created_at)
    VALUES ($1, $2, $3, $4, $5, $6, $7, 'open', $8, $9, 100, 0, $10, NOW())
    RETURNING *
  `, [
    orderId, 
    palletLabel, 
    sequence, 
    colorCode, 
    data.flowCode,
    data.routingId || null,
    data.carrierId || null,
    data.operationId || null,
    data.operationCode || null,
    data.notes || null
  ]);
  
  const row = result.rows[0];
  console.log(`📦 Created pallet "${palletLabel}" for order ${orderNumber} (flow: ${data.flowCode})`);
  
  return {
    id: row.id,
    palletLabel: row.pallet_label,
    sequence: row.sequence,
    flowCode: row.flow_code,
    colorCode: row.color_code,
    status: row.status,
    currentOperationCode: row.current_operation_code,
    currentLoad: row.current_load,
    maxCapacity: row.max_capacity,
    carrierId: row.carrier_id,
    createdAt: row.created_at,
  };
}

/**
 * Auto-create pallets after cutting operation - one per unique flowCode in the order
 * Uses atomic sequence increment via row locking to prevent duplicates
 */
export async function autoCreatePalletsAfterCutting(client: any, orderId: number, operationCode: string, operationId?: number): Promise<any[]> {
  // Note: production_orders row is already locked by the caller (completeWorkOrder uses FOR UPDATE OF wo, po)
  // Lock existing pallets for this order to prevent race conditions
  await client.query(`
    SELECT id FROM production.production_order_pallets 
    WHERE production_order_id = $1 
    FOR UPDATE
  `, [orderId]);
  
  // Get base sequence once before creating any pallets - within locked scope
  const baseSeqResult = await client.query(`
    SELECT COALESCE(MAX(sequence), 0) as max_seq
    FROM production.production_order_pallets
    WHERE production_order_id = $1
  `, [orderId]);
  let nextSequence = parseInt(baseSeqResult.rows[0].max_seq) + 1;
  
  // Get order info including routing_id and color_code
  const orderResult = await client.query(`
    SELECT po.routing_id, po.color_code, r.code as routing_code
    FROM production.production_orders po
    LEFT JOIN production.production_routings r ON r.id = po.routing_id
    WHERE po.id = $1
  `, [orderId]);
  const orderRoutingId = orderResult.rows[0]?.routing_id || null;
  const orderColorCode = orderResult.rows[0]?.color_code || '';
  const orderRoutingCode = orderResult.rows[0]?.routing_code || 'DEFAULT';
  
  // Get unique color codes from BOM items (each color gets its own pallet)
  const flowsResult = await client.query(`
    SELECT DISTINCT 
      pobi.color_code
    FROM production.production_order_bom_items pobi
    JOIN production.production_order_boms pob ON pob.id = pobi.production_order_bom_id
    WHERE pob.production_order_id = $1 AND pobi.color_code IS NOT NULL AND pobi.color_code != ''
  `, [orderId]);
  
  const createdPallets: any[] = [];
  
  // If no color codes found in BOM, create a default pallet with order color
  if (flowsResult.rows.length === 0) {
    const flowCode = orderRoutingCode;
    const palletLabel = `${orderColorCode || 'PALLET'}-${flowCode}-${nextSequence}`;
    
    // Check if pallet already exists
    const existingResult = await client.query(`
      SELECT id FROM production.production_order_pallets
      WHERE production_order_id = $1 AND flow_code = $2 AND color_code = $3
      LIMIT 1
    `, [orderId, flowCode, orderColorCode]);
    
    if (existingResult.rows.length === 0) {
      const result = await client.query(`
        INSERT INTO production.production_order_pallets 
        (production_order_id, pallet_label, sequence, color_code, flow_code, 
         routing_id, status, current_operation_code, current_operation_id, max_capacity, current_load, created_at)
        VALUES ($1, $2, $3, $4, $5, $6, 'open', $7, $8, 100, 0, NOW())
        RETURNING *
      `, [orderId, palletLabel, nextSequence, orderColorCode, flowCode, orderRoutingId, operationCode, operationId || null]);
      
      createdPallets.push({
        id: result.rows[0].id,
        palletLabel: result.rows[0].pallet_label,
        sequence: result.rows[0].sequence,
        flowCode: result.rows[0].flow_code,
        colorCode: result.rows[0].color_code,
        status: result.rows[0].status,
        currentOperationCode: result.rows[0].current_operation_code,
      });
      console.log(`📦 Auto-created default pallet "${palletLabel}" after ${operationCode}`);
    }
  } else {
    // Create one pallet per unique color code (using order routing)
    for (const colorRow of flowsResult.rows) {
      const colorCode = colorRow.color_code;
      const flowCode = orderRoutingCode;
      
      // Check if pallet already exists for this color in this order
      const existingResult = await client.query(`
        SELECT id FROM production.production_order_pallets
        WHERE production_order_id = $1 AND flow_code = $2 AND color_code = $3
        LIMIT 1
      `, [orderId, flowCode, colorCode]);
      
      if (existingResult.rows.length === 0) {
        const palletLabel = `${colorCode}-${flowCode}-${nextSequence}`;
        
        const result = await client.query(`
          INSERT INTO production.production_order_pallets 
          (production_order_id, pallet_label, sequence, color_code, flow_code, 
           routing_id, status, current_operation_code, current_operation_id, max_capacity, current_load, created_at)
          VALUES ($1, $2, $3, $4, $5, $6, 'open', $7, $8, 100, 0, NOW())
          RETURNING *
        `, [orderId, palletLabel, nextSequence, colorCode, flowCode, orderRoutingId, operationCode, operationId || null]);
        
        createdPallets.push({
          id: result.rows[0].id,
          palletLabel: result.rows[0].pallet_label,
          sequence: result.rows[0].sequence,
          flowCode: result.rows[0].flow_code,
          colorCode: result.rows[0].color_code,
          status: result.rows[0].status,
          currentOperationCode: result.rows[0].current_operation_code,
        });
        console.log(`📦 Auto-created pallet "${palletLabel}" for color ${colorCode} after ${operationCode}`);
        nextSequence++;
      }
    }
  }
  
  return createdPallets;
}

/**
 * Check if an operation is a cutting operation (uses routing_operation metadata or code heuristics)
 */
export function isCuttingOperation(operationCode: string, operationType?: string): boolean {
  // Check operation_type metadata first (preferred method)
  if (operationType && (operationType === 'cutting' || operationType === 'saw' || operationType === 'cut')) {
    return true;
  }
  // Fallback to code-based detection
  const code = operationCode.toLowerCase();
  return code.includes('cut') || code.includes('cieci') || code.includes('pila') || code === 'cutting';
}

/**
 * Check if an operation is an edging operation
 */
export function isEdgingOperation(operationCode: string, operationType?: string): boolean {
  if (operationType && (operationType === 'edging' || operationType === 'edge' || operationType === 'oklejanie')) {
    return true;
  }
  const code = operationCode.toLowerCase();
  return code.includes('oklej') || code.includes('edge') || code.includes('edging') || code.includes('obrzez');
}

/**
 * Check if an operation is a drilling operation
 */
export function isDrillingOperation(operationCode: string, operationType?: string): boolean {
  if (operationType && (operationType === 'drilling' || operationType === 'drill' || operationType === 'wiercenie')) {
    return true;
  }
  const code = operationCode.toLowerCase();
  return code.includes('wierc') || code.includes('drill') || code.includes('drilling') || code.includes('cnc');
}

/**
 * Check if an operation is an assembly operation
 */
export function isAssemblyOperation(operationCode: string): boolean {
  const code = operationCode.toLowerCase();
  return code.includes('assembly') || code.includes('kompletow');
}

/**
 * Check if an operation is a packing operation
 */
export function isPackingOperation(operationCode: string): boolean {
  const code = operationCode.toLowerCase();
  return code.includes('packing') || code.includes('pakow');
}

/**
 * Get plan-level assembly operations for grouping across all ZLP-colors
 * Returns work orders for assembly/packing operations from all ZLPs belonging to the same plan
 */
export interface PlanAssemblyWorkOrder {
  id: number;
  workOrderNumber: string;
  sequence: number;
  status: string;
  operationCode: string;
  operationName: string;
  zlpId: number;
  zlpOrderNumber: string;
  zlpColorCode: string;
  zlpStatus: string;
  quantityPlanned: number;
  quantityProduced: number;
  actualStartTime: string | null;
  actualEndTime: string | null;
}

export interface PlanAssemblyOperationGroup {
  operationCode: string;
  operationName: string;
  sequence: number;
  workOrders: PlanAssemblyWorkOrder[];
  allDone: boolean;
  anyInProgress: boolean;
  allPrerequisitesDone: boolean;
  totalQuantityPlanned: number;
  totalQuantityProduced: number;
}

export interface PlanZlpInfo {
  id: number;
  orderNumber: string;
  colorCode: string;
  status: string;
  quantityPlanned: number;
  allPreAssemblyDone: boolean;
}

export interface PlanAssemblyOperations {
  planId: number;
  planNumber: string;
  planName: string;
  zlps: PlanZlpInfo[];
  operationGroups: PlanAssemblyOperationGroup[];
  canStartAssembly: boolean;
  canStartPacking: boolean;
}

export async function getPlanAssemblyOperations(pool: Pool, planId: number): Promise<PlanAssemblyOperations | null> {
  // Get plan info
  const planResult = await pool.query(`
    SELECT id, plan_number, name
    FROM production.production_plans
    WHERE id = $1
  `, [planId]);
  
  if (planResult.rows.length === 0) {
    return null;
  }
  
  const plan = planResult.rows[0];
  const planNumber = `PLAN-${String(planId).padStart(4, '0')}`;
  
  // Get all ZLPs from this plan
  const zlpsResult = await pool.query(`
    SELECT 
      po.id,
      po.order_number,
      po.color_code,
      po.status,
      po.quantity_planned
    FROM production.production_orders po
    WHERE po.source_order_number = $1
    ORDER BY po.color_code
  `, [planNumber]);
  
  if (zlpsResult.rows.length === 0) {
    return null;
  }
  
  const zlpIds = zlpsResult.rows.map((z: any) => z.id);
  
  // Get all work orders for these ZLPs
  const workOrdersResult = await pool.query(`
    SELECT 
      wo.id,
      wo.work_order_number,
      wo.production_order_id as zlp_id,
      wo.sequence,
      wo.status,
      wo.quantity_planned,
      wo.quantity_produced,
      wo.actual_start_time,
      wo.actual_end_time,
      wo.routing_operation_id,
      ro.code as operation_code,
      ro.name as operation_name,
      po.order_number as zlp_order_number,
      po.color_code as zlp_color_code,
      po.status as zlp_status
    FROM production.production_work_orders wo
    LEFT JOIN production.production_routing_operations ro ON ro.id = wo.routing_operation_id
    JOIN production.production_orders po ON po.id = wo.production_order_id
    WHERE wo.production_order_id = ANY($1)
    ORDER BY po.color_code, wo.sequence
  `, [zlpIds]);
  
  // Build ZLP info with pre-assembly status
  const zlps: PlanZlpInfo[] = [];
  const assemblyOpCodes = ['assembly', 'kompletowanie'];
  const packingOpCodes = ['packing', 'pakowanie'];
  const mergeOpCodes = [...assemblyOpCodes, ...packingOpCodes];
  
  for (const zlpRow of zlpsResult.rows) {
    const zlpWorkOrders = workOrdersResult.rows.filter((wo: any) => wo.zlp_id === zlpRow.id);
    
    // Find assembly work order sequence
    const assemblyWo = zlpWorkOrders.find((wo: any) => 
      wo.operation_code && assemblyOpCodes.includes(wo.operation_code.toLowerCase())
    );
    
    // Check if all work orders BEFORE assembly are done
    let allPreAssemblyDone = true;
    if (assemblyWo) {
      const preAssemblyWos = zlpWorkOrders.filter((wo: any) => wo.sequence < assemblyWo.sequence);
      allPreAssemblyDone = preAssemblyWos.every((wo: any) => wo.status === 'done');
    }
    
    zlps.push({
      id: zlpRow.id,
      orderNumber: zlpRow.order_number,
      colorCode: zlpRow.color_code,
      status: zlpRow.status,
      quantityPlanned: parseFloat(zlpRow.quantity_planned) || 0,
      allPreAssemblyDone,
    });
  }
  
  // Group assembly/packing work orders by operation code
  const operationGroups: PlanAssemblyOperationGroup[] = [];
  const assemblyPackingWos = workOrdersResult.rows.filter((wo: any) => 
    wo.operation_code && mergeOpCodes.includes(wo.operation_code.toLowerCase())
  );
  
  // Group by operation code
  const groupedByOp: { [opCode: string]: any[] } = {};
  for (const wo of assemblyPackingWos) {
    const opCode = wo.operation_code.toLowerCase();
    if (!groupedByOp[opCode]) {
      groupedByOp[opCode] = [];
    }
    groupedByOp[opCode].push(wo);
  }
  
  // Sort operation codes (assembly before packing)
  const sortedOpCodes = Object.keys(groupedByOp).sort((a, b) => {
    if (assemblyOpCodes.includes(a)) return -1;
    if (assemblyOpCodes.includes(b)) return 1;
    return 0;
  });
  
  for (const opCode of sortedOpCodes) {
    const wos = groupedByOp[opCode];
    const firstWo = wos[0];
    
    // Check prerequisites based on operation type
    let allPrerequisitesDone = true;
    if (assemblyOpCodes.includes(opCode)) {
      // For assembly: all pre-assembly operations must be done for all ZLPs
      allPrerequisitesDone = zlps.every(z => z.allPreAssemblyDone);
    } else if (packingOpCodes.includes(opCode)) {
      // For packing: all assembly operations must be done
      const assemblyGroup = operationGroups.find(g => 
        assemblyOpCodes.includes(g.operationCode.toLowerCase())
      );
      allPrerequisitesDone = assemblyGroup ? assemblyGroup.allDone : false;
    }
    
    const workOrders: PlanAssemblyWorkOrder[] = wos.map((wo: any) => ({
      id: wo.id,
      workOrderNumber: wo.work_order_number,
      sequence: wo.sequence,
      status: wo.status,
      operationCode: wo.operation_code,
      operationName: wo.operation_name,
      zlpId: wo.zlp_id,
      zlpOrderNumber: wo.zlp_order_number,
      zlpColorCode: wo.zlp_color_code,
      zlpStatus: wo.zlp_status,
      quantityPlanned: parseFloat(wo.quantity_planned) || 0,
      quantityProduced: parseFloat(wo.quantity_produced) || 0,
      actualStartTime: wo.actual_start_time,
      actualEndTime: wo.actual_end_time,
    }));
    
    operationGroups.push({
      operationCode: firstWo.operation_code,
      operationName: firstWo.operation_name,
      sequence: Math.min(...wos.map((wo: any) => wo.sequence)),
      workOrders,
      allDone: workOrders.every(wo => wo.status === 'done'),
      anyInProgress: workOrders.some(wo => wo.status === 'in_progress'),
      allPrerequisitesDone,
      totalQuantityPlanned: workOrders.reduce((sum, wo) => sum + wo.quantityPlanned, 0),
      totalQuantityProduced: workOrders.reduce((sum, wo) => sum + wo.quantityProduced, 0),
    });
  }
  
  // Determine if assembly/packing can start
  const assemblyGroup = operationGroups.find(g => 
    assemblyOpCodes.includes(g.operationCode.toLowerCase())
  );
  const packingGroup = operationGroups.find(g => 
    packingOpCodes.includes(g.operationCode.toLowerCase())
  );
  
  const canStartAssembly = assemblyGroup 
    ? assemblyGroup.allPrerequisitesDone && !assemblyGroup.allDone 
    : false;
  const canStartPacking = packingGroup 
    ? packingGroup.allPrerequisitesDone && !packingGroup.allDone 
    : false;
  
  return {
    planId,
    planNumber: plan.plan_number,
    planName: plan.name,
    zlps,
    operationGroups,
    canStartAssembly,
    canStartPacking,
  };
}

/**
 * Batch start all pending work orders in a plan-level operation group
 */
export async function batchStartPlanAssemblyOperation(
  pool: Pool, 
  planId: number, 
  operationCode: string
): Promise<{ startedCount: number; workOrderIds: number[] }> {
  const client = await pool.connect();
  
  try {
    await client.query('BEGIN');
    
    const planNumber = `PLAN-${String(planId).padStart(4, '0')}`;
    
    // Get all pending work orders for this operation across all ZLPs in the plan
    const result = await client.query(`
      SELECT wo.id, wo.production_order_id
      FROM production.production_work_orders wo
      JOIN production.production_orders po ON po.id = wo.production_order_id
      JOIN production.production_routing_operations ro ON ro.id = wo.routing_operation_id
      WHERE po.source_order_number = $1
        AND LOWER(ro.code) = LOWER($2)
        AND wo.status IN ('pending', 'ready')
      FOR UPDATE OF wo
    `, [planNumber, operationCode]);
    
    const workOrderIds = result.rows.map((r: any) => r.id);
    
    if (workOrderIds.length === 0) {
      await client.query('ROLLBACK');
      return { startedCount: 0, workOrderIds: [] };
    }
    
    // Start all work orders
    await client.query(`
      UPDATE production.production_work_orders
      SET 
        status = 'in_progress',
        actual_start_time = NOW(),
        updated_at = NOW()
      WHERE id = ANY($1)
    `, [workOrderIds]);
    
    // Update ZLP statuses to in_progress if not already
    const zlpIds = Array.from(new Set(result.rows.map((r: any) => r.production_order_id)));
    await client.query(`
      UPDATE production.production_orders
      SET 
        status = 'in_progress',
        actual_start_date = COALESCE(actual_start_date, NOW()),
        updated_at = NOW()
      WHERE id = ANY($1) AND status NOT IN ('in_progress', 'done')
    `, [zlpIds]);
    
    // Log the batch start
    for (const zlpId of zlpIds) {
      await client.query(`
        INSERT INTO production.production_order_logs 
        (production_order_id, event_type, event_data, message, created_at)
        VALUES ($1, 'plan_batch_start', $2, $3, NOW())
      `, [
        zlpId,
        JSON.stringify({ planId, operationCode, workOrderIds }),
        `Plan-level batch start for ${operationCode}`
      ]);
    }
    
    await client.query('COMMIT');
    
    return { startedCount: workOrderIds.length, workOrderIds };
  } catch (error) {
    await client.query('ROLLBACK');
    throw error;
  } finally {
    client.release();
  }
}

/**
 * Batch complete all in-progress work orders in a plan-level operation group
 */
export async function batchCompletePlanAssemblyOperation(
  pool: Pool, 
  planId: number, 
  operationCode: string
): Promise<{ completedCount: number; workOrderIds: number[] }> {
  const client = await pool.connect();
  
  try {
    await client.query('BEGIN');
    
    const planNumber = `PLAN-${String(planId).padStart(4, '0')}`;
    
    // Get all in-progress work orders for this operation
    const result = await client.query(`
      SELECT wo.id, wo.production_order_id, wo.quantity_planned
      FROM production.production_work_orders wo
      JOIN production.production_orders po ON po.id = wo.production_order_id
      JOIN production.production_routing_operations ro ON ro.id = wo.routing_operation_id
      WHERE po.source_order_number = $1
        AND LOWER(ro.code) = LOWER($2)
        AND wo.status = 'in_progress'
      FOR UPDATE OF wo
    `, [planNumber, operationCode]);
    
    const workOrderIds = result.rows.map((r: any) => r.id);
    
    if (workOrderIds.length === 0) {
      await client.query('ROLLBACK');
      return { completedCount: 0, workOrderIds: [] };
    }
    
    // Complete all work orders
    await client.query(`
      UPDATE production.production_work_orders
      SET 
        status = 'done',
        actual_end_time = NOW(),
        quantity_produced = quantity_planned,
        updated_at = NOW()
      WHERE id = ANY($1)
    `, [workOrderIds]);
    
    // Check if ZLPs should be marked complete
    const zlpIds = Array.from(new Set(result.rows.map((r: any) => r.production_order_id)));
    for (const zlpId of zlpIds) {
      const remainingResult = await client.query(`
        SELECT COUNT(*) as remaining
        FROM production.production_work_orders
        WHERE production_order_id = $1 AND status NOT IN ('done', 'cancelled')
      `, [zlpId]);
      
      if (parseInt(remainingResult.rows[0].remaining) === 0) {
        await client.query(`
          UPDATE production.production_orders
          SET 
            status = 'done',
            actual_end_date = NOW(),
            quantity_produced = quantity_planned,
            updated_at = NOW()
          WHERE id = $1
        `, [zlpId]);
      }
    }
    
    // Log the batch complete
    for (const zlpId of zlpIds) {
      await client.query(`
        INSERT INTO production.production_order_logs 
        (production_order_id, event_type, event_data, message, created_at)
        VALUES ($1, 'plan_batch_complete', $2, $3, NOW())
      `, [
        zlpId,
        JSON.stringify({ planId, operationCode, workOrderIds }),
        `Plan-level batch complete for ${operationCode}`
      ]);
    }
    
    await client.query('COMMIT');
    
    return { completedCount: workOrderIds.length, workOrderIds };
  } catch (error) {
    await client.query('ROLLBACK');
    throw error;
  } finally {
    client.release();
  }
}
