import { Pool } from "pg";
import type { 
  ProductionNormGroup,
  InsertProductionNormGroup,
  ProductionNorm,
  InsertProductionNorm,
  ProductionMaterialConsumptionNorm,
  InsertProductionMaterialConsumptionNorm,
} from "@shared/schema";

// ===== Norm Groups =====

export async function getProductionNormGroups(pool: Pool): Promise<ProductionNormGroup[]> {
  const result = await pool.query(`
    SELECT * FROM production.production_norm_groups
    WHERE is_active = true
    ORDER BY sort_order ASC, name ASC
  `);
  
  return result.rows.map(row => ({
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    unitOfMeasure: row.unit_of_measure,
    isActive: row.is_active,
    sortOrder: row.sort_order,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

export async function getProductionNormGroupById(pool: Pool, id: number): Promise<ProductionNormGroup | null> {
  const result = await pool.query(`
    SELECT * FROM production.production_norm_groups
    WHERE id = $1
  `, [id]);
  
  if (result.rows.length === 0) return null;
  
  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    unitOfMeasure: row.unit_of_measure,
    isActive: row.is_active,
    sortOrder: row.sort_order,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function createProductionNormGroup(pool: Pool, group: InsertProductionNormGroup): Promise<ProductionNormGroup> {
  const result = await pool.query(`
    INSERT INTO production.production_norm_groups 
      (code, name, description, unit_of_measure, is_active, sort_order)
    VALUES ($1, $2, $3, $4, $5, $6)
    RETURNING *
  `, [
    group.code,
    group.name,
    group.description ?? null,
    group.unitOfMeasure ?? null,
    group.isActive ?? true,
    group.sortOrder ?? 0,
  ]);
  
  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    unitOfMeasure: row.unit_of_measure,
    isActive: row.is_active,
    sortOrder: row.sort_order,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function updateProductionNormGroup(
  pool: Pool, 
  id: number, 
  updates: Partial<InsertProductionNormGroup>
): Promise<ProductionNormGroup> {
  const setParts: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;
  
  if (updates.code !== undefined) {
    setParts.push(`code = $${paramIndex++}`);
    values.push(updates.code);
  }
  if (updates.name !== undefined) {
    setParts.push(`name = $${paramIndex++}`);
    values.push(updates.name);
  }
  if (updates.description !== undefined) {
    setParts.push(`description = $${paramIndex++}`);
    values.push(updates.description);
  }
  if (updates.unitOfMeasure !== undefined) {
    setParts.push(`unit_of_measure = $${paramIndex++}`);
    values.push(updates.unitOfMeasure);
  }
  if (updates.isActive !== undefined) {
    setParts.push(`is_active = $${paramIndex++}`);
    values.push(updates.isActive);
  }
  if (updates.sortOrder !== undefined) {
    setParts.push(`sort_order = $${paramIndex++}`);
    values.push(updates.sortOrder);
  }
  
  setParts.push(`updated_at = NOW()`);
  values.push(id);
  
  const result = await pool.query(`
    UPDATE production.production_norm_groups
    SET ${setParts.join(', ')}
    WHERE id = $${paramIndex}
    RETURNING *
  `, values);
  
  if (result.rows.length === 0) {
    throw new Error(`Production norm group with id ${id} not found`);
  }
  
  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    unitOfMeasure: row.unit_of_measure,
    isActive: row.is_active,
    sortOrder: row.sort_order,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function deleteProductionNormGroup(pool: Pool, id: number): Promise<void> {
  await pool.query(`
    DELETE FROM production.production_norm_groups
    WHERE id = $1
  `, [id]);
}

// ===== Norms =====

export async function getProductionNorms(pool: Pool, groupId?: number): Promise<ProductionNorm[]> {
  let query = `
    SELECT * FROM production.production_norms
    WHERE is_active = true
  `;
  const params: any[] = [];
  
  if (groupId !== undefined) {
    query += ' AND norm_group_id = $1';
    params.push(groupId);
  }
  
  query += ' ORDER BY norm_group_id, code ASC';
  
  const result = await pool.query(query, params);
  
  return result.rows.map(row => ({
    id: row.id,
    normGroupId: row.norm_group_id,
    code: row.code,
    name: row.name,
    description: row.description,
    defaultValue: row.default_value,
    minValue: row.min_value,
    maxValue: row.max_value,
    tolerance: row.tolerance,
    unitOfMeasure: row.unit_of_measure,
    applicableTo: row.applicable_to,
    isActive: row.is_active,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

export async function getProductionNormById(pool: Pool, id: number): Promise<ProductionNorm | null> {
  const result = await pool.query(`
    SELECT * FROM production.production_norms
    WHERE id = $1
  `, [id]);
  
  if (result.rows.length === 0) return null;
  
  const row = result.rows[0];
  return {
    id: row.id,
    normGroupId: row.norm_group_id,
    code: row.code,
    name: row.name,
    description: row.description,
    defaultValue: row.default_value,
    minValue: row.min_value,
    maxValue: row.max_value,
    tolerance: row.tolerance,
    unitOfMeasure: row.unit_of_measure,
    applicableTo: row.applicable_to,
    isActive: row.is_active,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function createProductionNorm(pool: Pool, norm: InsertProductionNorm): Promise<ProductionNorm> {
  const result = await pool.query(`
    INSERT INTO production.production_norms 
      (norm_group_id, code, name, description, default_value, min_value, max_value, 
       tolerance, unit_of_measure, applicable_to, is_active)
    VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
    RETURNING *
  `, [
    norm.normGroupId,
    norm.code,
    norm.name,
    norm.description ?? null,
    norm.defaultValue,
    norm.minValue ?? null,
    norm.maxValue ?? null,
    norm.tolerance ?? null,
    norm.unitOfMeasure ?? null,
    norm.applicableTo ? JSON.stringify(norm.applicableTo) : null,
    norm.isActive ?? true,
  ]);
  
  const row = result.rows[0];
  return {
    id: row.id,
    normGroupId: row.norm_group_id,
    code: row.code,
    name: row.name,
    description: row.description,
    defaultValue: row.default_value,
    minValue: row.min_value,
    maxValue: row.max_value,
    tolerance: row.tolerance,
    unitOfMeasure: row.unit_of_measure,
    applicableTo: row.applicable_to,
    isActive: row.is_active,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function updateProductionNorm(
  pool: Pool, 
  id: number, 
  updates: Partial<InsertProductionNorm>
): Promise<ProductionNorm> {
  const setParts: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;
  
  if (updates.normGroupId !== undefined) {
    setParts.push(`norm_group_id = $${paramIndex++}`);
    values.push(updates.normGroupId);
  }
  if (updates.code !== undefined) {
    setParts.push(`code = $${paramIndex++}`);
    values.push(updates.code);
  }
  if (updates.name !== undefined) {
    setParts.push(`name = $${paramIndex++}`);
    values.push(updates.name);
  }
  if (updates.description !== undefined) {
    setParts.push(`description = $${paramIndex++}`);
    values.push(updates.description);
  }
  if (updates.defaultValue !== undefined) {
    setParts.push(`default_value = $${paramIndex++}`);
    values.push(updates.defaultValue);
  }
  if (updates.minValue !== undefined) {
    setParts.push(`min_value = $${paramIndex++}`);
    values.push(updates.minValue);
  }
  if (updates.maxValue !== undefined) {
    setParts.push(`max_value = $${paramIndex++}`);
    values.push(updates.maxValue);
  }
  if (updates.tolerance !== undefined) {
    setParts.push(`tolerance = $${paramIndex++}`);
    values.push(updates.tolerance);
  }
  if (updates.unitOfMeasure !== undefined) {
    setParts.push(`unit_of_measure = $${paramIndex++}`);
    values.push(updates.unitOfMeasure);
  }
  if (updates.applicableTo !== undefined) {
    setParts.push(`applicable_to = $${paramIndex++}`);
    values.push(updates.applicableTo ? JSON.stringify(updates.applicableTo) : null);
  }
  if (updates.isActive !== undefined) {
    setParts.push(`is_active = $${paramIndex++}`);
    values.push(updates.isActive);
  }
  
  setParts.push(`updated_at = NOW()`);
  values.push(id);
  
  const result = await pool.query(`
    UPDATE production.production_norms
    SET ${setParts.join(', ')}
    WHERE id = $${paramIndex}
    RETURNING *
  `, values);
  
  if (result.rows.length === 0) {
    throw new Error(`Production norm with id ${id} not found`);
  }
  
  const row = result.rows[0];
  return {
    id: row.id,
    normGroupId: row.norm_group_id,
    code: row.code,
    name: row.name,
    description: row.description,
    defaultValue: row.default_value,
    minValue: row.min_value,
    maxValue: row.max_value,
    tolerance: row.tolerance,
    unitOfMeasure: row.unit_of_measure,
    applicableTo: row.applicable_to,
    isActive: row.is_active,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function deleteProductionNorm(pool: Pool, id: number): Promise<void> {
  await pool.query(`
    DELETE FROM production.production_norms
    WHERE id = $1
  `, [id]);
}

// ===== Material Consumption Norms =====

export async function getMaterialConsumptionNorms(pool: Pool): Promise<ProductionMaterialConsumptionNorm[]> {
  const result = await pool.query(`
    SELECT * FROM production.production_material_consumption_norms
    WHERE is_active = true
    ORDER BY material_type ASC, material_category ASC
  `);
  
  return result.rows.map(row => ({
    id: row.id,
    materialType: row.material_type,
    materialCategory: row.material_category,
    wastePercentage: row.waste_percentage,
    edgeOverageCm: row.edge_overage_cm,
    consumptionPerUnit: row.consumption_per_unit,
    inputUnit: row.input_unit,
    outputUnit: row.output_unit,
    isActive: row.is_active,
    description: row.description,
    notes: row.notes,
    metadata: row.metadata,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

export async function getMaterialConsumptionNormById(pool: Pool, id: number): Promise<ProductionMaterialConsumptionNorm | null> {
  const result = await pool.query(`
    SELECT * FROM production.production_material_consumption_norms
    WHERE id = $1
  `, [id]);
  
  if (result.rows.length === 0) return null;
  
  const row = result.rows[0];
  return {
    id: row.id,
    materialType: row.material_type,
    materialCategory: row.material_category,
    wastePercentage: row.waste_percentage,
    edgeOverageCm: row.edge_overage_cm,
    consumptionPerUnit: row.consumption_per_unit,
    inputUnit: row.input_unit,
    outputUnit: row.output_unit,
    isActive: row.is_active,
    description: row.description,
    notes: row.notes,
    metadata: row.metadata,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function createMaterialConsumptionNorm(pool: Pool, norm: InsertProductionMaterialConsumptionNorm): Promise<ProductionMaterialConsumptionNorm> {
  const result = await pool.query(`
    INSERT INTO production.production_material_consumption_norms 
      (material_type, material_category, waste_percentage, edge_overage_cm, consumption_per_unit, 
       input_unit, output_unit, is_active, description, notes, metadata)
    VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
    RETURNING *
  `, [
    norm.materialType,
    norm.materialCategory ?? null,
    norm.wastePercentage ?? null,
    norm.edgeOverageCm ?? null,
    norm.consumptionPerUnit ?? null,
    norm.inputUnit,
    norm.outputUnit,
    norm.isActive ?? true,
    norm.description ?? null,
    norm.notes ?? null,
    norm.metadata ? JSON.stringify(norm.metadata) : null,
  ]);
  
  const row = result.rows[0];
  return {
    id: row.id,
    materialType: row.material_type,
    materialCategory: row.material_category,
    wastePercentage: row.waste_percentage,
    edgeOverageCm: row.edge_overage_cm,
    consumptionPerUnit: row.consumption_per_unit,
    inputUnit: row.input_unit,
    outputUnit: row.output_unit,
    isActive: row.is_active,
    description: row.description,
    notes: row.notes,
    metadata: row.metadata,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function updateMaterialConsumptionNorm(
  pool: Pool, 
  id: number, 
  updates: Partial<InsertProductionMaterialConsumptionNorm>
): Promise<ProductionMaterialConsumptionNorm> {
  const setParts: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;
  
  if (updates.materialType !== undefined) {
    setParts.push(`material_type = $${paramIndex++}`);
    values.push(updates.materialType);
  }
  if (updates.materialCategory !== undefined) {
    setParts.push(`material_category = $${paramIndex++}`);
    values.push(updates.materialCategory);
  }
  if (updates.wastePercentage !== undefined) {
    setParts.push(`waste_percentage = $${paramIndex++}`);
    values.push(updates.wastePercentage);
  }
  if (updates.edgeOverageCm !== undefined) {
    setParts.push(`edge_overage_cm = $${paramIndex++}`);
    values.push(updates.edgeOverageCm);
  }
  if (updates.consumptionPerUnit !== undefined) {
    setParts.push(`consumption_per_unit = $${paramIndex++}`);
    values.push(updates.consumptionPerUnit);
  }
  if (updates.inputUnit !== undefined) {
    setParts.push(`input_unit = $${paramIndex++}`);
    values.push(updates.inputUnit);
  }
  if (updates.outputUnit !== undefined) {
    setParts.push(`output_unit = $${paramIndex++}`);
    values.push(updates.outputUnit);
  }
  if (updates.isActive !== undefined) {
    setParts.push(`is_active = $${paramIndex++}`);
    values.push(updates.isActive);
  }
  if (updates.description !== undefined) {
    setParts.push(`description = $${paramIndex++}`);
    values.push(updates.description);
  }
  if (updates.notes !== undefined) {
    setParts.push(`notes = $${paramIndex++}`);
    values.push(updates.notes);
  }
  if (updates.metadata !== undefined) {
    setParts.push(`metadata = $${paramIndex++}`);
    values.push(updates.metadata ? JSON.stringify(updates.metadata) : null);
  }
  
  setParts.push(`updated_at = NOW()`);
  values.push(id);
  
  const result = await pool.query(`
    UPDATE production.production_material_consumption_norms
    SET ${setParts.join(', ')}
    WHERE id = $${paramIndex}
    RETURNING *
  `, values);
  
  if (result.rows.length === 0) {
    throw new Error(`Material consumption norm with id ${id} not found`);
  }
  
  const row = result.rows[0];
  return {
    id: row.id,
    materialType: row.material_type,
    materialCategory: row.material_category,
    wastePercentage: row.waste_percentage,
    edgeOverageCm: row.edge_overage_cm,
    consumptionPerUnit: row.consumption_per_unit,
    inputUnit: row.input_unit,
    outputUnit: row.output_unit,
    isActive: row.is_active,
    description: row.description,
    notes: row.notes,
    metadata: row.metadata,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function deleteMaterialConsumptionNorm(pool: Pool, id: number): Promise<void> {
  await pool.query(`
    DELETE FROM production.production_material_consumption_norms
    WHERE id = $1
  `, [id]);
}
