import { Pool, PoolClient } from "pg";
import type { ProductionBufferReservation, InsertProductionBufferReservation } from "@shared/schema";
import { createBufferMovement } from "./buffer-movements";
import { getOrCreateBufferStock } from "./buffer-stock";

type PoolOrClient = Pool | PoolClient;

export async function getReservationsByZlp(pool: Pool, zlpId: number): Promise<ProductionBufferReservation[]> {
  const result = await pool.query(`
    SELECT 
      r.*,
      l.name as location_name,
      l.code as location_code
    FROM production.production_buffer_reservations r
    LEFT JOIN production.production_locations l ON r.location_id = l.id
    WHERE r.zlp_id = $1
    ORDER BY r.reserved_at DESC
  `, [zlpId]);
  
  return result.rows.map(row => ({
    id: row.id,
    zlpId: row.zlp_id,
    zlpItemId: row.zlp_item_id,
    productSku: row.product_sku,
    quantityReserved: row.quantity_reserved,
    quantityConsumed: row.quantity_consumed,
    unitOfMeasure: row.unit_of_measure,
    locationId: row.location_id,
    status: row.status,
    reservedAt: row.reserved_at,
    reservedBy: row.reserved_by,
    consumedAt: row.consumed_at,
    cancelledAt: row.cancelled_at,
    notes: row.notes,
  }));
}

export async function getReservationsBySku(pool: Pool, productSku: string): Promise<ProductionBufferReservation[]> {
  const result = await pool.query(`
    SELECT * FROM production.production_buffer_reservations 
    WHERE product_sku = $1 AND status = 'ACTIVE'
    ORDER BY reserved_at DESC
  `, [productSku]);
  
  return result.rows.map(row => ({
    id: row.id,
    zlpId: row.zlp_id,
    zlpItemId: row.zlp_item_id,
    productSku: row.product_sku,
    quantityReserved: row.quantity_reserved,
    quantityConsumed: row.quantity_consumed,
    unitOfMeasure: row.unit_of_measure,
    locationId: row.location_id,
    status: row.status,
    reservedAt: row.reserved_at,
    reservedBy: row.reserved_by,
    consumedAt: row.consumed_at,
    cancelledAt: row.cancelled_at,
    notes: row.notes,
  }));
}

export async function createReservation(
  pool: PoolOrClient,
  data: InsertProductionBufferReservation
): Promise<ProductionBufferReservation> {
  const isClient = 'release' in pool && typeof pool.release === 'function';
  const client = isClient ? (pool as PoolClient) : await (pool as Pool).connect();
  try {
    if (!isClient) await client.query('BEGIN');
    
    await getOrCreateBufferStock(client, data.productSku, data.locationId || null);
    
    const result = await client.query(`
      INSERT INTO production.production_buffer_reservations 
      (zlp_id, zlp_item_id, product_sku, quantity_reserved, unit_of_measure, 
       location_id, status, reserved_by, notes)
      VALUES ($1, $2, $3, $4, $5, $6, 'ACTIVE', $7, $8)
      RETURNING *
    `, [
      data.zlpId,
      data.zlpItemId || null,
      data.productSku,
      data.quantityReserved,
      data.unitOfMeasure || 'szt',
      data.locationId || null,
      data.reservedBy || null,
      data.notes || null,
    ]);
    
    const reservation = result.rows[0];
    
    await createBufferMovement(client, {
      movementType: 'RESERVE',
      productSku: data.productSku,
      quantity: data.quantityReserved,
      unitOfMeasure: data.unitOfMeasure || 'szt',
      locationId: data.locationId || null,
      sourceType: 'ZLP',
      sourceId: data.zlpId,
      zlpId: data.zlpId,
      reservationId: reservation.id,
      createdBy: data.reservedBy || null,
      notes: `Rezerwacja dla ZLP #${data.zlpId}`,
    });
    
    if (!isClient) await client.query('COMMIT');
    
    return {
      id: reservation.id,
      zlpId: reservation.zlp_id,
      zlpItemId: reservation.zlp_item_id,
      productSku: reservation.product_sku,
      quantityReserved: reservation.quantity_reserved,
      quantityConsumed: reservation.quantity_consumed,
      unitOfMeasure: reservation.unit_of_measure,
      locationId: reservation.location_id,
      status: reservation.status,
      reservedAt: reservation.reserved_at,
      reservedBy: reservation.reserved_by,
      consumedAt: reservation.consumed_at,
      cancelledAt: reservation.cancelled_at,
      notes: reservation.notes,
    };
  } catch (error) {
    if (!isClient) await client.query('ROLLBACK');
    throw error;
  } finally {
    if (!isClient) client.release();
  }
}

export async function consumeReservation(
  pool: PoolOrClient,
  reservationId: number,
  quantityConsumed: number,
  userId?: number
): Promise<ProductionBufferReservation> {
  const isClient = 'release' in pool && typeof pool.release === 'function';
  const client = isClient ? (pool as PoolClient) : await (pool as Pool).connect();
  try {
    if (!isClient) await client.query('BEGIN');
    
    const reservation = await client.query(`
      SELECT * FROM production.production_buffer_reservations WHERE id = $1
    `, [reservationId]);
    
    if (reservation.rows.length === 0) {
      throw new Error('Reservation not found');
    }
    
    const res = reservation.rows[0];
    const newConsumed = Number(res.quantity_consumed) + quantityConsumed;
    const isFullyConsumed = newConsumed >= Number(res.quantity_reserved);
    
    const newStatus = isFullyConsumed ? 'CONSUMED' : 'ACTIVE';
    
    const result = await client.query(`
      UPDATE production.production_buffer_reservations 
      SET 
        quantity_consumed = $2,
        status = $3::varchar,
        consumed_at = CASE WHEN $3 = 'CONSUMED' THEN NOW() ELSE consumed_at END
      WHERE id = $1
      RETURNING *
    `, [reservationId, newConsumed, newStatus]);
    
    await createBufferMovement(client, {
      movementType: 'OUT',
      productSku: res.product_sku,
      quantity: quantityConsumed.toString(),
      unitOfMeasure: res.unit_of_measure,
      locationId: res.location_id,
      sourceType: 'ZLP',
      sourceId: res.zlp_id,
      zlpId: res.zlp_id,
      reservationId: reservationId,
      createdBy: userId || null,
      notes: `Zużycie rezerwacji #${reservationId}`,
    });
    
    if (!isClient) await client.query('COMMIT');
    
    const row = result.rows[0];
    return {
      id: row.id,
      zlpId: row.zlp_id,
      zlpItemId: row.zlp_item_id,
      productSku: row.product_sku,
      quantityReserved: row.quantity_reserved,
      quantityConsumed: row.quantity_consumed,
      unitOfMeasure: row.unit_of_measure,
      locationId: row.location_id,
      status: row.status,
      reservedAt: row.reserved_at,
      reservedBy: row.reserved_by,
      consumedAt: row.consumed_at,
      cancelledAt: row.cancelled_at,
      notes: row.notes,
    };
  } catch (error) {
    if (!isClient) await client.query('ROLLBACK');
    throw error;
  } finally {
    if (!isClient) client.release();
  }
}

export async function cancelReservation(
  pool: PoolOrClient,
  reservationId: number,
  userId?: number
): Promise<ProductionBufferReservation> {
  const isClient = 'release' in pool && typeof pool.release === 'function';
  const client = isClient ? (pool as PoolClient) : await (pool as Pool).connect();
  try {
    if (!isClient) await client.query('BEGIN');
    
    const reservation = await client.query(`
      SELECT * FROM production.production_buffer_reservations WHERE id = $1
    `, [reservationId]);
    
    if (reservation.rows.length === 0) {
      throw new Error('Reservation not found');
    }
    
    const res = reservation.rows[0];
    const unconsumedQty = Number(res.quantity_reserved) - Number(res.quantity_consumed);
    
    const result = await client.query(`
      UPDATE production.production_buffer_reservations 
      SET 
        status = 'CANCELLED',
        cancelled_at = NOW()
      WHERE id = $1
      RETURNING *
    `, [reservationId]);
    
    if (unconsumedQty > 0) {
      await createBufferMovement(client, {
        movementType: 'RELEASE',
        productSku: res.product_sku,
        quantity: unconsumedQty.toString(),
        unitOfMeasure: res.unit_of_measure,
        locationId: res.location_id,
        sourceType: 'ZLP',
        sourceId: res.zlp_id,
        zlpId: res.zlp_id,
        reservationId: reservationId,
        createdBy: userId || null,
        notes: `Anulowanie rezerwacji #${reservationId}`,
      });
    }
    
    if (!isClient) await client.query('COMMIT');
    
    const row = result.rows[0];
    return {
      id: row.id,
      zlpId: row.zlp_id,
      zlpItemId: row.zlp_item_id,
      productSku: row.product_sku,
      quantityReserved: row.quantity_reserved,
      quantityConsumed: row.quantity_consumed,
      unitOfMeasure: row.unit_of_measure,
      locationId: row.location_id,
      status: row.status,
      reservedAt: row.reserved_at,
      reservedBy: row.reserved_by,
      consumedAt: row.consumed_at,
      cancelledAt: row.cancelled_at,
      notes: row.notes,
    };
  } catch (error) {
    if (!isClient) await client.query('ROLLBACK');
    throw error;
  } finally {
    if (!isClient) client.release();
  }
}
