import { Pool, PoolClient } from "pg";
import type { ProductionBufferMovement, InsertProductionBufferMovement } from "@shared/schema";
import { getOrCreateBufferStock, updateBufferQuantities } from "./buffer-stock";

export type BufferMovementType = 'IN' | 'OUT' | 'RESERVE' | 'RELEASE';
type PoolOrClient = Pool | PoolClient;

export async function getBufferMovements(pool: Pool, filters?: {
  productSku?: string;
  locationId?: number;
  zlpId?: number;
  movementType?: BufferMovementType;
  limit?: number;
}): Promise<ProductionBufferMovement[]> {
  let query = `
    SELECT 
      bm.*,
      l.name as location_name,
      l.code as location_code
    FROM production.production_buffer_movements bm
    LEFT JOIN production.production_locations l ON bm.location_id = l.id
    WHERE 1=1
  `;
  const params: any[] = [];
  let paramIndex = 1;
  
  if (filters?.productSku) {
    query += ` AND bm.product_sku = $${paramIndex++}`;
    params.push(filters.productSku);
  }
  
  if (filters?.locationId) {
    query += ` AND bm.location_id = $${paramIndex++}`;
    params.push(filters.locationId);
  }
  
  if (filters?.zlpId) {
    query += ` AND bm.zlp_id = $${paramIndex++}`;
    params.push(filters.zlpId);
  }
  
  if (filters?.movementType) {
    query += ` AND bm.movement_type = $${paramIndex++}`;
    params.push(filters.movementType);
  }
  
  query += ` ORDER BY bm.created_at DESC`;
  
  if (filters?.limit) {
    query += ` LIMIT $${paramIndex++}`;
    params.push(filters.limit);
  }
  
  const result = await pool.query(query, params);
  
  return result.rows.map(row => ({
    id: row.id,
    movementType: row.movement_type,
    productSku: row.product_sku,
    quantity: row.quantity,
    unitOfMeasure: row.unit_of_measure,
    locationId: row.location_id,
    sourceType: row.source_type,
    sourceId: row.source_id,
    zlpId: row.zlp_id,
    reservationId: row.reservation_id,
    createdBy: row.created_by,
    createdAt: row.created_at,
    notes: row.notes,
  }));
}

export async function createBufferMovement(
  pool: PoolOrClient,
  data: InsertProductionBufferMovement
): Promise<ProductionBufferMovement> {
  const isClient = 'release' in pool && typeof pool.release === 'function';
  const client = isClient ? (pool as PoolClient) : await (pool as Pool).connect();
  try {
    if (!isClient) {
      await client.query('BEGIN');
    }
    
    const stock = await getOrCreateBufferStock(client, data.productSku, data.locationId || null);
    
    let availableDelta = 0;
    let reservedDelta = 0;
    
    switch (data.movementType) {
      case 'IN':
        availableDelta = Number(data.quantity);
        break;
      case 'OUT':
        availableDelta = -Number(data.quantity);
        break;
      case 'RESERVE':
        availableDelta = -Number(data.quantity);
        reservedDelta = Number(data.quantity);
        break;
      case 'RELEASE':
        availableDelta = Number(data.quantity);
        reservedDelta = -Number(data.quantity);
        break;
    }
    
    await updateBufferQuantities(client, data.productSku, data.locationId || null, availableDelta, reservedDelta);
    
    const result = await client.query(`
      INSERT INTO production.production_buffer_movements 
      (movement_type, product_sku, quantity, unit_of_measure, location_id, 
       source_type, source_id, zlp_id, reservation_id, created_by, notes)
      VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
      RETURNING *
    `, [
      data.movementType,
      data.productSku,
      data.quantity,
      data.unitOfMeasure || 'szt',
      data.locationId || null,
      data.sourceType || null,
      data.sourceId || null,
      data.zlpId || null,
      data.reservationId || null,
      data.createdBy || null,
      data.notes || null,
    ]);
    
    if (!isClient) {
      await client.query('COMMIT');
    }
    
    const row = result.rows[0];
    return {
      id: row.id,
      movementType: row.movement_type,
      productSku: row.product_sku,
      quantity: row.quantity,
      unitOfMeasure: row.unit_of_measure,
      locationId: row.location_id,
      sourceType: row.source_type,
      sourceId: row.source_id,
      zlpId: row.zlp_id,
      reservationId: row.reservation_id,
      createdBy: row.created_by,
      createdAt: row.created_at,
      notes: row.notes,
    };
  } catch (error) {
    if (!isClient) {
      await client.query('ROLLBACK');
    }
    throw error;
  } finally {
    if (!isClient) {
      client.release();
    }
  }
}
