import { Express } from 'express';
import { Pool } from 'pg';

export function registerPackedProductItemsRoutes(app: Express, pool: Pool, isAuthenticated: any) {
  
  // GET /api/warehouse/packed-products/:id/items - Lista pojedynczych sztuk produktu
  app.get("/api/warehouse/packed-products/:id/items", isAuthenticated, async (req, res) => {
    try {
      const productId = parseInt(req.params.id);
      if (isNaN(productId)) {
        return res.status(400).json({ error: "Invalid product ID" });
      }
      
      const { status, limit = '100', offset = '0' } = req.query;
      
      let query = `
        SELECT 
          ppi.id,
          ppi.serial_number as "serialNumber",
          ppi.product_sku as "productSku",
          ppi.status,
          ppi.packed_at as "packedAt",
          ppi.produced_at as "producedAt",
          ppi.consumed_at as "consumedAt",
          ppi.reserved_at as "reservedAt",
          ppi.reserved_for_plan_line_id as "reservedForPlanLineId",
          ppi.location_id as "locationId",
          ppi.carrier_id as "carrierId",
          ppi.defect_description as "defectDescription",
          ppi.notes,
          ppi.bom_id as "bomId",
          ppi.production_plan_line_id as "productionPlanLineId",
          ppi.metadata,
          ppi.source_type as "sourceType",
          ppi.source_id as "sourceId",
          ppi.created_at as "createdAt",
          ppi.updated_at as "updatedAt",
          -- Location and carrier names
          pl.name as "locationName",
          pc.name as "carrierName",
          -- Plan line info if reserved
          ppl.plan_id as "reservedForPlanId"
        FROM warehouse.packed_product_items ppi
        LEFT JOIN production.production_locations pl ON pl.id = ppi.location_id
        LEFT JOIN production.production_carriers pc ON pc.id = ppi.carrier_id
        LEFT JOIN production.production_plan_lines ppl ON ppl.id = ppi.reserved_for_plan_line_id
        WHERE ppi.packed_product_id = $1
      `;
      
      const params: any[] = [productId];
      let paramIndex = 2;
      
      if (status && status !== 'all') {
        query += ` AND ppi.status = $${paramIndex++}`;
        params.push(status);
      }
      
      query += ` ORDER BY ppi.packed_at ASC LIMIT $${paramIndex++} OFFSET $${paramIndex++}`;
      params.push(parseInt(limit as string), parseInt(offset as string));
      
      const itemsResult = await pool.query(query, params);
      
      // Count stats per status
      const statsResult = await pool.query(`
        SELECT 
          status,
          COUNT(*) as count
        FROM warehouse.packed_product_items
        WHERE packed_product_id = $1
        GROUP BY status
      `, [productId]);
      
      // Initialize all valid statuses
      const stats: Record<string, number> = {
        available: 0,
        reserved: 0,
        shipped: 0,
        returned: 0,
        defective: 0,
        scrapped: 0,
        total: 0
      };
      
      for (const row of statsResult.rows) {
        if (stats.hasOwnProperty(row.status)) {
          stats[row.status] = parseInt(row.count);
        }
        stats.total += parseInt(row.count);
      }
      
      // Calculate filtered total for pagination when status filter is active
      let filteredTotal = stats.total;
      if (status && status !== 'all' && stats.hasOwnProperty(status as string)) {
        filteredTotal = stats[status as string];
      }
      
      res.json({
        items: itemsResult.rows,
        stats,
        pagination: {
          limit: parseInt(limit as string),
          offset: parseInt(offset as string),
          total: filteredTotal
        }
      });
    } catch (error) {
      console.error("❌ Error fetching packed product items:", error);
      res.status(500).json({ error: "Failed to fetch packed product items" });
    }
  });
  
  // GET /api/warehouse/packed-products/items/:itemId - Pojedyncza sztuka
  app.get("/api/warehouse/packed-products/items/:itemId", isAuthenticated, async (req, res) => {
    try {
      const itemId = parseInt(req.params.itemId);
      if (isNaN(itemId)) {
        return res.status(400).json({ error: "Invalid item ID" });
      }
      
      const result = await pool.query(`
        SELECT 
          ppi.*,
          pp.product_name as "productName",
          pl.name as "locationName",
          pc.name as "carrierName"
        FROM warehouse.packed_product_items ppi
        JOIN warehouse.packed_products pp ON pp.id = ppi.packed_product_id
        LEFT JOIN production.production_locations pl ON pl.id = ppi.location_id
        LEFT JOIN production.production_carriers pc ON pc.id = ppi.carrier_id
        WHERE ppi.id = $1
      `, [itemId]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Item not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error fetching packed product item:", error);
      res.status(500).json({ error: "Failed to fetch packed product item" });
    }
  });
  
  // Valid status values
  const VALID_STATUSES = ['available', 'reserved', 'shipped', 'returned', 'defective', 'scrapped'];
  
  // Valid status transitions to prevent invalid state changes
  const VALID_TRANSITIONS: Record<string, string[]> = {
    'available': ['reserved', 'shipped', 'defective', 'scrapped'],
    'reserved': ['available', 'shipped', 'defective', 'scrapped'],
    'shipped': ['returned', 'defective', 'scrapped'],
    'returned': ['available', 'defective', 'scrapped'],
    'defective': ['scrapped', 'available'],
    'scrapped': []
  };
  
  // PATCH /api/warehouse/packed-products/items/:itemId - Aktualizuj pojedynczą sztukę
  app.patch("/api/warehouse/packed-products/items/:itemId", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    try {
      const itemId = parseInt(req.params.itemId);
      if (isNaN(itemId)) {
        return res.status(400).json({ error: "Invalid item ID" });
      }
      
      const { status, locationId, carrierId, defectDescription, notes } = req.body;
      
      // Validate status if provided
      if (status !== undefined && !VALID_STATUSES.includes(status)) {
        return res.status(400).json({ 
          error: `Invalid status. Must be one of: ${VALID_STATUSES.join(', ')}`
        });
      }
      
      await client.query('BEGIN');
      
      // Lock the item row to prevent race conditions
      const currentItemResult = await client.query(
        `SELECT id, status, reserved_for_plan_line_id, packed_product_id 
         FROM warehouse.packed_product_items 
         WHERE id = $1 
         FOR UPDATE`,
        [itemId]
      );
      
      if (currentItemResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Item not found" });
      }
      
      const currentItem = currentItemResult.rows[0];
      
      // Validate status transition if changing status
      if (status !== undefined && status !== currentItem.status) {
        const allowedTransitions = VALID_TRANSITIONS[currentItem.status] || [];
        if (!allowedTransitions.includes(status)) {
          await client.query('ROLLBACK');
          return res.status(400).json({ 
            error: `Invalid status transition from '${currentItem.status}' to '${status}'. Allowed: ${allowedTransitions.join(', ') || 'none'}`
          });
        }
        
        // If changing from reserved to available, clear reservation link
        if (currentItem.status === 'reserved' && status === 'available') {
          if (currentItem.reserved_for_plan_line_id) {
            await client.query('ROLLBACK');
            return res.status(400).json({ 
              error: "Cannot change reserved item to available while it has a plan line reservation. Use unreserve endpoint instead."
            });
          }
        }
      }
      
      const fields: string[] = [];
      const values: any[] = [];
      let paramIndex = 1;
      
      if (status !== undefined) {
        fields.push(`status = $${paramIndex++}`);
        values.push(status);
        
        // Auto-set dates based on status changes
        if (status === 'shipped') {
          fields.push(`consumed_at = CURRENT_TIMESTAMP`);
        }
      }
      
      if (locationId !== undefined) {
        fields.push(`location_id = $${paramIndex++}`);
        values.push(locationId);
      }
      
      if (carrierId !== undefined) {
        fields.push(`carrier_id = $${paramIndex++}`);
        values.push(carrierId);
      }
      
      if (defectDescription !== undefined) {
        fields.push(`defect_description = $${paramIndex++}`);
        values.push(defectDescription);
      }
      
      if (notes !== undefined) {
        fields.push(`notes = $${paramIndex++}`);
        values.push(notes);
      }
      
      if (fields.length === 0) {
        await client.query('ROLLBACK');
        return res.status(400).json({ error: "No fields to update" });
      }
      
      fields.push(`updated_at = CURRENT_TIMESTAMP`);
      values.push(itemId);
      
      const result = await client.query(
        `UPDATE warehouse.packed_product_items 
         SET ${fields.join(', ')} 
         WHERE id = $${paramIndex}
         RETURNING *`,
        values
      );
      
      // Update summary table counts if status changed
      if (status !== undefined && status !== currentItem.status) {
        const oldIsAvailable = currentItem.status === 'available';
        const newIsAvailable = status === 'available';
        const oldIsReserved = currentItem.status === 'reserved';
        const newIsReserved = status === 'reserved';
        
        if (oldIsAvailable !== newIsAvailable || oldIsReserved !== newIsReserved) {
          // Recalculate counts from items table
          await client.query(`
            UPDATE warehouse.packed_products pp
            SET 
              quantity = (SELECT COUNT(*) FROM warehouse.packed_product_items ppi WHERE ppi.packed_product_id = pp.id),
              reserved_quantity = (SELECT COUNT(*) FROM warehouse.packed_product_items ppi WHERE ppi.packed_product_id = pp.id AND ppi.status = 'reserved')
            WHERE pp.id = $1
          `, [currentItem.packed_product_id]);
        }
      }
      
      await client.query('COMMIT');
      
      res.json(result.rows[0]);
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error updating packed product item:", error);
      res.status(500).json({ error: "Failed to update packed product item" });
    } finally {
      client.release();
    }
  });
  
  // POST /api/warehouse/packed-products/:id/items - Dodaj nową sztukę produktu
  app.post("/api/warehouse/packed-products/:id/items", isAuthenticated, async (req, res) => {
    try {
      const productId = parseInt(req.params.id);
      if (isNaN(productId)) {
        return res.status(400).json({ error: "Invalid product ID" });
      }
      
      // Get product info
      const productResult = await pool.query(
        'SELECT id, catalog_product_id, catalog_set_id, product_sku FROM warehouse.packed_products WHERE id = $1',
        [productId]
      );
      
      if (productResult.rows.length === 0) {
        return res.status(404).json({ error: "Product not found" });
      }
      
      const product = productResult.rows[0];
      const { quantity = 1, locationId, carrierId, notes, producedAt } = req.body;
      
      // Generate serial numbers
      const today = new Date().toISOString().slice(0, 10).replace(/-/g, '');
      const productIdStr = product.catalog_product_id?.toString() || '0';
      
      // Get next sequence number
      const seqResult = await pool.query(`
        SELECT COALESCE(MAX(CAST(SPLIT_PART(serial_number, '-', 3) AS INTEGER)), 0) + 1 as next_seq
        FROM warehouse.packed_product_items
        WHERE serial_number LIKE $1 || '-' || $2 || '-%'
      `, [productIdStr, today]);
      
      let nextSeq = seqResult.rows[0].next_seq;
      
      const insertedItems: any[] = [];
      
      for (let i = 0; i < quantity; i++) {
        const serialNumber = `${productIdStr}-${today}-${String(nextSeq++).padStart(5, '0')}`;
        
        const insertResult = await pool.query(`
          INSERT INTO warehouse.packed_product_items (
            packed_product_id, catalog_product_id, catalog_set_id, product_sku,
            serial_number, status, location_id, carrier_id, notes, produced_at,
            packed_at, created_at, updated_at
          ) VALUES ($1, $2, $3, $4, $5, 'available', $6, $7, $8, $9, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
          RETURNING *
        `, [
          productId,
          product.catalog_product_id,
          product.catalog_set_id,
          product.product_sku,
          serialNumber,
          locationId || null,
          carrierId || null,
          notes || null,
          producedAt || null
        ]);
        
        insertedItems.push(insertResult.rows[0]);
      }
      
      // Update aggregate quantity
      await pool.query(`
        UPDATE warehouse.packed_products 
        SET quantity = quantity + $1, updated_at = CURRENT_TIMESTAMP
        WHERE id = $2
      `, [quantity, productId]);
      
      res.json({
        items: insertedItems,
        count: insertedItems.length
      });
    } catch (error) {
      console.error("❌ Error creating packed product items:", error);
      res.status(500).json({ error: "Failed to create packed product items" });
    }
  });
  
  // POST /api/warehouse/packed-products/items/:itemId/mark-defective - Oznacz jako wadliwy
  app.post("/api/warehouse/packed-products/items/:itemId/mark-defective", isAuthenticated, async (req, res) => {
    try {
      const itemId = parseInt(req.params.itemId);
      if (isNaN(itemId)) {
        return res.status(400).json({ error: "Invalid item ID" });
      }
      
      const { defectDescription } = req.body;
      
      if (!defectDescription) {
        return res.status(400).json({ error: "Defect description is required" });
      }
      
      const result = await pool.query(`
        UPDATE warehouse.packed_product_items
        SET status = 'defective',
            defect_description = $1,
            updated_at = CURRENT_TIMESTAMP
        WHERE id = $2 AND status IN ('available', 'reserved')
        RETURNING *
      `, [defectDescription, itemId]);
      
      if (result.rows.length === 0) {
        return res.status(400).json({ error: "Item not found or cannot be marked as defective" });
      }
      
      // Update aggregate if it was reserved
      const item = result.rows[0];
      if (item.reserved_for_plan_line_id) {
        await pool.query(`
          UPDATE warehouse.packed_products 
          SET reserved_quantity = GREATEST(0, reserved_quantity - 1), updated_at = CURRENT_TIMESTAMP
          WHERE id = $1
        `, [item.packed_product_id]);
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error marking item as defective:", error);
      res.status(500).json({ error: "Failed to mark item as defective" });
    }
  });
  
  // POST /api/warehouse/packed-products/:id/create-inventory-count - Utwórz spis inwentaryzacyjny dla produktu
  app.post("/api/warehouse/packed-products/:id/create-inventory-count", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    
    try {
      const productId = parseInt(req.params.id);
      if (isNaN(productId)) {
        return res.status(400).json({ error: "Invalid product ID" });
      }
      
      const { name } = req.body;
      const username = (req.user as any)?.username || 'system';
      
      await client.query('BEGIN');
      
      // Get packed product info
      const productResult = await client.query(
        'SELECT id, product_name, product_sku, quantity FROM warehouse.packed_products WHERE id = $1',
        [productId]
      );
      
      if (productResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Packed product not found" });
      }
      
      const product = productResult.rows[0];
      
      // Count actual items
      const itemsCountResult = await client.query(
        'SELECT COUNT(*)::int as count FROM warehouse.packed_product_items WHERE packed_product_id = $1',
        [productId]
      );
      const currentQuantity = itemsCountResult.rows[0].count;
      
      // Create inventory count
      const countName = name || `Spis ${product.product_name} ${new Date().toLocaleDateString('pl-PL')}`;
      
      const countResult = await client.query(`
        INSERT INTO warehouse.inventory_counts (name, status, notes, created_by)
        VALUES ($1, 'draft', $2, $3)
        RETURNING id
      `, [countName, `Spis dla produktu: ${product.product_name} (${product.product_sku})`, username]);
      
      const inventoryCountId = countResult.rows[0].id;
      
      // Add packed product as inventory count item
      await client.query(`
        INSERT INTO warehouse.inventory_count_items (
          inventory_count_id, packed_product_id, system_quantity, counted_quantity, difference
        ) VALUES ($1, $2, $3, NULL, NULL)
      `, [inventoryCountId, productId, currentQuantity]);
      
      await client.query('COMMIT');
      
      res.json({
        inventoryCountId,
        message: `Utworzono spis inwentaryzacyjny "${countName}"`,
        product: {
          id: product.id,
          name: product.product_name,
          sku: product.product_sku,
          systemQuantity: currentQuantity
        }
      });
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error creating inventory count:", error);
      res.status(500).json({ error: "Failed to create inventory count" });
    } finally {
      client.release();
    }
  });
  
  // POST /api/warehouse/packed-products/items/:itemId/scrap - Zezłomuj sztukę
  app.post("/api/warehouse/packed-products/items/:itemId/scrap", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    try {
      const itemId = parseInt(req.params.itemId);
      if (isNaN(itemId)) {
        return res.status(400).json({ error: "Invalid item ID" });
      }
      
      const { reason } = req.body;
      const username = (req.user as any)?.username || 'system';
      
      if (!reason || reason.trim() === '') {
        return res.status(400).json({ error: "Scrap reason is required" });
      }
      
      await client.query('BEGIN');
      
      // Get item info with lock
      const itemResult = await client.query(
        `SELECT id, packed_product_id, status, serial_number, product_sku 
         FROM warehouse.packed_product_items 
         WHERE id = $1 
         FOR UPDATE`,
        [itemId]
      );
      
      if (itemResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Item not found" });
      }
      
      const item = itemResult.rows[0];
      
      // Don't allow scrapping of reserved items
      if (item.status === 'reserved') {
        await client.query('ROLLBACK');
        return res.status(400).json({ error: "Cannot scrap reserved item. Release reservation first." });
      }
      
      // Don't allow scrapping already scrapped items
      if (item.status === 'scrapped') {
        await client.query('ROLLBACK');
        return res.status(400).json({ error: "Item is already scrapped." });
      }
      
      // Check for lingering reservation links even if status changed
      const reservationCheck = await client.query(
        `SELECT reserved_for_plan_line_id FROM warehouse.packed_product_items WHERE id = $1`,
        [itemId]
      );
      if (reservationCheck.rows[0]?.reserved_for_plan_line_id) {
        await client.query('ROLLBACK');
        return res.status(400).json({ 
          error: "Cannot scrap item with active plan line reservation. Release reservation first." 
        });
      }
      
      const trimmedReason = reason.trim();
      
      // Build audit entry
      const auditEntry = {
        action: 'scrapped',
        timestamp: new Date().toISOString(),
        performedBy: username,
        reason: trimmedReason,
        previousStatus: item.status
      };
      
      // Update item to scrapped status with reason
      // Use || operator to preserve existing metadata while appending to auditHistory
      const result = await client.query(`
        UPDATE warehouse.packed_product_items
        SET status = 'scrapped',
            notes = CASE 
              WHEN notes IS NULL OR notes = '' THEN $1
              ELSE notes || E'\\n---\\n' || $1
            END,
            metadata = COALESCE(metadata, '{}'::jsonb) || jsonb_build_object(
              'auditHistory', COALESCE(metadata->'auditHistory', '[]'::jsonb) || $2::jsonb
            ),
            updated_at = CURRENT_TIMESTAMP
        WHERE id = $3
        RETURNING *
      `, [
        `[${new Date().toISOString().slice(0, 10)}] Zezłomowano: ${trimmedReason} (przez: ${username})`,
        JSON.stringify(auditEntry),
        itemId
      ]);
      
      // Recalculate aggregate counts from canonical items table to ensure consistency
      await client.query(`
        UPDATE warehouse.packed_products 
        SET quantity = (SELECT COUNT(*) FROM warehouse.packed_product_items WHERE packed_product_id = $1),
            reserved_quantity = (SELECT COUNT(*) FROM warehouse.packed_product_items WHERE packed_product_id = $1 AND status = 'reserved'),
            updated_at = CURRENT_TIMESTAMP
        WHERE id = $1
      `, [item.packed_product_id]);
      
      await client.query('COMMIT');
      
      console.log(`📦 Item ${item.serial_number} (${item.product_sku}) scrapped by ${username}: ${reason}`);
      
      res.json(result.rows[0]);
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error scrapping packed product item:", error);
      res.status(500).json({ error: "Failed to scrap packed product item" });
    } finally {
      client.release();
    }
  });

  // GET /api/warehouse/packed-products/:id/history - Zbiorcza historia wszystkich sztuk produktu
  app.get("/api/warehouse/packed-products/:id/history", isAuthenticated, async (req, res) => {
    try {
      const productId = parseInt(req.params.id);
      if (isNaN(productId)) {
        return res.status(400).json({ error: "Invalid product ID" });
      }
      
      // Get all items with their metadata
      const itemsResult = await pool.query(`
        SELECT 
          ppi.id,
          ppi.serial_number as "serialNumber",
          ppi.status,
          ppi.packed_at as "packedAt",
          ppi.reserved_at as "reservedAt",
          ppi.consumed_at as "consumedAt",
          ppi.reserved_for_plan_line_id as "reservedForPlanLineId",
          ppi.metadata,
          ppi.notes,
          ppi.source_type as "sourceType",
          ppi.source_id as "sourceId",
          ppi.updated_at as "updatedAt",
          pl.name as "locationName",
          ppl.plan_id as "planId"
        FROM warehouse.packed_product_items ppi
        LEFT JOIN production.production_locations pl ON pl.id = ppi.location_id
        LEFT JOIN production.production_plan_lines ppl ON ppl.id = ppi.reserved_for_plan_line_id
        WHERE ppi.packed_product_id = $1
        ORDER BY ppi.updated_at DESC
      `, [productId]);
      
      // Build aggregated history from all items
      const allEvents: Array<{
        id: number;
        serialNumber: string;
        action: string;
        timestamp: string;
        performedBy?: string;
        details?: string;
        status: string;
        planId?: number;
        planLineId?: number;
        orderId?: string;
        sourceType?: string;
        sourceId?: number;
      }> = [];
      
      for (const item of itemsResult.rows) {
        // Add creation event
        if (item.packedAt) {
          const sourceLabel = item.sourceType === 'inventory_count' 
            ? `ze spisu #${item.sourceId}` 
            : item.sourceType === 'production' 
              ? `z produkcji #${item.sourceId}` 
              : 'ręcznie';
          allEvents.push({
            id: item.id,
            serialNumber: item.serialNumber,
            action: 'created',
            timestamp: item.packedAt,
            details: `Dodano do magazynu ${sourceLabel}`,
            status: 'available',
            sourceType: item.sourceType,
            sourceId: item.sourceId
          });
        }
        
        // Add audit history events
        if (item.metadata?.auditHistory) {
          for (const entry of item.metadata.auditHistory) {
            let details = '';
            switch (entry.action) {
              case 'reserved':
                details = entry.planId 
                  ? `Zarezerwowano do planu #${entry.planId}` 
                  : 'Zarezerwowano';
                break;
              case 'unreserved':
                details = entry.planId 
                  ? `Anulowano rezerwację z planu #${entry.planId}` 
                  : 'Anulowano rezerwację';
                break;
              case 'shipped':
                details = entry.orderId 
                  ? `Wysłano z zamówieniem ${entry.orderId}` 
                  : 'Wysłano';
                break;
              case 'returned':
                details = entry.reason || 'Zwrócono';
                break;
              case 'moved':
                details = entry.locationFrom && entry.locationTo
                  ? `Przeniesiono z ${entry.locationFrom} do ${entry.locationTo}`
                  : 'Przeniesiono';
                break;
              case 'scrapped':
                details = entry.reason?.replace(/^\[.*?\]\s*/, '') || 'Zezłomowano';
                break;
              case 'defective':
                details = entry.reason || 'Oznaczono jako wadliwy';
                break;
              default:
                details = entry.action;
            }
            
            allEvents.push({
              id: item.id,
              serialNumber: item.serialNumber,
              action: entry.action,
              timestamp: entry.timestamp,
              performedBy: entry.performedBy,
              details,
              status: entry.action === 'scrapped' ? 'scrapped' : 
                      entry.action === 'reserved' ? 'reserved' : 
                      entry.action === 'unreserved' ? 'available' :
                      entry.action === 'shipped' ? 'shipped' : item.status,
              planId: entry.planId,
              planLineId: entry.planLineId,
              orderId: entry.orderId
            });
          }
        }
        
        // Add current reservation if not in audit history
        if (item.reservedAt && item.status === 'reserved' && 
            !item.metadata?.auditHistory?.some((e: any) => e.action === 'reserved')) {
          allEvents.push({
            id: item.id,
            serialNumber: item.serialNumber,
            action: 'reserved',
            timestamp: item.reservedAt,
            details: item.planId 
              ? `Zarezerwowano do planu #${item.planId}` 
              : 'Zarezerwowano',
            status: 'reserved',
            planId: item.planId,
            planLineId: item.reservedForPlanLineId
          });
        }
      }
      
      // Sort all events by timestamp descending (newest first)
      allEvents.sort((a, b) => new Date(b.timestamp).getTime() - new Date(a.timestamp).getTime());
      
      // Get stats
      const statsResult = await pool.query(`
        SELECT 
          status,
          COUNT(*) as count
        FROM warehouse.packed_product_items
        WHERE packed_product_id = $1
        GROUP BY status
      `, [productId]);
      
      const stats: Record<string, number> = {
        available: 0,
        reserved: 0,
        shipped: 0,
        returned: 0,
        defective: 0,
        scrapped: 0,
        total: 0
      };
      
      for (const row of statsResult.rows) {
        if (stats.hasOwnProperty(row.status)) {
          stats[row.status] = parseInt(row.count);
        }
        stats.total += parseInt(row.count);
      }
      
      res.json({
        events: allEvents,
        stats,
        totalItems: itemsResult.rows.length
      });
    } catch (error) {
      console.error("❌ Error fetching packed product history:", error);
      res.status(500).json({ error: "Failed to fetch packed product history" });
    }
  });
}
