import type { Express } from "express";
import { z } from "zod";
import { EventEmitter } from 'events';
import { pool } from "../../postgres";
import * as planningService from "../../services/production/planning";
import * as batchEngine from "../../services/production/batch-engine";
import { requirePermission } from "../../auth";
import { generateProductionOrdersFromPlan, ProgressCallback, ProductionOrderGenerationResult } from "../../services/production/production-order-generator";

// EventEmitter for ZLP generation progress streaming
const zlpProgressEmitter = new EventEmitter();
zlpProgressEmitter.setMaxListeners(50);

// Store active generation sessions
interface GenerationSession {
  planId: number;
  status: 'pending' | 'running' | 'completed' | 'failed';
  startedAt: Date;
  completedAt?: Date;
  result?: ProductionOrderGenerationResult;
  error?: string;
}
const generationSessions = new Map<string, GenerationSession>();

// Helper: Generate unique session ID
function generateSessionId(): string {
  return `zlp-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
}

// Merge Point schemas
const createMergePointSchema = z.object({
  mergeStage: z.string().min(1, "Merge stage is required"),
  requiredFamilies: z.array(z.string()).min(1, "At least one family is required"),
  completionPolicy: z.enum(['all_required', 'any_one', 'majority']).default('all_required'),
  manualRelease: z.boolean().default(false),
  routingOperationId: z.number().int().positive().optional().nullable(),
  notes: z.string().optional().nullable(),
});

const updateMergePointSchema = z.object({
  mergeStage: z.string().min(1).optional(),
  requiredFamilies: z.array(z.string()).min(1).optional(),
  completionPolicy: z.enum(['all_required', 'any_one', 'majority']).optional(),
  manualRelease: z.boolean().optional(),
  routingOperationId: z.number().int().positive().optional().nullable(),
  notes: z.string().optional().nullable(),
}).refine(data => Object.keys(data).length > 0, {
  message: "At least one field must be provided for update"
});

const createPlanSchema = z.object({
  planNumber: z.string().optional(),
  name: z.string().min(1),
  description: z.string().optional().nullable(),
  plannedStartDate: z.string().optional().nullable(),
  plannedEndDate: z.string().optional().nullable(),
  status: z.enum(['draft', 'approved', 'in_progress', 'completed', 'cancelled']).optional(),
  priority: z.enum(['low', 'normal', 'high', 'urgent']).optional(),
  notes: z.string().optional().nullable(),
  metadata: z.any().optional(),
  createdBy: z.number().optional().nullable(),
  autoAssignRoutings: z.boolean().optional().default(false),
});

const createPlanLineSchema = z.object({
  planId: z.number(),
  productId: z.number(),
  quantity: z.number().positive(),
  sourceType: z.enum(['sales_order', 'forecast', 'buffer_stock', 'manual', 'order_demand', 'catalog_internal', 'cutting_pattern']).optional().nullable(),
  sourceId: z.number().optional().nullable(),
  sourceReference: z.string().optional().nullable(),
  productionOrderId: z.number().optional().nullable(),
  routingId: z.number().optional().nullable(),
  routingOverride: z.any().optional(),
  bomId: z.number().optional().nullable(),
  plannedStartDate: z.string().optional().nullable(),
  plannedEndDate: z.string().optional().nullable(),
  status: z.enum(['pending', 'scheduled', 'in_progress', 'completed', 'cancelled']).optional(),
  sequence: z.number().optional().nullable(),
  notes: z.string().optional().nullable(),
  metadata: z.any().optional(),
});

const filtersSchema = z.object({
  status: z.string().optional(),
  priority: z.string().optional(),
  createdBy: z.coerce.number().optional(),
  startDate: z.string().optional(),
  endDate: z.string().optional(),
  search: z.string().optional(),
  limit: z.coerce.number().optional(),
  offset: z.coerce.number().optional(),
});

const demandFiltersSchema = z.object({
  startDate: z.string().optional(),
  endDate: z.string().optional(),
  marketplace: z.string().optional(),
  orderStatus: z.string().optional(),
  paymentStatus: z.string().optional(),
});

const availableOrdersFiltersSchema = z.object({
  search: z.string().optional(),
  color: z.string().optional(),
  sku: z.string().optional(),
  minLength: z.coerce.number().optional(),
  maxLength: z.coerce.number().optional(),
  minWidth: z.coerce.number().optional(),
  maxWidth: z.coerce.number().optional(),
  orderNumber: z.string().optional(),
  customerName: z.string().optional(),
  marketplace: z.string().optional(),
  showSetsOnly: z.string().optional(), // 'true' or 'false'
  showCatalogLinked: z.string().optional(), // 'true' or 'false'
  showInPlans: z.string().optional(), // 'true' or 'false' - filter items already in any plan
  dateFilter: z.enum(['all', 'today', 'yesterday', 'day-before', 'custom-days']).optional().default('all'),
  customDays: z.coerce.number().optional(), // Number of days ago for custom-days filter
  limit: z.coerce.number().min(1).max(10000).optional().default(100), // Increased max to 10000 for full data loading
  offset: z.coerce.number().min(0).optional().default(0),
  sortBy: z.enum(['order_date', 'order_number', 'buyer_name', 'total_amount', 'product_sku']).optional().default('order_date'),
  sortOrder: z.enum(['asc', 'desc']).optional().default('desc'),
});

// Helper: Parse semicolon-delimited search tokens into structured filters
interface ParsedSearchTokens {
  marketplaces: string[];
  paymentStatuses: string[];
  searchTerms: string[];
}

function parseSearchTokens(rawSearch: string | undefined): ParsedSearchTokens {
  if (!rawSearch) {
    return { marketplaces: [], paymentStatuses: [], searchTerms: [] };
  }

  // Split by semicolon, trim, uppercase, deduplicate
  const tokens = Array.from(new Set(
    rawSearch.split(';')
      .map(t => t.trim().toUpperCase())
      .filter(Boolean)
  ));

  // Whitelist recognized tokens
  const MARKETPLACE_TOKENS = ['ALLEGRO', 'SHOPER'];
  const PAYMENT_STATUS_TOKENS = ['PAID', 'UNPAID', 'PENDING'];

  const marketplaces: string[] = [];
  const paymentStatuses: string[] = [];
  const searchTerms: string[] = [];

  tokens.forEach(token => {
    if (MARKETPLACE_TOKENS.includes(token)) {
      marketplaces.push(token.toLowerCase()); // Store as lowercase for DB comparison
    } else if (PAYMENT_STATUS_TOKENS.includes(token)) {
      paymentStatuses.push(token.toLowerCase());
    } else {
      searchTerms.push(token); // Keep original case for search
    }
  });

  return { marketplaces, paymentStatuses, searchTerms };
}

export function registerPlanningRoutes(app: Express) {
  // GET /api/production/planning/plans - Get all plans with filtering
  app.get("/api/production/planning/plans", requirePermission('view_production'), async (req, res) => {
    try {
      const filters = filtersSchema.parse(req.query);
      
      const processedFilters = {
        ...filters,
        startDate: filters.startDate ? new Date(filters.startDate) : undefined,
        endDate: filters.endDate ? new Date(filters.endDate) : undefined,
      };

      const plans = await planningService.getPlans(pool, processedFilters);
      res.json(plans);
    } catch (error) {
      console.error("Error fetching production plans:", error);
      res.status(500).json({ message: "Failed to fetch production plans" });
    }
  });

  // GET /api/production/planning/plans/:id - Get plan by ID
  app.get("/api/production/planning/plans/:id", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const plan = await planningService.getPlanById(pool, id);
      if (!plan) {
        return res.status(404).json({ message: "Production plan not found" });
      }

      res.json(plan);
    } catch (error) {
      console.error("Error fetching production plan:", error);
      res.status(500).json({ message: "Failed to fetch production plan" });
    }
  });

  // GET /api/production/planning/plans/:id/items - Get plan lines with full details
  app.get("/api/production/planning/plans/:id/items", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const items = await planningService.getPlanLinesWithDetails(pool, id);
      
      // DEBUG: Log how many items we're returning
      console.log(`📦 [PLAN ITEMS] Returning ${items.length} items for plan ${id}`);
      if (items.length > 0) {
        console.log(`📦 [SAMPLE] First item:`, {
          id: items[0].id,
          product_sku: items[0].product_sku,
          product_title: items[0].product_title,
          order_number: items[0].order_number
        });
      }
      
      // Disable HTTP caching and ETag to ensure fresh data
      res.setHeader('Cache-Control', 'no-store, no-cache, must-revalidate, private, max-age=0');
      res.setHeader('Pragma', 'no-cache');
      res.setHeader('Expires', '0');
      res.removeHeader('ETag'); // Remove Express auto-generated ETag
      
      res.json(items);
    } catch (error) {
      console.error("Error fetching plan items:", error);
      res.status(500).json({ message: "Failed to fetch plan items" });
    }
  });

  // GET /api/production/planning/plans/:id/zlp-dashboard - Get ZLP dashboard data for a plan
  app.get("/api/production/planning/plans/:id/zlp-dashboard", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const dashboard = await planningService.getZlpDashboardData(pool, id);
      
      if (!dashboard) {
        return res.status(404).json({ message: "Production plan not found" });
      }
      
      res.json(dashboard);
    } catch (error) {
      console.error("Error fetching ZLP dashboard:", error);
      res.status(500).json({ message: "Failed to fetch ZLP dashboard" });
    }
  });

  // GET /api/production/demand/aggregate - Aggregate order demand by catalog product
  app.get("/api/production/demand/aggregate", requirePermission('view_production'), async (req, res) => {
    try {
      const filters = demandFiltersSchema.parse(req.query);
      
      const processedFilters: planningService.DemandFilters = {
        startDate: filters.startDate ? new Date(filters.startDate) : undefined,
        endDate: filters.endDate ? new Date(filters.endDate) : undefined,
        marketplace: filters.marketplace,
        orderStatus: filters.orderStatus ? filters.orderStatus.split(',') : undefined,
        paymentStatus: filters.paymentStatus ? filters.paymentStatus.split(',') : undefined,
      };

      const aggregatedDemand = await planningService.aggregateDemand(pool, processedFilters);
      res.json(aggregatedDemand);
    } catch (error) {
      console.error("Error aggregating demand:", error);
      res.status(500).json({ message: "Failed to aggregate demand" });
    }
  });

  // POST /api/production/planning/plans - Create new plan
  app.post("/api/production/planning/plans", requirePermission('manage_production'), async (req, res) => {
    try {
      const data = createPlanSchema.parse(req.body);
      
      const plan = await planningService.createPlan(pool, {
        ...data,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : null,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : null,
      });
      
      res.status(201).json(plan);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Plan with this number already exists" });
      }
      console.error("Error creating production plan:", error);
      res.status(500).json({ message: "Failed to create production plan" });
    }
  });

  // PATCH /api/production/planning/plans/:id - Update plan
  app.patch("/api/production/planning/plans/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const data = createPlanSchema.partial().parse(req.body);
      const plan = await planningService.updatePlan(pool, id, {
        ...data,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : undefined,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : undefined,
      });
      
      if (!plan) {
        return res.status(404).json({ message: "Production plan not found" });
      }

      res.json(plan);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      console.error("Error updating production plan:", error);
      res.status(500).json({ message: "Failed to update production plan" });
    }
  });

  // DELETE /api/production/planning/plans/:id - Delete plan
  app.delete("/api/production/planning/plans/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const success = await planningService.deletePlan(pool, id);
      if (!success) {
        return res.status(404).json({ message: "Production plan not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting production plan:", error);
      res.status(500).json({ message: "Failed to delete production plan" });
    }
  });

  // POST /api/production/planning/plans/:id/lock-zlp - Lock ZLP (prevent further editing)
  // Also starts all production orders (changes status to 'in_progress')
  app.post("/api/production/planning/plans/:id/lock-zlp", requirePermission('manage_production'), async (req, res) => {
    const client = await pool.connect();
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const userId = req.user?.id || null;
      
      await client.query('BEGIN');
      
      // Update plan with ZLP lock
      const result = await client.query(
        `UPDATE production.production_plans 
         SET zlp_locked = true, zlp_locked_by = $2, zlp_locked_at = NOW(), updated_at = NOW()
         WHERE id = $1
         RETURNING *`,
        [id, userId]
      );

      if (result.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ message: "Production plan not found" });
      }

      // Start all production orders linked to this plan
      // Change their status to 'in_progress' (as if "Rozpocznij" button was clicked)
      const ordersResult = await client.query(`
        UPDATE production.production_orders po
        SET 
          status = 'in_progress',
          actual_start_date = COALESCE(actual_start_date, NOW()),
          updated_at = NOW()
        WHERE po.id IN (
          SELECT DISTINCT pob.production_order_id
          FROM production.production_order_boms pob
          WHERE pob.source_plan_id = $1
        )
        AND po.status IN ('draft', 'confirmed', 'planned')
        RETURNING po.id, po.order_number
      `, [id]);

      const startedOrdersCount = ordersResult.rowCount || 0;
      
      if (startedOrdersCount > 0) {
        console.log(`✅ Started ${startedOrdersCount} production orders for plan ${id} during lock-zlp`);
      }

      await client.query('COMMIT');

      res.json({ 
        message: "ZLP locked successfully", 
        plan: result.rows[0],
        startedOrdersCount,
        startedOrders: ordersResult.rows
      });
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("Error locking ZLP:", error);
      res.status(500).json({ message: "Failed to lock ZLP" });
    } finally {
      client.release();
    }
  });

  // POST /api/production/planning/plans/:id/unlock-zlp - Unlock ZLP (allow editing again)
  app.post("/api/production/planning/plans/:id/unlock-zlp", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // Update plan to remove ZLP lock
      const result = await pool.query(
        `UPDATE production.production_plans 
         SET zlp_locked = false, zlp_locked_by = NULL, zlp_locked_at = NULL, updated_at = NOW()
         WHERE id = $1
         RETURNING *`,
        [id]
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Production plan not found" });
      }

      res.json({ 
        message: "ZLP unlocked successfully", 
        plan: result.rows[0] 
      });
    } catch (error) {
      console.error("Error unlocking ZLP:", error);
      res.status(500).json({ message: "Failed to unlock ZLP" });
    }
  });

  // POST /api/production/planning/plans/:id/generate-orders - Start async generation (returns session ID)
  app.post("/api/production/planning/plans/:id/generate-orders", requirePermission('manage_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // Generate session ID and start async job
      const sessionId = generateSessionId();
      
      // Store session
      generationSessions.set(sessionId, {
        planId,
        status: 'pending',
        startedAt: new Date(),
      });

      console.log(`🏭 [ZLP GENERATION] Starting async generation for plan ${planId}, session: ${sessionId}`);
      
      // Return session ID immediately
      res.status(202).json({ 
        sessionId, 
        message: "Generation started",
        planId,
      });

      // Start generation in background (non-blocking)
      setImmediate(async () => {
        const session = generationSessions.get(sessionId);
        try {
          // Update session status
          if (session) {
            session.status = 'running';
          }

          // Create progress callback that emits to SSE
          const onProgress: ProgressCallback = (progress) => {
            // The 'done' step will be handled separately with result
            if (progress.step !== 'done') {
              zlpProgressEmitter.emit(sessionId, {
                ...progress,
                timestamp: new Date().toISOString(),
              });
            }
          };

          // Run generation with progress callback
          const result = await generateProductionOrdersFromPlan(planId, onProgress);

          // Update session with result BEFORE emitting completion
          if (session) {
            session.status = result.success ? 'completed' : 'failed';
            session.completedAt = new Date();
            session.result = result;
            if (!result.success) {
              session.error = result.errors.join(', ');
            }
          }

          // Emit completion event with result
          zlpProgressEmitter.emit(sessionId, {
            step: 'done',
            current: 100,
            total: 100,
            message: result.success 
              ? `Wygenerowano ${result.summary.totalOrders} zleceń ZLP` 
              : `Błędy: ${result.errors.join(', ')}`,
            type: result.success ? 'success' : 'error',
            timestamp: new Date().toISOString(),
            result,
          });

          console.log(`✅ [ZLP GENERATION] Async completed for plan ${planId}:`, {
            success: result.success,
            totalOrders: result.summary.totalOrders,
            totalComponents: result.summary.totalComponents,
          });

          // Clean up session after 5 minutes
          setTimeout(() => {
            generationSessions.delete(sessionId);
          }, 5 * 60 * 1000);

        } catch (error: any) {
          console.error(`❌ [ZLP GENERATION] Async error for plan ${planId}:`, error);
          
          // Update session with error
          if (session) {
            session.status = 'failed';
            session.completedAt = new Date();
            session.error = error.message;
          }

          // Emit error event
          zlpProgressEmitter.emit(sessionId, {
            step: 'error',
            current: 0,
            total: 100,
            message: `Błąd: ${error.message}`,
            type: 'error',
            timestamp: new Date().toISOString(),
            error: error.message,
          });

          // Clean up session after 5 minutes
          setTimeout(() => {
            generationSessions.delete(sessionId);
          }, 5 * 60 * 1000);
        }
      });

    } catch (error: any) {
      console.error("❌ [ZLP GENERATION] Error starting generation:", error);
      res.status(500).json({ 
        message: "Failed to start generation",
        error: error.message,
      });
    }
  });

  // GET /api/production/planning/zlp-progress/:sessionId - SSE endpoint for progress streaming
  app.get("/api/production/planning/zlp-progress/:sessionId", requirePermission('view_production'), (req, res) => {
    const { sessionId } = req.params;

    // Check if session exists
    const session = generationSessions.get(sessionId);
    if (!session) {
      return res.status(404).json({ message: "Session not found" });
    }

    // Set SSE headers
    res.setHeader('Content-Type', 'text/event-stream');
    res.setHeader('Cache-Control', 'no-cache');
    res.setHeader('Connection', 'keep-alive');
    res.setHeader('X-Accel-Buffering', 'no');

    // Send initial connection message
    res.write(`data: ${JSON.stringify({ 
      type: 'info', 
      message: 'Połączono ze streamem postępu ZLP', 
      timestamp: new Date().toISOString(),
      step: 'connected',
      current: 0,
      total: 100,
    })}\n\n`);

    // If session already completed, send final result
    if (session.status === 'completed' || session.status === 'failed') {
      res.write(`data: ${JSON.stringify({
        step: 'done',
        current: 100,
        total: 100,
        message: session.status === 'completed' 
          ? 'Generowanie zakończone' 
          : `Błąd: ${session.error}`,
        type: session.status === 'completed' ? 'success' : 'error',
        timestamp: new Date().toISOString(),
        result: session.result,
      })}\n\n`);
      return res.end();
    }

    // Create listener for progress updates
    const progressListener = (data: any) => {
      res.write(`data: ${JSON.stringify(data)}\n\n`);
      
      // Close connection when done
      if (data.step === 'done' || data.step === 'error') {
        res.end();
      }
    };

    zlpProgressEmitter.on(sessionId, progressListener);

    // Cleanup on client disconnect
    req.on('close', () => {
      zlpProgressEmitter.removeListener(sessionId, progressListener);
      console.log(`📡 ZLP SSE client disconnected for session: ${sessionId}`);
    });

    console.log(`📡 ZLP SSE client connected for session: ${sessionId}`);
  });

  // GET /api/production/planning/zlp-status/:sessionId - Get session status
  app.get("/api/production/planning/zlp-status/:sessionId", requirePermission('view_production'), (req, res) => {
    const { sessionId } = req.params;
    const session = generationSessions.get(sessionId);
    
    if (!session) {
      return res.status(404).json({ message: "Session not found" });
    }

    res.json({
      sessionId,
      status: session.status,
      planId: session.planId,
      startedAt: session.startedAt,
      completedAt: session.completedAt,
      result: session.result,
      error: session.error,
    });
  });

  // GET /api/production/planning/plans/:id/lines - Get all lines for a plan
  app.get("/api/production/planning/plans/:id/lines", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const lines = await planningService.getPlanLines(pool, planId);
      res.json(lines);
    } catch (error) {
      console.error("Error fetching plan lines:", error);
      res.status(500).json({ message: "Failed to fetch plan lines" });
    }
  });

  // GET /api/production/planning/lines/:id - Get plan line by ID
  app.get("/api/production/planning/lines/:id", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const line = await planningService.getPlanLineById(pool, id);
      if (!line) {
        return res.status(404).json({ message: "Plan line not found" });
      }

      res.json(line);
    } catch (error) {
      console.error("Error fetching plan line:", error);
      res.status(500).json({ message: "Failed to fetch plan line" });
    }
  });

  // POST /api/production/planning/lines - Create new plan line
  app.post("/api/production/planning/lines", requirePermission('manage_production'), async (req, res) => {
    try {
      const data = createPlanLineSchema.parse(req.body);
      
      // Check for duplicate based on metadata type
      if (data.planId) {
        let duplicateCheck;
        
        // For components from sets: check by set_id + component_id + order_number
        // Note: Same component from DIFFERENT orders = DIFFERENT products (user requirement)
        if (data.metadata?.set_id && data.metadata?.component_id) {
          if (data.metadata.order_number) {
            // New behavior: Check only rows with same order_number
            // This allows same component from different orders to be added separately
            duplicateCheck = await pool.query(
              `SELECT id FROM production.production_plan_lines 
               WHERE plan_id = $1 
               AND (metadata->>'set_id')::int = $2
               AND (metadata->>'component_id')::int = $3
               AND (metadata->>'order_number') = $4
               AND deleted_at IS NULL
               LIMIT 1`,
              [data.planId, data.metadata.set_id, data.metadata.component_id, data.metadata.order_number]
            );
          } 
          // Legacy behavior: Check only set_id + component_id (for old plan_lines without order_number)
          else {
            duplicateCheck = await pool.query(
              `SELECT id FROM production.production_plan_lines 
               WHERE plan_id = $1 
               AND (metadata->>'set_id')::int = $2
               AND (metadata->>'component_id')::int = $3
               AND deleted_at IS NULL
               LIMIT 1`,
              [data.planId, data.metadata.set_id, data.metadata.component_id]
            );
          }
        }
        // For regular order items: check by catalog_product_id + order_number
        // Note: Same product from DIFFERENT orders = DIFFERENT products (user requirement)
        // Only check for order_demand source type (allow manual/forecast/buffer to add same product)
        else if (data.productId && data.sourceType === 'order_demand') {
          if (data.metadata?.order_number) {
            // Strict matching: Check only rows with same order_number
            // This allows same product from different orders to be added separately
            duplicateCheck = await pool.query(
              `SELECT id FROM production.production_plan_lines 
               WHERE plan_id = $1 
               AND product_id = $2
               AND source_type = 'order_demand'
               AND (metadata->>'order_number') = $3
               AND deleted_at IS NULL
               LIMIT 1`,
              [data.planId, data.productId, data.metadata.order_number]
            );
          } else {
            // Fallback: Check only product_id (should not happen as all new items have order_number)
            duplicateCheck = await pool.query(
              `SELECT id FROM production.production_plan_lines 
               WHERE plan_id = $1 
               AND product_id = $2
               AND source_type = 'order_demand'
               AND deleted_at IS NULL
               LIMIT 1`,
              [data.planId, data.productId]
            );
          }
        }
        
        if (duplicateCheck && duplicateCheck.rows.length > 0) {
          return res.status(409).json({ 
            message: "Product already in the plan" 
          });
        }
      }
      
      // Use split function to create individual records for each unit
      const quantity = data.quantity || 1;
      
      if (quantity > 1) {
        // Create multiple plan lines (one per unit) for individual tracking
        const result = await planningService.createPlanLinesWithSplit(pool, {
          ...data,
          plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : null,
          plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : null,
        });
        
        // Return summary for multiple lines
        res.status(201).json({
          message: `Utworzono ${result.planLines.length} osobnych rekordów`,
          planLines: result.planLines,
          totalReservations: result.totalReservations,
          count: result.planLines.length,
        });
      } else {
        // Single unit - use original function
        const result = await planningService.createPlanLine(pool, {
          ...data,
          plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : null,
          plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : null,
        });
        
        // Return plan line with reservation info for frontend display
        res.status(201).json({
          ...result.planLine,
          reservationInfo: result.reservationInfo,
        });
      }
    } catch (error: any) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if (error?.code === '23503') {
        return res.status(400).json({ message: "Invalid plan ID or product ID" });
      }
      // Handle duplicate detection error from createPlanLine
      if (error?.message?.includes('już istnieje w planie') || error?.message?.includes('already exists in plan')) {
        return res.status(409).json({ message: error.message });
      }
      console.error("Error creating plan line:", error);
      res.status(500).json({ message: "Failed to create plan line" });
    }
  });

  // PATCH /api/production/planning/lines/:id - Update plan line
  app.patch("/api/production/planning/lines/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const data = createPlanLineSchema.partial().parse(req.body);
      const line = await planningService.updatePlanLine(pool, id, {
        ...data,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : undefined,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : undefined,
      });
      
      if (!line) {
        return res.status(404).json({ message: "Plan line not found" });
      }

      res.json(line);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      console.error("Error updating plan line:", error);
      res.status(500).json({ message: "Failed to update plan line" });
    }
  });

  // DELETE /api/production/planning/lines/:id - Delete plan line

  // POST /api/production/planning/lines/:id/reserve - Manually reserve packed product
  app.post("/api/production/planning/lines/:id/reserve", requirePermission('manage_production'), async (req, res) => {
    try {
      const lineId = parseInt(req.params.id);
      if (isNaN(lineId)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const { quantity: requestedQty } = req.body;
      
      const client = await pool.connect();
      try {
        await client.query('BEGIN');
        
        // Get line info with row lock
        const lineResult = await client.query(
          `SELECT product_id, quantity, reserved_quantity FROM production.production_plan_lines WHERE id = $1 FOR UPDATE`,
          [lineId]
        );
        
        if (lineResult.rows.length === 0) {
          await client.query('ROLLBACK');
          return res.status(404).json({ message: "Plan line not found" });
        }
        
        const { product_id, quantity: lineQty, reserved_quantity: currentReserved } = lineResult.rows[0];
        
        // Check if it's a packed product with row lock
        const packedProductResult = await client.query(
          `SELECT id, product_sku, quantity, reserved_quantity FROM warehouse.packed_products WHERE catalog_product_id = $1 FOR UPDATE`,
          [product_id]
        );
        
        if (packedProductResult.rows.length === 0) {
          await client.query('ROLLBACK');
          return res.status(400).json({ message: "This product is not a packed product" });
        }
        
        const packedProduct = packedProductResult.rows[0];
        const availableQty = packedProduct.quantity - packedProduct.reserved_quantity;
        
        if (availableQty <= 0) {
          await client.query('ROLLBACK');
          return res.status(400).json({ message: "No available stock to reserve" });
        }
        
        // Calculate quantity to reserve - use requested or calculate needed amount
        const neededQty = lineQty - (currentReserved || 0);
        const qtyToReserve = Math.min(requestedQty || neededQty, availableQty, neededQty);
        
        if (qtyToReserve <= 0) {
          await client.query('ROLLBACK');
          return res.status(400).json({ message: "Invalid quantity to reserve" });
        }
        
        // Get plan info for audit history
        const planResult = await client.query(
          `SELECT pp.id as plan_id, pp.name as plan_name 
           FROM production.production_plan_lines ppl 
           JOIN production.production_plans pp ON pp.id = ppl.plan_id 
           WHERE ppl.id = $1`,
          [lineId]
        );
        const planInfo = planResult.rows[0];
        const username = (req.user as any)?.username || 'system';
        
        // Build audit entry for reservation
        const auditEntry = {
          action: 'reserved',
          timestamp: new Date().toISOString(),
          performedBy: username,
          previousStatus: 'available',
          planId: planInfo?.plan_id,
          planLineId: lineId
        };
        
        // Reserve individual items using FIFO (oldest first) with audit history
        const reservedItemsResult = await client.query(
          `UPDATE warehouse.packed_product_items
           SET status = 'reserved',
               reserved_for_plan_line_id = $1,
               reserved_at = CURRENT_TIMESTAMP,
               updated_at = CURRENT_TIMESTAMP,
               metadata = COALESCE(metadata, '{}'::jsonb) || jsonb_build_object(
                 'auditHistory', COALESCE(metadata->'auditHistory', '[]'::jsonb) || $4::jsonb
               )
           WHERE id IN (
             SELECT id FROM warehouse.packed_product_items
             WHERE packed_product_id = $2
               AND status = 'available'
             ORDER BY packed_at ASC
             LIMIT $3
             FOR UPDATE SKIP LOCKED
           )
           RETURNING id, serial_number`,
          [lineId, packedProduct.id, qtyToReserve, JSON.stringify([auditEntry])]
        );
        
        const actualReservedCount = reservedItemsResult.rows.length;
        
        if (actualReservedCount === 0) {
          await client.query('ROLLBACK');
          return res.status(400).json({ message: "No available items to reserve" });
        }
        
        // Update summary table
        await client.query(
          `UPDATE warehouse.packed_products 
           SET reserved_quantity = reserved_quantity + $1 
           WHERE id = $2`,
          [actualReservedCount, packedProduct.id]
        );
        
        // Update plan line reserved_quantity
        await client.query(
          `UPDATE production.production_plan_lines
           SET reserved_quantity = reserved_quantity + $1, updated_at = CURRENT_TIMESTAMP
           WHERE id = $2`,
          [actualReservedCount, lineId]
        );
        
        // Auto-zwolnij rezerwacje formatek W TEJ SAMEJ TRANSAKCJI (produkt spakowany jest zarezerwowany, formatki niepotrzebne)
        const formatkaRelease = await planningService.releaseFormatkiReservationsForPlanLine(client, lineId);
        
        await client.query('COMMIT');
        
        const serialNumbers = reservedItemsResult.rows.map((r: any) => r.serial_number);
        console.log(`📦 [RĘCZNA REZERWACJA] Zarezerwowano ${actualReservedCount} szt. produktu ${packedProduct.product_sku} dla ZLP line #${lineId}`);
        console.log(`📦 [ZAREZERWOWANE SZTUKI]: ${serialNumbers.join(', ')}`);
        
        res.json({ 
          message: "Reservation created successfully",
          productSku: packedProduct.product_sku,
          quantityReserved: actualReservedCount,
          totalReserved: (currentReserved || 0) + actualReservedCount,
          reservedSerialNumbers: serialNumbers,
          formatkiReleased: formatkaRelease.releasedCount,
          formatkiLogs: formatkaRelease.logs
        });
      } catch (error) {
        await client.query('ROLLBACK');
        throw error;
      } finally {
        client.release();
      }
    } catch (error) {
      console.error("Error creating reservation:", error);
      res.status(500).json({ message: "Failed to create reservation" });
    }
  });

  // POST /api/production/planning/lines/:id/unreserve - Manually release packed product reservation
  app.post("/api/production/planning/lines/:id/unreserve", requirePermission('manage_production'), async (req, res) => {
    try {
      const lineId = parseInt(req.params.id);
      if (isNaN(lineId)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const client = await pool.connect();
      try {
        await client.query('BEGIN');
        
        // Get line info
        const lineResult = await client.query(
          `SELECT product_id, quantity, reserved_quantity FROM production.production_plan_lines WHERE id = $1`,
          [lineId]
        );
        
        if (lineResult.rows.length === 0) {
          await client.query('ROLLBACK');
          return res.status(404).json({ message: "Plan line not found" });
        }
        
        const { product_id, quantity, reserved_quantity } = lineResult.rows[0];
        
        // Check if it's a packed product
        const packedProductResult = await client.query(
          `SELECT id, product_sku, reserved_quantity FROM warehouse.packed_products WHERE catalog_product_id = $1`,
          [product_id]
        );
        
        if (packedProductResult.rows.length === 0) {
          await client.query('ROLLBACK');
          return res.status(400).json({ message: "This product is not a packed product" });
        }
        
        const packedProduct = packedProductResult.rows[0];
        
        // Get plan info for audit history
        const planResult = await client.query(
          `SELECT pp.id as plan_id, pp.name as plan_name 
           FROM production.production_plan_lines ppl 
           JOIN production.production_plans pp ON pp.id = ppl.plan_id 
           WHERE ppl.id = $1`,
          [lineId]
        );
        const planInfo = planResult.rows[0];
        const username = (req.user as any)?.username || 'system';
        
        // Build audit entry for unreservation
        const auditEntry = {
          action: 'unreserved',
          timestamp: new Date().toISOString(),
          performedBy: username,
          previousStatus: 'reserved',
          planId: planInfo?.plan_id,
          planLineId: lineId
        };
        
        // Zwolnij pojedyncze sztuki zarezerwowane dla tej linii planu z historią audytu
        const releasedItemsResult = await client.query(
          `UPDATE warehouse.packed_product_items
           SET status = 'available',
               reserved_for_plan_line_id = NULL,
               reserved_at = NULL,
               updated_at = CURRENT_TIMESTAMP,
               metadata = COALESCE(metadata, '{}'::jsonb) || jsonb_build_object(
                 'auditHistory', COALESCE(metadata->'auditHistory', '[]'::jsonb) || $2::jsonb
               )
           WHERE reserved_for_plan_line_id = $1
             AND status = 'reserved'
           RETURNING id, serial_number`,
          [lineId, JSON.stringify([auditEntry])]
        );
        
        const releasedCount = releasedItemsResult.rows.length;
        
        // Aktualizuj licznik w tabeli agregacyjnej (dla kompatybilności wstecznej)
        await client.query(
          `UPDATE warehouse.packed_products 
           SET reserved_quantity = GREATEST(0, reserved_quantity - $1) 
           WHERE id = $2`,
          [releasedCount, packedProduct.id]
        );
        
        // Zero out plan line reserved_quantity
        await client.query(
          `UPDATE production.production_plan_lines
           SET reserved_quantity = 0, updated_at = CURRENT_TIMESTAMP
           WHERE id = $1`,
          [lineId]
        );
        
        // Auto-zarezerwuj formatki z BOM W TEJ SAMEJ TRANSAKCJI (produkt spakowany zwolniony, trzeba wyprodukować z formatek)
        const userId = (req.user as any)?.id;
        const formatkaReserve = await planningService.reserveFormatkiForPlanLine(client, lineId, userId);
        
        await client.query('COMMIT');
        
        const serialNumbers = releasedItemsResult.rows.map((r: any) => r.serial_number);
        console.log(`📦 [RĘCZNE ZWOLNIENIE] Zwolniono ${releasedCount} szt. rezerwacji produktu ${packedProduct.product_sku} dla ZLP line #${lineId}`);
        console.log(`📦 [ZWOLNIONE SZTUKI]: ${serialNumbers.join(', ')}`);
        
        res.json({ 
          message: "Reservation released successfully",
          productSku: packedProduct.product_sku,
          quantityReleased: quantity,
          previousReserved: packedProduct.reserved_quantity,
          formatkiReserved: formatkaReserve.success,
          formatkiLogs: formatkaReserve.logs
        });
      } catch (error) {
        await client.query('ROLLBACK');
        throw error;
      } finally {
        client.release();
      }
    } catch (error) {
      console.error("Error releasing reservation:", error);
      res.status(500).json({ message: "Failed to release reservation" });
    }
  });

  // GET /api/production/planning/lines/:id/bom-formatki - Get BOM formatki status for plan line
  app.get("/api/production/planning/lines/:id/bom-formatki", requirePermission('view_production'), async (req, res) => {
    try {
      const lineId = parseInt(req.params.id);
      if (isNaN(lineId)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      // Get plan line with BOM and packed product reservation status
      const lineResult = await pool.query(`
        SELECT ppl.id, ppl.product_id, ppl.bom_id, ppl.quantity as line_quantity,
               ppl.reserved_quantity as packed_reserved_qty,
               cp.sku as product_sku, cp.title as product_title,
               wpp.id as packed_product_id
        FROM production.production_plan_lines ppl
        JOIN catalog.products cp ON cp.id = ppl.product_id
        LEFT JOIN warehouse.packed_products wpp ON wpp.catalog_product_id = ppl.product_id
        WHERE ppl.id = $1 AND ppl.deleted_at IS NULL
      `, [lineId]);

      if (lineResult.rows.length === 0) {
        return res.status(404).json({ message: "Plan line not found" });
      }

      const line = lineResult.rows[0];
      
      // Check if product is reserved from packed_products warehouse (already produced)
      const isPackedProductReserved = line.packed_product_id && (line.packed_reserved_qty || 0) > 0;
      
      if (!line.bom_id) {
        return res.json({
          lineId,
          productSku: line.product_sku,
          productTitle: line.product_title,
          bomId: null,
          isPackedProductReserved,
          formatki: [],
          summary: { total: 0, produced: 0, toProduction: 0 }
        });
      }

      // Get BOM components (formatki)
      const componentsResult = await pool.query(`
        SELECT 
          pc.id as component_id,
          pc.generated_name,
          pc.component_type,
          pc.calculated_length,
          pc.calculated_width,
          pc.thickness,
          pc.quantity as component_qty,
          pc.color,
          pc.edging_pattern
        FROM bom.product_components pc
        WHERE pc.product_bom_id = $1
        ORDER BY pc.id
      `, [line.bom_id]);

      // Get color dictionary for badge colors
      const colorsResult = await pool.query(`
        SELECT code, name, color as hex_color
        FROM product_creator.dictionaries
        WHERE dictionary_type = 'color' AND is_active = true
      `);
      
      const colorDict = new Map<string, string>();
      for (const c of colorsResult.rows) {
        colorDict.set(c.code?.toUpperCase(), c.hex_color);
        colorDict.set(c.name?.toUpperCase(), c.hex_color);
      }

      // Get existing reservations for this plan line (ALL statuses for history)
      // UWAGA: Formatki są w warehouse.stock_panels, NIE w warehouse.materials
      const reservationsResult = await pool.query(`
        SELECT 
          r.id as reservation_id,
          r.product_sku,
          r.quantity_reserved,
          r.quantity_consumed,
          r.status,
          r.reserved_at,
          r.consumed_at,
          r.cancelled_at,
          r.notes,
          sp.id as material_id,
          sp.generated_name as material_name,
          sp.generated_name as internal_code
        FROM production.production_buffer_reservations r
        LEFT JOIN warehouse.stock_panels sp ON sp.generated_name = r.product_sku
        WHERE r.zlp_item_id = $1
        ORDER BY r.reserved_at DESC
      `, [lineId]);

      // Separate active reservations (for status) and all reservations (for history)
      const activeReservationsMap = new Map();
      const reservationHistoryMap = new Map<string, any[]>();
      
      for (const r of reservationsResult.rows) {
        // Track all reservations as history per SKU
        const historyList = reservationHistoryMap.get(r.product_sku) || [];
        historyList.push({
          id: r.reservation_id,
          status: r.status,
          quantityReserved: parseFloat(r.quantity_reserved),
          quantityConsumed: r.quantity_consumed ? parseFloat(r.quantity_consumed) : null,
          reservedAt: r.reserved_at,
          consumedAt: r.consumed_at,
          cancelledAt: r.cancelled_at,
          notes: r.notes
        });
        reservationHistoryMap.set(r.product_sku, historyList);
        
        // Only keep ACTIVE reservation in the main map for status determination
        if (r.status === 'ACTIVE') {
          activeReservationsMap.set(r.product_sku, r);
        }
      }

      // Build formatki list with production status
      // KEY LOGIC: If packed product is reserved from warehouse, all formatki are ALREADY PRODUCED
      // Otherwise, check individual formatki reservations from formatki warehouse
      const formatki = [];
      let totalComponents = 0;
      let producedCount = 0;

      for (const comp of componentsResult.rows) {
        // Check if this is a formatka (has dimensions)
        const isFormatka = comp.calculated_length && comp.calculated_width;
        if (!isFormatka) continue;

        totalComponents++;
        const componentQty = (comp.component_qty || 1) * line.line_quantity;

        // Extract color from generated_name (last part after last dash)
        const nameParts = (comp.generated_name || '').split('-');
        const color = nameParts[nameParts.length - 1]?.toUpperCase() || comp.color || 'UNKNOWN';

        // Helper function to fully normalize a string for comparison
        const normalizeForMatch = (str: string): string => {
          return str
            .toLowerCase()
            .replace(/formatka-/gi, '') // Remove ALL occurrences of formatka-
            .replace(/×/g, 'x')        // Normalize × to x
            .replace(/[-_\s]+/g, '')   // Remove hyphens, underscores, spaces
            .trim();
        };
        
        // Extract key parts from a SKU/component name
        const extractKeyParts = (str: string): { dims: string | null; color: string | null } => {
          const normalized = str.toLowerCase().replace(/×/g, 'x');
          
          // Extract dimensions: e.g., "500x300"
          const dimsMatch = normalized.match(/(\d+)x(\d+)/i);
          const dims = dimsMatch ? `${dimsMatch[1]}x${dimsMatch[2]}` : null;
          
          // Extract color: last non-numeric token after removing dimensions
          const withoutDims = normalized.replace(/\d+x\d+/g, '').replace(/[-_×\s]+/g, ' ').trim();
          const tokens = withoutDims.split(/\s+/).filter(t => t && !/^\d+$/.test(t));
          const color = tokens.length > 0 ? tokens[tokens.length - 1].replace(/formatka/gi, '') : null;
          
          return { dims, color: color || null };
        };
        
        // Helper function to check if SKU matches component name
        const skuMatchesComponent = (sku: string, componentName: string): boolean => {
          if (!componentName || !sku) return false;
          
          const normalizedSku = normalizeForMatch(sku);
          const normalizedComp = normalizeForMatch(componentName);
          
          // Direct match after full normalization
          if (normalizedSku === normalizedComp) return true;
          
          // Bidirectional includes check with normalized strings
          if (normalizedSku.includes(normalizedComp)) return true;
          if (normalizedComp.includes(normalizedSku)) return true;
          
          // Extract and compare key parts (dimensions + color)
          const skuParts = extractKeyParts(sku);
          const compParts = extractKeyParts(componentName);
          
          // Must have matching dimensions
          if (skuParts.dims && compParts.dims && skuParts.dims === compParts.dims) {
            // If both have colors, they must match
            if (skuParts.color && compParts.color) {
              return skuParts.color === compParts.color;
            }
            // If only one has color or neither, consider it a match
            return true;
          }
          
          return false;
        };

        // FIRST: Check history map for matching SKUs (includes all statuses)
        let reservationHistory: any[] = [];
        let matchedSku: string | null = null;
        for (const entry of Array.from(reservationHistoryMap.entries())) {
          const [sku, history] = entry;
          if (skuMatchesComponent(sku, comp.generated_name)) {
            reservationHistory = history;
            matchedSku = sku;
            break;
          }
        }
        
        // SECOND: Check if there's an ACTIVE reservation for this component
        let formatkiReservation = null;
        if (matchedSku && activeReservationsMap.has(matchedSku)) {
          formatkiReservation = activeReservationsMap.get(matchedSku);
        } else {
          // Fallback: check all active reservations
          for (const entry of Array.from(activeReservationsMap.entries())) {
            const [sku, activeRes] = entry;
            if (skuMatchesComponent(sku, comp.generated_name)) {
              formatkiReservation = activeRes;
              // If we didn't find history via matched SKU, get it now
              if (reservationHistory.length === 0) {
                reservationHistory = reservationHistoryMap.get(sku) || [];
              }
              break;
            }
          }
        }

        // Determine production status:
        // - If packed product is reserved → formatka is PRODUCED (part of finished product)
        // - If formatka is reserved from formatki warehouse → formatka is RESERVED (ready for production)
        // - Otherwise → formatka needs TO BE PRODUCED
        let status: 'produced' | 'reserved' | 'to_production';
        if (isPackedProductReserved) {
          status = 'produced';
          producedCount++;
        } else if (formatkiReservation) {
          status = 'reserved';
          producedCount++; // Reserved formatki are also "ready", not needing production
        } else {
          status = 'to_production';
        }

        // Determine routing variant (simplified)
        let routingVariant = 'SUROWA';
        if (color && !['SUROWA', 'SUROWY', 'RAW'].includes(color.toUpperCase())) {
          routingVariant = 'OKLEINOWANA';
        }

        // Get color hex from dictionary
        const colorHex = colorDict.get(color) || null;

        formatki.push({
          componentId: comp.component_id,
          generatedName: comp.generated_name,
          componentType: comp.component_type || nameParts[0] || 'PANEL',
          length: parseFloat(comp.calculated_length),
          width: parseFloat(comp.calculated_width),
          thickness: comp.thickness ? parseFloat(comp.thickness) : 18,
          color,
          colorHex,
          quantity: componentQty,
          edgingPattern: comp.edging_pattern,
          routingVariant,
          status,
          formatkiReservation: formatkiReservation ? {
            id: formatkiReservation.reservation_id,
            sku: formatkiReservation.product_sku,
            quantityReserved: parseFloat(formatkiReservation.quantity_reserved),
            materialId: formatkiReservation.material_id,
            materialName: formatkiReservation.material_name,
            reservedAt: formatkiReservation.reserved_at
          } : null,
          reservationHistory: reservationHistory.length > 0 ? reservationHistory : null
        });
      }

      // Collect all reservations history for this line (not per-component, but for the whole line)
      const allLineReservations: any[] = [];
      for (const entry of Array.from(reservationHistoryMap.entries())) {
        const [sku, history] = entry;
        for (const item of history) {
          allLineReservations.push({
            ...item,
            sku
          });
        }
      }
      // Sort by reservedAt descending
      allLineReservations.sort((a, b) => 
        new Date(b.reservedAt).getTime() - new Date(a.reservedAt).getTime()
      );

      res.json({
        lineId,
        productSku: line.product_sku,
        productTitle: line.product_title,
        bomId: line.bom_id,
        isPackedProductReserved,
        formatki,
        lineReservationHistory: allLineReservations.length > 0 ? allLineReservations : null,
        summary: {
          total: totalComponents,
          produced: producedCount,
          toProduction: totalComponents - producedCount
        }
      });
    } catch (error) {
      console.error("Error fetching BOM formatki:", error);
      res.status(500).json({ message: "Failed to fetch BOM formatki" });
    }
  });

  app.delete("/api/production/planning/lines/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }


      
      const success = await planningService.deletePlanLine(pool, id);
      if (!success) {
        return res.status(404).json({ message: "Plan line not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting plan line:", error);
      res.status(500).json({ message: "Failed to delete plan line" });
    }
  });

  // POST /api/production/planning/lines/:id/transfer - Transfer plan line to another plan (keeps reservations)
  app.post("/api/production/planning/lines/:id/transfer", requirePermission('manage_production'), async (req, res) => {
    try {
      const lineId = parseInt(req.params.id);
      if (isNaN(lineId)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const { targetPlanId } = req.body;
      if (!targetPlanId || isNaN(parseInt(targetPlanId))) {
        return res.status(400).json({ message: "Target plan ID is required" });
      }

      const transferredLine = await planningService.transferPlanLine(pool, lineId, parseInt(targetPlanId));
      
      res.json({
        message: "Linia została przeniesiona do nowego planu",
        line: transferredLine
      });
    } catch (error: any) {
      console.error("Error transferring plan line:", error);
      
      if (error.message?.includes('nie istnieje') || error.message?.includes('not found')) {
        return res.status(404).json({ message: error.message });
      }
      if (error.message?.includes('już istnieje') || error.message?.includes('already')) {
        return res.status(409).json({ message: error.message });
      }
      if (error.message?.includes('już w tym planie')) {
        return res.status(400).json({ message: error.message });
      }
      
      res.status(500).json({ message: error.message || "Failed to transfer plan line" });
    }
  });

  // POST /api/production/planning/lines/:id/generate-batches - Generate component batches for plan line
  app.post("/api/production/planning/lines/:id/generate-batches", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const batches = await batchEngine.generateBatchesForPlanLine(pool, id);
      
      res.status(201).json({
        message: `Generated ${batches.length} batches`,
        batches: batches.map(b => b.batch),
      });
    } catch (error) {
      console.error("Error generating batches:", error);
      
      if (error instanceof Error) {
        if (error.message.includes('not found')) {
          return res.status(404).json({ message: error.message });
        }
        if (error.message.includes('No active BOM')) {
          return res.status(400).json({ message: error.message });
        }
        if (error.message.includes('No components found')) {
          return res.status(400).json({ message: error.message });
        }
      }
      
      res.status(500).json({ message: "Failed to generate batches" });
    }
  });

  // GET /api/production/planning/lines/:id/batches - Get all batches for a plan line
  app.get("/api/production/planning/lines/:id/batches", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid line ID" });
      }

      const batches = await batchEngine.getBatchesForPlanLine(pool, id);
      res.json(batches);
    } catch (error) {
      console.error("Error fetching batches:", error);
      res.status(500).json({ message: "Failed to fetch batches" });
    }
  });

  // GET /api/production/planning/plans/:id/available-orders - Get available marketplace orders for production planning
  app.get("/api/production/planning/plans/:id/available-orders", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // Coerce empty strings to undefined before parsing (URL params issue)
      const normalizedQuery = Object.fromEntries(
        Object.entries(req.query).map(([key, value]) => [key, value === '' ? undefined : value])
      );

      // Parse and validate query parameters
      const filters = availableOrdersFiltersSchema.parse(normalizedQuery);

      // Parse search tokens into structured filters
      const parsedTokens = parseSearchTokens(filters.search);

      // Whitelist map for safe sorting (prevents SQL injection)
      const sortColumnMap: Record<string, string> = {
        'order_date': 'o.order_date',
        'order_number': 'o.order_number',
        'buyer_name': 'LOWER(o.buyer_last_name || \' \' || o.buyer_first_name)',
        'total_amount': 'o.total_to_pay_amount',
        'product_sku': 'MIN(cp.sku)',
      };

      const orderByColumn = sortColumnMap[filters.sortBy];
      const orderByDirection = filters.sortOrder.toUpperCase(); // 'ASC' or 'DESC'

      // Build WHERE clauses for filtering - use validated filters
      const conditions: string[] = ['1=1'];
      const params: any[] = [];
      let paramIndex = 1; // Start at 1 (no planId in params for count query)

      // Marketplace filter from parsed tokens (supports multiple)
      if (parsedTokens.marketplaces.length > 0) {
        const placeholders = parsedTokens.marketplaces.map((_, i) => `$${paramIndex + i}`).join(', ');
        conditions.push(`o.source IN (${placeholders})`);
        params.push(...parsedTokens.marketplaces);
        paramIndex += parsedTokens.marketplaces.length;
      }

      // Payment status filter from parsed tokens (supports multiple)
      if (parsedTokens.paymentStatuses.length > 0) {
        const placeholders = parsedTokens.paymentStatuses.map((_, i) => `$${paramIndex + i}`).join(', ');
        conditions.push(`o.payment_status IN (${placeholders})`);
        params.push(...parsedTokens.paymentStatuses);
        paramIndex += parsedTokens.paymentStatuses.length;
      }

      // Search terms filter (each term AND-ed together)
      if (parsedTokens.searchTerms.length > 0) {
        parsedTokens.searchTerms.forEach(term => {
          conditions.push(`(
            o.order_number ILIKE $${paramIndex} OR
            o.buyer_first_name ILIKE $${paramIndex} OR
            o.buyer_last_name ILIKE $${paramIndex} OR
            o.buyer_email ILIKE $${paramIndex} OR
            COALESCE(mp.sku, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.sku, ps.sku, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.product_type, sm.product_group, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.title, ps.title, '')::text ILIKE $${paramIndex} OR
            COALESCE(oi.name, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.color, ps.color, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.doors, sm.doors, '')::text ILIKE $${paramIndex} OR
            COALESCE(cp.legs, sm.legs, '')::text ILIKE $${paramIndex} OR
            EXISTS (
              SELECT 1 FROM unnest(COALESCE(cp.color_options, ps.color_options, ARRAY[]::text[])) AS co 
              WHERE co ILIKE $${paramIndex}
            )
          )`);
          params.push(`%${term}%`);
          paramIndex++;
        });
      }

      // Order number filter
      if (filters.orderNumber) {
        conditions.push(`o.order_number ILIKE $${paramIndex}`);
        params.push(`%${filters.orderNumber}%`);
        paramIndex++;
      }

      // Customer name filter
      if (filters.customerName) {
        conditions.push(`(
          o.buyer_first_name ILIKE $${paramIndex} OR
          o.buyer_last_name ILIKE $${paramIndex}
        )`);
        params.push(`%${filters.customerName}%`);
        paramIndex++;
      }

      // SKU filter
      if (filters.sku) {
        conditions.push(`(
          COALESCE(mp.sku, '')::text ILIKE $${paramIndex} OR
          COALESCE(cp.sku, ps.sku, '')::text ILIKE $${paramIndex}
        )`);
        params.push(`%${filters.sku}%`);
        paramIndex++;
      }

      // Color filter
      if (filters.color) {
        conditions.push(`COALESCE(cp.color, ps.color, '') ILIKE $${paramIndex}`);
        params.push(`%${filters.color}%`);
        paramIndex++;
      }

      // Length filters
      if (filters.minLength) {
        conditions.push(`cp.length >= $${paramIndex}`);
        params.push(filters.minLength);
        paramIndex++;
      }
      if (filters.maxLength) {
        conditions.push(`cp.length <= $${paramIndex}`);
        params.push(filters.maxLength);
        paramIndex++;
      }

      // Width filters
      if (filters.minWidth) {
        conditions.push(`cp.width >= $${paramIndex}`);
        params.push(filters.minWidth);
        paramIndex++;
      }
      if (filters.maxWidth) {
        conditions.push(`cp.width <= $${paramIndex}`);
        params.push(filters.maxWidth);
        paramIndex++;
      }

      // Legacy marketplace filter (kept for backward compatibility if passed explicitly)
      if (filters.marketplace && filters.marketplace !== 'all' && parsedTokens.marketplaces.length === 0) {
        conditions.push(`o.source = $${paramIndex}`);
        params.push(filters.marketplace);
        paramIndex++;
      }

      // Filter: Show only catalog-linked products or sets
      if (filters.showCatalogLinked === 'true') {
        conditions.push(`(cp.id IS NOT NULL OR ps.id IS NOT NULL)`);
      }

      // Filter: Show only products that are sets (linked to catalog.product_sets)
      if (filters.showSetsOnly === 'true') {
        conditions.push(`ps.id IS NOT NULL`);
      }

      // Filter: Show only items that are already in any plan (exclude soft-deleted lines)
      // Match specific order items: must match both order_number AND (product_id OR set_id)
      if (filters.showInPlans === 'true') {
        conditions.push(`(
          EXISTS (
            SELECT 1 FROM production.production_plan_lines pl 
            WHERE pl.deleted_at IS NULL
              AND pl.metadata->>'order_number' = o.order_number
              AND (
                pl.product_id = cp.id
                OR (pl.metadata->>'set_id')::int = ps.id
              )
          )
        )`);
      }

      // Date filter - filter orders by date
      if (filters.dateFilter && filters.dateFilter !== 'all') {
        switch (filters.dateFilter) {
          case 'today':
            conditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw')`);
            break;
          case 'yesterday':
            conditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw' - INTERVAL '1 day')`);
            break;
          case 'day-before':
            conditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw' - INTERVAL '2 days')`);
            break;
          case 'custom-days':
            if (filters.customDays && filters.customDays > 0) {
              conditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw' - INTERVAL '${filters.customDays} days')`);
            }
            break;
        }
      }

      const whereClause = conditions.join(' AND ');

      // Step 1: Get total count for pagination (with same core joins to ensure cp and ps aliases exist)
      // Note: catalog products can be linked via product_platform_data OR directly via oi.catalog_product_id
      const countResult = await pool.query(`
        SELECT COUNT(DISTINCT o.id)::int as total
        FROM commerce.orders o
        LEFT JOIN commerce.order_items oi ON o.id = oi.order_id
        LEFT JOIN commerce.marketplace_products mp ON oi.offer_external_id = mp.offer_external_id AND mp.source = o.source
        LEFT JOIN LATERAL (
          SELECT DISTINCT ON (external_id) *
          FROM catalog.product_platform_data
          WHERE external_id = oi.offer_external_id
          ORDER BY external_id, CASE WHEN link_type = 'product' THEN 0 ELSE 1 END, id DESC
        ) ppd ON oi.catalog_product_id IS NULL AND oi.catalog_set_id IS NULL
        LEFT JOIN catalog.products cp ON cp.id = COALESCE(oi.catalog_product_id, ppd.product_id)
        LEFT JOIN product_creator.product_sets ps ON ps.id = COALESCE(oi.catalog_set_id, ppd.set_id)
        LEFT JOIN product_creator.set_matrices sm ON ps.set_matrix_id = sm.id
        WHERE ${whereClause}
          AND oi.id IS NOT NULL
      `, params);

      const total = countResult.rows[0]?.total || 0;

      // Step 2: Build WHERE clause for data query with adjusted param indices
      // Data query needs planId as $1, so we rebuild conditions with paramIndex starting from 2
      const dataConditions: string[] = ['1=1'];
      const dataParams: any[] = [planId]; // Start with planId at $1
      let dataParamIndex = 2; // Start filter params at $2

      // Rebuild all filter conditions with adjusted indices
      // Marketplace filter from parsed tokens (supports multiple)
      if (parsedTokens.marketplaces.length > 0) {
        const placeholders = parsedTokens.marketplaces.map((_, i) => `$${dataParamIndex + i}`).join(', ');
        dataConditions.push(`o.source IN (${placeholders})`);
        dataParams.push(...parsedTokens.marketplaces);
        dataParamIndex += parsedTokens.marketplaces.length;
      }

      // Payment status filter from parsed tokens (supports multiple)
      if (parsedTokens.paymentStatuses.length > 0) {
        const placeholders = parsedTokens.paymentStatuses.map((_, i) => `$${dataParamIndex + i}`).join(', ');
        dataConditions.push(`o.payment_status IN (${placeholders})`);
        dataParams.push(...parsedTokens.paymentStatuses);
        dataParamIndex += parsedTokens.paymentStatuses.length;
      }

      // Search terms filter (each term AND-ed together)
      if (parsedTokens.searchTerms.length > 0) {
        parsedTokens.searchTerms.forEach(term => {
          dataConditions.push(`(
            o.order_number ILIKE $${dataParamIndex} OR
            o.buyer_first_name ILIKE $${dataParamIndex} OR
            o.buyer_last_name ILIKE $${dataParamIndex} OR
            o.buyer_email ILIKE $${dataParamIndex} OR
            COALESCE(mp.sku, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.sku, ps.sku, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.product_type, sm.product_group, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.title, ps.title, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(oi.name, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.color, ps.color, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.doors, sm.doors, '')::text ILIKE $${dataParamIndex} OR
            COALESCE(cp.legs, sm.legs, '')::text ILIKE $${dataParamIndex} OR
            EXISTS (
              SELECT 1 FROM unnest(COALESCE(cp.color_options, ps.color_options, ARRAY[]::text[])) AS co 
              WHERE co ILIKE $${dataParamIndex}
            )
          )`);
          dataParams.push(`%${term}%`);
          dataParamIndex++;
        });
      }
      if (filters.orderNumber) {
        dataConditions.push(`o.order_number ILIKE $${dataParamIndex}`);
        dataParams.push(`%${filters.orderNumber}%`);
        dataParamIndex++;
      }
      if (filters.customerName) {
        dataConditions.push(`(
          o.buyer_first_name ILIKE $${dataParamIndex} OR
          o.buyer_last_name ILIKE $${dataParamIndex}
        )`);
        dataParams.push(`%${filters.customerName}%`);
        dataParamIndex++;
      }
      if (filters.sku) {
        dataConditions.push(`(
          COALESCE(mp.sku, '')::text ILIKE $${dataParamIndex} OR
          COALESCE(cp.sku, ps.sku, '')::text ILIKE $${dataParamIndex}
        )`);
        dataParams.push(`%${filters.sku}%`);
        dataParamIndex++;
      }
      if (filters.color) {
        dataConditions.push(`COALESCE(cp.color, ps.color, '') ILIKE $${dataParamIndex}`);
        dataParams.push(`%${filters.color}%`);
        dataParamIndex++;
      }
      if (filters.minLength) {
        dataConditions.push(`cp.length >= $${dataParamIndex}`);
        dataParams.push(filters.minLength);
        dataParamIndex++;
      }
      if (filters.maxLength) {
        dataConditions.push(`cp.length <= $${dataParamIndex}`);
        dataParams.push(filters.maxLength);
        dataParamIndex++;
      }
      if (filters.minWidth) {
        dataConditions.push(`cp.width >= $${dataParamIndex}`);
        dataParams.push(filters.minWidth);
        dataParamIndex++;
      }
      if (filters.maxWidth) {
        dataConditions.push(`cp.width <= $${dataParamIndex}`);
        dataParams.push(filters.maxWidth);
        dataParamIndex++;
      }
      // Legacy marketplace filter (kept for backward compatibility if passed explicitly)
      if (filters.marketplace && filters.marketplace !== 'all' && parsedTokens.marketplaces.length === 0) {
        dataConditions.push(`o.source = $${dataParamIndex}`);
        dataParams.push(filters.marketplace);
        dataParamIndex++;
      }

      // Filter: Show only catalog-linked products or sets
      if (filters.showCatalogLinked === 'true') {
        dataConditions.push(`(cp.id IS NOT NULL OR ps.id IS NOT NULL)`);
      }

      // Filter: Show only products that are sets (linked to catalog.product_sets)
      if (filters.showSetsOnly === 'true') {
        dataConditions.push(`ps.id IS NOT NULL`);
      }

      // Filter: Show only items that are already in any plan (exclude soft-deleted lines)
      // Match specific order items: must match both order_number AND (product_id OR set_id)
      if (filters.showInPlans === 'true') {
        dataConditions.push(`(
          EXISTS (
            SELECT 1 FROM production.production_plan_lines pl 
            WHERE pl.deleted_at IS NULL
              AND pl.metadata->>'order_number' = o.order_number
              AND (
                pl.product_id = cp.id
                OR (pl.metadata->>'set_id')::int = ps.id
              )
          )
        )`);
      }

      // Date filter - filter orders by date
      if (filters.dateFilter && filters.dateFilter !== 'all') {
        switch (filters.dateFilter) {
          case 'today':
            dataConditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw')`);
            break;
          case 'yesterday':
            dataConditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw' - INTERVAL '1 day')`);
            break;
          case 'day-before':
            dataConditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw' - INTERVAL '2 days')`);
            break;
          case 'custom-days':
            if (filters.customDays && filters.customDays > 0) {
              dataConditions.push(`DATE(o.order_date AT TIME ZONE 'Europe/Warsaw') = DATE(NOW() AT TIME ZONE 'Europe/Warsaw' - INTERVAL '${filters.customDays} days')`);
            }
            break;
        }
      }

      const dataWhereClause = dataConditions.join(' AND ');
      
      // Include MIN(cp.sku) in SELECT/GROUP BY when sorting by product_sku
      const includeMinSku = filters.sortBy === 'product_sku';
      
      const dataResult = await pool.query(`
        SELECT 
          o.id as order_id,
          o.order_number,
          o.source as marketplace,
          o.buyer_first_name,
          o.buyer_last_name,
          o.buyer_email,
          o.order_date,
          o.payment_status,
          o.total_to_pay_amount,
          o.total_to_pay_currency as currency,
          ${includeMinSku ? 'MIN(cp.sku) as min_sku,' : ''}
          json_agg(
            json_build_object(
              'item_id', oi.id,
              'offer_external_id', oi.offer_external_id,
              'name', oi.name,
              'quantity', oi.quantity,
              'unit_price', oi.unit_price,
              'price', oi.price,
              'image_url', oi.image_url,
              'product_length', COALESCE(cp.length::text, sm.length),
              'product_width', COALESCE(cp.width::text, sm.width),
              'product_height', COALESCE(cp.height::text, sm.height),
              'product_color', COALESCE(cp.color, ps.color),
              'product_color_options', COALESCE(cp.color_options, ps.color_options),
              'product_sku', COALESCE(cp.sku, ps.sku),
              'product_type', COALESCE(cp.product_type, sm.product_group),
              'product_doors', COALESCE(cp.doors, sm.doors),
              'product_legs', COALESCE(cp.legs, sm.legs),
              'warehouse_total_qty', wpp.quantity,
              'warehouse_reserved_qty', wpp.reserved_quantity,
              'line_reserved_qty', ppl_any.reserved_quantity,
              'marketplace_product_id', mp.id,
              'link_type', ppd.link_type,
              'catalog_product_id', cp.id,
              'catalog_product_sku', cp.sku,
              'catalog_product_title', cp.title,
              'catalog_set_id', ps.id,
              'catalog_set_sku', ps.sku,
              'catalog_set_title', ps.title,
              'platform_link_id', ppd.id,
              'bom_component_count', COALESCE(pb_comp_count.component_count, 0),
              'product_group_id', ppg.id,
              'product_group_name', ppg.name,
              'product_group_color_hex', ppg.color_hex,
              'is_in_plan', CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END,
              'in_plan_number', ppl_any.plan_number,
              'in_plan_id', ppl_any.pp_id,
              'in_current_plan', CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END,
              'set_components', set_components.components
            ) ORDER BY oi.id
          ) FILTER (WHERE oi.id IS NOT NULL) as items
        FROM commerce.orders o
        LEFT JOIN commerce.order_items oi ON o.id = oi.order_id
        LEFT JOIN commerce.marketplace_products mp ON oi.offer_external_id = mp.offer_external_id AND mp.source = o.source
        LEFT JOIN LATERAL (
          -- Get only ONE platform data row per order item, preferring 'product' type over 'set'
          SELECT DISTINCT ON (external_id) *
          FROM catalog.product_platform_data
          WHERE external_id = oi.offer_external_id
          ORDER BY external_id, CASE WHEN link_type = 'product' THEN 0 ELSE 1 END, id DESC
        ) ppd ON oi.catalog_product_id IS NULL AND oi.catalog_set_id IS NULL
        -- Support both direct catalog_product_id (from replace feature) and platform_data mapping
        LEFT JOIN catalog.products cp ON cp.id = COALESCE(oi.catalog_product_id, ppd.product_id)
        LEFT JOIN product_creator.product_sets ps ON ps.id = COALESCE(oi.catalog_set_id, ppd.set_id)
        LEFT JOIN product_creator.set_matrices sm ON ps.set_matrix_id = sm.id
        LEFT JOIN bom.product_boms pb ON pb.product_id = cp.id AND pb.is_active = true
        LEFT JOIN LATERAL (
          SELECT COUNT(*)::int as component_count
          FROM bom.product_components pc
          WHERE pc.product_bom_id = pb.id
        ) pb_comp_count ON true
        LEFT JOIN warehouse.packed_products wpp ON wpp.catalog_product_id = cp.id
        LEFT JOIN production.production_product_group_items ppgi ON ppgi.product_id = cp.id
        LEFT JOIN production.production_product_groups ppg ON ppg.id = ppgi.group_id AND ppg.is_active = true
        LEFT JOIN LATERAL (
          SELECT id, plan_id, reserved_quantity
          FROM production.production_plan_lines
          WHERE plan_id = $1 
            AND source_type = 'order_demand'
            AND product_id = cp.id
            AND (metadata->>'order_number') = o.order_number
            AND deleted_at IS NULL
          ORDER BY id DESC
          LIMIT 1
        ) ppl_current ON ps.id IS NULL AND cp.id IS NOT NULL
        LEFT JOIN LATERAL (
          SELECT ppl.id, ppl.plan_id, ppl.reserved_quantity, pp.plan_number, pp.id as pp_id
          FROM production.production_plan_lines ppl
          LEFT JOIN production.production_plans pp ON pp.id = ppl.plan_id
          WHERE ppl.source_type = 'order_demand'
            AND ppl.product_id = cp.id
            AND (ppl.metadata->>'order_number') = o.order_number
            AND ppl.deleted_at IS NULL
          ORDER BY ppl.id DESC
          LIMIT 1
        ) ppl_any ON ps.id IS NULL AND cp.id IS NOT NULL
        LEFT JOIN LATERAL (
          SELECT json_agg(
            json_build_object(
              'component_id', comp_data.component_id,
              'component_sku', comp_data.component_sku,
              'component_title', comp_data.component_title,
              'component_color', comp_data.component_color,
              'component_length', comp_data.component_length,
              'component_width', comp_data.component_width,
              'component_height', comp_data.component_height,
              'component_product_type', comp_data.component_product_type,
              'component_doors', comp_data.component_doors,
              'component_legs', comp_data.component_legs,
              'quantity', comp_data.quantity,
              'primary_image_url', comp_data.primary_image_url,
              'parent_set_image_url', comp_data.parent_set_image_url,
              'is_in_current_plan', comp_data.is_in_current_plan,
              'is_in_any_plan', comp_data.is_in_any_plan,
              'in_plan_number', comp_data.in_plan_number,
              'in_plan_id', comp_data.in_plan_id
            ) ORDER BY comp_data.spl_id
          ) as components
          FROM (
            SELECT DISTINCT ON (comp.id)
              comp.id as component_id,
              comp.sku as component_sku,
              comp.title as component_title,
              comp.color as component_color,
              comp.length as component_length,
              comp.width as component_width,
              comp.height as component_height,
              comp.product_type as component_product_type,
              comp.doors as component_doors,
              comp.legs as component_legs,
              spl.quantity,
              COALESCE(comp.image_url, comp_img.url, oi.image_url) as primary_image_url,
              oi.image_url as parent_set_image_url,
              CASE WHEN ppl_comp_current.id IS NOT NULL THEN true ELSE false END as is_in_current_plan,
              CASE WHEN ppl_comp_any.id IS NOT NULL THEN true ELSE false END as is_in_any_plan,
              pp_comp_any.plan_number as in_plan_number,
              pp_comp_any.id as in_plan_id,
              spl.id as spl_id
            FROM product_creator.set_product_links spl
            JOIN catalog.products comp ON comp.id = spl.product_id
            LEFT JOIN catalog.product_images comp_img ON comp_img.product_id = comp.id AND comp_img.is_primary = true
            LEFT JOIN production.production_plan_lines ppl_comp_current ON (
              ppl_comp_current.plan_id = $1
              AND ppl_comp_current.source_type = 'order_demand'
              AND (ppl_comp_current.metadata->>'set_id')::int = ps.id
              AND (ppl_comp_current.metadata->>'component_id')::int = comp.id
              AND (ppl_comp_current.metadata->>'order_number') = o.order_number
              AND ppl_comp_current.deleted_at IS NULL
            )
            LEFT JOIN production.production_plan_lines ppl_comp_any ON (
              ppl_comp_any.source_type = 'order_demand'
              AND (ppl_comp_any.metadata->>'set_id')::int = ps.id
              AND (ppl_comp_any.metadata->>'component_id')::int = comp.id
              AND (ppl_comp_any.metadata->>'order_number') = o.order_number
              AND ppl_comp_any.deleted_at IS NULL
            )
            LEFT JOIN production.production_plans pp_comp_any ON pp_comp_any.id = ppl_comp_any.plan_id
            WHERE spl.set_id = ps.id
            ORDER BY comp.id, ppl_comp_any.id DESC NULLS LAST
          ) comp_data
        ) set_components ON ps.id IS NOT NULL
        WHERE ${dataWhereClause}
        GROUP BY o.id, o.order_number, o.source, o.buyer_first_name, o.buyer_last_name, 
                 o.buyer_email, o.order_date, o.payment_status, o.total_to_pay_amount, o.total_to_pay_currency
        HAVING COUNT(oi.id) > 0
        ORDER BY ${includeMinSku ? 'min_sku' : orderByColumn} ${orderByDirection}
        LIMIT $${dataParamIndex} OFFSET $${dataParamIndex + 1}
      `, [...dataParams, filters.limit, filters.offset]);

      // Debug: Check warehouse data for specific SKU
      const debugItems = dataResult.rows.flatMap((order: any) => 
        (order.items || []).filter((item: any) => item?.product_sku === '5905806202199')
      );
      if (debugItems.length > 0) {
        console.log('📦 [DEBUG WAREHOUSE] Items with SKU 5905806202199:', JSON.stringify(debugItems.slice(0, 2), null, 2));
      }

      res.json({
        orders: dataResult.rows,
        total,
        limit: filters.limit,
        offset: filters.offset,
      });
    } catch (error) {
      console.error("Error fetching available orders:", error);
      res.status(500).json({ message: "Failed to fetch available orders" });
    }
  });

  // GET /api/production/planning/plans/:id/available-catalog-products - Get catalog products for internal orders
  app.get("/api/production/planning/plans/:id/available-catalog-products", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const normalizedQuery = Object.fromEntries(
        Object.entries(req.query).map(([key, value]) => [key, value === '' ? undefined : value])
      );

      const filters = z.object({
        search: z.string().optional(),
        color: z.string().optional(),
        sku: z.string().optional(),
        minLength: z.coerce.number().optional(),
        maxLength: z.coerce.number().optional(),
        minWidth: z.coerce.number().optional(),
        maxWidth: z.coerce.number().optional(),
        limit: z.coerce.number().min(1).max(500).optional().default(100),
        offset: z.coerce.number().min(0).optional().default(0),
        sortBy: z.enum(['title', 'sku', 'color', 'created_at']).optional().default('created_at'),
        sortOrder: z.enum(['asc', 'desc']).optional().default('desc'),
      }).parse(normalizedQuery);

      const sortColumnMap: Record<string, string> = {
        'title': 'cp.title',
        'sku': 'cp.sku',
        'color': 'cp.color',
        'created_at': 'cp.created_at',
      };

      const orderByColumn = sortColumnMap[filters.sortBy];
      const orderByDirection = filters.sortOrder.toUpperCase();

      const conditions: string[] = ['cp.is_active = true'];
      const params: any[] = [];
      let paramIndex = 1;

      if (filters.search) {
        conditions.push(`(
          cp.title ILIKE $${paramIndex} OR
          cp.sku ILIKE $${paramIndex} OR
          cp.color ILIKE $${paramIndex}
        )`);
        params.push(`%${filters.search}%`);
        paramIndex++;
      }

      if (filters.color) {
        conditions.push(`cp.color ILIKE $${paramIndex}`);
        params.push(`%${filters.color}%`);
        paramIndex++;
      }

      if (filters.sku) {
        conditions.push(`cp.sku ILIKE $${paramIndex}`);
        params.push(`%${filters.sku}%`);
        paramIndex++;
      }

      if (filters.minLength) {
        conditions.push(`cp.length::numeric >= $${paramIndex}`);
        params.push(filters.minLength);
        paramIndex++;
      }

      if (filters.maxLength) {
        conditions.push(`cp.length::numeric <= $${paramIndex}`);
        params.push(filters.maxLength);
        paramIndex++;
      }

      if (filters.minWidth) {
        conditions.push(`cp.width::numeric >= $${paramIndex}`);
        params.push(filters.minWidth);
        paramIndex++;
      }

      if (filters.maxWidth) {
        conditions.push(`cp.width::numeric <= $${paramIndex}`);
        params.push(filters.maxWidth);
        paramIndex++;
      }

      const whereClause = conditions.join(' AND ');

      const countResult = await pool.query(`
        SELECT COUNT(*)::int as total
        FROM catalog.products cp
        WHERE ${whereClause}
      `, params);

      const total = countResult.rows[0].total;

      const dataParams = [...params, planId];
      let dataParamIndex = dataParams.length + 1;

      const dataResult = await pool.query(`
        SELECT DISTINCT ON (cp.id)
          cp.id,
          cp.sku,
          cp.title,
          cp.color,
          cp.color_options,
          cp.length,
          cp.width,
          cp.height,
          cp.product_type,
          cp.doors,
          cp.legs,
          cp.created_at,
          (
            SELECT pi.url
            FROM catalog.product_images pi
            WHERE pi.product_id = cp.id
            ORDER BY pi.is_primary DESC, pi.sort_order ASC
            LIMIT 1
          ) as image_url,
          COALESCE(pb_comp_count.component_count, 0) as bom_component_count,
          (
            SELECT ppg.id
            FROM production.production_product_group_items ppgi
            LEFT JOIN production.production_product_groups ppg ON ppg.id = ppgi.group_id AND ppg.is_active = true
            WHERE ppgi.product_id = cp.id
            LIMIT 1
          ) as product_group_id,
          (
            SELECT ppg.name
            FROM production.production_product_group_items ppgi
            LEFT JOIN production.production_product_groups ppg ON ppg.id = ppgi.group_id AND ppg.is_active = true
            WHERE ppgi.product_id = cp.id
            LIMIT 1
          ) as product_group_name,
          (
            SELECT ppg.color_hex
            FROM production.production_product_group_items ppgi
            LEFT JOIN production.production_product_groups ppg ON ppg.id = ppgi.group_id AND ppg.is_active = true
            WHERE ppgi.product_id = cp.id
            LIMIT 1
          ) as product_group_color_hex,
          CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END as is_in_plan,
          (
            SELECT pp_any.plan_number
            FROM production.production_plan_lines ppl_any
            LEFT JOIN production.production_plans pp_any ON pp_any.id = ppl_any.plan_id
            WHERE ppl_any.source_type = 'catalog_internal'
              AND ppl_any.product_id = cp.id
              AND ppl_any.deleted_at IS NULL
            LIMIT 1
          ) as in_plan_number,
          CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END as in_current_plan,
          wpp.quantity as warehouse_total_qty,
          wpp.reserved_quantity as warehouse_reserved_qty
        FROM catalog.products cp
        LEFT JOIN bom.product_boms pb ON pb.product_id = cp.id AND pb.is_active = true
        LEFT JOIN LATERAL (
          SELECT COUNT(*)::int as component_count
          FROM bom.product_components pc
          WHERE pc.product_bom_id = pb.id
        ) pb_comp_count ON true
        LEFT JOIN warehouse.packed_products wpp ON wpp.catalog_product_id = cp.id
        LEFT JOIN production.production_plan_lines ppl_current ON (
          ppl_current.plan_id = $${dataParams.length}
          AND ppl_current.source_type = 'catalog_internal'
          AND ppl_current.product_id = cp.id
          AND ppl_current.deleted_at IS NULL
        )
        WHERE ${whereClause}
        ORDER BY cp.id, ${orderByColumn} ${orderByDirection}
        LIMIT $${dataParamIndex} OFFSET $${dataParamIndex + 1}
      `, [...dataParams, filters.limit, filters.offset]);

      // Disable caching for fresh warehouse data
      res.setHeader('Cache-Control', 'no-cache, no-store, must-revalidate');
      res.setHeader('Pragma', 'no-cache');
      res.setHeader('Expires', '0');
      
      res.json({
        products: dataResult.rows,
        total,
        limit: filters.limit,
        offset: filters.offset,
      });
    } catch (error) {
      console.error("Error fetching catalog products:", error);
      res.status(500).json({ message: "Failed to fetch catalog products" });
    }
  });

  // GET /api/production/planning/plans/:id/available-cutting-patterns - Get cutting patterns for internal orders
  app.get("/api/production/planning/plans/:id/available-cutting-patterns", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const normalizedQuery = Object.fromEntries(
        Object.entries(req.query).map(([key, value]) => [key, value === '' ? undefined : value])
      );

      const filters = z.object({
        search: z.string().optional(),
        status: z.string().optional(),
        limit: z.coerce.number().min(1).max(500).optional().default(100),
        offset: z.coerce.number().min(0).optional().default(0),
        sortBy: z.enum(['name', 'code', 'created_at']).optional().default('created_at'),
        sortOrder: z.enum(['asc', 'desc']).optional().default('desc'),
      }).parse(normalizedQuery);

      const sortColumnMap: Record<string, string> = {
        'name': 'cpt.name',
        'code': 'cpt.code',
        'created_at': 'cpt.created_at',
      };

      const orderByColumn = sortColumnMap[filters.sortBy];
      const orderByDirection = filters.sortOrder.toUpperCase();

      const conditions: string[] = ['cpt.is_active = true'];
      const params: any[] = [];
      let paramIndex = 1;

      if (filters.search) {
        conditions.push(`(
          cpt.name ILIKE $${paramIndex} OR
          cpt.code ILIKE $${paramIndex} OR
          cpt.description ILIKE $${paramIndex}
        )`);
        params.push(`%${filters.search}%`);
        paramIndex++;
      }

      const whereClause = conditions.join(' AND ');

      const countResult = await pool.query(`
        SELECT COUNT(*)::int as total
        FROM production.cut_pattern_templates cpt
        WHERE ${whereClause}
      `, params);

      const total = countResult.rows[0].total;

      const dataParams = [...params, planId];
      let dataParamIndex = dataParams.length + 1;

      const dataResult = await pool.query(`
        SELECT 
          cpt.id,
          cpt.code,
          cpt.name,
          cpt.description,
          cpt.is_active,
          cpt.board_length,
          cpt.board_width,
          cpt.board_thickness,
          cpt.kerf,
          cpt.created_at,
          COALESCE(SUM(cpti.quantity_default), 0)::int as total_quantity,
          COUNT(cpti.id)::int as items_count,
          CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END as is_in_plan,
          pp_any.plan_number as in_plan_number,
          CASE WHEN ppl_current.id IS NOT NULL THEN true ELSE false END as in_current_plan
        FROM production.cut_pattern_templates cpt
        LEFT JOIN production.cut_pattern_template_items cpti ON cpti.template_id = cpt.id
        LEFT JOIN production.production_plan_lines ppl_current ON (
          ppl_current.plan_id = $${dataParams.length}
          AND ppl_current.source_type = 'cutting_pattern'
          AND (ppl_current.metadata->>'cutting_pattern_id')::int = cpt.id
        )
        LEFT JOIN production.production_plan_lines ppl_any ON (
          ppl_any.source_type = 'cutting_pattern'
          AND (ppl_any.metadata->>'cutting_pattern_id')::int = cpt.id
        )
        LEFT JOIN production.production_plans pp_any ON pp_any.id = ppl_any.plan_id
        WHERE ${whereClause}
        GROUP BY cpt.id, cpt.code, cpt.name, cpt.description, cpt.is_active, 
                 cpt.board_length, cpt.board_width, cpt.board_thickness, cpt.kerf, 
                 cpt.created_at, ppl_current.id, pp_any.plan_number
        ORDER BY ${orderByColumn} ${orderByDirection}
        LIMIT $${dataParamIndex} OFFSET $${dataParamIndex + 1}
      `, [...dataParams, filters.limit, filters.offset]);

      res.json({
        patterns: dataResult.rows,
        total,
        limit: filters.limit,
        offset: filters.offset,
      });
    } catch (error) {
      console.error("Error fetching cutting patterns:", error);
      res.status(500).json({ message: "Failed to fetch cutting patterns" });
    }
  });

  // ============================================================================
  // Production Plan Flow Graph - Visual Flow Tree
  // ============================================================================

  app.get("/api/production/planning/plans/:id/flow", async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const flowGraph = await planningService.getPlanFlowGraph(pool, planId);
      
      if (!flowGraph) {
        return res.status(404).json({ message: "Plan not found" });
      }

      res.json(flowGraph);
    } catch (error) {
      console.error("Error fetching plan flow graph:", error);
      res.status(500).json({ message: "Failed to fetch plan flow graph" });
    }
  });

  // Get operation items (formatki) for a flow tree node
  app.get("/api/production/planning/:planId/operation-items", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId, 10);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const zlpIdsParam = req.query.zlpIds as string;
      const materialColor = req.query.materialColor as string;
      const operationCode = req.query.operationCode as string;

      if (!zlpIdsParam) {
        return res.status(400).json({ message: "zlpIds parameter is required" });
      }

      const zlpIds = zlpIdsParam.split(',').map(id => parseInt(id, 10)).filter(id => !isNaN(id));
      
      if (zlpIds.length === 0) {
        return res.status(400).json({ message: "At least one valid zlpId is required" });
      }

      const items = await planningService.getOperationItems(pool, planId, zlpIds, materialColor, operationCode);
      res.json(items);
    } catch (error) {
      console.error("Error fetching operation items:", error);
      res.status(500).json({ message: "Failed to fetch operation items" });
    }
  });

  // ============================================================================
  // Merge Points API - Punkty zbieżności ścieżek produkcyjnych
  // ============================================================================

  // Get merge points for a plan
  app.get("/api/production/planning/plans/:id/merge-points", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const result = await pool.query(`
        SELECT 
          mp.*,
          ro.code as operation_code,
          ro.name as operation_name
        FROM production.production_plan_merge_points mp
        LEFT JOIN production.production_routing_operations ro ON ro.id = mp.routing_operation_id
        WHERE mp.plan_id = $1
        ORDER BY mp.merge_stage, mp.id
      `, [planId]);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching merge points:", error);
      res.status(500).json({ message: "Failed to fetch merge points" });
    }
  });

  // Create merge point
  app.post("/api/production/planning/plans/:id/merge-points", requirePermission('manage_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const parseResult = createMergePointSchema.safeParse(req.body);
      if (!parseResult.success) {
        return res.status(400).json({ 
          message: "Validation failed", 
          errors: parseResult.error.errors 
        });
      }

      const { 
        mergeStage, 
        requiredFamilies, 
        completionPolicy,
        manualRelease,
        routingOperationId,
        notes 
      } = parseResult.data;

      const result = await pool.query(`
        INSERT INTO production.production_plan_merge_points 
        (plan_id, merge_stage, required_families, completion_policy, manual_release, routing_operation_id, notes)
        VALUES ($1, $2, $3, $4, $5, $6, $7)
        RETURNING *
      `, [planId, mergeStage, JSON.stringify(requiredFamilies), completionPolicy, manualRelease, routingOperationId || null, notes || null]);

      res.status(201).json(result.rows[0]);
    } catch (error) {
      console.error("Error creating merge point:", error);
      res.status(500).json({ message: "Failed to create merge point" });
    }
  });

  // Update merge point - only configuration fields, not status/progress
  app.patch("/api/production/planning/merge-points/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid merge point ID" });
      }

      const parseResult = updateMergePointSchema.safeParse(req.body);
      if (!parseResult.success) {
        return res.status(400).json({ 
          message: "Validation failed", 
          errors: parseResult.error.errors 
        });
      }

      const { 
        mergeStage, 
        requiredFamilies, 
        completionPolicy,
        manualRelease,
        routingOperationId,
        notes
      } = parseResult.data;

      const updates: string[] = [];
      const values: any[] = [];
      let paramIndex = 1;

      if (mergeStage !== undefined) {
        updates.push(`merge_stage = $${paramIndex++}`);
        values.push(mergeStage);
      }
      if (requiredFamilies !== undefined) {
        updates.push(`required_families = $${paramIndex++}`);
        values.push(JSON.stringify(requiredFamilies));
      }
      if (completionPolicy !== undefined) {
        updates.push(`completion_policy = $${paramIndex++}`);
        values.push(completionPolicy);
      }
      if (manualRelease !== undefined) {
        updates.push(`manual_release = $${paramIndex++}`);
        values.push(manualRelease);
      }
      if (routingOperationId !== undefined) {
        updates.push(`routing_operation_id = $${paramIndex++}`);
        values.push(routingOperationId);
      }
      if (notes !== undefined) {
        updates.push(`notes = $${paramIndex++}`);
        values.push(notes);
      }

      if (updates.length === 0) {
        return res.status(400).json({ message: "No fields to update" });
      }

      updates.push(`updated_at = NOW()`);
      values.push(id);

      const result = await pool.query(`
        UPDATE production.production_plan_merge_points 
        SET ${updates.join(', ')}
        WHERE id = $${paramIndex}
        RETURNING *
      `, values);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Merge point not found" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Error updating merge point:", error);
      res.status(500).json({ message: "Failed to update merge point" });
    }
  });

  // Delete merge point
  app.delete("/api/production/planning/merge-points/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid merge point ID" });
      }

      const result = await pool.query(`
        DELETE FROM production.production_plan_merge_points 
        WHERE id = $1
        RETURNING id
      `, [id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Merge point not found" });
      }

      res.json({ message: "Merge point deleted", id });
    } catch (error) {
      console.error("Error deleting merge point:", error);
      res.status(500).json({ message: "Failed to delete merge point" });
    }
  });

  // Release merge point (manual release)
  app.post("/api/production/planning/merge-points/:id/release", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid merge point ID" });
      }

      const userId = req.user?.id || null;

      const result = await pool.query(`
        UPDATE production.production_plan_merge_points 
        SET status = 'released', released_at = NOW(), released_by = $2, updated_at = NOW()
        WHERE id = $1 AND status IN ('ready', 'partial')
        RETURNING *
      `, [id, userId]);

      if (result.rows.length === 0) {
        return res.status(400).json({ message: "Merge point not found or cannot be released" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Error releasing merge point:", error);
      res.status(500).json({ message: "Failed to release merge point" });
    }
  });

  // Get suggested merge points based on plan's component families
  app.get("/api/production/planning/plans/:id/merge-points/suggestions", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // Get component families present in this plan from ZLPs
      const familiesResult = await pool.query(`
        SELECT DISTINCT 
          CASE 
            WHEN po.component_type ILIKE '%HDF%' THEN 'formatki_hdf'
            WHEN po.component_type ILIKE '%BIAŁ%' OR po.component_type ILIKE '%BIAL%' THEN 'formatki_biale'
            WHEN po.component_type ILIKE '%KOLOR%' THEN 'formatki_kolor'
            WHEN po.component_type ILIKE '%TAPIC%' THEN 'tapicerowane'
            WHEN po.component_type ILIKE '%SIEDZISK%' OR po.component_type ILIKE '%SUROW%' THEN 'siedziska'
            WHEN po.component_type ILIKE '%AKCES%' THEN 'akcesoria'
            ELSE 'mixed'
          END as family
        FROM production.production_orders po
        WHERE po.plan_id = $1
      `, [planId]);

      const families = familiesResult.rows.map(r => r.family).filter(f => f !== 'mixed');
      const uniqueFamilies = Array.from(new Set(families));

      // Suggest merge points at kompletowanie and pakowanie stages
      const suggestions = [];

      if (uniqueFamilies.length > 1) {
        suggestions.push({
          mergeStage: 'kompletowanie',
          requiredFamilies: uniqueFamilies,
          completionPolicy: 'all_required',
          manualRelease: false,
          description: `Punkt zbieżności na etapie kompletowania - łączy ${uniqueFamilies.length} rodzin komponentów`
        });

        suggestions.push({
          mergeStage: 'pakowanie',
          requiredFamilies: uniqueFamilies,
          completionPolicy: 'all_required',
          manualRelease: true,
          description: `Punkt zbieżności na etapie pakowania - wymaga ręcznego zatwierdzenia`
        });
      }

      res.json({
        families: uniqueFamilies,
        suggestions
      });
    } catch (error) {
      console.error("Error getting merge point suggestions:", error);
      res.status(500).json({ message: "Failed to get merge point suggestions" });
    }
  });

  // ============================================================================
  // Transport Batches API - Partie transportowe
  // ============================================================================

  const createTransportBatchSchema = z.object({
    batchNumber: z.string().optional().nullable(), // Auto-generated if not provided
    label: z.string().optional().nullable(),
    targetStage: z.string().optional().nullable(),
    destinationLocationId: z.number().int().positive().optional().nullable(),
    sortOrder: z.number().int().default(0),
    plannedDepartureAt: z.string().optional().nullable(),
    familiesRequired: z.array(z.string()).optional(),
    notes: z.string().optional().nullable(),
  });

  const updateTransportBatchSchema = z.object({
    batchNumber: z.string().min(1).optional(),
    label: z.string().optional().nullable(),
    targetStage: z.string().optional().nullable(),
    destinationLocationId: z.number().int().positive().optional().nullable(),
    sortOrder: z.number().int().optional(),
    plannedDepartureAt: z.string().optional().nullable(),
    status: z.enum(['pending', 'loading', 'in_transit', 'delivered', 'cancelled']).optional(),
    familiesRequired: z.array(z.string()).optional(),
    notes: z.string().optional().nullable(),
  });

  // Get transport batches for a plan
  app.get("/api/production/planning/plans/:id/transport-batches", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const result = await pool.query(`
        SELECT 
          tb.*,
          loc.name as destination_location_name,
          (
            SELECT COUNT(*) 
            FROM production.transport_batch_items tbi 
            WHERE tbi.batch_id = tb.id
          ) as actual_item_count,
          (
            SELECT COUNT(*) 
            FROM production.transport_batch_items tbi 
            WHERE tbi.batch_id = tb.id AND tbi.status = 'ready'
          ) as ready_item_count
        FROM production.transport_batches tb
        LEFT JOIN production.production_locations loc ON loc.id = tb.destination_location_id
        WHERE tb.plan_id = $1
        ORDER BY tb.sort_order, tb.id
      `, [planId]);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching transport batches:", error);
      res.status(500).json({ message: "Failed to fetch transport batches" });
    }
  });

  // Create transport batch
  app.post("/api/production/planning/plans/:id/transport-batches", requirePermission('manage_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const parseResult = createTransportBatchSchema.safeParse(req.body);
      if (!parseResult.success) {
        return res.status(400).json({ 
          message: "Validation failed", 
          errors: parseResult.error.errors 
        });
      }

      const { 
        batchNumber: providedBatchNumber, 
        label, 
        targetStage, 
        destinationLocationId,
        sortOrder,
        plannedDepartureAt,
        familiesRequired,
        notes 
      } = parseResult.data;

      // Auto-generate batch number if not provided
      let batchNumber = providedBatchNumber;
      if (!batchNumber) {
        const countResult = await pool.query(`
          SELECT COUNT(*) + 1 as next_num 
          FROM production.transport_batches 
          WHERE plan_id = $1
        `, [planId]);
        const nextNum = countResult.rows[0].next_num;
        batchNumber = `T${nextNum}`;
      }

      const result = await pool.query(`
        INSERT INTO production.transport_batches 
        (plan_id, batch_number, label, target_stage, destination_location_id, sort_order, planned_departure_at, families_required, notes)
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
        RETURNING *
      `, [
        planId, 
        batchNumber, 
        label || null, 
        targetStage || null, 
        destinationLocationId || null,
        sortOrder || 0,
        plannedDepartureAt || null,
        familiesRequired ? JSON.stringify(familiesRequired) : null,
        notes || null
      ]);

      res.status(201).json(result.rows[0]);
    } catch (error: any) {
      console.error("Error creating transport batch:", error);
      if (error.code === '23505') { // Unique constraint violation
        return res.status(400).json({ message: "Partia o takim numerze już istnieje w tym planie" });
      }
      res.status(500).json({ message: "Failed to create transport batch" });
    }
  });

  // Update transport batch
  app.patch("/api/production/planning/transport-batches/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid batch ID" });
      }

      const parseResult = updateTransportBatchSchema.safeParse(req.body);
      if (!parseResult.success) {
        return res.status(400).json({ 
          message: "Validation failed", 
          errors: parseResult.error.errors 
        });
      }

      const updates: string[] = [];
      const values: any[] = [];
      let paramIndex = 1;

      const data = parseResult.data;
      
      if (data.batchNumber !== undefined) {
        updates.push(`batch_number = $${paramIndex++}`);
        values.push(data.batchNumber);
      }
      if (data.label !== undefined) {
        updates.push(`label = $${paramIndex++}`);
        values.push(data.label);
      }
      if (data.targetStage !== undefined) {
        updates.push(`target_stage = $${paramIndex++}`);
        values.push(data.targetStage);
      }
      if (data.destinationLocationId !== undefined) {
        updates.push(`destination_location_id = $${paramIndex++}`);
        values.push(data.destinationLocationId);
      }
      if (data.sortOrder !== undefined) {
        updates.push(`sort_order = $${paramIndex++}`);
        values.push(data.sortOrder);
      }
      if (data.plannedDepartureAt !== undefined) {
        updates.push(`planned_departure_at = $${paramIndex++}`);
        values.push(data.plannedDepartureAt);
      }
      if (data.status !== undefined) {
        updates.push(`status = $${paramIndex++}`);
        values.push(data.status);
        // Set actual departure time when moving to in_transit
        if (data.status === 'in_transit') {
          updates.push(`actual_departure_at = NOW()`);
        }
      }
      if (data.familiesRequired !== undefined) {
        updates.push(`families_required = $${paramIndex++}`);
        values.push(JSON.stringify(data.familiesRequired));
      }
      if (data.notes !== undefined) {
        updates.push(`notes = $${paramIndex++}`);
        values.push(data.notes);
      }

      if (updates.length === 0) {
        return res.status(400).json({ message: "No fields to update" });
      }

      updates.push(`updated_at = NOW()`);
      values.push(id);

      const result = await pool.query(`
        UPDATE production.transport_batches 
        SET ${updates.join(', ')}
        WHERE id = $${paramIndex}
        RETURNING *
      `, values);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Transport batch not found" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Error updating transport batch:", error);
      res.status(500).json({ message: "Failed to update transport batch" });
    }
  });

  // Delete transport batch
  app.delete("/api/production/planning/transport-batches/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid batch ID" });
      }

      const result = await pool.query(`
        DELETE FROM production.transport_batches 
        WHERE id = $1
        RETURNING id
      `, [id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Transport batch not found" });
      }

      res.json({ message: "Transport batch deleted", id });
    } catch (error) {
      console.error("Error deleting transport batch:", error);
      res.status(500).json({ message: "Failed to delete transport batch" });
    }
  });

  // Get items in a transport batch
  app.get("/api/production/planning/transport-batches/:id/items", requirePermission('view_production'), async (req, res) => {
    try {
      const batchId = parseInt(req.params.id);
      if (isNaN(batchId)) {
        return res.status(400).json({ message: "Invalid batch ID" });
      }

      const result = await pool.query(`
        SELECT 
          tbi.*,
          po.order_number as zlp_number,
          po.status as zlp_status,
          po.color_code,
          po.cutting_sequence,
          po.cutting_priority,
          cp.name as product_name
        FROM production.transport_batch_items tbi
        LEFT JOIN production.production_orders po ON po.id = tbi.production_order_id
        LEFT JOIN catalog.products cp ON cp.id = po.product_id
        WHERE tbi.batch_id = $1
        ORDER BY tbi.sort_index, tbi.id
      `, [batchId]);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching transport batch items:", error);
      res.status(500).json({ message: "Failed to fetch transport batch items" });
    }
  });

  // Add item to transport batch
  app.post("/api/production/planning/transport-batches/:id/items", requirePermission('manage_production'), async (req, res) => {
    try {
      const batchId = parseInt(req.params.id);
      if (isNaN(batchId)) {
        return res.status(400).json({ message: "Invalid batch ID" });
      }

      const { productionOrderId, planLineId, componentFamily, colorCode, quantity, sourceType, requiresCutting } = req.body;

      if (!productionOrderId && !planLineId) {
        return res.status(400).json({ message: "productionOrderId or planLineId is required" });
      }

      const result = await pool.query(`
        INSERT INTO production.transport_batch_items 
        (batch_id, production_order_id, plan_line_id, component_family, color_code, quantity, source_type, requires_cutting)
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
        RETURNING *
      `, [
        batchId,
        productionOrderId || null,
        planLineId || null,
        componentFamily || null,
        colorCode || null,
        quantity || 1,
        sourceType || 'cutting',
        requiresCutting !== false
      ]);

      // Update ZLP with transport batch assignment
      if (productionOrderId) {
        await pool.query(`
          UPDATE production.production_orders 
          SET transport_batch_id = $1
          WHERE id = $2
        `, [batchId, productionOrderId]);
      }

      // Update batch item count
      await pool.query(`
        UPDATE production.transport_batches 
        SET item_count = (SELECT COUNT(*) FROM production.transport_batch_items WHERE batch_id = $1),
            updated_at = NOW()
        WHERE id = $1
      `, [batchId]);

      res.status(201).json(result.rows[0]);
    } catch (error) {
      console.error("Error adding item to transport batch:", error);
      res.status(500).json({ message: "Failed to add item to transport batch" });
    }
  });

  // Remove item from transport batch
  app.delete("/api/production/planning/transport-batch-items/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid item ID" });
      }

      // Get batch ID and production order ID before deletion
      const itemResult = await pool.query(`
        SELECT batch_id, production_order_id FROM production.transport_batch_items WHERE id = $1
      `, [id]);

      if (itemResult.rows.length === 0) {
        return res.status(404).json({ message: "Item not found" });
      }

      const { batch_id: batchId, production_order_id: productionOrderId } = itemResult.rows[0];

      // Delete the item
      await pool.query(`DELETE FROM production.transport_batch_items WHERE id = $1`, [id]);

      // Clear transport batch assignment from ZLP
      if (productionOrderId) {
        await pool.query(`
          UPDATE production.production_orders 
          SET transport_batch_id = NULL
          WHERE id = $1
        `, [productionOrderId]);
      }

      // Update batch item count
      await pool.query(`
        UPDATE production.transport_batches 
        SET item_count = (SELECT COUNT(*) FROM production.transport_batch_items WHERE batch_id = $1),
            updated_at = NOW()
        WHERE id = $1
      `, [batchId]);

      res.json({ message: "Item removed from transport batch", id });
    } catch (error) {
      console.error("Error removing item from transport batch:", error);
      res.status(500).json({ message: "Failed to remove item from transport batch" });
    }
  });

  // ============================================================================
  // Cutting Sequence API - Kolejność cięcia w ZLP
  // ============================================================================

  // Get ZLPs with cutting info for a plan
  app.get("/api/production/planning/plans/:id/cutting-sequence", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const result = await pool.query(`
        SELECT 
          po.id,
          po.order_number,
          po.color_code,
          po.cutting_sequence,
          po.cutting_priority,
          po.requires_cutting,
          po.transport_batch_id,
          po.status,
          po.workflow_stage,
          tb.batch_number as transport_batch_number,
          tb.label as transport_batch_label,
          cp.name as product_name,
          cp.sku as product_sku
        FROM production.production_orders po
        LEFT JOIN production.transport_batches tb ON tb.id = po.transport_batch_id
        LEFT JOIN catalog.products cp ON cp.id = po.product_id
        WHERE EXISTS (
          SELECT 1 FROM production.production_plan_lines ppl 
          WHERE ppl.production_order_id = po.id AND ppl.plan_id = $1 AND ppl.deleted_at IS NULL
        )
        ORDER BY 
          CASE WHEN po.requires_cutting = false THEN 1 ELSE 0 END,
          po.cutting_sequence NULLS LAST,
          po.cutting_priority DESC,
          po.id
      `, [planId]);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching cutting sequence:", error);
      res.status(500).json({ message: "Failed to fetch cutting sequence" });
    }
  });

  // Update cutting sequence for multiple ZLPs
  app.patch("/api/production/planning/plans/:id/cutting-sequence", requirePermission('manage_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const { items } = req.body;
      if (!Array.isArray(items)) {
        return res.status(400).json({ message: "items array is required" });
      }

      // Validate items structure
      for (const item of items) {
        if (typeof item.id !== 'number') {
          return res.status(400).json({ message: "Each item must have an id" });
        }
      }

      // Update each ZLP's cutting sequence
      for (const item of items) {
        const updates: string[] = [];
        const values: any[] = [];
        let paramIndex = 1;

        if (item.cuttingSequence !== undefined) {
          updates.push(`cutting_sequence = $${paramIndex++}`);
          values.push(item.cuttingSequence);
        }
        if (item.cuttingPriority !== undefined) {
          updates.push(`cutting_priority = $${paramIndex++}`);
          values.push(item.cuttingPriority);
        }
        if (item.requiresCutting !== undefined) {
          updates.push(`requires_cutting = $${paramIndex++}`);
          values.push(item.requiresCutting);
        }
        if (item.transportBatchId !== undefined) {
          updates.push(`transport_batch_id = $${paramIndex++}`);
          values.push(item.transportBatchId);
        }

        if (updates.length > 0) {
          values.push(item.id);
          await pool.query(`
            UPDATE production.production_orders 
            SET ${updates.join(', ')}, updated_at = NOW()
            WHERE id = $${paramIndex}
          `, values);
        }
      }

      res.json({ message: "Cutting sequence updated", count: items.length });
    } catch (error) {
      console.error("Error updating cutting sequence:", error);
      res.status(500).json({ message: "Failed to update cutting sequence" });
    }
  });

  // Update single ZLP cutting info
  app.patch("/api/production/planning/zlp/:id/cutting", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid ZLP ID" });
      }

      const { cuttingSequence, cuttingPriority, requiresCutting, transportBatchId } = req.body;

      const updates: string[] = [];
      const values: any[] = [];
      let paramIndex = 1;

      if (cuttingSequence !== undefined) {
        updates.push(`cutting_sequence = $${paramIndex++}`);
        values.push(cuttingSequence);
      }
      if (cuttingPriority !== undefined) {
        updates.push(`cutting_priority = $${paramIndex++}`);
        values.push(cuttingPriority);
      }
      if (requiresCutting !== undefined) {
        updates.push(`requires_cutting = $${paramIndex++}`);
        values.push(requiresCutting);
      }
      if (transportBatchId !== undefined) {
        updates.push(`transport_batch_id = $${paramIndex++}`);
        values.push(transportBatchId);
      }

      if (updates.length === 0) {
        return res.status(400).json({ message: "No fields to update" });
      }

      values.push(id);
      const result = await pool.query(`
        UPDATE production.production_orders 
        SET ${updates.join(', ')}, updated_at = NOW()
        WHERE id = $${paramIndex}
        RETURNING id, order_number, cutting_sequence, cutting_priority, requires_cutting, transport_batch_id
      `, values);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "ZLP not found" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Error updating ZLP cutting info:", error);
      res.status(500).json({ message: "Failed to update ZLP cutting info" });
    }
  });

  // Auto-assign cutting sequence based on color grouping
  app.post("/api/production/planning/plans/:id/auto-cutting-sequence", requirePermission('manage_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.id);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const { colorOrder } = req.body;
      // colorOrder is array of colors in desired cutting order, e.g. ['BIALY', 'SONOMA', 'SUROWA', 'HDF']

      if (!Array.isArray(colorOrder) || colorOrder.length === 0) {
        return res.status(400).json({ message: "colorOrder array is required" });
      }

      // Get all ZLPs for this plan
      const zlpsResult = await pool.query(`
        SELECT po.id, po.color_code
        FROM production.production_orders po
        WHERE po.requires_cutting = true AND EXISTS (
          SELECT 1 FROM production.production_plan_lines ppl 
          WHERE ppl.production_order_id = po.id AND ppl.plan_id = $1 AND ppl.deleted_at IS NULL
        )
        ORDER BY po.id
      `, [planId]);

      // Group by color and assign sequence
      let sequence = 1;
      const updates: { id: number; sequence: number }[] = [];

      for (const color of colorOrder) {
        const zlpsForColor = zlpsResult.rows.filter(z => 
          (z.color_code || '').toUpperCase() === color.toUpperCase()
        );
        for (const zlp of zlpsForColor) {
          updates.push({ id: zlp.id, sequence });
          sequence++;
        }
      }

      // Also add any ZLPs not in colorOrder at the end
      const assignedIds = new Set(updates.map(u => u.id));
      for (const zlp of zlpsResult.rows) {
        if (!assignedIds.has(zlp.id)) {
          updates.push({ id: zlp.id, sequence });
          sequence++;
        }
      }

      // Apply updates
      for (const update of updates) {
        await pool.query(`
          UPDATE production.production_orders 
          SET cutting_sequence = $1, updated_at = NOW()
          WHERE id = $2
        `, [update.sequence, update.id]);
      }

      res.json({ 
        message: "Cutting sequence auto-assigned", 
        count: updates.length,
        colorOrder,
        assignments: updates
      });
    } catch (error) {
      console.error("Error auto-assigning cutting sequence:", error);
      res.status(500).json({ message: "Failed to auto-assign cutting sequence" });
    }
  });

  // GET /api/production/planning/plans/:planId/work-order-operators - Get work orders with operators for a plan
  app.get("/api/production/planning/plans/:planId/work-order-operators", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // Get all work orders for this plan with their operators
      const result = await pool.query(`
        SELECT 
          wo.id as work_order_id,
          wo.work_order_number,
          wo.work_center_id,
          wc.name as work_center_name,
          ro.name as operation_name,
          ro.code as operation_code,
          po.order_number as zlp_number,
          po.color_code,
          COALESCE(
            (
              SELECT json_agg(json_build_object(
                'operatorId', woo.operator_id,
                'operatorName', op.full_name,
                'isPrimary', woo.is_primary
              ) ORDER BY woo.is_primary DESC, op.full_name)
              FROM production.work_order_operators woo
              JOIN production.production_operators op ON woo.operator_id = op.id
              WHERE woo.work_order_id = wo.id
            ),
            '[]'::json
          ) as assigned_operators
        FROM production.production_work_orders wo
        JOIN production.production_orders po ON wo.production_order_id = po.id
        JOIN production.production_plan_lines ppl ON ppl.production_order_id = po.id
        LEFT JOIN production.production_routing_operations ro ON wo.routing_operation_id = ro.id
        LEFT JOIN production.production_work_centers wc ON wo.work_center_id = wc.id
        WHERE ppl.plan_id = $1 AND ppl.deleted_at IS NULL
        ORDER BY wc.name NULLS LAST, wo.sequence
      `, [planId]);

      const workOrders = result.rows.map(row => ({
        workOrderId: row.work_order_id,
        workOrderNumber: row.work_order_number,
        workCenterId: row.work_center_id,
        workCenterName: row.work_center_name,
        operationName: row.operation_name,
        operationCode: row.operation_code,
        zlpNumber: row.zlp_number,
        colorCode: row.color_code,
        assignedOperators: row.assigned_operators || [],
      }));

      res.json({ workOrders });
    } catch (error) {
      console.error("Error fetching work order operators:", error);
      res.status(500).json({ message: "Failed to fetch work order operators" });
    }
  });

  // ============================================================================
  // WAREHOUSE ROUTING RULES - configurable routing for warehouse inventory
  // ============================================================================

  // GET /api/production/planning/routing-rules - Get all warehouse routing rules
  app.get("/api/production/planning/routing-rules", requirePermission('view_production'), async (req, res) => {
    try {
      const { planId, isActive } = req.query;
      
      let query = `
        SELECT 
          wrr.*,
          pp.plan_number,
          pp.name as plan_name,
          cp.title as catalog_product_name,
          cp.sku as catalog_product_sku,
          pro.name as destination_operation_name,
          pl.name as destination_location_name,
          u.username as created_by_username
        FROM production.warehouse_routing_rules wrr
        LEFT JOIN production.production_plans pp ON pp.id = wrr.plan_id
        LEFT JOIN catalog.products cp ON cp.id = wrr.catalog_product_id
        LEFT JOIN production.production_routing_operations pro ON pro.id = wrr.destination_operation_id
        LEFT JOIN production.production_locations pl ON pl.id = wrr.destination_location_id
        LEFT JOIN users u ON u.id = wrr.created_by
        WHERE 1=1
      `;
      const params: any[] = [];
      let paramIndex = 1;
      
      if (planId && planId !== 'all') {
        if (planId === 'global') {
          query += ` AND wrr.plan_id IS NULL`;
        } else {
          query += ` AND (wrr.plan_id = $${paramIndex++} OR wrr.plan_id IS NULL)`;
          params.push(parseInt(planId as string));
        }
      }
      
      if (isActive !== undefined && isActive !== '') {
        query += ` AND wrr.is_active = $${paramIndex++}`;
        params.push(isActive === 'true');
      }
      
      query += ` ORDER BY wrr.priority DESC, wrr.created_at DESC`;
      
      const result = await pool.query(query, params);
      
      const rules = result.rows.map(row => ({
        id: row.id,
        planId: row.plan_id,
        planNumber: row.plan_number,
        planName: row.plan_name,
        catalogProductId: row.catalog_product_id,
        catalogProductName: row.catalog_product_name,
        catalogProductSku: row.catalog_product_sku,
        productSku: row.product_sku,
        productType: row.product_type,
        completionStatus: row.completion_status,
        destination: row.destination,
        destinationOperationId: row.destination_operation_id,
        destinationOperationName: row.destination_operation_name,
        destinationLocationId: row.destination_location_id,
        destinationLocationName: row.destination_location_name,
        destinationName: row.destination_name,
        priority: row.priority,
        isActive: row.is_active,
        name: row.name,
        description: row.description,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
        createdBy: row.created_by,
        createdByUsername: row.created_by_username,
      }));
      
      res.json({ rules });
    } catch (error) {
      console.error("Error fetching warehouse routing rules:", error);
      res.status(500).json({ message: "Failed to fetch routing rules" });
    }
  });

  // GET /api/production/planning/routing-rules/:id - Get a single routing rule
  app.get("/api/production/planning/routing-rules/:id", requirePermission('view_production'), async (req, res) => {
    try {
      const ruleId = parseInt(req.params.id);
      if (isNaN(ruleId)) {
        return res.status(400).json({ message: "Invalid rule ID" });
      }
      
      const result = await pool.query(`
        SELECT 
          wrr.*,
          pp.plan_number,
          pp.name as plan_name,
          cp.title as catalog_product_name,
          pro.name as destination_operation_name,
          pl.name as destination_location_name
        FROM production.warehouse_routing_rules wrr
        LEFT JOIN production.production_plans pp ON pp.id = wrr.plan_id
        LEFT JOIN catalog.products cp ON cp.id = wrr.catalog_product_id
        LEFT JOIN production.production_routing_operations pro ON pro.id = wrr.destination_operation_id
        LEFT JOIN production.production_locations pl ON pl.id = wrr.destination_location_id
        WHERE wrr.id = $1
      `, [ruleId]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Routing rule not found" });
      }
      
      const row = result.rows[0];
      res.json({
        id: row.id,
        planId: row.plan_id,
        planNumber: row.plan_number,
        planName: row.plan_name,
        catalogProductId: row.catalog_product_id,
        catalogProductName: row.catalog_product_name,
        productSku: row.product_sku,
        productType: row.product_type,
        completionStatus: row.completion_status,
        destination: row.destination,
        destinationOperationId: row.destination_operation_id,
        destinationOperationName: row.destination_operation_name,
        destinationLocationId: row.destination_location_id,
        destinationLocationName: row.destination_location_name,
        destinationName: row.destination_name,
        priority: row.priority,
        isActive: row.is_active,
        name: row.name,
        description: row.description,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
        createdBy: row.created_by,
      });
    } catch (error) {
      console.error("Error fetching routing rule:", error);
      res.status(500).json({ message: "Failed to fetch routing rule" });
    }
  });

  // POST /api/production/planning/routing-rules - Create a new routing rule
  app.post("/api/production/planning/routing-rules", requirePermission('manage_production'), async (req, res) => {
    try {
      const schema = z.object({
        planId: z.number().int().positive().nullable().optional(),
        catalogProductId: z.number().int().positive().nullable().optional(),
        productSku: z.string().max(100).nullable().optional(),
        productType: z.string().max(100).nullable().optional(),
        completionStatus: z.enum(['incomplete', 'complete', 'any']).default('incomplete'),
        destination: z.string().min(1).max(50),
        destinationOperationId: z.number().int().positive().nullable().optional(),
        destinationLocationId: z.number().int().positive().nullable().optional(),
        destinationName: z.string().max(255).nullable().optional(),
        priority: z.number().int().default(0),
        isActive: z.boolean().default(true),
        name: z.string().max(255).nullable().optional(),
        description: z.string().nullable().optional(),
      });
      
      const data = schema.parse(req.body);
      const userId = (req as any).user?.id;
      
      const result = await pool.query(`
        INSERT INTO production.warehouse_routing_rules (
          plan_id, catalog_product_id, product_sku, product_type,
          completion_status, destination, destination_operation_id, destination_location_id,
          destination_name, priority, is_active, name, description, created_by,
          created_at, updated_at
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, NOW(), NOW())
        RETURNING id
      `, [
        data.planId || null,
        data.catalogProductId || null,
        data.productSku || null,
        data.productType || null,
        data.completionStatus,
        data.destination,
        data.destinationOperationId || null,
        data.destinationLocationId || null,
        data.destinationName || null,
        data.priority,
        data.isActive,
        data.name || null,
        data.description || null,
        userId || null,
      ]);
      
      res.status(201).json({ 
        id: result.rows[0].id,
        message: "Routing rule created successfully" 
      });
    } catch (error: any) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Invalid data", errors: error.errors });
      }
      console.error("Error creating routing rule:", error);
      res.status(500).json({ message: "Failed to create routing rule" });
    }
  });

  // PATCH /api/production/planning/routing-rules/:id - Update a routing rule
  app.patch("/api/production/planning/routing-rules/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const ruleId = parseInt(req.params.id);
      if (isNaN(ruleId)) {
        return res.status(400).json({ message: "Invalid rule ID" });
      }
      
      const schema = z.object({
        planId: z.number().int().positive().nullable().optional(),
        catalogProductId: z.number().int().positive().nullable().optional(),
        productSku: z.string().max(100).nullable().optional(),
        productType: z.string().max(100).nullable().optional(),
        completionStatus: z.enum(['incomplete', 'complete', 'any']).optional(),
        destination: z.string().min(1).max(50).optional(),
        destinationOperationId: z.number().int().positive().nullable().optional(),
        destinationLocationId: z.number().int().positive().nullable().optional(),
        destinationName: z.string().max(255).nullable().optional(),
        priority: z.number().int().optional(),
        isActive: z.boolean().optional(),
        name: z.string().max(255).nullable().optional(),
        description: z.string().nullable().optional(),
      });
      
      const data = schema.parse(req.body);
      
      const fields: string[] = [];
      const values: any[] = [];
      let paramIndex = 1;
      
      if (data.planId !== undefined) {
        fields.push(`plan_id = $${paramIndex++}`);
        values.push(data.planId);
      }
      if (data.catalogProductId !== undefined) {
        fields.push(`catalog_product_id = $${paramIndex++}`);
        values.push(data.catalogProductId);
      }
      if (data.productSku !== undefined) {
        fields.push(`product_sku = $${paramIndex++}`);
        values.push(data.productSku);
      }
      if (data.productType !== undefined) {
        fields.push(`product_type = $${paramIndex++}`);
        values.push(data.productType);
      }
      if (data.completionStatus !== undefined) {
        fields.push(`completion_status = $${paramIndex++}`);
        values.push(data.completionStatus);
      }
      if (data.destination !== undefined) {
        fields.push(`destination = $${paramIndex++}`);
        values.push(data.destination);
      }
      if (data.destinationOperationId !== undefined) {
        fields.push(`destination_operation_id = $${paramIndex++}`);
        values.push(data.destinationOperationId);
      }
      if (data.destinationLocationId !== undefined) {
        fields.push(`destination_location_id = $${paramIndex++}`);
        values.push(data.destinationLocationId);
      }
      if (data.destinationName !== undefined) {
        fields.push(`destination_name = $${paramIndex++}`);
        values.push(data.destinationName);
      }
      if (data.priority !== undefined) {
        fields.push(`priority = $${paramIndex++}`);
        values.push(data.priority);
      }
      if (data.isActive !== undefined) {
        fields.push(`is_active = $${paramIndex++}`);
        values.push(data.isActive);
      }
      if (data.name !== undefined) {
        fields.push(`name = $${paramIndex++}`);
        values.push(data.name);
      }
      if (data.description !== undefined) {
        fields.push(`description = $${paramIndex++}`);
        values.push(data.description);
      }
      
      if (fields.length === 0) {
        return res.status(400).json({ message: "No fields to update" });
      }
      
      fields.push(`updated_at = NOW()`);
      values.push(ruleId);
      
      const result = await pool.query(`
        UPDATE production.warehouse_routing_rules
        SET ${fields.join(', ')}
        WHERE id = $${paramIndex}
        RETURNING id
      `, values);
      
      if (result.rowCount === 0) {
        return res.status(404).json({ message: "Routing rule not found" });
      }
      
      res.json({ message: "Routing rule updated successfully" });
    } catch (error: any) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Invalid data", errors: error.errors });
      }
      console.error("Error updating routing rule:", error);
      res.status(500).json({ message: "Failed to update routing rule" });
    }
  });

  // DELETE /api/production/planning/routing-rules/:id - Delete a routing rule
  app.delete("/api/production/planning/routing-rules/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const ruleId = parseInt(req.params.id);
      if (isNaN(ruleId)) {
        return res.status(400).json({ message: "Invalid rule ID" });
      }
      
      const result = await pool.query(`
        DELETE FROM production.warehouse_routing_rules
        WHERE id = $1
        RETURNING id
      `, [ruleId]);
      
      if (result.rowCount === 0) {
        return res.status(404).json({ message: "Routing rule not found" });
      }
      
      res.json({ message: "Routing rule deleted successfully" });
    } catch (error) {
      console.error("Error deleting routing rule:", error);
      res.status(500).json({ message: "Failed to delete routing rule" });
    }
  });

  // GET /api/production/planning/routing-rules/destinations - Get available destinations
  app.get("/api/production/planning/routing-rules/destinations", requirePermission('view_production'), async (req, res) => {
    try {
      // Get all routing operations that could be destinations
      const operationsResult = await pool.query(`
        SELECT id, code, name
        FROM production.production_routing_operations
        WHERE is_active = true
        ORDER BY sequence, name
      `);
      
      // Get all production locations that could be destinations
      const locationsResult = await pool.query(`
        SELECT id, code, name, location_type
        FROM production.production_locations
        WHERE is_active = true
        ORDER BY name
      `);
      
      const destinations = {
        predefined: [
          { value: 'packing', label: 'Pakowanie', description: 'Produkt wymaga uzupełnienia/pakowania' },
          { value: 'shipping_buffer', label: 'Bufor wysyłkowy', description: 'Produkt gotowy do wysyłki' },
          { value: 'assembly', label: 'Montaż/Kompletowanie', description: 'Produkt wymaga montażu' },
        ],
        operations: operationsResult.rows.map(row => ({
          id: row.id,
          code: row.code,
          name: row.name,
          value: `operation:${row.id}`,
        })),
        locations: locationsResult.rows.map(row => ({
          id: row.id,
          code: row.code,
          name: row.name,
          locationType: row.location_type,
          value: `location:${row.id}`,
        })),
      };
      
      res.json(destinations);
    } catch (error) {
      console.error("Error fetching destinations:", error);
      res.status(500).json({ message: "Failed to fetch destinations" });
    }
  });

  // ============= ASSEMBLY STATION ENDPOINTS =============

  // GET /api/production/plans/:planId/assembly-station - Get assembly station data for a plan
  app.get("/api/production/plans/:planId/assembly-station", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // Get plan info
      const planResult = await pool.query(`
        SELECT id, plan_number, name, status
        FROM production.production_plans
        WHERE id = $1
      `, [planId]);

      if (planResult.rows.length === 0) {
        return res.status(404).json({ message: "Plan not found" });
      }

      const plan = planResult.rows[0];

      // Get all ZLP orders for this plan with their status
      const zlpOrdersResult = await pool.query(`
        SELECT 
          po.id as order_id,
          po.order_number,
          po.status,
          pob.source_plan_id,
          COALESCE(
            (SELECT color_code FROM production.production_order_bom_items pobi 
             WHERE pobi.production_order_bom_id = pob.id 
             LIMIT 1), 
            'UNKNOWN'
          ) as color_code
        FROM production.production_orders po
        JOIN production.production_order_boms pob ON pob.production_order_id = po.id
        WHERE pob.source_plan_id = $1
        ORDER BY po.order_number
      `, [planId]);

      // Get work orders status for each ZLP to determine readiness
      const zlpColors = await Promise.all(zlpOrdersResult.rows.map(async (zlp) => {
        // Check if all operations before assembly are done
        const preAssemblyResult = await pool.query(`
          SELECT 
            COUNT(*) FILTER (WHERE ro.code NOT IN ('assembly', 'packing') AND wo.status != 'done') as pending_before_assembly,
            COUNT(*) FILTER (WHERE ro.code = 'assembly' AND wo.status = 'done') as assembly_done,
            COUNT(*) FILTER (WHERE ro.code = 'packing' AND wo.status = 'done') as packing_done,
            COUNT(*) as total_operations
          FROM production.production_work_orders wo
          LEFT JOIN production.production_routing_operations ro ON ro.id = wo.routing_operation_id
          WHERE wo.production_order_id = $1
        `, [zlp.order_id]);

        const stats = preAssemblyResult.rows[0] || {};
        const assemblyReady = parseInt(stats.pending_before_assembly || '0') === 0;
        const packingReady = parseInt(stats.assembly_done || '0') > 0 || assemblyReady;

        // Count formatki
        const formatkiResult = await pool.query(`
          SELECT 
            COUNT(*) as total,
            COUNT(*) FILTER (WHERE last_warehouse_doc_id IS NOT NULL) as done
          FROM production.production_order_bom_items pobi
          JOIN production.production_order_boms pob ON pob.id = pobi.production_order_bom_id
          WHERE pob.production_order_id = $1
        `, [zlp.order_id]);

        const formatkiStats = formatkiResult.rows[0] || {};

        return {
          orderId: zlp.order_id,
          orderNumber: zlp.order_number,
          colorCode: zlp.color_code,
          colorName: zlp.color_code,
          status: zlp.status,
          assemblyReady,
          packingReady,
          componentsTotal: parseInt(formatkiStats.total || '0'),
          componentsDone: parseInt(formatkiStats.done || '0'),
        };
      }));

      // Get all formatki counts
      const allFormatkiResult = await pool.query(`
        SELECT 
          COUNT(*) as total,
          COUNT(*) FILTER (WHERE COALESCE(pobi.is_damaged, false) = false AND pobi.last_warehouse_doc_id IS NOT NULL) as done,
          COUNT(*) FILTER (WHERE pobi.is_damaged = true) as damaged
        FROM production.production_order_bom_items pobi
        JOIN production.production_order_boms pob ON pob.id = pobi.production_order_bom_id
        WHERE pob.source_plan_id = $1
      `, [planId]);

      const formatkiStats = allFormatkiResult.rows[0] || {};

      // Get assembly and packing operations status
      const assemblyOpsResult = await pool.query(`
        SELECT 
          ro.code as operation_code,
          ro.name as operation_name,
          COUNT(*) as total_items,
          COUNT(*) FILTER (WHERE wo.status = 'done') as completed_items,
          MIN(wo.status) as min_status
        FROM production.production_work_orders wo
        JOIN production.production_routing_operations ro ON ro.id = wo.routing_operation_id
        JOIN production.production_orders po ON po.id = wo.production_order_id
        JOIN production.production_order_boms pob ON pob.production_order_id = po.id
        WHERE pob.source_plan_id = $1
          AND ro.code = 'assembly'
        GROUP BY ro.code, ro.name
      `, [planId]);

      const packingOpsResult = await pool.query(`
        SELECT 
          ro.code as operation_code,
          ro.name as operation_name,
          COUNT(*) as total_items,
          COUNT(*) FILTER (WHERE wo.status = 'done') as completed_items,
          MIN(wo.status) as min_status
        FROM production.production_work_orders wo
        JOIN production.production_routing_operations ro ON ro.id = wo.routing_operation_id
        JOIN production.production_orders po ON po.id = wo.production_order_id
        JOIN production.production_order_boms pob ON pob.production_order_id = po.id
        WHERE pob.source_plan_id = $1
          AND ro.code = 'packing'
        GROUP BY ro.code, ro.name
      `, [planId]);

      const assemblyOperations = assemblyOpsResult.rows.map(row => ({
        operationCode: row.operation_code,
        operationName: row.operation_name,
        status: row.min_status === 'done' ? 'done' : (row.completed_items > 0 ? 'in_progress' : 'pending'),
        totalItems: parseInt(row.total_items),
        completedItems: parseInt(row.completed_items),
        canStart: zlpColors.every(z => z.assemblyReady),
      }));

      const packingOperations = packingOpsResult.rows.map(row => ({
        operationCode: row.operation_code,
        operationName: row.operation_name,
        status: row.min_status === 'done' ? 'done' : (row.completed_items > 0 ? 'in_progress' : 'pending'),
        totalItems: parseInt(row.total_items),
        completedItems: parseInt(row.completed_items),
        canStart: assemblyOperations.every(a => a.status === 'done'),
      }));

      res.json({
        planId: plan.id,
        planNumber: plan.plan_number,
        planName: plan.name,
        zlpColors,
        assemblyOperations,
        packingOperations,
        allFormatkiCount: parseInt(formatkiStats.total || '0'),
        allFormatkiDone: parseInt(formatkiStats.done || '0'),
        allFormatkiDamaged: parseInt(formatkiStats.damaged || '0'),
      });
    } catch (error) {
      console.error("Error fetching assembly station data:", error);
      res.status(500).json({ message: "Failed to fetch assembly station data" });
    }
  });

  // GET /api/production/plans/:planId/all-formatki - Get all formatki for a plan with filtering
  app.get("/api/production/plans/:planId/all-formatki", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // Get filter parameters
      const { colorCode, zlpOrderNumber, isAssembled, isDamaged, searchText, length, width } = req.query;

      let whereConditions = ['pob.source_plan_id = $1'];
      let params: any[] = [planId];
      let paramIndex = 2;

      // Search text with semicolon separator (e.g., "bo;cza" searches for items containing "bo" OR "cza")
      if (searchText && typeof searchText === 'string' && searchText.trim()) {
        const searchTokens = searchText.split(';').map(t => t.trim().toLowerCase()).filter(t => t.length > 0);
        if (searchTokens.length > 0) {
          const searchConditions = searchTokens.map(token => {
            params.push(`%${token}%`);
            const idx = paramIndex++;
            return `(LOWER(pobi.component_name) LIKE $${idx} OR LOWER(pobi.color_code) LIKE $${idx} OR LOWER(po.order_number) LIKE $${idx})`;
          });
          whereConditions.push(`(${searchConditions.join(' OR ')})`);
        }
      }

      // Color code filter with semicolon separator
      if (colorCode && colorCode !== 'all' && typeof colorCode === 'string') {
        const colorTokens = colorCode.split(';').map(c => c.trim()).filter(c => c.length > 0);
        if (colorTokens.length === 1) {
          whereConditions.push(`pobi.color_code = $${paramIndex}`);
          params.push(colorTokens[0]);
          paramIndex++;
        } else if (colorTokens.length > 1) {
          whereConditions.push(`pobi.color_code = ANY($${paramIndex})`);
          params.push(colorTokens);
          paramIndex++;
        }
      }

      if (zlpOrderNumber && zlpOrderNumber !== 'all') {
        whereConditions.push(`po.order_number = $${paramIndex}`);
        params.push(zlpOrderNumber);
        paramIndex++;
      }

      // Length filter (exact value)
      if (length && !isNaN(parseFloat(length as string))) {
        whereConditions.push(`pobi.length = $${paramIndex}`);
        params.push(parseFloat(length as string));
        paramIndex++;
      }

      // Width filter (exact value)
      if (width && !isNaN(parseFloat(width as string))) {
        whereConditions.push(`pobi.width = $${paramIndex}`);
        params.push(parseFloat(width as string));
        paramIndex++;
      }

      if (isAssembled === 'true') {
        whereConditions.push('pobi.is_assembled = true');
      } else if (isAssembled === 'false') {
        whereConditions.push('(pobi.is_assembled = false OR pobi.is_assembled IS NULL)');
      }

      if (isDamaged === 'true') {
        whereConditions.push('pobi.is_damaged = true');
      } else if (isDamaged === 'false') {
        whereConditions.push('(pobi.is_damaged = false OR pobi.is_damaged IS NULL)');
      }

      const result = await pool.query(`
        SELECT 
          pobi.id,
          pobi.component_name,
          pobi.component_type,
          pobi.color_code,
          pobi.length,
          pobi.width,
          pobi.thickness,
          pobi.quantity,
          pobi.is_damaged,
          pobi.damage_type,
          pobi.damage_notes,
          pobi.is_assembled,
          pobi.assembled_at,
          pobi.is_cut,
          pobi.is_edged,
          pobi.is_drilled,
          po.order_number as zlp_order_number
        FROM production.production_order_bom_items pobi
        JOIN production.production_order_boms pob ON pob.id = pobi.production_order_bom_id
        JOIN production.production_orders po ON po.id = pob.production_order_id
        WHERE ${whereConditions.join(' AND ')}
        ORDER BY po.order_number, pobi.color_code, pobi.component_name
      `, params);

      res.json(result.rows.map(row => ({
        id: row.id,
        componentName: row.component_name,
        componentType: row.component_type,
        colorCode: row.color_code,
        length: parseFloat(row.length) || 0,
        width: parseFloat(row.width) || 0,
        thickness: parseFloat(row.thickness) || 0,
        quantity: parseInt(row.quantity) || 1,
        isDamaged: row.is_damaged || false,
        damageType: row.damage_type,
        damageNotes: row.damage_notes,
        isAssembled: row.is_assembled || false,
        assembledAt: row.assembled_at,
        isCut: row.is_cut || false,
        isEdged: row.is_edged || false,
        isDrilled: row.is_drilled || false,
        zlpOrderNumber: row.zlp_order_number,
      })));
    } catch (error) {
      console.error("Error fetching plan formatki:", error);
      res.status(500).json({ message: "Failed to fetch formatki" });
    }
  });

  // GET /api/production/plans/:planId/packing-products - Get products with their formatki for packing view
  app.get("/api/production/plans/:planId/packing-products", requirePermission('view_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // Verify plan exists
      const planCheck = await pool.query(`
        SELECT id FROM production.production_plans WHERE id = $1
      `, [planId]);
      
      if (planCheck.rows.length === 0) {
        return res.status(404).json({ message: "Plan not found" });
      }

      // Get all plan lines (products) with their formatki
      // Formatki are linked to plan lines via production_order_bom_items.source_plan_line_id
      const result = await pool.query(`
        SELECT 
          ppl.id as line_id,
          ppl.product_id,
          ppl.quantity,
          ppl.reserved_quantity,
          ppl.bom_id,
          ppl.source_reference,
          cp.sku as product_sku,
          cp.title as product_title,
          COALESCE(
            (SELECT pi.url FROM catalog.product_images pi WHERE pi.product_id = cp.id AND pi.is_primary = true LIMIT 1),
            cp.image_url
          ) as product_image,
          (
            SELECT json_agg(json_build_object(
              'id', pobi.id,
              'componentName', pobi.component_name,
              'componentType', pobi.component_type,
              'colorCode', pobi.color_code,
              'length', pobi.length,
              'width', pobi.width,
              'thickness', pobi.thickness,
              'quantity', pobi.quantity,
              'isDamaged', COALESCE(pobi.is_damaged, false),
              'isAssembled', COALESCE(pobi.is_assembled, false),
              'isCut', COALESCE(pobi.is_cut, false),
              'isEdged', COALESCE(pobi.is_edged, false),
              'isDrilled', COALESCE(pobi.is_drilled, false),
              'isFromWarehouse', false,
              'zlpOrderNumber', po.order_number
            ) ORDER BY pobi.component_name)
            FROM production.production_order_bom_items pobi
            JOIN production.production_order_boms pob ON pob.id = pobi.production_order_bom_id
            JOIN production.production_orders po ON po.id = pob.production_order_id
            WHERE pobi.source_plan_line_id = ppl.id
          ) as formatki
        FROM production.production_plan_lines ppl
        JOIN catalog.products cp ON cp.id = ppl.product_id
        WHERE ppl.plan_id = $1 AND ppl.deleted_at IS NULL
        ORDER BY ppl.sequence NULLS LAST, ppl.id
      `, [planId]);

      // Process each row and fetch BOM formatki for warehouse products
      const products = await Promise.all(result.rows.map(async (row) => {
        const reservedQuantity = parseInt(row.reserved_quantity) || 0;
        let formatki = row.formatki || [];
        
        // If product is from warehouse (reservedQuantity > 0) and has no formatki from production,
        // fetch them from the BOM directly and mark as already assembled (from warehouse)
        if (reservedQuantity > 0 && formatki.length === 0 && row.bom_id) {
          // Fetch BOM components for this product
          const bomComponentsResult = await pool.query(`
            SELECT 
              pc.id as component_id,
              pc.generated_name as component_name,
              pc.component_type,
              pc.color as color_code,
              pc.calculated_length as length,
              pc.calculated_width as width,
              pc.thickness,
              pc.quantity
            FROM bom.product_components pc
            WHERE pc.product_bom_id = $1
            ORDER BY pc.id
          `, [row.bom_id]);
          
          // Transform BOM components to formatki format, marked as from warehouse
          formatki = bomComponentsResult.rows.map((comp, idx) => ({
            id: -(row.line_id * 1000 + idx), // Negative ID to distinguish from real formatki
            componentName: comp.component_name,
            componentType: comp.component_type,
            colorCode: comp.color_code,
            length: parseFloat(comp.length) || 0,
            width: parseFloat(comp.width) || 0,
            thickness: parseFloat(comp.thickness) || 0,
            quantity: parseInt(comp.quantity) || 1,
            isDamaged: false,
            isAssembled: true, // Already assembled when packed to warehouse
            isCut: true,
            isEdged: true,
            isDrilled: true,
            isFromWarehouse: true, // Flag to disable editing in frontend
            zlpOrderNumber: null
          }));
        }
        
        return {
          lineId: row.line_id,
          productId: row.product_id,
          productSku: row.product_sku,
          productTitle: row.product_title,
          productImage: row.product_image,
          quantity: parseInt(row.quantity) || 1,
          reservedQuantity,
          sourceReference: row.source_reference,
          formatki,
          formatkiTotal: formatki.length,
          formatkiAssembled: formatki.filter((f: any) => f.isAssembled).length,
          formatkiDamaged: formatki.filter((f: any) => f.isDamaged).length,
        };
      }));

      res.json(products);
    } catch (error) {
      console.error("Error fetching packing products:", error);
      res.status(500).json({ message: "Failed to fetch packing products" });
    }
  });

  // PATCH /api/production/plans/:planId/formatki/bulk-assemble - Bulk mark formatki as assembled
  app.patch("/api/production/plans/:planId/formatki/bulk-assemble", requirePermission('manage_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const { ids, isAssembled } = req.body;
      if (!Array.isArray(ids) || ids.length === 0) {
        return res.status(400).json({ message: "Nie podano ID formatek" });
      }

      // Validate all IDs are numbers
      const validIds = ids.filter(id => typeof id === 'number' && Number.isInteger(id) && id > 0);
      if (validIds.length !== ids.length) {
        return res.status(400).json({ message: "Nieprawidłowe ID formatek" });
      }

      const userId = (req as any).user?.id ? parseInt((req as any).user.id) : null;

      // Update only formatki that belong to this plan (secure update with JOIN)
      const result = await pool.query(`
        UPDATE production.production_order_bom_items pobi
        SET 
          is_assembled = $1,
          assembled_at = CASE WHEN $1 = true THEN NOW() ELSE NULL END,
          assembled_by_user_id = CASE WHEN $1 = true THEN $2::integer ELSE NULL END,
          updated_at = NOW()
        FROM production.production_order_boms pob
        WHERE pobi.production_order_bom_id = pob.id
          AND pob.source_plan_id = $3
          AND pobi.id = ANY($4)
        RETURNING pobi.id
      `, [isAssembled !== false, userId, planId, validIds]);

      if (result.rowCount === 0) {
        return res.status(404).json({ 
          message: "Nie znaleziono formatek do aktualizacji lub nie należą do tego planu",
          updatedCount: 0
        });
      }

      if (result.rowCount !== validIds.length) {
        // Some formatki were not in this plan - partial update
        res.json({ 
          message: isAssembled !== false 
            ? `Oznaczono ${result.rowCount} z ${validIds.length} formatek jako skompletowane (niektóre nie należą do tego planu)` 
            : `Cofnięto oznaczenie ${result.rowCount} z ${validIds.length} formatek`,
          updatedCount: result.rowCount,
          requestedCount: validIds.length
        });
      } else {
        res.json({ 
          message: isAssembled !== false 
            ? `Oznaczono ${result.rowCount} formatek jako skompletowane` 
            : `Cofnięto oznaczenie ${result.rowCount} formatek`,
          updatedCount: result.rowCount 
        });
      }
    } catch (error) {
      console.error("Error bulk updating formatki:", error);
      res.status(500).json({ message: "Failed to update formatki" });
    }
  });

  // POST /api/production/plans/:planId/assembly-operations/start - Start assembly for all ZLP colors
  app.post("/api/production/plans/:planId/assembly-operations/start", requirePermission('manage_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // First check current status of assembly work orders
      const statusCheck = await pool.query(`
        SELECT wo.status, COUNT(*) as count
        FROM production.production_work_orders wo
        JOIN production.production_routing_operations ro ON wo.routing_operation_id = ro.id
        JOIN production.production_orders po ON wo.production_order_id = po.id
        JOIN production.production_order_boms pob ON pob.production_order_id = po.id
        WHERE pob.source_plan_id = $1
          AND ro.code = 'assembly'
        GROUP BY wo.status
      `, [planId]);

      const statusMap: Record<string, number> = {};
      statusCheck.rows.forEach((row: any) => {
        statusMap[row.status] = parseInt(row.count);
      });

      const pending = statusMap['pending'] || 0;
      const inProgress = statusMap['in_progress'] || 0;
      const done = statusMap['done'] || 0;
      const total = pending + inProgress + done;

      if (total === 0) {
        return res.json({ 
          message: "Brak operacji kompletowania dla tego planu", 
          updatedCount: 0,
          status: { pending: 0, inProgress: 0, done: 0 }
        });
      }

      if (done === total) {
        return res.json({ 
          message: "Wszystkie operacje kompletowania są już zakończone", 
          updatedCount: 0,
          status: { pending, inProgress, done }
        });
      }

      if (inProgress > 0 && pending === 0) {
        return res.json({ 
          message: `Kompletowanie już trwa (${inProgress} w trakcie, ${done} zakończonych)`, 
          updatedCount: 0,
          status: { pending, inProgress, done }
        });
      }

      // Update pending assembly work orders to in_progress
      const result = await pool.query(`
        UPDATE production.production_work_orders wo
        SET status = 'in_progress', actual_start_time = COALESCE(wo.actual_start_time, NOW())
        FROM production.production_routing_operations ro,
             production.production_orders po,
             production.production_order_boms pob
        WHERE wo.routing_operation_id = ro.id
          AND wo.production_order_id = po.id
          AND pob.production_order_id = po.id
          AND pob.source_plan_id = $1
          AND ro.code = 'assembly'
          AND wo.status = 'pending'
        RETURNING wo.id
      `, [planId]);

      res.json({ 
        message: `Rozpoczęto kompletowanie (${result.rowCount} operacji)`, 
        updatedCount: result.rowCount,
        status: { pending: 0, inProgress: inProgress + (result.rowCount || 0), done }
      });
    } catch (error) {
      console.error("Error starting assembly operations:", error);
      res.status(500).json({ message: "Failed to start assembly operations" });
    }
  });

  // POST /api/production/plans/:planId/assembly-operations/complete - Complete assembly for all ZLP colors
  app.post("/api/production/plans/:planId/assembly-operations/complete", requirePermission('manage_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // Update all assembly work orders for this plan to done
      const result = await pool.query(`
        UPDATE production.production_work_orders wo
        SET status = 'done', actual_end_time = NOW()
        FROM production.production_routing_operations ro,
             production.production_orders po,
             production.production_order_boms pob
        WHERE wo.routing_operation_id = ro.id
          AND wo.production_order_id = po.id
          AND pob.production_order_id = po.id
          AND pob.source_plan_id = $1
          AND ro.code = 'assembly'
          AND wo.status IN ('pending', 'in_progress')
        RETURNING wo.id
      `, [planId]);

      res.json({ 
        message: "Assembly operations completed", 
        updatedCount: result.rowCount 
      });
    } catch (error) {
      console.error("Error completing assembly operations:", error);
      res.status(500).json({ message: "Failed to complete assembly operations" });
    }
  });

  // POST /api/production/plans/:planId/packing-operations/start - Start packing for all ZLP colors
  app.post("/api/production/plans/:planId/packing-operations/start", requirePermission('manage_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // First check current status of packing work orders
      const statusCheck = await pool.query(`
        SELECT wo.status, COUNT(*) as count
        FROM production.production_work_orders wo
        JOIN production.production_routing_operations ro ON wo.routing_operation_id = ro.id
        JOIN production.production_orders po ON wo.production_order_id = po.id
        JOIN production.production_order_boms pob ON pob.production_order_id = po.id
        WHERE pob.source_plan_id = $1
          AND ro.code = 'packing'
        GROUP BY wo.status
      `, [planId]);

      const statusMap: Record<string, number> = {};
      statusCheck.rows.forEach((row: any) => {
        statusMap[row.status] = parseInt(row.count);
      });

      const pending = statusMap['pending'] || 0;
      const inProgress = statusMap['in_progress'] || 0;
      const done = statusMap['done'] || 0;
      const total = pending + inProgress + done;

      if (total === 0) {
        return res.json({ 
          message: "Brak operacji pakowania dla tego planu", 
          updatedCount: 0,
          status: { pending: 0, inProgress: 0, done: 0 }
        });
      }

      if (done === total) {
        return res.json({ 
          message: "Wszystkie operacje pakowania są już zakończone", 
          updatedCount: 0,
          status: { pending, inProgress, done }
        });
      }

      if (inProgress > 0 && pending === 0) {
        return res.json({ 
          message: `Pakowanie już trwa (${inProgress} w trakcie, ${done} zakończonych)`, 
          updatedCount: 0,
          status: { pending, inProgress, done }
        });
      }

      // Update pending packing work orders to in_progress
      const result = await pool.query(`
        UPDATE production.production_work_orders wo
        SET status = 'in_progress', actual_start_time = COALESCE(wo.actual_start_time, NOW())
        FROM production.production_routing_operations ro,
             production.production_orders po,
             production.production_order_boms pob
        WHERE wo.routing_operation_id = ro.id
          AND wo.production_order_id = po.id
          AND pob.production_order_id = po.id
          AND pob.source_plan_id = $1
          AND ro.code = 'packing'
          AND wo.status = 'pending'
        RETURNING wo.id
      `, [planId]);

      res.json({ 
        message: `Rozpoczęto pakowanie (${result.rowCount} operacji)`, 
        updatedCount: result.rowCount,
        status: { pending: 0, inProgress: inProgress + (result.rowCount || 0), done }
      });
    } catch (error) {
      console.error("Error starting packing operations:", error);
      res.status(500).json({ message: "Failed to start packing operations" });
    }
  });

  // POST /api/production/plans/:planId/packing-operations/complete - Complete packing for all ZLP colors
  app.post("/api/production/plans/:planId/packing-operations/complete", requirePermission('manage_production'), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // Update all packing work orders for this plan to done
      const result = await pool.query(`
        UPDATE production.production_work_orders wo
        SET status = 'done', actual_end_time = NOW()
        FROM production.production_routing_operations ro,
             production.production_orders po,
             production.production_order_boms pob
        WHERE wo.routing_operation_id = ro.id
          AND wo.production_order_id = po.id
          AND pob.production_order_id = po.id
          AND pob.source_plan_id = $1
          AND ro.code = 'packing'
          AND wo.status IN ('pending', 'in_progress')
        RETURNING wo.id
      `, [planId]);

      res.json({ 
        message: "Packing operations completed", 
        updatedCount: result.rowCount 
      });
    } catch (error) {
      console.error("Error completing packing operations:", error);
      res.status(500).json({ message: "Failed to complete packing operations" });
    }
  });
}
