import type { Express } from "express";
import { requirePermission } from "../../auth";
import { z } from "zod";
import { pool } from "../../postgres";

// Helper function to check if plan is ZLP locked
async function checkPlanZlpLocked(planId: number): Promise<{ locked: boolean; message?: string }> {
  const result = await pool.query(
    `SELECT zlp_locked FROM production.production_plans WHERE id = $1`,
    [planId]
  );
  
  if (result.rows.length === 0) {
    return { locked: false };
  }
  
  if (result.rows[0].zlp_locked) {
    return { 
      locked: true, 
      message: "ZLP jest zatwierdzone - modyfikacje są zablokowane. Odblokuj ZLP aby edytować." 
    };
  }
  
  return { locked: false };
}

// Helper to get plan ID from assignment ID
async function getPlanIdFromAssignment(assignmentId: number): Promise<number | null> {
  const result = await pool.query(
    `SELECT plan_id FROM production.plan_routing_assignments WHERE id = $1`,
    [assignmentId]
  );
  return result.rows[0]?.plan_id || null;
}

const assignmentFormSchema = z.object({
  planId: z.number(),
  materialType: z.string().nullable().optional(),
  colorCode: z.string().nullable().optional(),
  componentPattern: z.string().nullable().optional(),
  routingId: z.number(),
  locationId: z.number().nullable().optional(),
  overrideWorkCenterId: z.number().nullable().optional(),
  templateId: z.number().nullable().optional(),
  priority: z.number().default(0),
  notes: z.string().nullable().optional(),
});

export function registerPlanRoutingAssignmentRoutes(app: Express) {
  // GET /api/production/plans/:planId/routing-assignments - Get all routing assignments for a plan
  app.get("/api/production/plans/:planId/routing-assignments", requirePermission("view_production"), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      const result = await pool.query(`
        SELECT 
          a.*,
          r.code as routing_code,
          r.name as routing_name,
          l.code as location_code,
          l.name as location_name,
          wc.code as work_center_code,
          wc.name as work_center_name,
          t.name as template_name
        FROM production.plan_routing_assignments a
        LEFT JOIN production.production_routings r ON r.id = a.routing_id
        LEFT JOIN production.production_locations l ON l.id = a.location_id
        LEFT JOIN production.production_work_centers wc ON wc.id = a.override_work_center_id
        LEFT JOIN production.component_routing_templates t ON t.id = a.template_id
        WHERE a.plan_id = $1
        ORDER BY a.priority DESC, a.id ASC
      `, [planId]);

      const assignments = result.rows.map(row => ({
        id: row.id,
        planId: row.plan_id,
        materialType: row.material_type,
        colorCode: row.color_code,
        componentPattern: row.component_pattern,
        routingId: row.routing_id,
        routingCode: row.routing_code,
        routingName: row.routing_name,
        locationId: row.location_id,
        locationCode: row.location_code,
        locationName: row.location_name,
        overrideWorkCenterId: row.override_work_center_id,
        workCenterCode: row.work_center_code,
        workCenterName: row.work_center_name,
        templateId: row.template_id,
        templateName: row.template_name,
        priority: row.priority,
        notes: row.notes,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
      }));

      res.json(assignments);
    } catch (error) {
      console.error("Error fetching plan routing assignments:", error);
      res.status(500).json({ message: "Failed to fetch routing assignments" });
    }
  });

  // POST /api/production/plans/:planId/routing-assignments - Create new routing assignment
  app.post("/api/production/plans/:planId/routing-assignments", requirePermission("manage_production"), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // Check if ZLP is locked
      const lockCheck = await checkPlanZlpLocked(planId);
      if (lockCheck.locked) {
        return res.status(403).json({ message: lockCheck.message });
      }

      const data = assignmentFormSchema.parse({ ...req.body, planId });

      const result = await pool.query(`
        INSERT INTO production.plan_routing_assignments 
          (plan_id, material_type, color_code, component_pattern, routing_id, location_id, override_work_center_id, template_id, priority, notes)
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
        RETURNING *
      `, [
        data.planId,
        data.materialType || null,
        data.colorCode || null,
        data.componentPattern || null,
        data.routingId,
        data.locationId || null,
        data.overrideWorkCenterId || null,
        data.templateId || null,
        data.priority,
        data.notes || null,
      ]);

      const row = result.rows[0];
      res.status(201).json({
        id: row.id,
        planId: row.plan_id,
        materialType: row.material_type,
        colorCode: row.color_code,
        componentPattern: row.component_pattern,
        routingId: row.routing_id,
        locationId: row.location_id,
        overrideWorkCenterId: row.override_work_center_id,
        templateId: row.template_id,
        priority: row.priority,
        notes: row.notes,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
      });
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      console.error("Error creating routing assignment:", error);
      res.status(500).json({ message: "Failed to create routing assignment" });
    }
  });

  // PATCH /api/production/plan-routing-assignments/:id - Update routing assignment
  app.patch("/api/production/plan-routing-assignments/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid assignment ID" });
      }

      // Check if ZLP is locked
      const planId = await getPlanIdFromAssignment(id);
      if (planId) {
        const lockCheck = await checkPlanZlpLocked(planId);
        if (lockCheck.locked) {
          return res.status(403).json({ message: lockCheck.message });
        }
      }

      const data = assignmentFormSchema.partial().parse(req.body);

      const setClauses: string[] = [];
      const values: any[] = [];
      let paramIndex = 1;

      if (data.materialType !== undefined) {
        setClauses.push(`material_type = $${paramIndex++}`);
        values.push(data.materialType);
      }
      if (data.colorCode !== undefined) {
        setClauses.push(`color_code = $${paramIndex++}`);
        values.push(data.colorCode);
      }
      if (data.componentPattern !== undefined) {
        setClauses.push(`component_pattern = $${paramIndex++}`);
        values.push(data.componentPattern);
      }
      if (data.routingId !== undefined) {
        setClauses.push(`routing_id = $${paramIndex++}`);
        values.push(data.routingId);
      }
      if (data.locationId !== undefined) {
        setClauses.push(`location_id = $${paramIndex++}`);
        values.push(data.locationId);
      }
      if (data.overrideWorkCenterId !== undefined) {
        setClauses.push(`override_work_center_id = $${paramIndex++}`);
        values.push(data.overrideWorkCenterId);
      }
      if (data.templateId !== undefined) {
        setClauses.push(`template_id = $${paramIndex++}`);
        values.push(data.templateId);
      }
      if (data.priority !== undefined) {
        setClauses.push(`priority = $${paramIndex++}`);
        values.push(data.priority);
      }
      if (data.notes !== undefined) {
        setClauses.push(`notes = $${paramIndex++}`);
        values.push(data.notes);
      }

      if (setClauses.length === 0) {
        return res.status(400).json({ message: "No fields to update" });
      }

      setClauses.push(`updated_at = NOW()`);
      values.push(id);

      const result = await pool.query(`
        UPDATE production.plan_routing_assignments 
        SET ${setClauses.join(', ')}
        WHERE id = $${paramIndex}
        RETURNING *
      `, values);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Routing assignment not found" });
      }

      const row = result.rows[0];
      res.json({
        id: row.id,
        planId: row.plan_id,
        materialType: row.material_type,
        colorCode: row.color_code,
        componentPattern: row.component_pattern,
        routingId: row.routing_id,
        locationId: row.location_id,
        overrideWorkCenterId: row.override_work_center_id,
        templateId: row.template_id,
        priority: row.priority,
        notes: row.notes,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
      });
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      console.error("Error updating routing assignment:", error);
      res.status(500).json({ message: "Failed to update routing assignment" });
    }
  });

  // DELETE /api/production/plan-routing-assignments/:id - Delete routing assignment
  app.delete("/api/production/plan-routing-assignments/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid assignment ID" });
      }

      // Check if ZLP is locked
      const planId = await getPlanIdFromAssignment(id);
      if (planId) {
        const lockCheck = await checkPlanZlpLocked(planId);
        if (lockCheck.locked) {
          return res.status(403).json({ message: lockCheck.message });
        }
      }

      const result = await pool.query(`
        DELETE FROM production.plan_routing_assignments WHERE id = $1 RETURNING id
      `, [id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Routing assignment not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting routing assignment:", error);
      res.status(500).json({ message: "Failed to delete routing assignment" });
    }
  });

  // POST /api/production/plans/:planId/routing-assignments/bulk - Bulk create/update assignments
  app.post("/api/production/plans/:planId/routing-assignments/bulk", requirePermission("manage_production"), async (req, res) => {
    try {
      const planId = parseInt(req.params.planId);
      if (isNaN(planId)) {
        return res.status(400).json({ message: "Invalid plan ID" });
      }

      // Check if ZLP is locked
      const lockCheck = await checkPlanZlpLocked(planId);
      if (lockCheck.locked) {
        return res.status(403).json({ message: lockCheck.message });
      }

      const { assignments } = req.body;
      if (!Array.isArray(assignments)) {
        return res.status(400).json({ message: "assignments must be an array" });
      }

      const client = await pool.connect();
      try {
        await client.query('BEGIN');

        // Delete existing assignments for this plan
        await client.query(`DELETE FROM production.plan_routing_assignments WHERE plan_id = $1`, [planId]);

        // Insert new assignments
        const insertedAssignments = [];
        for (const assignment of assignments) {
          const data = assignmentFormSchema.parse({ ...assignment, planId });
          const result = await client.query(`
            INSERT INTO production.plan_routing_assignments 
              (plan_id, material_type, color_code, component_pattern, routing_id, location_id, priority, notes)
            VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
            RETURNING *
          `, [
            data.planId,
            data.materialType || null,
            data.colorCode || null,
            data.componentPattern || null,
            data.routingId,
            data.locationId || null,
            data.priority,
            data.notes || null,
          ]);
          insertedAssignments.push(result.rows[0]);
        }

        await client.query('COMMIT');

        res.json({
          success: true,
          count: insertedAssignments.length,
          assignments: insertedAssignments.map(row => ({
            id: row.id,
            planId: row.plan_id,
            materialType: row.material_type,
            colorCode: row.color_code,
            componentPattern: row.component_pattern,
            routingId: row.routing_id,
            locationId: row.location_id,
            priority: row.priority,
            notes: row.notes,
          })),
        });
      } catch (error) {
        await client.query('ROLLBACK');
        throw error;
      } finally {
        client.release();
      }
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      console.error("Error bulk creating routing assignments:", error);
      res.status(500).json({ message: "Failed to create routing assignments" });
    }
  });

  // ============================================================================
  // OPERATION OVERRIDES - nadpisania operacji na poziomie przypisania marszruty
  // ============================================================================

  // GET /api/production/plan-routing-assignments/:id/operations - Get routing operations with overrides
  app.get("/api/production/plan-routing-assignments/:id/operations", requirePermission("view_production"), async (req, res) => {
    try {
      const assignmentId = parseInt(req.params.id);
      if (isNaN(assignmentId)) {
        return res.status(400).json({ message: "Invalid assignment ID" });
      }

      // Get assignment with routing info
      const assignmentResult = await pool.query(`
        SELECT a.*, r.id as routing_id, r.code as routing_code, r.name as routing_name
        FROM production.plan_routing_assignments a
        JOIN production.production_routings r ON r.id = a.routing_id
        WHERE a.id = $1
      `, [assignmentId]);

      if (assignmentResult.rows.length === 0) {
        return res.status(404).json({ message: "Assignment not found" });
      }

      const assignment = assignmentResult.rows[0];

      // Get routing operations with overrides
      const operationsResult = await pool.query(`
        SELECT 
          ro.id,
          ro.sequence,
          ro.code,
          ro.name,
          ro.description,
          ro.icon,
          ro.color,
          ro.work_center_id as default_work_center_id,
          wc.code as default_work_center_code,
          wc.name as default_work_center_name,
          ro.buffer_before_id as default_buffer_before_id,
          bb.name as default_buffer_before_name,
          ro.buffer_after_id as default_buffer_after_id,
          ba.name as default_buffer_after_name,
          ro.creates_buffer,
          ro.buffer_location_id,
          bl.name as buffer_location_name,
          -- Overrides
          ov.id as override_id,
          ov.override_work_center_id,
          owc.code as override_work_center_code,
          owc.name as override_work_center_name,
          ov.override_buffer_before_id,
          obb.name as override_buffer_before_name,
          ov.override_buffer_after_id,
          oba.name as override_buffer_after_name,
          ov.skip_operation,
          ov.notes as override_notes
        FROM production.production_routing_operations ro
        LEFT JOIN production.production_work_centers wc ON wc.id = ro.work_center_id
        LEFT JOIN production.production_locations bb ON bb.id = ro.buffer_before_id
        LEFT JOIN production.production_locations ba ON ba.id = ro.buffer_after_id
        LEFT JOIN production.production_locations bl ON bl.id = ro.buffer_location_id
        LEFT JOIN production.plan_routing_operation_overrides ov 
          ON ov.routing_operation_id = ro.id AND ov.plan_routing_assignment_id = $1
        LEFT JOIN production.production_work_centers owc ON owc.id = ov.override_work_center_id
        LEFT JOIN production.production_locations obb ON obb.id = ov.override_buffer_before_id
        LEFT JOIN production.production_locations oba ON oba.id = ov.override_buffer_after_id
        WHERE ro.routing_id = $2 AND ro.is_active = true
        ORDER BY ro.sequence
      `, [assignmentId, assignment.routing_id]);

      const operations = operationsResult.rows.map(row => ({
        id: row.id,
        sequence: row.sequence,
        code: row.code,
        name: row.name,
        description: row.description,
        icon: row.icon,
        color: row.color,
        // Default values from routing
        defaultWorkCenterId: row.default_work_center_id,
        defaultWorkCenterCode: row.default_work_center_code,
        defaultWorkCenterName: row.default_work_center_name,
        defaultBufferBeforeId: row.default_buffer_before_id,
        defaultBufferBeforeName: row.default_buffer_before_name,
        defaultBufferAfterId: row.default_buffer_after_id,
        defaultBufferAfterName: row.default_buffer_after_name,
        createsBuffer: row.creates_buffer,
        bufferLocationId: row.buffer_location_id,
        bufferLocationName: row.buffer_location_name,
        // Overrides (null if not overridden)
        overrideId: row.override_id,
        overrideWorkCenterId: row.override_work_center_id,
        overrideWorkCenterCode: row.override_work_center_code,
        overrideWorkCenterName: row.override_work_center_name,
        overrideBufferBeforeId: row.override_buffer_before_id,
        overrideBufferBeforeName: row.override_buffer_before_name,
        overrideBufferAfterId: row.override_buffer_after_id,
        overrideBufferAfterName: row.override_buffer_after_name,
        skipOperation: row.skip_operation || false,
        overrideNotes: row.override_notes,
        // Effective values (override or default)
        effectiveWorkCenterId: row.override_work_center_id || row.default_work_center_id,
        effectiveWorkCenterName: row.override_work_center_name || row.default_work_center_name,
        effectiveBufferBeforeId: row.override_buffer_before_id ?? row.default_buffer_before_id,
        effectiveBufferAfterId: row.override_buffer_after_id ?? row.default_buffer_after_id,
        hasOverride: row.override_id !== null,
      }));

      res.json({
        assignment: {
          id: assignment.id,
          planId: assignment.plan_id,
          routingId: assignment.routing_id,
          routingCode: assignment.routing_code,
          routingName: assignment.routing_name,
          materialType: assignment.material_type,
          colorCode: assignment.color_code,
          componentPattern: assignment.component_pattern,
        },
        operations,
      });
    } catch (error) {
      console.error("Error fetching assignment operations:", error);
      res.status(500).json({ message: "Failed to fetch operations" });
    }
  });

  // PUT /api/production/plan-routing-assignments/:id/operations/:opId - Set operation override
  app.put("/api/production/plan-routing-assignments/:id/operations/:opId", requirePermission("manage_production"), async (req, res) => {
    try {
      const assignmentId = parseInt(req.params.id);
      const operationId = parseInt(req.params.opId);
      
      if (isNaN(assignmentId) || isNaN(operationId)) {
        return res.status(400).json({ message: "Invalid IDs" });
      }

      const overrideSchema = z.object({
        overrideWorkCenterId: z.number().nullable().optional(),
        overrideBufferBeforeId: z.number().nullable().optional(),
        overrideBufferAfterId: z.number().nullable().optional(),
        skipOperation: z.boolean().optional(),
        notes: z.string().nullable().optional(),
      });

      const data = overrideSchema.parse(req.body);

      // Upsert override
      const result = await pool.query(`
        INSERT INTO production.plan_routing_operation_overrides 
          (plan_routing_assignment_id, routing_operation_id, override_work_center_id, 
           override_buffer_before_id, override_buffer_after_id, skip_operation, notes)
        VALUES ($1, $2, $3, $4, $5, $6, $7)
        ON CONFLICT (plan_routing_assignment_id, routing_operation_id) 
        DO UPDATE SET
          override_work_center_id = EXCLUDED.override_work_center_id,
          override_buffer_before_id = EXCLUDED.override_buffer_before_id,
          override_buffer_after_id = EXCLUDED.override_buffer_after_id,
          skip_operation = EXCLUDED.skip_operation,
          notes = EXCLUDED.notes,
          updated_at = NOW()
        RETURNING *
      `, [
        assignmentId,
        operationId,
        data.overrideWorkCenterId ?? null,
        data.overrideBufferBeforeId ?? null,
        data.overrideBufferAfterId ?? null,
        data.skipOperation ?? false,
        data.notes ?? null,
      ]);

      res.json({
        id: result.rows[0].id,
        planRoutingAssignmentId: result.rows[0].plan_routing_assignment_id,
        routingOperationId: result.rows[0].routing_operation_id,
        overrideWorkCenterId: result.rows[0].override_work_center_id,
        overrideBufferBeforeId: result.rows[0].override_buffer_before_id,
        overrideBufferAfterId: result.rows[0].override_buffer_after_id,
        skipOperation: result.rows[0].skip_operation,
        notes: result.rows[0].notes,
      });
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      console.error("Error setting operation override:", error);
      res.status(500).json({ message: "Failed to set operation override" });
    }
  });

  // DELETE /api/production/plan-routing-assignments/:id/operations/:opId - Remove operation override
  app.delete("/api/production/plan-routing-assignments/:id/operations/:opId", requirePermission("manage_production"), async (req, res) => {
    try {
      const assignmentId = parseInt(req.params.id);
      const operationId = parseInt(req.params.opId);
      
      if (isNaN(assignmentId) || isNaN(operationId)) {
        return res.status(400).json({ message: "Invalid IDs" });
      }

      await pool.query(`
        DELETE FROM production.plan_routing_operation_overrides 
        WHERE plan_routing_assignment_id = $1 AND routing_operation_id = $2
      `, [assignmentId, operationId]);

      res.status(204).send();
    } catch (error) {
      console.error("Error removing operation override:", error);
      res.status(500).json({ message: "Failed to remove operation override" });
    }
  });

  // PUT /api/production/plan-routing-assignments/:id/operations/bulk - Bulk set operation overrides
  app.put("/api/production/plan-routing-assignments/:id/operations/bulk", requirePermission("manage_production"), async (req, res) => {
    try {
      const assignmentId = parseInt(req.params.id);
      if (isNaN(assignmentId)) {
        return res.status(400).json({ message: "Invalid assignment ID" });
      }

      const bulkSchema = z.object({
        operations: z.array(z.object({
          routingOperationId: z.number(),
          overrideWorkCenterId: z.number().nullable(),
          overrideBufferBeforeId: z.number().nullable(),
          overrideBufferAfterId: z.number().nullable(),
          skipOperation: z.boolean(),
          notes: z.string().nullable().optional(),
        })),
      });

      const { operations } = bulkSchema.parse(req.body);

      const client = await pool.connect();
      try {
        await client.query('BEGIN');

        // Validate that assignment exists and get its routing_id
        const assignmentResult = await client.query(`
          SELECT a.id, a.routing_id 
          FROM production.plan_routing_assignments a
          WHERE a.id = $1
        `, [assignmentId]);

        if (assignmentResult.rows.length === 0) {
          await client.query('ROLLBACK');
          return res.status(404).json({ message: "Assignment not found" });
        }

        const routingId = assignmentResult.rows[0].routing_id;

        // Validate that all operation IDs belong to this routing
        const validOpsResult = await client.query(`
          SELECT id FROM production.production_routing_operations
          WHERE routing_id = $1 AND is_active = true
        `, [routingId]);

        const validOpIds = new Set(validOpsResult.rows.map(r => r.id));
        const invalidOps = operations.filter(op => !validOpIds.has(op.routingOperationId));
        
        if (invalidOps.length > 0) {
          await client.query('ROLLBACK');
          return res.status(400).json({ 
            message: "Invalid operation IDs", 
            invalidIds: invalidOps.map(op => op.routingOperationId) 
          });
        }

        // Delete existing overrides for this assignment
        await client.query(`
          DELETE FROM production.plan_routing_operation_overrides 
          WHERE plan_routing_assignment_id = $1
        `, [assignmentId]);

        // Insert new overrides only if there's actual override data (not all null/false)
        let insertedCount = 0;
        for (const op of operations) {
          const hasOverride = op.overrideWorkCenterId !== null || 
                              op.overrideBufferBeforeId !== null || 
                              op.overrideBufferAfterId !== null || 
                              op.skipOperation === true;
          
          if (hasOverride) {
            await client.query(`
              INSERT INTO production.plan_routing_operation_overrides 
                (plan_routing_assignment_id, routing_operation_id, override_work_center_id, 
                 override_buffer_before_id, override_buffer_after_id, skip_operation, notes)
              VALUES ($1, $2, $3, $4, $5, $6, $7)
            `, [
              assignmentId,
              op.routingOperationId,
              op.overrideWorkCenterId,
              op.overrideBufferBeforeId,
              op.overrideBufferAfterId,
              op.skipOperation,
              op.notes ?? null,
            ]);
            insertedCount++;
          }
        }

        await client.query('COMMIT');
        res.json({ success: true, count: insertedCount });
      } catch (error) {
        await client.query('ROLLBACK');
        throw error;
      } finally {
        client.release();
      }
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      console.error("Error bulk setting operation overrides:", error);
      res.status(500).json({ message: "Failed to set operation overrides" });
    }
  });

  // GET /api/production/material-types - Get distinct material types from BOM items
  app.get("/api/production/material-types", requirePermission("view_production"), async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT DISTINCT material_type 
        FROM (
          -- From BOM items
          SELECT DISTINCT 
            CASE 
              WHEN thickness = 18 AND material_type ILIKE '%płyt%' THEN 'plyta_18mm'
              WHEN thickness = 25 AND material_type ILIKE '%płyt%' THEN 'plyta_25mm'
              WHEN thickness = 3 AND material_type ILIKE '%hdf%' THEN 'hdf_3mm'
              WHEN thickness = 3 THEN 'hdf_3mm'
              WHEN thickness = 18 THEN 'plyta_18mm'
              ELSE COALESCE(material_type, 'inne')
            END as material_type
          FROM bom.product_bom_items
          WHERE material_type IS NOT NULL
          
          UNION
          
          -- Predefined common types
          SELECT 'plyta_18mm' as material_type
          UNION SELECT 'plyta_25mm'
          UNION SELECT 'hdf_3mm'
          UNION SELECT 'hdf_8mm'
        ) types
        WHERE material_type IS NOT NULL
        ORDER BY material_type
      `);

      res.json(result.rows.map(r => r.material_type));
    } catch (error) {
      console.error("Error fetching material types:", error);
      res.status(500).json({ message: "Failed to fetch material types" });
    }
  });
}
