import { Express, Request, Response } from "express";
import { pool } from "../../postgres";
import { insertProductionOperatorSchema } from "@shared/schema";
import { requirePermission } from "../../auth";

export function registerOperatorRoutes(app: Express) {
  app.get(
    "/api/production/operators",
    requirePermission("view_production"),
    async (req: Request, res: Response) => {
      try {
        const { search, is_active, work_center_id, limit = "50", offset = "0" } = req.query;

        let whereConditions: string[] = [];
        const params: any[] = [];
        let paramIndex = 1;

        if (search && typeof search === "string") {
          whereConditions.push(`(
            full_name ILIKE $${paramIndex} OR 
            short_code ILIKE $${paramIndex} OR 
            badge_id ILIKE $${paramIndex}
          )`);
          params.push(`%${search}%`);
          paramIndex++;
        }

        if (is_active !== undefined) {
          whereConditions.push(`is_active = $${paramIndex}`);
          params.push(is_active === "true");
          paramIndex++;
        }

        if (work_center_id && typeof work_center_id === "string") {
          whereConditions.push(`work_center_id = $${paramIndex}`);
          params.push(parseInt(work_center_id));
          paramIndex++;
        }

        const whereClause = whereConditions.length > 0 
          ? `WHERE ${whereConditions.join(" AND ")}` 
          : "";

        const limitValue = parseInt(limit as string);
        const offsetValue = parseInt(offset as string);

        const [operatorsResult, countResult] = await Promise.all([
          pool.query(`
            SELECT * FROM production.production_operators
            ${whereClause}
            ORDER BY is_active DESC, full_name ASC
            LIMIT $${paramIndex} OFFSET $${paramIndex + 1}
          `, [...params, limitValue, offsetValue]),
          pool.query(`
            SELECT COUNT(*)::int as count FROM production.production_operators
            ${whereClause}
          `, params),
        ]);

        // Pobierz gniazda robocze dla każdego operatora
        const operatorIds = operatorsResult.rows.map((op: any) => op.id);
        let workCentersMap: Record<number, any[]> = {};
        
        if (operatorIds.length > 0) {
          const workCentersResult = await pool.query(`
            SELECT 
              owc.operator_id,
              owc.work_center_id,
              owc.is_primary,
              owc.proficiency_level,
              wc.code as work_center_code,
              wc.name as work_center_name
            FROM production.operator_work_centers owc
            JOIN production.production_work_centers wc ON owc.work_center_id = wc.id
            WHERE owc.operator_id = ANY($1)
            ORDER BY owc.is_primary DESC, wc.name ASC
          `, [operatorIds]);

          for (const row of workCentersResult.rows) {
            if (!workCentersMap[row.operator_id]) {
              workCentersMap[row.operator_id] = [];
            }
            workCentersMap[row.operator_id].push({
              id: row.work_center_id,
              code: row.work_center_code,
              name: row.work_center_name,
              isPrimary: row.is_primary,
              proficiencyLevel: row.proficiency_level,
            });
          }
        }

        // Dodaj gniazda do każdego operatora
        const operatorsWithWorkCenters = operatorsResult.rows.map((op: any) => ({
          ...op,
          work_centers: workCentersMap[op.id] || [],
        }));

        res.json({
          operators: operatorsWithWorkCenters,
          total: countResult.rows[0]?.count || 0,
          limit: limitValue,
          offset: offsetValue,
        });
      } catch (error) {
        console.error("Error fetching operators:", error);
        res.status(500).json({ error: "Failed to fetch operators" });
      }
    }
  );

  app.get(
    "/api/production/operators/:id",
    requirePermission("view_production"),
    async (req: Request, res: Response) => {
      try {
        const { id } = req.params;

        const result = await pool.query(
          `SELECT * FROM production.production_operators WHERE id = $1`,
          [parseInt(id)]
        );

        if (result.rows.length === 0) {
          return res.status(404).json({ error: "Operator not found" });
        }

        res.json(result.rows[0]);
      } catch (error) {
        console.error("Error fetching operator:", error);
        res.status(500).json({ error: "Failed to fetch operator" });
      }
    }
  );

  app.post(
    "/api/production/operators",
    requirePermission("manage_production"),
    async (req: Request, res: Response) => {
      try {
        const validatedData = insertProductionOperatorSchema.parse(req.body);

        const result = await pool.query(
          `INSERT INTO production.production_operators 
           (full_name, short_code, badge_id, user_id, work_center_id, is_active, notes)
           VALUES ($1, $2, $3, $4, $5, $6, $7)
           RETURNING *`,
          [
            validatedData.fullName,
            validatedData.shortCode || null,
            validatedData.badgeId || null,
            validatedData.userId || null,
            validatedData.workCenterId || null,
            validatedData.isActive ?? true,
            validatedData.notes || null,
          ]
        );

        res.status(201).json(result.rows[0]);
      } catch (error) {
        console.error("Error creating operator:", error);
        if (error instanceof Error && error.name === "ZodError") {
          return res.status(400).json({ error: "Invalid operator data", details: error });
        }
        res.status(500).json({ error: "Failed to create operator" });
      }
    }
  );

  app.patch(
    "/api/production/operators/:id",
    requirePermission("manage_production"),
    async (req: Request, res: Response) => {
      try {
        const { id } = req.params;
        const validatedData = insertProductionOperatorSchema.partial().parse(req.body);

        const updateFields: string[] = [];
        const params: any[] = [];
        let paramIndex = 1;

        if (validatedData.fullName !== undefined) {
          updateFields.push(`full_name = $${paramIndex++}`);
          params.push(validatedData.fullName);
        }
        if (validatedData.shortCode !== undefined) {
          updateFields.push(`short_code = $${paramIndex++}`);
          params.push(validatedData.shortCode);
        }
        if (validatedData.badgeId !== undefined) {
          updateFields.push(`badge_id = $${paramIndex++}`);
          params.push(validatedData.badgeId);
        }
        if (validatedData.userId !== undefined) {
          updateFields.push(`user_id = $${paramIndex++}`);
          params.push(validatedData.userId);
        }
        if (validatedData.workCenterId !== undefined) {
          updateFields.push(`work_center_id = $${paramIndex++}`);
          params.push(validatedData.workCenterId);
        }
        if (validatedData.isActive !== undefined) {
          updateFields.push(`is_active = $${paramIndex++}`);
          params.push(validatedData.isActive);
        }
        if (validatedData.notes !== undefined) {
          updateFields.push(`notes = $${paramIndex++}`);
          params.push(validatedData.notes);
        }

        updateFields.push(`updated_at = NOW()`);

        if (updateFields.length === 1) {
          return res.status(400).json({ error: "No fields to update" });
        }

        params.push(parseInt(id));

        const result = await pool.query(
          `UPDATE production.production_operators 
           SET ${updateFields.join(", ")}
           WHERE id = $${paramIndex}
           RETURNING *`,
          params
        );

        if (result.rows.length === 0) {
          return res.status(404).json({ error: "Operator not found" });
        }

        res.json(result.rows[0]);
      } catch (error) {
        console.error("Error updating operator:", error);
        if (error instanceof Error && error.name === "ZodError") {
          return res.status(400).json({ error: "Invalid operator data", details: error });
        }
        res.status(500).json({ error: "Failed to update operator" });
      }
    }
  );

  app.delete(
    "/api/production/operators/:id",
    requirePermission("manage_production"),
    async (req: Request, res: Response) => {
      try {
        const { id } = req.params;

        const result = await pool.query(
          `DELETE FROM production.production_operators WHERE id = $1 RETURNING *`,
          [parseInt(id)]
        );

        if (result.rows.length === 0) {
          return res.status(404).json({ error: "Operator not found" });
        }

        res.json({ success: true, deleted: result.rows[0] });
      } catch (error) {
        console.error("Error deleting operator:", error);
        res.status(500).json({ error: "Failed to delete operator" });
      }
    }
  );

  app.get(
    "/api/production/operators-select",
    requirePermission("view_production"),
    async (req: Request, res: Response) => {
      try {
        const result = await pool.query(`
          SELECT id, full_name, short_code, work_center_id
          FROM production.production_operators
          WHERE is_active = true
          ORDER BY full_name ASC
        `);

        res.json(result.rows);
      } catch (error) {
        console.error("Error fetching operators for select:", error);
        res.status(500).json({ error: "Failed to fetch operators" });
      }
    }
  );

  // Pobierz gniazda robocze przypisane do operatora
  app.get(
    "/api/production/operators/:id/work-centers",
    requirePermission("view_production"),
    async (req: Request, res: Response) => {
      try {
        const { id } = req.params;
        
        const result = await pool.query(`
          SELECT 
            owc.id,
            owc.work_center_id,
            owc.is_primary,
            owc.proficiency_level,
            owc.notes,
            wc.code as work_center_code,
            wc.name as work_center_name
          FROM production.operator_work_centers owc
          JOIN production.production_work_centers wc ON owc.work_center_id = wc.id
          WHERE owc.operator_id = $1
          ORDER BY owc.is_primary DESC, wc.name ASC
        `, [parseInt(id)]);

        res.json(result.rows);
      } catch (error) {
        console.error("Error fetching operator work centers:", error);
        res.status(500).json({ error: "Failed to fetch operator work centers" });
      }
    }
  );

  // Zaktualizuj gniazda robocze operatora (bulk update)
  app.put(
    "/api/production/operators/:id/work-centers",
    requirePermission("manage_production"),
    async (req: Request, res: Response) => {
      try {
        const operatorId = parseInt(req.params.id);
        const { workCenterIds, primaryWorkCenterId } = req.body;

        // Sprawdź czy operator istnieje
        const operatorCheck = await pool.query(
          `SELECT id FROM production.production_operators WHERE id = $1`,
          [operatorId]
        );
        if (operatorCheck.rows.length === 0) {
          return res.status(404).json({ error: "Operator not found" });
        }

        // Usuń wszystkie obecne przypisania
        await pool.query(
          `DELETE FROM production.operator_work_centers WHERE operator_id = $1`,
          [operatorId]
        );

        // Dodaj nowe przypisania
        if (workCenterIds && workCenterIds.length > 0) {
          const insertValues = workCenterIds.map((wcId: number) => {
            const isPrimary = wcId === primaryWorkCenterId;
            return `(${operatorId}, ${wcId}, ${isPrimary}, 'standard')`;
          }).join(", ");

          await pool.query(`
            INSERT INTO production.operator_work_centers 
            (operator_id, work_center_id, is_primary, proficiency_level)
            VALUES ${insertValues}
          `);
        }

        // Zaktualizuj też pole work_center_id w głównej tabeli (dla kompatybilności wstecznej)
        await pool.query(`
          UPDATE production.production_operators 
          SET work_center_id = $1, updated_at = NOW()
          WHERE id = $2
        `, [primaryWorkCenterId || null, operatorId]);

        // Pobierz zaktualizowane dane
        const result = await pool.query(`
          SELECT 
            owc.id,
            owc.work_center_id,
            owc.is_primary,
            owc.proficiency_level,
            wc.code as work_center_code,
            wc.name as work_center_name
          FROM production.operator_work_centers owc
          JOIN production.production_work_centers wc ON owc.work_center_id = wc.id
          WHERE owc.operator_id = $1
          ORDER BY owc.is_primary DESC, wc.name ASC
        `, [operatorId]);

        res.json({ success: true, workCenters: result.rows });
      } catch (error) {
        console.error("Error updating operator work centers:", error);
        res.status(500).json({ error: "Failed to update operator work centers" });
      }
    }
  );

  // Pobierz operatorów dla danego gniazda roboczego
  app.get(
    "/api/production/work-centers/:id/operators",
    requirePermission("view_production"),
    async (req: Request, res: Response) => {
      try {
        const { id } = req.params;
        
        const result = await pool.query(`
          SELECT 
            po.id,
            po.full_name,
            po.short_code,
            po.badge_id,
            po.is_active,
            owc.is_primary,
            owc.proficiency_level
          FROM production.operator_work_centers owc
          JOIN production.production_operators po ON owc.operator_id = po.id
          WHERE owc.work_center_id = $1 AND po.is_active = true
          ORDER BY owc.is_primary DESC, po.full_name ASC
        `, [parseInt(id)]);

        res.json(result.rows);
      } catch (error) {
        console.error("Error fetching work center operators:", error);
        res.status(500).json({ error: "Failed to fetch work center operators" });
      }
    }
  );
}
