import type { Express } from "express";
import { requirePermission } from "../../auth";
import { pool } from "../../postgres";
import { 
  getLocationGroups, 
  getLocationGroupById, 
  createLocationGroup, 
  updateLocationGroup, 
  deleteLocationGroup
} from "../../services/production/location-groups";
import { insertProductionLocationGroupSchema } from "@shared/schema";

const partialGroupSchema = insertProductionLocationGroupSchema.partial();

export function setupLocationGroupRoutes(app: Express) {
  // GET /api/production/location-groups - Get all location groups
  app.get("/api/production/location-groups", requirePermission("view_production"), async (req, res) => {
    try {
      const groups = await getLocationGroups(pool);
      res.json(groups);
    } catch (error: any) {
      console.error("Error fetching location groups:", error);
      res.status(500).json({ error: "Failed to fetch location groups" });
    }
  });

  // GET /api/production/location-groups/:id - Get single location group
  app.get("/api/production/location-groups/:id", requirePermission("view_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const group = await getLocationGroupById(pool, id);
      
      if (!group) {
        return res.status(404).json({ error: "Location group not found" });
      }
      
      res.json(group);
    } catch (error: any) {
      console.error("Error fetching location group:", error);
      res.status(500).json({ error: "Failed to fetch location group" });
    }
  });

  // POST /api/production/location-groups - Create location group
  app.post("/api/production/location-groups", requirePermission("manage_production"), async (req, res) => {
    try {
      const parsed = insertProductionLocationGroupSchema.safeParse(req.body);
      
      if (!parsed.success) {
        return res.status(400).json({ 
          error: "Invalid request data", 
          details: parsed.error.errors 
        });
      }
      
      const group = await createLocationGroup(pool, parsed.data);
      res.status(201).json(group);
    } catch (error: any) {
      console.error("Error creating location group:", error);
      res.status(500).json({ error: "Failed to create location group" });
    }
  });

  // PATCH /api/production/location-groups/:id - Update location group
  app.patch("/api/production/location-groups/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const parsed = partialGroupSchema.safeParse(req.body);
      
      if (!parsed.success) {
        return res.status(400).json({ 
          error: "Invalid request data", 
          details: parsed.error.errors 
        });
      }
      
      const group = await updateLocationGroup(pool, id, parsed.data);
      
      if (!group) {
        return res.status(404).json({ error: "Location group not found" });
      }
      
      res.json(group);
    } catch (error: any) {
      console.error("Error updating location group:", error);
      res.status(500).json({ error: "Failed to update location group" });
    }
  });

  // DELETE /api/production/location-groups/:id - Delete location group
  app.delete("/api/production/location-groups/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      const deleted = await deleteLocationGroup(pool, id);
      
      if (!deleted) {
        return res.status(404).json({ error: "Location group not found" });
      }
      
      res.status(204).send();
    } catch (error: any) {
      console.error("Error deleting location group:", error);
      res.status(500).json({ error: "Failed to delete location group" });
    }
  });
}
