import type { Express } from "express";
import { requirePermission } from "../../auth";
import { z } from "zod";
import { pool } from "../../postgres";

const templateFormSchema = z.object({
  name: z.string().min(1, "Nazwa jest wymagana"),
  description: z.string().nullable().optional(),
  componentPattern: z.string().nullable().optional(),
  materialType: z.string().nullable().optional(),
  colorCode: z.string().nullable().optional(),
  defaultRoutingId: z.number({ required_error: "Marszruta jest wymagana" }),
  defaultWorkCenterId: z.number().nullable().optional(),
  locationId: z.number().nullable().optional(),
  priority: z.number().default(100),
  isActive: z.boolean().default(true),
});

export function registerComponentRoutingTemplateRoutes(app: Express) {
  // GET /api/production/component-routing-templates - Lista wszystkich szablonów
  app.get("/api/production/component-routing-templates", requirePermission("view_production"), async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          t.*,
          r.code as routing_code,
          r.name as routing_name,
          wc.code as work_center_code,
          wc.name as work_center_name,
          l.code as location_code,
          l.name as location_name
        FROM production.component_routing_templates t
        LEFT JOIN production.production_routings r ON r.id = t.default_routing_id
        LEFT JOIN production.production_work_centers wc ON wc.id = t.default_work_center_id
        LEFT JOIN production.production_locations l ON l.id = t.location_id
        ORDER BY t.priority DESC, t.name
      `);

      const templates = result.rows.map(row => ({
        id: row.id,
        name: row.name,
        description: row.description,
        componentPattern: row.component_pattern,
        materialType: row.material_type,
        colorCode: row.color_code,
        defaultRoutingId: row.default_routing_id,
        defaultRoutingCode: row.routing_code,
        defaultRoutingName: row.routing_name,
        defaultWorkCenterId: row.default_work_center_id,
        defaultWorkCenterCode: row.work_center_code,
        defaultWorkCenterName: row.work_center_name,
        locationId: row.location_id,
        locationCode: row.location_code,
        locationName: row.location_name,
        priority: row.priority,
        isActive: row.is_active,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
      }));

      res.json(templates);
    } catch (error) {
      console.error("Error fetching component routing templates:", error);
      res.status(500).json({ message: "Failed to fetch templates" });
    }
  });

  // GET /api/production/component-routing-templates/:id - Szczegóły szablonu
  app.get("/api/production/component-routing-templates/:id", requirePermission("view_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid template ID" });
      }

      const result = await pool.query(`
        SELECT 
          t.*,
          r.code as routing_code,
          r.name as routing_name,
          wc.code as work_center_code,
          wc.name as work_center_name,
          l.code as location_code,
          l.name as location_name
        FROM production.component_routing_templates t
        LEFT JOIN production.production_routings r ON r.id = t.default_routing_id
        LEFT JOIN production.production_work_centers wc ON wc.id = t.default_work_center_id
        LEFT JOIN production.production_locations l ON l.id = t.location_id
        WHERE t.id = $1
      `, [id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Template not found" });
      }

      const row = result.rows[0];
      res.json({
        id: row.id,
        name: row.name,
        description: row.description,
        componentPattern: row.component_pattern,
        materialType: row.material_type,
        colorCode: row.color_code,
        defaultRoutingId: row.default_routing_id,
        defaultRoutingCode: row.routing_code,
        defaultRoutingName: row.routing_name,
        defaultWorkCenterId: row.default_work_center_id,
        defaultWorkCenterCode: row.work_center_code,
        defaultWorkCenterName: row.work_center_name,
        locationId: row.location_id,
        locationCode: row.location_code,
        locationName: row.location_name,
        priority: row.priority,
        isActive: row.is_active,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
      });
    } catch (error) {
      console.error("Error fetching template:", error);
      res.status(500).json({ message: "Failed to fetch template" });
    }
  });

  // POST /api/production/component-routing-templates - Utwórz szablon
  app.post("/api/production/component-routing-templates", requirePermission("manage_production"), async (req, res) => {
    try {
      const parsed = templateFormSchema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ message: "Invalid data", errors: parsed.error.errors });
      }

      const data = parsed.data;
      const result = await pool.query(`
        INSERT INTO production.component_routing_templates (
          name, description, component_pattern, material_type, color_code,
          default_routing_id, default_work_center_id, location_id,
          priority, is_active
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
        RETURNING id
      `, [
        data.name,
        data.description || null,
        data.componentPattern || null,
        data.materialType || null,
        data.colorCode || null,
        data.defaultRoutingId,
        data.defaultWorkCenterId || null,
        data.locationId || null,
        data.priority,
        data.isActive,
      ]);

      res.status(201).json({ id: result.rows[0].id, message: "Template created" });
    } catch (error) {
      console.error("Error creating template:", error);
      res.status(500).json({ message: "Failed to create template" });
    }
  });

  // PUT /api/production/component-routing-templates/:id - Aktualizuj szablon
  app.put("/api/production/component-routing-templates/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid template ID" });
      }

      const parsed = templateFormSchema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ message: "Invalid data", errors: parsed.error.errors });
      }

      const data = parsed.data;
      await pool.query(`
        UPDATE production.component_routing_templates SET
          name = $1,
          description = $2,
          component_pattern = $3,
          material_type = $4,
          color_code = $5,
          default_routing_id = $6,
          default_work_center_id = $7,
          location_id = $8,
          priority = $9,
          is_active = $10,
          updated_at = NOW()
        WHERE id = $11
      `, [
        data.name,
        data.description || null,
        data.componentPattern || null,
        data.materialType || null,
        data.colorCode || null,
        data.defaultRoutingId,
        data.defaultWorkCenterId || null,
        data.locationId || null,
        data.priority,
        data.isActive,
        id,
      ]);

      res.json({ message: "Template updated" });
    } catch (error) {
      console.error("Error updating template:", error);
      res.status(500).json({ message: "Failed to update template" });
    }
  });

  // DELETE /api/production/component-routing-templates/:id - Usuń szablon
  app.delete("/api/production/component-routing-templates/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid template ID" });
      }

      // Check if template is used in any plan assignments
      const usageCheck = await pool.query(`
        SELECT COUNT(*) as count FROM production.plan_routing_assignments WHERE template_id = $1
      `, [id]);

      if (parseInt(usageCheck.rows[0].count) > 0) {
        return res.status(400).json({ 
          message: "Cannot delete template - it is used in plan assignments. Deactivate it instead." 
        });
      }

      await pool.query(`DELETE FROM production.component_routing_templates WHERE id = $1`, [id]);
      res.json({ message: "Template deleted" });
    } catch (error) {
      console.error("Error deleting template:", error);
      res.status(500).json({ message: "Failed to delete template" });
    }
  });

  // POST /api/production/component-routing-templates/test-pattern - Test dopasowania wzorca
  app.post("/api/production/component-routing-templates/test-pattern", requirePermission("view_production"), async (req, res) => {
    try {
      const { componentName, pattern } = req.body;
      if (!componentName || !pattern) {
        return res.status(400).json({ message: "componentName and pattern are required" });
      }

      const result = await pool.query(`SELECT $1 LIKE $2 as matches`, [componentName, pattern]);
      res.json({ matches: result.rows[0].matches });
    } catch (error) {
      console.error("Error testing pattern:", error);
      res.status(500).json({ message: "Failed to test pattern" });
    }
  });

  // GET /api/production/component-routing-templates/match - Znajdź pasujące szablony dla komponentu
  app.get("/api/production/component-routing-templates/match", requirePermission("view_production"), async (req, res) => {
    try {
      const { componentName, materialType, colorCode } = req.query;

      if (!componentName) {
        return res.status(400).json({ message: "componentName is required" });
      }

      const result = await pool.query(`
        SELECT 
          t.*,
          r.code as routing_code,
          r.name as routing_name,
          wc.code as work_center_code,
          wc.name as work_center_name,
          l.code as location_code,
          l.name as location_name
        FROM production.component_routing_templates t
        LEFT JOIN production.production_routings r ON r.id = t.default_routing_id
        LEFT JOIN production.production_work_centers wc ON wc.id = t.default_work_center_id
        LEFT JOIN production.production_locations l ON l.id = t.location_id
        WHERE t.is_active = true
          AND (t.component_pattern IS NULL OR $1 LIKE t.component_pattern)
          AND (t.material_type IS NULL OR t.material_type = $2)
          AND (t.color_code IS NULL OR t.color_code = $3)
        ORDER BY t.priority DESC
        LIMIT 1
      `, [componentName, materialType || null, colorCode || null]);

      if (result.rows.length === 0) {
        return res.json({ matched: false, template: null });
      }

      const row = result.rows[0];
      res.json({
        matched: true,
        template: {
          id: row.id,
          name: row.name,
          componentPattern: row.component_pattern,
          defaultRoutingId: row.default_routing_id,
          defaultRoutingCode: row.routing_code,
          defaultRoutingName: row.routing_name,
          defaultWorkCenterId: row.default_work_center_id,
          defaultWorkCenterCode: row.work_center_code,
          defaultWorkCenterName: row.work_center_name,
          locationId: row.location_id,
          locationCode: row.location_code,
          locationName: row.location_name,
        },
      });
    } catch (error) {
      console.error("Error matching template:", error);
      res.status(500).json({ message: "Failed to match template" });
    }
  });

  // GET /api/production/work-centers - Lista gniazd roboczych (dla combobox)
  app.get("/api/production/work-centers", requirePermission("view_production"), async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT id, code, name, description, is_active
        FROM production.production_work_centers
        WHERE is_active = true
        ORDER BY name
      `);

      res.json(result.rows.map(row => ({
        id: row.id,
        code: row.code,
        name: row.name,
        description: row.description,
        isActive: row.is_active,
      })));
    } catch (error) {
      console.error("Error fetching work centers:", error);
      res.status(500).json({ message: "Failed to fetch work centers" });
    }
  });
}
