import type { Express } from "express";
import { pool } from "../../postgres";
import { requirePermission } from "../../auth";

interface ZlpStatusSummary {
  status: string;
  count: number;
  itemCount: number;
}

interface PlanSummary {
  id: number;
  name: string;
  status: string;
  zlpCount: number;
  itemCount: number;
  completedCount: number;
}

interface WorkCenterStats {
  id: number;
  name: string;
  code: string;
  totalOperations: number;
  completedOperations: number;
  inProgressOperations: number;
  pendingOperations: number;
}

interface DailyStats {
  date: string;
  completedZlp: number;
  createdZlp: number;
}

interface TopProduct {
  id: number;
  name: string;
  sku: string;
  zlpCount: number;
  totalQuantity: number;
}

interface RecentZlp {
  id: number;
  name: string;
  status: string;
  itemCount: number;
  createdAt: string;
}

interface AnalyticsDashboard {
  zlpByStatus: ZlpStatusSummary[];
  totalZlp: number;
  totalItems: number;
  completedItems: number;
  pendingItems: number;
  inProgressItems: number;
  plans: PlanSummary[];
  recentPlans: PlanSummary[];
  workCenterStats: WorkCenterStats[];
  dailyStats: DailyStats[];
  topProducts: TopProduct[];
  recentZlps: RecentZlp[];
  bomStats: {
    totalBoms: number;
    bomsWithFormatki: number;
    averageFormatkiPerBom: number;
  };
  formatkiStats: {
    totalFormatki: number;
    uniqueFormatki: number;
    formatkiInProduction: number;
  };
}

interface CytoscapeNode {
  data: {
    id: string;
    label: string;
    type: "product" | "set" | "formatka" | "bom" | "order" | "zlp";
    sku?: string;
    color?: string;
    status?: string;
  };
}

interface CytoscapeEdge {
  data: {
    id: string;
    source: string;
    target: string;
    label: string;
  };
}

interface CytoscapeGraph {
  nodes: CytoscapeNode[];
  edges: CytoscapeEdge[];
  stats: {
    totalProducts: number;
    totalSets: number;
    totalBoms: number;
    totalZlps: number;
    totalFormatki: number;
  };
}

export function registerProductionAnalyticsRoutes(app: Express) {
  app.get("/api/production/analytics/dashboard", requirePermission("view_orders"), async (req, res) => {
    try {
      const zlpStatusQuery = await pool.query(`
        SELECT 
          po.status,
          COUNT(DISTINCT po.id) as zlp_count,
          COALESCE(SUM(
            (SELECT COUNT(*) 
             FROM production.production_order_boms pob 
             JOIN production.production_order_bom_items pobi ON pobi.production_order_bom_id = pob.id 
             WHERE pob.production_order_id = po.id)
          ), 0) as item_count
        FROM production.production_orders po
        GROUP BY po.status
        ORDER BY 
          CASE po.status 
            WHEN 'in_progress' THEN 1
            WHEN 'pending' THEN 2
            WHEN 'ready' THEN 3
            WHEN 'generated' THEN 4
            WHEN 'approved' THEN 5
            WHEN 'completed' THEN 6
            WHEN 'draft' THEN 7
            WHEN 'cancelled' THEN 8
            ELSE 9
          END
      `);

      const totalZlpQuery = await pool.query(`
        SELECT COUNT(*) as total FROM production.production_orders
      `);

      const itemStatsQuery = await pool.query(`
        SELECT 
          COUNT(*) as total_items,
          SUM(CASE WHEN wo.status = 'done' THEN 1 ELSE 0 END) as completed_items,
          SUM(CASE WHEN wo.status = 'in_progress' THEN 1 ELSE 0 END) as in_progress_items,
          SUM(CASE WHEN wo.status IN ('pending', 'ready') THEN 1 ELSE 0 END) as pending_items
        FROM production.production_work_orders wo
      `);

      const plansQuery = await pool.query(`
        SELECT 
          pp.id,
          pp.name,
          pp.status,
          (
            SELECT COUNT(DISTINCT ppl.production_order_id)
            FROM production.production_plan_lines ppl
            WHERE ppl.plan_id = pp.id AND ppl.production_order_id IS NOT NULL
          ) as zlp_count,
          (
            SELECT COALESCE(SUM(ppl.quantity), 0)
            FROM production.production_plan_lines ppl
            WHERE ppl.plan_id = pp.id
          ) as item_count,
          (
            SELECT COUNT(*)
            FROM production.production_plan_lines ppl
            JOIN production.production_orders po ON ppl.production_order_id = po.id
            WHERE ppl.plan_id = pp.id AND po.status = 'completed'
          ) as completed_count
        FROM production.production_plans pp
        WHERE pp.status NOT IN ('cancelled')
        ORDER BY pp.created_at DESC
        LIMIT 10
      `);

      // Work center statistics
      const workCenterStatsQuery = await pool.query(`
        SELECT 
          wc.id,
          wc.name,
          wc.code,
          COUNT(DISTINCT pwo.id) as total_operations,
          SUM(CASE WHEN pwo.status = 'done' THEN 1 ELSE 0 END) as completed_operations,
          SUM(CASE WHEN pwo.status = 'in_progress' THEN 1 ELSE 0 END) as in_progress_operations,
          SUM(CASE WHEN pwo.status IN ('pending', 'ready') THEN 1 ELSE 0 END) as pending_operations
        FROM production.work_centers wc
        LEFT JOIN production.production_work_orders pwo ON pwo.work_center_id = wc.id
        GROUP BY wc.id, wc.name, wc.code
        ORDER BY total_operations DESC
        LIMIT 10
      `);

      // Daily stats for last 14 days
      const dailyStatsQuery = await pool.query(`
        WITH dates AS (
          SELECT generate_series(
            CURRENT_DATE - INTERVAL '13 days',
            CURRENT_DATE,
            INTERVAL '1 day'
          )::date as date
        )
        SELECT 
          d.date::text,
          COALESCE(created.count, 0) as created_zlp,
          COALESCE(completed.count, 0) as completed_zlp
        FROM dates d
        LEFT JOIN (
          SELECT DATE(created_at) as date, COUNT(*) as count
          FROM production.production_orders
          WHERE created_at >= CURRENT_DATE - INTERVAL '13 days'
          GROUP BY DATE(created_at)
        ) created ON d.date = created.date
        LEFT JOIN (
          SELECT DATE(updated_at) as date, COUNT(*) as count
          FROM production.production_orders
          WHERE status = 'completed' AND updated_at >= CURRENT_DATE - INTERVAL '13 days'
          GROUP BY DATE(updated_at)
        ) completed ON d.date = completed.date
        ORDER BY d.date
      `);

      // Top products in production
      const topProductsQuery = await pool.query(`
        SELECT 
          cp.id,
          cp.name,
          cp.sku,
          COUNT(DISTINCT po.id) as zlp_count,
          COALESCE(SUM(ppl.quantity), 0) as total_quantity
        FROM catalog.products cp
        JOIN production.production_plan_lines ppl ON ppl.catalog_product_id = cp.id
        LEFT JOIN production.production_orders po ON ppl.production_order_id = po.id
        GROUP BY cp.id, cp.name, cp.sku
        ORDER BY zlp_count DESC, total_quantity DESC
        LIMIT 10
      `);

      // Recent ZLPs
      const recentZlpsQuery = await pool.query(`
        SELECT 
          po.id,
          po.name,
          po.status,
          po.created_at,
          (
            SELECT COUNT(*) 
            FROM production.production_order_boms pob 
            JOIN production.production_order_bom_items pobi ON pobi.production_order_bom_id = pob.id 
            WHERE pob.production_order_id = po.id
          ) as item_count
        FROM production.production_orders po
        ORDER BY po.created_at DESC
        LIMIT 8
      `);

      // BOM statistics
      const bomStatsQuery = await pool.query(`
        SELECT 
          COUNT(DISTINCT b.id) as total_boms,
          COUNT(DISTINCT CASE WHEN bc.id IS NOT NULL THEN b.id END) as boms_with_formatki,
          ROUND(AVG(component_counts.component_count)::numeric, 2) as avg_formatki_per_bom
        FROM catalog.boms b
        LEFT JOIN catalog.bom_components bc ON bc.bom_id = b.id
        LEFT JOIN (
          SELECT bom_id, COUNT(*) as component_count 
          FROM catalog.bom_components 
          GROUP BY bom_id
        ) component_counts ON component_counts.bom_id = b.id
      `);

      // Formatki statistics
      const formatkiStatsQuery = await pool.query(`
        SELECT 
          (SELECT COUNT(*) FROM warehouse.components) as total_formatki,
          (SELECT COUNT(DISTINCT component_id) FROM catalog.bom_components) as unique_formatki_in_boms,
          (
            SELECT COUNT(DISTINCT bc.component_id)
            FROM catalog.bom_components bc
            JOIN catalog.boms b ON bc.bom_id = b.id
            JOIN production.production_order_boms pob ON pob.bom_id = b.id
            JOIN production.production_orders po ON pob.production_order_id = po.id
            WHERE po.status IN ('pending', 'in_progress', 'ready')
          ) as formatki_in_production
      `);

      const zlpByStatus: ZlpStatusSummary[] = zlpStatusQuery.rows.map(row => ({
        status: row.status,
        count: parseInt(row.zlp_count) || 0,
        itemCount: parseInt(row.item_count) || 0,
      }));

      const itemStats = itemStatsQuery.rows[0] || {};
      const bomStats = bomStatsQuery.rows[0] || {};
      const formatkiStats = formatkiStatsQuery.rows[0] || {};

      const dashboard: AnalyticsDashboard = {
        zlpByStatus,
        totalZlp: parseInt(totalZlpQuery.rows[0]?.total) || 0,
        totalItems: parseInt(itemStats.total_items) || 0,
        completedItems: parseInt(itemStats.completed_items) || 0,
        pendingItems: parseInt(itemStats.pending_items) || 0,
        inProgressItems: parseInt(itemStats.in_progress_items) || 0,
        plans: plansQuery.rows.map(row => ({
          id: row.id,
          name: row.name,
          status: row.status,
          zlpCount: parseInt(row.zlp_count) || 0,
          itemCount: parseInt(row.item_count) || 0,
          completedCount: parseInt(row.completed_count) || 0,
        })),
        recentPlans: plansQuery.rows.slice(0, 5).map(row => ({
          id: row.id,
          name: row.name,
          status: row.status,
          zlpCount: parseInt(row.zlp_count) || 0,
          itemCount: parseInt(row.item_count) || 0,
          completedCount: parseInt(row.completed_count) || 0,
        })),
        workCenterStats: workCenterStatsQuery.rows.map(row => ({
          id: row.id,
          name: row.name,
          code: row.code,
          totalOperations: parseInt(row.total_operations) || 0,
          completedOperations: parseInt(row.completed_operations) || 0,
          inProgressOperations: parseInt(row.in_progress_operations) || 0,
          pendingOperations: parseInt(row.pending_operations) || 0,
        })),
        dailyStats: dailyStatsQuery.rows.map(row => ({
          date: row.date,
          createdZlp: parseInt(row.created_zlp) || 0,
          completedZlp: parseInt(row.completed_zlp) || 0,
        })),
        topProducts: topProductsQuery.rows.map(row => ({
          id: row.id,
          name: row.name,
          sku: row.sku || "",
          zlpCount: parseInt(row.zlp_count) || 0,
          totalQuantity: parseInt(row.total_quantity) || 0,
        })),
        recentZlps: recentZlpsQuery.rows.map(row => ({
          id: row.id,
          name: row.name,
          status: row.status,
          itemCount: parseInt(row.item_count) || 0,
          createdAt: row.created_at,
        })),
        bomStats: {
          totalBoms: parseInt(bomStats.total_boms) || 0,
          bomsWithFormatki: parseInt(bomStats.boms_with_formatki) || 0,
          averageFormatkiPerBom: parseFloat(bomStats.avg_formatki_per_bom) || 0,
        },
        formatkiStats: {
          totalFormatki: parseInt(formatkiStats.total_formatki) || 0,
          uniqueFormatki: parseInt(formatkiStats.unique_formatki_in_boms) || 0,
          formatkiInProduction: parseInt(formatkiStats.formatki_in_production) || 0,
        },
      };

      res.json(dashboard);
    } catch (error) {
      console.error("Error fetching analytics dashboard:", error);
      res.status(500).json({ error: "Failed to fetch analytics dashboard" });
    }
  });

  app.get("/api/production/analytics/connection-graph", requirePermission("view_orders"), async (req, res) => {
    try {
      const { search, type, limit: queryLimit } = req.query;
      const searchQuery = (search as string || "").trim();
      const nodeType = type as string || "all";
      const limitNodes = Math.min(parseInt(queryLimit as string) || 500, 1000);

      const nodes: CytoscapeNode[] = [];
      const edges: CytoscapeEdge[] = [];
      const nodeIds = new Set<string>();
      let edgeCounter = 0;

      const productsQuery = await pool.query(`
        SELECT 
          cp.id,
          cp.name,
          cp.sku,
          cp.color,
          cp.matrix_id
        FROM catalog.products cp
        WHERE ($1 = '' OR cp.name ILIKE $2 OR cp.sku ILIKE $2)
        ORDER BY cp.id
        LIMIT $3
      `, [searchQuery, `%${searchQuery}%`, limitNodes]);

      for (const row of productsQuery.rows) {
        const productId = `product-${row.id}`;
        if (!nodeIds.has(productId)) {
          nodeIds.add(productId);
          nodes.push({
            data: {
              id: productId,
              label: row.name || `Produkt ${row.id}`,
              type: "product",
              sku: row.sku,
              color: row.color,
            },
          });
        }
      }

      const setsQuery = await pool.query(`
        SELECT 
          ps.id,
          ps.sku,
          ps.title,
          ps.color,
          ps.is_active
        FROM product_creator.product_sets ps
        WHERE ($1 = '' OR ps.title ILIKE $2 OR ps.sku ILIKE $2)
        ORDER BY ps.id
        LIMIT $3
      `, [searchQuery, `%${searchQuery}%`, Math.min(limitNodes, 200)]);

      for (const row of setsQuery.rows) {
        const setId = `set-${row.id}`;
        if (!nodeIds.has(setId)) {
          nodeIds.add(setId);
          nodes.push({
            data: {
              id: setId,
              label: row.title || row.sku || `Zestaw ${row.id}`,
              type: "set",
              sku: row.sku,
              color: row.color,
              status: row.is_active ? "active" : "inactive",
            },
          });
        }
      }

      const setLinksQuery = await pool.query(`
        SELECT 
          spl.set_id,
          spl.product_id,
          spl.component_type,
          spl.quantity
        FROM product_creator.set_product_links spl
        WHERE spl.set_id = ANY(
          SELECT ps.id FROM product_creator.product_sets ps
          WHERE ($1 = '' OR ps.title ILIKE $2 OR ps.sku ILIKE $2)
          LIMIT $3
        )
      `, [searchQuery, `%${searchQuery}%`, Math.min(limitNodes, 200)]);

      for (const row of setLinksQuery.rows) {
        const setId = `set-${row.set_id}`;
        const productId = `product-${row.product_id}`;

        if (!nodeIds.has(productId)) {
          const productData = await pool.query(`SELECT id, name, sku FROM catalog.products WHERE id = $1`, [row.product_id]);
          if (productData.rows[0]) {
            nodeIds.add(productId);
            nodes.push({
              data: {
                id: productId,
                label: productData.rows[0].name || `Produkt ${row.product_id}`,
                type: "product",
                sku: productData.rows[0].sku,
              },
            });
          }
        }

        if (nodeIds.has(setId) && nodeIds.has(productId)) {
          edges.push({
            data: {
              id: `edge-${++edgeCounter}`,
              source: setId,
              target: productId,
              label: `zawiera (${row.quantity || 1}x)`,
            },
          });
        }
      }

      const bomsQuery = await pool.query(`
        SELECT 
          pb.id,
          pb.name,
          pb.product_id,
          pb.status,
          pb.is_active
        FROM bom.product_boms pb
        WHERE ($1 = '' OR pb.product_id = ANY(
          SELECT cp.id FROM catalog.products cp
          WHERE cp.name ILIKE $2 OR cp.sku ILIKE $2
        ) OR pb.name ILIKE $2)
        ORDER BY pb.id
        LIMIT $3
      `, [searchQuery, `%${searchQuery}%`, Math.min(limitNodes, 300)]);

      for (const row of bomsQuery.rows) {
        const bomId = `bom-${row.id}`;
        const productId = `product-${row.product_id}`;

        if (!nodeIds.has(bomId)) {
          nodeIds.add(bomId);
          nodes.push({
            data: {
              id: bomId,
              label: row.name || `BOM ${row.id}`,
              type: "bom",
              status: row.status,
            },
          });
        }

        if (nodeIds.has(productId)) {
          edges.push({
            data: {
              id: `edge-${++edgeCounter}`,
              source: productId,
              target: bomId,
              label: "ma BOM",
            },
          });
        }
      }

      const bomIds = bomsQuery.rows.map(r => r.id);
      
      console.log(`[CONNECTION GRAPH] Querying components for ${bomIds.length} BOMs`);
      
      if (bomIds.length > 0) {
        const componentsQuery = await pool.query(`
          SELECT 
            pc.id,
            pc.product_bom_id,
            pc.generated_name,
            pc.component_type,
            pc.color,
            pc.calculated_length,
            pc.calculated_width,
            pc.thickness
          FROM bom.product_components pc
          WHERE pc.product_bom_id = ANY($1)
          ORDER BY pc.product_bom_id, pc.position_in_bom
        `, [bomIds]);
        
        console.log(`[CONNECTION GRAPH] Found ${componentsQuery.rows.length} components for ${bomIds.length} BOMs`);

        for (const row of componentsQuery.rows) {
          const formatkaId = `formatka-${row.id}`;
          const bomId = `bom-${row.product_bom_id}`;

          if (!nodeIds.has(formatkaId)) {
            nodeIds.add(formatkaId);
            const dimensions = row.calculated_length && row.calculated_width 
              ? `${row.calculated_length}x${row.calculated_width}` 
              : null;
            nodes.push({
              data: {
                id: formatkaId,
                label: row.generated_name || `Formatka ${row.id}`,
                type: "formatka",
                color: row.color,
                sku: dimensions || undefined,
              },
            });
          }

          if (nodeIds.has(bomId)) {
            edges.push({
              data: {
                id: `edge-${++edgeCounter}`,
                source: bomId,
                target: formatkaId,
                label: row.component_type || "komponent",
              },
            });
          }
        }
      }

      const zlpQuery = await pool.query(`
        SELECT 
          po.id,
          po.order_number,
          po.status,
          ppl.product_id
        FROM production.production_orders po
        JOIN production.production_plan_lines ppl ON ppl.production_order_id = po.id
        WHERE ppl.product_id = ANY(
          SELECT cp.id FROM catalog.products cp
          WHERE ($1 = '' OR cp.name ILIKE $2 OR cp.sku ILIKE $2)
          LIMIT $3
        )
        AND po.status NOT IN ('cancelled')
        LIMIT 100
      `, [searchQuery, `%${searchQuery}%`, limitNodes]);

      for (const row of zlpQuery.rows) {
        const zlpId = `zlp-${row.id}`;
        const productId = `product-${row.product_id}`;

        if (!nodeIds.has(zlpId)) {
          nodeIds.add(zlpId);
          nodes.push({
            data: {
              id: zlpId,
              label: row.order_number || `ZLP-${row.id}`,
              type: "zlp",
              status: row.status,
            },
          });
        }

        if (nodeIds.has(productId)) {
          edges.push({
            data: {
              id: `edge-${++edgeCounter}`,
              source: productId,
              target: zlpId,
              label: "produkowany",
            },
          });
        }
      }

      const stats = {
        totalProducts: nodes.filter(n => n.data.type === "product").length,
        totalSets: nodes.filter(n => n.data.type === "set").length,
        totalBoms: nodes.filter(n => n.data.type === "bom").length,
        totalZlps: nodes.filter(n => n.data.type === "zlp").length,
        totalFormatki: nodes.filter(n => n.data.type === "formatka").length,
      };

      console.log(`[CONNECTION GRAPH] Stats: products=${stats.totalProducts}, sets=${stats.totalSets}, boms=${stats.totalBoms}, formatki=${stats.totalFormatki}, zlps=${stats.totalZlps}, edges=${edges.length}`);

      const graph: CytoscapeGraph = { nodes, edges, stats };
      res.json(graph);
    } catch (error) {
      console.error("Error fetching connection graph:", error);
      res.status(500).json({ error: "Failed to fetch connection graph" });
    }
  });
}
