-- Migration: Add Production Buffer/Warehouse Reservation System
-- Description: Adds support for semi-finished product storage between operations

-- Add buffer columns to production_routing_operations
ALTER TABLE production.production_routing_operations 
ADD COLUMN IF NOT EXISTS creates_buffer BOOLEAN DEFAULT FALSE,
ADD COLUMN IF NOT EXISTS buffer_location_id INTEGER REFERENCES production.production_locations(id) ON DELETE SET NULL;

-- Create index for buffer operations
CREATE INDEX IF NOT EXISTS idx_production_routing_ops_buffer 
ON production.production_routing_operations(creates_buffer) 
WHERE creates_buffer = TRUE;

-- Production Buffer Stock - stany magazynowe półproduktów
CREATE TABLE IF NOT EXISTS production.production_buffer_stock (
  id SERIAL PRIMARY KEY,
  product_sku VARCHAR(255) NOT NULL,
  product_name VARCHAR(500),
  location_id INTEGER REFERENCES production.production_locations(id) ON DELETE SET NULL,
  quantity_available DECIMAL(10,2) NOT NULL DEFAULT 0,
  quantity_reserved DECIMAL(10,2) NOT NULL DEFAULT 0,
  quantity_total DECIMAL(10,2) NOT NULL DEFAULT 0,
  unit_of_measure VARCHAR(20) DEFAULT 'szt',
  created_at TIMESTAMP DEFAULT NOW(),
  updated_at TIMESTAMP DEFAULT NOW(),
  UNIQUE(product_sku, location_id)
);

CREATE INDEX idx_buffer_stock_sku ON production.production_buffer_stock(product_sku);
CREATE INDEX idx_buffer_stock_location ON production.production_buffer_stock(location_id);

-- Production Buffer Movements - ruchy magazynowe półproduktów
CREATE TABLE IF NOT EXISTS production.production_buffer_movements (
  id SERIAL PRIMARY KEY,
  movement_type VARCHAR(50) NOT NULL CHECK (movement_type IN ('IN', 'RESERVE', 'RELEASE', 'OUT')),
  product_sku VARCHAR(255) NOT NULL,
  quantity DECIMAL(10,2) NOT NULL,
  unit_of_measure VARCHAR(20) DEFAULT 'szt',
  location_id INTEGER REFERENCES production.production_locations(id) ON DELETE SET NULL,
  source_type VARCHAR(50),
  source_id INTEGER,
  zlp_id INTEGER REFERENCES production.production_plans(id) ON DELETE SET NULL,
  reservation_id INTEGER,
  created_by INTEGER REFERENCES users(id) ON DELETE SET NULL,
  created_at TIMESTAMP DEFAULT NOW(),
  notes TEXT
);

CREATE INDEX idx_buffer_movements_sku ON production.production_buffer_movements(product_sku);
CREATE INDEX idx_buffer_movements_type ON production.production_buffer_movements(movement_type);
CREATE INDEX idx_buffer_movements_zlp ON production.production_buffer_movements(zlp_id);
CREATE INDEX idx_buffer_movements_source ON production.production_buffer_movements(source_type, source_id);
CREATE INDEX idx_buffer_movements_created_at ON production.production_buffer_movements(created_at);

-- Production Buffer Reservations - rezerwacje półproduktów pod ZLP
CREATE TABLE IF NOT EXISTS production.production_buffer_reservations (
  id SERIAL PRIMARY KEY,
  zlp_id INTEGER NOT NULL REFERENCES production.production_plans(id) ON DELETE CASCADE,
  zlp_item_id INTEGER,
  product_sku VARCHAR(255) NOT NULL,
  quantity_reserved DECIMAL(10,2) NOT NULL,
  quantity_consumed DECIMAL(10,2) DEFAULT 0,
  unit_of_measure VARCHAR(20) DEFAULT 'szt',
  location_id INTEGER REFERENCES production.production_locations(id) ON DELETE SET NULL,
  status VARCHAR(50) DEFAULT 'ACTIVE' CHECK (status IN ('ACTIVE', 'CONSUMED', 'CANCELLED')),
  reserved_at TIMESTAMP DEFAULT NOW(),
  reserved_by INTEGER REFERENCES users(id) ON DELETE SET NULL,
  consumed_at TIMESTAMP,
  cancelled_at TIMESTAMP,
  notes TEXT
);

CREATE INDEX idx_buffer_reservations_zlp ON production.production_buffer_reservations(zlp_id);
CREATE INDEX idx_buffer_reservations_sku ON production.production_buffer_reservations(product_sku);
CREATE INDEX idx_buffer_reservations_status ON production.production_buffer_reservations(status);
CREATE INDEX idx_buffer_reservations_location ON production.production_buffer_reservations(location_id);

-- Add comment to describe buffer system
COMMENT ON TABLE production.production_buffer_stock IS 'Stan magazynowy półproduktów produkowanych na zapas między operacjami';
COMMENT ON TABLE production.production_buffer_movements IS 'Historia ruchów magazynowych półproduktów (IN/OUT/RESERVE/RELEASE)';
COMMENT ON TABLE production.production_buffer_reservations IS 'Rezerwacje półproduktów pod konkretne ZLP';

COMMENT ON COLUMN production.production_routing_operations.creates_buffer IS 'Czy operacja tworzy półprodukt składowany w magazynie';
COMMENT ON COLUMN production.production_routing_operations.buffer_location_id IS 'Domyślna lokalizacja składowania półproduktu';
