import { pool } from "./postgres";

/**
 * Migration script to rebuild combination_key for all matrix-derived products
 * 
 * Problem: Products have inconsistent combination_key formats:
 *   - OLD: "500×360×450:ARTISAN" (missing productType)
 *   - NEW: "VB:500×360×450:SONOMA:no-options" (correct format)
 * 
 * Solution: Rebuild combination_key using canonical buildCombinationKey logic
 */

function buildCombinationKey(
  productType: string | null,
  length: number | null,
  width: number | null,
  height: number | null,
  color: string | null,
  colorOptions: string[]
): string {
  const comboParts = [
    length !== null ? length.toString() : 'null',
    width !== null ? width.toString() : 'null',
    height !== null ? height.toString() : 'null',
  ];
  
  // Build base key
  let key = `${productType || 'NO_TYPE'}:${comboParts.join('×')}:${color || 'no-color'}`;
  
  // Add color options ONLY if they exist (non-empty array)
  if (colorOptions && colorOptions.length > 0) {
    // Sort color options for deterministic key
    const sortedOptions = colorOptions.slice().sort().join(',');
    key += `:${sortedOptions}`;
  }
  
  return key;
}

async function migrateCombinationKeys() {
  console.log('🔄 Starting combination_key migration...\n');
  
  try {
    // Get all products with matrix_id
    const productsResult = await pool.query(
      `SELECT 
        p.id,
        p.sku,
        p.matrix_id,
        p.combination_key as old_combination_key,
        p.length,
        p.width,
        p.height,
        p.color,
        m.product_type,
        m.colors,
        m.color_options
       FROM catalog.products p
       JOIN product_creator.product_matrices m ON p.matrix_id = m.id
       WHERE p.matrix_id IS NOT NULL
       ORDER BY p.id`
    );
    
    console.log(`📊 Found ${productsResult.rows.length} products to migrate\n`);
    
    let updatedCount = 0;
    let unchangedCount = 0;
    const errors: Array<{ id: number; sku: string; error: string }> = [];
    
    for (const product of productsResult.rows) {
      try {
        const {
          id,
          sku,
          old_combination_key,
          length,
          width,
          height,
          color,
          product_type,
          colors,
          color_options
        } = product;
        
        // Parse dimensions
        const lengthNum = length ? parseFloat(length) : null;
        const widthNum = width ? parseFloat(width) : null;
        const heightNum = height ? parseFloat(height) : null;
        
        // Find color index to get color options
        const colorIndex = colors ? colors.indexOf(color) : -1;
        const colorOptionsForProduct = (color_options && colorIndex >= 0 && color_options[colorIndex.toString()])
          ? color_options[colorIndex.toString()]
          : [];
        
        // Build new combination_key
        const newCombinationKey = buildCombinationKey(
          product_type,
          lengthNum,
          widthNum,
          heightNum,
          color,
          colorOptionsForProduct
        );
        
        // Update if changed
        if (old_combination_key !== newCombinationKey) {
          await pool.query(
            `UPDATE catalog.products 
             SET combination_key = $1, updated_at = CURRENT_TIMESTAMP
             WHERE id = $2`,
            [newCombinationKey, id]
          );
          
          updatedCount++;
          console.log(`✅ ${id.toString().padStart(4)} | ${sku.padEnd(30)} | ${old_combination_key} → ${newCombinationKey}`);
        } else {
          unchangedCount++;
        }
      } catch (error: any) {
        errors.push({
          id: product.id,
          sku: product.sku,
          error: error.message
        });
        console.error(`❌ Error processing product ${product.id} (${product.sku}):`, error.message);
      }
    }
    
    console.log('\n📈 Migration Summary:');
    console.log(`   ✅ Updated: ${updatedCount} products`);
    console.log(`   ⏭️  Unchanged: ${unchangedCount} products`);
    console.log(`   ❌ Errors: ${errors.length} products`);
    
    if (errors.length > 0) {
      console.log('\n❌ Errors:');
      errors.forEach(({ id, sku, error }) => {
        console.log(`   ${id} | ${sku} | ${error}`);
      });
    }
    
    console.log('\n✅ Migration completed!');
    
  } catch (error) {
    console.error('❌ Migration failed:', error);
    throw error;
  }
}

// Run migration
migrateCombinationKeys()
  .then(() => {
    console.log('\n🎉 Done!');
    process.exit(0);
  })
  .catch((error) => {
    console.error('\n💥 Fatal error:', error);
    process.exit(1);
  });
