import { pool } from './postgres';
import { processImage, generateImageVersionFilenames } from './image-processor';
import { getFileStorageAdapter } from './file-storage-adapter';
import https from 'https';
import http from 'http';

/**
 * Simplified migration script - ONLY processes catalog.product_images
 * Skips matrices entirely to avoid timeout issues
 */

async function downloadImage(url: string): Promise<Buffer> {
  return new Promise((resolve, reject) => {
    const protocol = url.startsWith('https') ? https : http;
    
    protocol.get(url, (response) => {
      if (response.statusCode !== 200) {
        reject(new Error(`Failed to download image: ${response.statusCode}`));
        return;
      }
      
      const chunks: Buffer[] = [];
      response.on('data', (chunk) => chunks.push(chunk));
      response.on('end', () => resolve(Buffer.concat(chunks)));
      response.on('error', reject);
    }).on('error', reject);
  });
}

async function main() {
  console.log('🟣 === MIGRACJA CATALOG.PRODUCT_IMAGES ONLY ===\n');
  
  const adapter = await getFileStorageAdapter();
  
  // Count total images without thumbnails
  const countResult = await pool.query(`
    SELECT COUNT(*) as total
    FROM catalog.product_images
    WHERE thumbnail_url IS NULL OR medium_url IS NULL
  `);
  
  const totalImages = parseInt(countResult.rows[0].total);
  console.log(`📊 Znaleziono ${totalImages} zdjęć bez miniatur\n`);
  
  if (totalImages === 0) {
    console.log('✅ Wszystkie zdjęcia katalogowe mają już miniatury!');
    await pool.end();
    return;
  }
  
  const CHUNK_SIZE = 15;
  const DELAY_MS = 3000; // 3 second pause between chunks
  let processedCount = 0;
  let errorCount = 0;
  
  try {
    // Process in chunks to avoid timeout
    for (let offset = 0; offset < totalImages; offset += CHUNK_SIZE) {
      const chunkNum = Math.floor(offset / CHUNK_SIZE) + 1;
      const totalChunks = Math.ceil(totalImages / CHUNK_SIZE);
      
      console.log(`\n📦 Chunk ${chunkNum}/${totalChunks} (offset: ${offset})`);
      
      // Get next chunk
      const result = await pool.query(`
        SELECT id, url
        FROM catalog.product_images
        WHERE (thumbnail_url IS NULL OR medium_url IS NULL)
          AND url IS NOT NULL
        ORDER BY id ASC
        LIMIT $1 OFFSET $2
      `, [CHUNK_SIZE, offset]);
      
      if (result.rows.length === 0) {
        console.log('   ℹ️  Brak więcej zdjęć do przetworzenia');
        break;
      }
      
      console.log(`   Przetwarzam ${result.rows.length} zdjęć...`);
      
      for (const row of result.rows) {
        const { id, url: image_url } = row;
        
        try {
          const shortUrl = image_url.length > 60 ? image_url.substring(0, 60) + '...' : image_url;
          console.log(`   🔄 [${id}] ${shortUrl}`);
          
          // Get image buffer - either from local filesystem or via HTTP
          let imageBuffer: Buffer;
          
          if (image_url.startsWith('/uploads/')) {
            // Local file - use adapter to read from ./uploads directory
            console.log(`     📁 Reading from local filesystem...`);
            imageBuffer = await adapter.getBuffer(image_url);
          } else if (image_url.startsWith('http://') || image_url.startsWith('https://')) {
            // Already absolute URL - download via HTTP
            console.log(`     🌐 Downloading from ${image_url.substring(0, 40)}...`);
            imageBuffer = await downloadImage(image_url);
          } else {
            // Relative path without /uploads/ - prepend base URL
            const absoluteUrl = `https://files.alpsys.pl/OMS/${image_url}`;
            console.log(`     🌐 Downloading from ${absoluteUrl.substring(0, 40)}...`);
            imageBuffer = await downloadImage(absoluteUrl);
          }
          
          // Extract filename
          const originalFilename = image_url.split('/').pop() || `image-${id}.jpg`;
          const baseFilename = originalFilename.replace(/\.(jpg|jpeg|png|webp)$/i, '');
          const filenames = generateImageVersionFilenames(baseFilename, 'webp');
          
          // Process image
          const processed = await processImage(imageBuffer, {
            thumbnailSize: 80,
            mediumSize: 400,
            quality: 90,
            format: 'webp',
          });
          
          // Upload versions
          const folderPrefix = 'catalog-products';
          
          await adapter.upload({
            filename: filenames.thumbnail,
            buffer: processed.buffers.thumbnail,
            mimetype: 'image/webp',
            subfolder: `${folderPrefix}/thumbnails`,
          });
          
          await adapter.upload({
            filename: filenames.medium,
            buffer: processed.buffers.medium,
            mimetype: 'image/webp',
            subfolder: `${folderPrefix}/medium`,
          });
          
          // Build URLs
          const thumbnailUrl = adapter.getUrl(`${folderPrefix}/thumbnails/${filenames.thumbnail}`);
          const mediumUrl = adapter.getUrl(`${folderPrefix}/medium/${filenames.medium}`);
          
          // Update database
          await pool.query(`
            UPDATE catalog.product_images
            SET thumbnail_url = $1, medium_url = $2
            WHERE id = $3
          `, [thumbnailUrl, mediumUrl, id]);
          
          processedCount++;
          const pct = Math.round((processedCount / totalImages) * 100);
          console.log(`   ✅ [${id}] OK! (${processedCount}/${totalImages} = ${pct}%)`);
          
        } catch (error: any) {
          console.error(`   ❌ [${id}] Błąd:`, error.message);
          errorCount++;
        }
      }
      
      // Progress summary
      const pct = Math.round((processedCount / totalImages) * 100);
      console.log(`\n   📊 Postęp: ${processedCount}/${totalImages} OK (${pct}%), ${errorCount} błędów`);
      
      // Pause between chunks to avoid CPU timeout
      if (offset + CHUNK_SIZE < totalImages) {
        console.log(`   ⏸️  Pauza ${DELAY_MS}ms przed następnym chunkiem...\n`);
        await new Promise(resolve => setTimeout(resolve, DELAY_MS));
      }
    }
    
    console.log(`\n✅ === MIGRACJA ZAKOŃCZONA ===`);
    console.log(`📊 Wynik: ${processedCount} OK, ${errorCount} błędów`);
    
  } catch (error: any) {
    console.error('\n❌ === BŁĄD PODCZAS MIGRACJI ===');
    console.error(error);
    process.exit(1);
  } finally {
    await pool.end();
  }
}

// Run migration
main();
