import { pool } from './postgres.js';
import { renderFormatkaVisualization } from './formatka-visualizer.js';

const POLLING_INTERVAL = 5000; // 5 seconds
let isProcessing = false;
let isShuttingDown = false;

async function processNextJob() {
  if (isProcessing || isShuttingDown) {
    return;
  }

  const client = await pool.connect();
  try {
    isProcessing = true;

    // Pobierz najstarszy pending job z najwyższym priorytetem
    const jobResult = await client.query(`
      SELECT * FROM jobs
      WHERE status = 'pending' AND attempts < max_attempts
      ORDER BY priority DESC, created_at ASC
      LIMIT 1
      FOR UPDATE SKIP LOCKED
    `);

    if (jobResult.rows.length === 0) {
      return; // Brak zadań do przetworzenia
    }

    const job = jobResult.rows[0];
    console.log(`[JOB PROCESSOR] Processing job #${job.id} (${job.job_type})`);

    // Oznacz jako processing
    await client.query(`
      UPDATE jobs
      SET status = 'processing', started_at = NOW(), attempts = attempts + 1
      WHERE id = $1
    `, [job.id]);

    try {
      // Przetwórz job według typu
      if (job.job_type === 'render_formatka') {
        await processRenderFormatka(client, job);
      } else if (job.job_type === 'generate_bom') {
        // To będzie zaimplementowane później dla bulk generation
        console.log(`[JOB PROCESSOR] Job type ${job.job_type} not yet implemented`);
        throw new Error(`Job type ${job.job_type} not yet implemented`);
      } else {
        throw new Error(`Unknown job type: ${job.job_type}`);
      }

      // Oznacz jako completed
      await client.query(`
        UPDATE jobs
        SET status = 'completed', completed_at = NOW(), error = NULL
        WHERE id = $1
      `, [job.id]);

      console.log(`[JOB PROCESSOR] ✅ Job #${job.id} completed successfully`);

    } catch (error: any) {
      console.error(`[JOB PROCESSOR] ❌ Job #${job.id} failed:`, error.message);

      // Sprawdź czy przekroczono max_attempts
      if (job.attempts + 1 >= job.max_attempts) {
        // Oznacz jako failed
        await client.query(`
          UPDATE jobs
          SET status = 'failed', completed_at = NOW(), error = $1
          WHERE id = $2
        `, [error.message, job.id]);
        console.log(`[JOB PROCESSOR] Job #${job.id} failed permanently (max attempts reached)`);
      } else {
        // Cofnij do pending dla ponowienia
        await client.query(`
          UPDATE jobs
          SET status = 'pending', error = $1
          WHERE id = $2
        `, [error.message, job.id]);
        console.log(`[JOB PROCESSOR] Job #${job.id} will be retried (attempt ${job.attempts + 1}/${job.max_attempts})`);
      }
    }

  } finally {
    client.release();
    isProcessing = false;
  }
}

async function processRenderFormatka(client: any, job: any) {
  const { productId, bomId } = job.payload;
  console.log(`[RENDER FORMATKA] Processing product ${productId}, BOM ${bomId}`);

  // Pobierz wszystkie komponenty z pełnymi danymi z templates
  const fullComponentsResult = await client.query(`
    SELECT 
      pc.id,
      pc.generated_name,
      ct.cz1,
      ct.thickness,
      ct.edge1,
      ct.edge2,
      ct.edge3,
      ct.edge4,
      ct.edging_material
    FROM bom.product_components pc
    JOIN bom.component_templates ct ON ct.id = pc.component_template_id
    WHERE pc.product_bom_id = $1
  `, [bomId]);

  let visualizationsGenerated = 0;
  const totalComponents = fullComponentsResult.rows.length;

  for (const comp of fullComponentsResult.rows) {
    try {
      // Wydobądź wymiary i kolor z generated_name (np. "BOK-L-VB-390x340-BIALY")
      const dimensionMatch = comp.generated_name.match(/(\d+)x(\d+)/);
      const colorMatch = comp.generated_name.match(/-([^-]+)$/);
      
      if (!dimensionMatch) {
        console.warn(`⚠️  No dimensions found in ${comp.generated_name}, skipping visualization`);
        continue;
      }
      
      const colorCode = colorMatch ? colorMatch[1] : 'UNKNOWN';
      
      // Pobierz kolor hex formatki ze słownika
      let colorHex: string | undefined;
      if (colorCode) {
        const colorResult = await client.query(`
          SELECT color FROM product_creator.dictionaries 
          WHERE dictionary_type = 'color' AND code = $1
        `, [colorCode]);
        if (colorResult.rows.length > 0) {
          colorHex = colorResult.rows[0].color;
        }
      }

      // Pobierz kolor hex obrzeża ze słownika
      let edgingColorHex: string | undefined;
      if (comp.edging_material) {
        // Wyciągnij kod koloru z edging_material (np. "0.8_BIALY" -> "BIALY")
        const parts = comp.edging_material.split('_');
        const edgingColorCode = parts.length >= 2 ? parts[parts.length - 1] : comp.edging_material;
        
        const edgingColorResult = await client.query(`
          SELECT color FROM product_creator.dictionaries 
          WHERE dictionary_type = 'color' AND code = $1
        `, [edgingColorCode]);
        if (edgingColorResult.rows.length > 0) {
          edgingColorHex = edgingColorResult.rows[0].color;
        }
      }
      
      const formatkaData = {
        name: comp.generated_name,
        length: parseInt(dimensionMatch[1]),
        width: parseInt(dimensionMatch[2]),
        thickness: parseFloat(comp.thickness) || 18,
        color: colorCode,
        colorHex: colorHex,
        edgingColorHex: edgingColorHex,
        componentType: comp.cz1 || 'formatka',
        edge1: comp.edge1 || false,
        edge2: comp.edge2 || false,
        edge3: comp.edge3 || false,
        edge4: comp.edge4 || false
      };
      
      const visualizationUrl = await renderFormatkaVisualization(formatkaData);
      
      // Zapisz URL w bazie
      await client.query(`
        UPDATE bom.product_components
        SET visualization_url = $1
        WHERE id = $2
      `, [visualizationUrl, comp.id]);
      
      visualizationsGenerated++;
      console.log(`  ✅ [${visualizationsGenerated}/${totalComponents}] ${comp.generated_name}`);
      
    } catch (vizError: any) {
      console.error(`❌ Failed to generate visualization for ${comp.generated_name}:`, vizError.message);
      // Kontynuuj mimo błędów - wizualizacje są opcjonalne
    }
  }
  
  console.log(`✅ [RENDER FORMATKA] Generated ${visualizationsGenerated}/${totalComponents} visualizations for product ${productId}`);
}

async function startProcessor() {
  console.log('[JOB PROCESSOR] Starting background job processor...');
  console.log(`[JOB PROCESSOR] Polling interval: ${POLLING_INTERVAL}ms`);

  // Uruchom polling loop
  const intervalId = setInterval(async () => {
    try {
      await processNextJob();
    } catch (error: any) {
      console.error('[JOB PROCESSOR] Error in polling loop:', error.message);
    }
  }, POLLING_INTERVAL);

  // Graceful shutdown
  process.on('SIGINT', () => {
    console.log('[JOB PROCESSOR] Received SIGINT, shutting down...');
    isShuttingDown = true;
    clearInterval(intervalId);
    process.exit(0);
  });

  process.on('SIGTERM', () => {
    console.log('[JOB PROCESSOR] Received SIGTERM, shutting down...');
    isShuttingDown = true;
    clearInterval(intervalId);
    process.exit(0);
  });
}

// Auto-start jeśli uruchomiony jako standalone proces
if (import.meta.url === `file://${process.argv[1]}`) {
  startProcessor().catch((error) => {
    console.error('[JOB PROCESSOR] Fatal error:', error);
    process.exit(1);
  });
}

export { startProcessor, processNextJob };
