import crypto from 'crypto';
// Puppeteer is imported dynamically to avoid loading Chromium binaries in production
import { getFileStorageAdapter } from './file-storage-adapter.js';
import { pool } from './postgres.js';

/**
 * Pobiera kod hex koloru ze słownika dictionaries
 */
async function getColorHexFromDictionary(colorCode: string): Promise<string | null> {
  if (!colorCode) return null;
  
  try {
    const result = await pool.query(`
      SELECT color FROM product_creator.dictionaries 
      WHERE dictionary_type = 'color' AND code = $1
    `, [colorCode]);
    
    if (result.rows.length > 0) {
      return result.rows[0].color;
    }
    
    console.warn(`⚠️  Color code "${colorCode}" not found in dictionaries`);
    return null;
  } catch (error) {
    console.error(`❌ Error fetching color hex for "${colorCode}":`, error);
    return null;
  }
}

/**
 * Wyciąga kod koloru z pola edging_material (np. "0.8_BIALY" -> "BIALY")
 */
function extractEdgingColorCode(edgingMaterial: string | null): string | null {
  if (!edgingMaterial) return null;
  
  // Format: "thickness_COLOR" lub "COLOR"
  const parts = edgingMaterial.split('_');
  if (parts.length >= 2) {
    // Weź ostatnią część po podkreślniku
    return parts[parts.length - 1];
  }
  
  // Jeśli nie ma podkreślnika, zwróć całość
  return edgingMaterial;
}

interface FormatkaData {
  name: string;
  length: number;
  width: number;
  thickness: number;
  color: string;
  colorHex?: string;        // Kod hex koloru formatki (z dictionaries)
  edgingColorHex?: string;  // Kod hex koloru obrzeża (z dictionaries)
  edgingPattern?: string;
  componentType: string;
  edge1?: boolean;
  edge2?: boolean;
  edge3?: boolean;
  edge4?: boolean;
}

/**
 * Generate simple HTML for formatka visualization (Odoo export)
 * Only shape with edging indicators, no text
 */
function generateSimpleFormatkaHTML(formatka: FormatkaData): string {
  const { length, width, edge1, edge2, edge3, edge4, colorHex, edgingColorHex } = formatka;

  // Validate dimensions to avoid division by zero
  const validLength = Math.max(1, length || 1);
  const validWidth = Math.max(1, width || 1);

  // Calculate SVG dimensions - fixed size 800x600
  const svgWidth = 800;
  const svgHeight = 600;
  const padding = 100; // Odstęp od krawędzi canvas (dla obrzeży)
  
  // Dostępne miejsce na prostokąt formatki
  const maxRectWidth = svgWidth - padding * 2;
  const maxRectHeight = svgHeight - padding * 2;
  
  // Oblicz aspect ratio formatki (stosunek długości do szerokości)
  const aspectRatio = validLength / validWidth;
  
  // Skaluj prostokąt zachowując proporcje
  let rectWidth: number, rectHeight: number;
  
  // Najpierw spróbuj dopasować do szerokości
  rectWidth = maxRectWidth;
  rectHeight = rectWidth / aspectRatio;
  
  // Jeśli wysokość przekracza dostępne miejsce, dopasuj do wysokości
  if (rectHeight > maxRectHeight) {
    rectHeight = maxRectHeight;
    rectWidth = rectHeight * aspectRatio;
  }

  // Wycentruj prostokąt
  const rectX = (svgWidth - rectWidth) / 2;
  const rectY = (svgHeight - rectHeight) / 2;

  // Kolor tła formatki - MUSI BYĆ Z BAZY DANYCH!
  const fillColor = colorHex || '#cccccc'; // szary fallback jeśli brak colorHex
  
  // Sprawdź czy formatka jest biała
  const isWhiteBackground = colorHex && (colorHex.toLowerCase() === '#ffffff' || colorHex.toLowerCase() === '#fff');
  
  // ZAWSZE CZARNE OBRZEŻA I CZARNY STROKE (dla świętego spokoju!)
  const strokeColor = '#000000'; // zawsze czarny
  const strokeWidth = isWhiteBackground ? 8 : 3; // grubszy dla białej (lepiej widoczny)
  const edgingColor = '#000000'; // ZAWSZE czarny
  const edgingStrokeWidth = 20; // Grubość obrzeża 2x większa (było 10px)
  const edgeOffset = isWhiteBackground ? 80 : 64; // 2x większy offset - dalej od formatki
  
  console.log(`🎨 [FORMATKA] Rendering: ${formatka.name}, fill=${fillColor}, stroke=${strokeColor}, edging=${edgingColor}, edgingWidth=${edgingStrokeWidth}, offset=${edgeOffset}`);

  return `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <style>
    * {
      margin: 0;
      padding: 0;
      box-sizing: border-box;
    }
    body {
      width: ${svgWidth}px;
      height: ${svgHeight}px;
      background: #ffffff;
      display: flex;
      align-items: center;
      justify-content: center;
    }
  </style>
</head>
<body>
  <svg width="${svgWidth}" height="${svgHeight}" viewBox="0 0 ${svgWidth} ${svgHeight}">
    <!-- Main rectangle -->
    <rect
      x="${rectX}"
      y="${rectY}"
      width="${rectWidth}"
      height="${rectHeight}"
      fill="${fillColor}"
      stroke="${strokeColor}"
      stroke-width="${strokeWidth}"
    />
    
    <!-- Edge 1 (top) -->
    ${edge1 ? `
    <line
      x1="${rectX}"
      y1="${rectY - edgeOffset}"
      x2="${rectX + rectWidth}"
      y2="${rectY - edgeOffset}"
      stroke="${edgingColor}"
      stroke-width="${edgingStrokeWidth}"
    />
    ` : ''}
    
    <!-- Edge 2 (right) -->
    ${edge2 ? `
    <line
      x1="${rectX + rectWidth + edgeOffset}"
      y1="${rectY}"
      x2="${rectX + rectWidth + edgeOffset}"
      y2="${rectY + rectHeight}"
      stroke="${edgingColor}"
      stroke-width="${edgingStrokeWidth}"
    />
    ` : ''}
    
    <!-- Edge 3 (bottom) -->
    ${edge3 ? `
    <line
      x1="${rectX}"
      y1="${rectY + rectHeight + edgeOffset}"
      x2="${rectX + rectWidth}"
      y2="${rectY + rectHeight + edgeOffset}"
      stroke="${edgingColor}"
      stroke-width="${edgingStrokeWidth}"
    />
    ` : ''}
    
    <!-- Edge 4 (left) -->
    ${edge4 ? `
    <line
      x1="${rectX - edgeOffset}"
      y1="${rectY}"
      x2="${rectX - edgeOffset}"
      y2="${rectY + rectHeight}"
      stroke="${edgingColor}"
      stroke-width="${edgingStrokeWidth}"
    />
    ` : ''}
  </svg>
</body>
</html>
  `.trim();
}

/**
 * Generate detailed HTML for formatka visualization
 * Full visualization with dimensions and details
 */
function generateDetailedFormatkaHTML(formatka: FormatkaData): string {
  const { name, length, width, thickness, color, edgingPattern } = formatka;

  return `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <style>
    * {
      margin: 0;
      padding: 0;
      box-sizing: border-box;
    }
    
    body {
      width: 800px;
      height: 600px;
      display: flex;
      align-items: center;
      justify-content: center;
      background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
      font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }
    
    .formatka-container {
      background: white;
      border-radius: 12px;
      padding: 40px;
      box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
      text-align: center;
      max-width: 700px;
    }
    
    .formatka-title {
      font-size: 24px;
      font-weight: 600;
      color: #2c3e50;
      margin-bottom: 30px;
      word-wrap: break-word;
    }
    
    .formatka-rectangle {
      width: ${Math.min(500, length * 0.8)}px;
      height: ${Math.min(300, width * 0.8)}px;
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      margin: 0 auto 30px;
      border-radius: 8px;
      display: flex;
      align-items: center;
      justify-content: center;
      position: relative;
      box-shadow: 0 4px 20px rgba(102, 126, 234, 0.3);
    }
    
    .formatka-dimensions {
      color: white;
      font-size: 32px;
      font-weight: 700;
      text-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
    }
    
    .formatka-details {
      display: grid;
      grid-template-columns: repeat(2, 1fr);
      gap: 15px;
      margin-top: 25px;
    }
    
    .detail-item {
      background: #f8f9fa;
      padding: 12px 20px;
      border-radius: 8px;
      border-left: 4px solid #667eea;
    }
    
    .detail-label {
      font-size: 12px;
      color: #6c757d;
      text-transform: uppercase;
      letter-spacing: 0.5px;
      margin-bottom: 4px;
    }
    
    .detail-value {
      font-size: 16px;
      font-weight: 600;
      color: #2c3e50;
    }
  </style>
</head>
<body>
  <div class="formatka-container">
    <div class="formatka-title">${name}</div>
    
    <div class="formatka-rectangle">
      <div class="formatka-dimensions">${length} × ${width} mm</div>
    </div>
    
    <div class="formatka-details">
      <div class="detail-item">
        <div class="detail-label">Długość</div>
        <div class="detail-value">${length} mm</div>
      </div>
      
      <div class="detail-item">
        <div class="detail-label">Szerokość</div>
        <div class="detail-value">${width} mm</div>
      </div>
      
      <div class="detail-item">
        <div class="detail-label">Grubość</div>
        <div class="detail-value">${thickness} mm</div>
      </div>
      
      <div class="detail-item">
        <div class="detail-label">Kolor</div>
        <div class="detail-value">${color}</div>
      </div>
      
      ${edgingPattern ? `
      <div class="detail-item" style="grid-column: 1 / -1;">
        <div class="detail-label">Obrzeża</div>
        <div class="detail-value">${edgingPattern}</div>
      </div>
      ` : ''}
    </div>
  </div>
</body>
</html>
  `;
}

/**
 * Generate cache key for formatka visualization
 */
function generateCacheKey(formatka: FormatkaData): string {
  const edges = `${formatka.edge1 ? '1' : '0'}${formatka.edge2 ? '1' : '0'}${formatka.edge3 ? '1' : '0'}${formatka.edge4 ? '1' : '0'}`;
  const key = `${formatka.name}_${formatka.length}_${formatka.width}_${formatka.thickness}_${formatka.color}_${formatka.colorHex || ''}_${formatka.edgingColorHex || ''}_${edges}`;
  return crypto.createHash('md5').update(key).digest('hex');
}

/**
 * Render formatka visualization as JPG and upload to storage
 * @returns Public URL of the generated JPG image
 */
export async function renderFormatkaVisualization(formatka: FormatkaData): Promise<string> {
  const cacheKey = generateCacheKey(formatka);
  const filename = `formatka-${cacheKey}.jpg`;
  const storagePath = `bom-visualizations/${filename}`;
  
  // Check if already exists in cache
  const fileStorage = await getFileStorageAdapter();
  const cached = await fileStorage.exists(storagePath);
  
  if (cached) {
    console.log(`✅ Cache hit for formatka visualization: ${filename}`);
    return fileStorage.getUrl(storagePath);
  }
  
  console.log(`🎨 Rendering new formatka visualization: ${filename} (${formatka.length}×${formatka.width}mm)`);
  
  // Generate simple HTML (only shape + edges, no text) for Odoo
  const html = generateSimpleFormatkaHTML(formatka);
  
  // Find Chromium executable
  const { execSync } = await import('child_process');
  let chromiumPath: string | undefined;
  
  try {
    chromiumPath = execSync('which chromium-browser 2>/dev/null || which chromium 2>/dev/null || which google-chrome 2>/dev/null', { encoding: 'utf-8' }).trim();
  } catch (error) {
    const nixStorePaths = [
      '/nix/store/*/bin/chromium-browser',
      '/nix/store/*/bin/chromium'
    ];
    
    for (const pattern of nixStorePaths) {
      try {
        const paths = execSync(`ls ${pattern} 2>/dev/null | head -1`, { encoding: 'utf-8' }).trim();
        if (paths) {
          chromiumPath = paths;
          break;
        }
      } catch {}
    }
  }
  
  console.log(`🌐 Using Chromium executable: ${chromiumPath || 'bundled'}`);
  
  // Create unique temp dir for this browser instance
  const tempDir = `/tmp/chrome-formatka-${Date.now()}-${Math.random().toString(36).slice(2, 8)}`;
  
  let browser;
  try {
    // Dynamically import puppeteer to avoid loading Chromium binaries at module load time
    const puppeteer = (await import('puppeteer')).default;
    
    // Launch browser with isolated temp dir
    browser = await puppeteer.launch({
      headless: true,
      args: [
        '--no-sandbox',
        '--disable-setuid-sandbox',
        '--disable-dev-shm-usage',
        '--disable-gpu',
        `--user-data-dir=${tempDir}`,
        `--disk-cache-dir=${tempDir}/cache`
      ],
      executablePath: chromiumPath || undefined,
    });
    
    const page = await browser.newPage();
    await page.setViewport({ width: 800, height: 600 });
    await page.setContent(html, { waitUntil: 'networkidle0' });
    
    // Take screenshot as JPG
    const screenshotBuffer = await page.screenshot({
      type: 'jpeg',
      quality: 90,
      fullPage: false,
    });
    
    await browser.close();
    
    // Upload to storage using upload method
    const uploadedUrl = await fileStorage.upload({
      filename,
      buffer: screenshotBuffer as Buffer,
      mimetype: 'image/jpeg',
      subfolder: 'bom-visualizations'
    });
    
    console.log(`✅ Formatka visualization rendered and uploaded: ${uploadedUrl}`);
    
    // Cleanup temp dir
    try {
      const fs = await import('fs/promises');
      await fs.rm(tempDir, { recursive: true, force: true });
      console.log(`🧹 Cleaned up temp dir: ${tempDir}`);
    } catch (cleanupError) {
      console.warn(`⚠️  Failed to cleanup temp dir:`, cleanupError);
    }
    
    return uploadedUrl;
    
  } catch (error) {
    if (browser) {
      await browser.close();
    }
    
    // Cleanup temp dir on error
    try {
      const fs = await import('fs/promises');
      await fs.rm(tempDir, { recursive: true, force: true });
      console.log(`🧹 Cleaned up temp dir after error: ${tempDir}`);
    } catch (cleanupError) {
      console.warn(`⚠️  Failed to cleanup temp dir:`, cleanupError);
    }
    
    const errorMessage = error instanceof Error ? error.message : String(error);
    throw new Error(`Failed to render formatka visualization for ${formatka.name}: ${errorMessage}`);
  }
}
