import axios from 'axios';

const ERLI_API_BASE = 'https://erli.pl/svc/shop-api';

export interface ErliOrder {
  id: string;
  status: string;
  updated?: string;
  purchasedAt?: string;
  calculatedParcelsCount?: number;
  items?: ErliOrderItem[];
  trackingNumber?: string;
  deliveryStatus?: string;
  buyer?: {
    login?: string;
    email?: string;
    firstName?: string;
    lastName?: string;
    phone?: string;
  };
  delivery?: {
    method?: string;
    address?: {
      street?: string;
      city?: string;
      zipCode?: string;
      countryCode?: string;
    };
    cost?: {
      amount?: string;
      currency?: string;
    };
  };
  payment?: {
    status?: string;
    type?: string;
    amount?: {
      amount?: string;
      currency?: string;
    };
  };
  summary?: {
    totalToPay?: {
      amount?: string;
      currency?: string;
    };
  };
  notes?: string;
}

export interface ErliOrderItem {
  id: string;
  externalId?: string;
  name: string;
  quantity: number;
  price?: {
    amount?: string;
    currency?: string;
  };
  weight?: number;
  imageUrl?: string;
}

export interface ErliSearchParams {
  status?: string;
  updatedAfter?: string;
  dateFrom?: string;
  cursor?: string;
  limit?: number;
  offset?: number;
}

function getErliApiKey(): string {
  const apiKey = process.env.ERLI_API_KEY;
  if (!apiKey) {
    throw new Error('ERLI_API_KEY not configured. Please add it to your secrets.');
  }
  return apiKey;
}

function getHeaders(): Record<string, string> {
  return {
    'Authorization': `Bearer ${getErliApiKey()}`,
    'Accept': 'application/json',
    'Content-Type': 'application/json',
    'User-Agent': 'OMS-Integration/1.0'
  };
}

export async function testErliConnection(): Promise<{ success: boolean; message: string; error?: string; shopInfo?: any }> {
  try {
    const response = await axios.get(
      `${ERLI_API_BASE}/me`,
      { headers: getHeaders(), timeout: 10000 }
    );
    
    console.log('✅ Erli connection test successful:', response.data);
    return {
      success: true,
      message: 'Połączenie z Erli działa poprawnie',
      shopInfo: response.data
    };
  } catch (error: any) {
    const errorMessage = error.response?.data?.message || error.message;
    console.error('❌ Erli connection test failed:', errorMessage);
    return {
      success: false,
      message: `Błąd połączenia z Erli: ${errorMessage}`,
      error: errorMessage
    };
  }
}

export async function searchErliOrders(params: ErliSearchParams = {}): Promise<ErliOrder[]> {
  try {
    // Erli uses cursor-based pagination with format: timestamp;orderId (e.g., "1234;221201x12345")
    // First request: no 'after' parameter to get newest orders
    // Subsequent requests: use cursor from last order
    const searchBody: any = {
      pagination: {
        sortField: 'updated',
        order: 'DESC',
        limit: params.limit || 100
      }
    };
    
    // If we have a cursor from previous request, use it
    if (params.cursor) {
      searchBody.pagination.after = params.cursor;
    }
    
    if (params.status) {
      searchBody.status = params.status;
    }

    console.log(`📦 [ERLI] Searching orders with params:`, JSON.stringify(searchBody));

    const response = await axios.post(
      `${ERLI_API_BASE}/orders/_search`,
      searchBody,
      { headers: getHeaders(), timeout: 30000 }
    );

    // Erli API returns array directly
    let orders: any[] = [];
    if (Array.isArray(response.data)) {
      orders = response.data;
    } else if (response.data?.orders) {
      orders = response.data.orders;
    } else if (response.data?.data) {
      orders = response.data.data;
    } else if (response.data?.items) {
      orders = response.data.items;
    }
    
    console.log(`📦 [ERLI] Found ${orders.length} orders from API (first: ${orders[0]?.created}, last: ${orders[orders.length-1]?.created})`);
    
    return orders;
  } catch (error: any) {
    const errorMessage = error.response?.data?.message || error.message;
    console.error('❌ [ERLI] Failed to search orders:', errorMessage, error.response?.data);
    throw new Error(`Failed to search Erli orders: ${errorMessage}`);
  }
}

export async function fetchErliOrderDetails(orderId: string): Promise<ErliOrder | null> {
  try {
    console.log(`📦 [ERLI] Fetching order details for: ${orderId}`);

    const response = await axios.get(
      `${ERLI_API_BASE}/orders/${orderId}`,
      { headers: getHeaders(), timeout: 15000 }
    );

    return response.data;
  } catch (error: any) {
    const errorMessage = error.response?.data?.message || error.message;
    console.error(`❌ [ERLI] Failed to fetch order ${orderId}:`, errorMessage);
    return null;
  }
}

export async function updateErliOrder(
  orderId: string,
  updates: { trackingNumber?: string; status?: string }
): Promise<boolean> {
  try {
    console.log(`📝 [ERLI] Updating order ${orderId}:`, updates);

    await axios.patch(
      `${ERLI_API_BASE}/orders/${orderId}`,
      updates,
      { headers: getHeaders(), timeout: 15000 }
    );

    console.log(`✅ [ERLI] Order ${orderId} updated successfully`);
    return true;
  } catch (error: any) {
    const errorMessage = error.response?.data?.message || error.message;
    console.error(`❌ [ERLI] Failed to update order ${orderId}:`, errorMessage);
    return false;
  }
}

export async function fetchAllErliOrders(
  updatedAfter?: Date,
  maxOrders: number = 1000
): Promise<ErliOrder[]> {
  const allOrders: ErliOrder[] = [];
  let offset = 0;
  const limit = 100;

  try {
    while (allOrders.length < maxOrders) {
      const params: ErliSearchParams = { limit, offset };
      
      const orders = await searchErliOrders(params);
      
      if (orders.length === 0) {
        break;
      }

      if (updatedAfter) {
        const filteredOrders = orders.filter(order => {
          if (!order.updated) return true;
          return new Date(order.updated) >= updatedAfter;
        });
        allOrders.push(...filteredOrders);
        
        if (filteredOrders.length < orders.length) {
          break;
        }
      } else {
        allOrders.push(...orders);
      }

      if (orders.length < limit) {
        break;
      }

      offset += limit;
      
      await new Promise(resolve => setTimeout(resolve, 200));
    }

    console.log(`📦 [ERLI] Total orders fetched: ${allOrders.length}`);
    return allOrders;
  } catch (error: any) {
    console.error('❌ [ERLI] Failed to fetch all orders:', error.message);
    throw error;
  }
}

export async function fetchErliInbox(): Promise<any[]> {
  try {
    console.log('📬 [ERLI] Fetching inbox events');

    const response = await axios.get(
      `${ERLI_API_BASE}/inbox`,
      { headers: getHeaders(), timeout: 15000 }
    );

    const events = response.data?.events || [];
    console.log(`📬 [ERLI] Found ${events.length} inbox events`);
    
    return events;
  } catch (error: any) {
    const errorMessage = error.response?.data?.message || error.message;
    console.error('❌ [ERLI] Failed to fetch inbox:', errorMessage);
    return [];
  }
}

export function mapErliStatusToInternal(erliStatus: string): string {
  const statusMap: Record<string, string> = {
    'pending': 'NEW',
    'confirmed': 'PROCESSING',
    'shipped': 'SENT',
    'delivered': 'DELIVERED',
    'cancelled': 'CANCELLED',
    'returned': 'RETURNED',
    'claimed': 'CLAIMED'
  };
  
  return statusMap[erliStatus?.toLowerCase()] || erliStatus?.toUpperCase() || 'NEW';
}

export function mapErliPaymentStatus(erliPaymentStatus: string): string {
  const statusMap: Record<string, string> = {
    'paid': 'PAID',
    'unpaid': 'UNPAID',
    'pending': 'PENDING',
    'refunded': 'REFUNDED',
    'partial': 'PARTIAL'
  };
  
  return statusMap[erliPaymentStatus?.toLowerCase()] || erliPaymentStatus?.toUpperCase() || 'PENDING';
}
