import { pool } from './postgres';
import archiver from 'archiver';
import { createWriteStream, promises as fs } from 'fs';
import { pipeline, finished } from 'stream/promises';
import path from 'path';
import { getFileStorageAdapter } from './file-storage-adapter';
import SftpClient from 'ssh2-sftp-client';

interface BackupMetadata {
  id: string;
  type: 'database' | 'files' | 'full';
  timestamp: string;
  filename: string;
  size: number;
  status: 'pending' | 'completed' | 'failed';
  error?: string;
}

const BACKUP_DIR = path.join(process.cwd(), 'backups');

// Ensure backup directory exists
async function ensureBackupDir() {
  try {
    await fs.mkdir(BACKUP_DIR, { recursive: true });
  } catch (error) {
    console.error('Error creating backup directory:', error);
  }
}

/**
 * Export database to SQL dump (streaming to file to avoid memory limits)
 */
export async function exportDatabaseToSQL(): Promise<string> {
  const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
  const filename = `database-backup-${timestamp}.sql`;
  const filepath = path.join(BACKUP_DIR, filename);

  await ensureBackupDir();

  const client = await pool.connect();
  const writeStream = createWriteStream(filepath, { encoding: 'utf-8' });
  
  const writeLine = (line: string) => {
    writeStream.write(line + '\n');
  };

  try {
    writeLine('-- Alpma OMS Database Backup');
    writeLine(`-- Generated: ${new Date().toISOString()}`);
    writeLine('-- Compatible with PostgreSQL');
    writeLine('');
    writeLine('BEGIN;');
    writeLine('');

    // Get all schemas except system schemas
    const schemasResult = await client.query(`
      SELECT schema_name 
      FROM information_schema.schemata 
      WHERE schema_name NOT IN ('pg_catalog', 'information_schema', 'pg_toast')
      ORDER BY schema_name
    `);

    // Store foreign keys and sequence ownership for later
    const allForeignKeys: Array<{
      schema: string;
      tableName: string;
      constraintName: string;
      columns: string[];
      refSchema: string;
      refTable: string;
      refColumns: string[];
      updateRule: string;
      deleteRule: string;
    }> = [];

    const allSequenceOwnerships: Array<{
      schemaname: string;
      sequencename: string;
      ownedSchema: string;
      ownedTable: string;
      ownedColumn: string;
    }> = [];

    const allSequenceSetvals: Array<{
      schemaname: string;
      sequencename: string;
      lastValue: number;
    }> = [];

    for (const schemaRow of schemasResult.rows) {
      const schema = schemaRow.schema_name;
      writeLine(`-- Schema: ${schema}`);
      writeLine(`CREATE SCHEMA IF NOT EXISTS ${schema};\n`);

      // Export sequences first using pg_sequences
      const sequencesResult = await client.query(`
        SELECT 
          schemaname,
          sequencename,
          start_value,
          min_value,
          max_value,
          increment_by,
          cycle,
          cache_size,
          last_value
        FROM pg_sequences
        WHERE schemaname = $1
        ORDER BY sequencename
      `, [schema]);

      for (const seq of sequencesResult.rows) {
        writeLine(`-- Sequence: ${seq.schemaname}.${seq.sequencename}`);
        
        const cycleStr = seq.cycle ? 'CYCLE' : 'NO CYCLE';
        writeLine(`CREATE SEQUENCE IF NOT EXISTS ${seq.schemaname}.${seq.sequencename}`);
        writeLine(`  INCREMENT BY ${seq.increment_by}`);
        writeLine(`  MINVALUE ${seq.min_value}`);
        writeLine(`  MAXVALUE ${seq.max_value}`);
        writeLine(`  START WITH ${seq.start_value}`);
        writeLine(`  CACHE ${seq.cache_size}`);
        writeLine(`  ${cycleStr};`);
        
        // Get OWNED BY information (defer execution until after tables are created)
        const ownedByResult = await client.query(`
          SELECT 
            n.nspname AS schema_name,
            t.relname AS table_name,
            a.attname AS column_name
          FROM pg_class c
          JOIN pg_namespace ns ON c.relnamespace = ns.oid
          JOIN pg_depend d ON d.objid = c.oid
          JOIN pg_class t ON d.refobjid = t.oid
          JOIN pg_attribute a ON a.attrelid = t.oid AND a.attnum = d.refobjsubid
          JOIN pg_namespace n ON t.relnamespace = n.oid
          WHERE c.relkind = 'S'
            AND ns.nspname = $1
            AND c.relname = $2
            AND d.deptype = 'a'
        `, [seq.schemaname, seq.sequencename]);

        if (ownedByResult.rows.length > 0) {
          const owned = ownedByResult.rows[0];
          allSequenceOwnerships.push({
            schemaname: seq.schemaname,
            sequencename: seq.sequencename,
            ownedSchema: owned.schema_name,
            ownedTable: owned.table_name,
            ownedColumn: owned.column_name
          });
        }
        
        // Store setval for later (after OWNED BY)
        if (seq.last_value) {
          allSequenceSetvals.push({
            schemaname: seq.schemaname,
            sequencename: seq.sequencename,
            lastValue: seq.last_value
          });
        }
      }
      writeLine(''); // Add blank line after sequences

      // Get all tables in this schema
      const tablesResult = await client.query(`
        SELECT table_name 
        FROM information_schema.tables 
        WHERE table_schema = $1 
        AND table_type = 'BASE TABLE'
        ORDER BY table_name
      `, [schema]);

      for (const tableRow of tablesResult.rows) {
        const tableName = tableRow.table_name;
        const fullTableName = `${schema}.${tableName}`;

        writeLine(`\n-- Table: ${fullTableName}`);

        // Get table structure
        const columnsResult = await client.query(`
          SELECT 
            column_name,
            data_type,
            character_maximum_length,
            is_nullable,
            column_default
          FROM information_schema.columns
          WHERE table_schema = $1 AND table_name = $2
          ORDER BY ordinal_position
        `, [schema, tableName]);

        // Create table statement
        const columnDefs = columnsResult.rows.map(col => {
          let def = `  ${col.column_name} ${col.data_type}`;
          if (col.character_maximum_length) {
            def += `(${col.character_maximum_length})`;
          }
          if (col.column_default) {
            def += ` DEFAULT ${col.column_default}`;
          }
          if (col.is_nullable === 'NO') {
            def += ' NOT NULL';
          }
          return def;
        }).join(',\n');

        writeLine(`CREATE TABLE IF NOT EXISTS ${fullTableName} (\n${columnDefs}\n);`);

        // Add primary key constraints
        const pkResult = await client.query(`
          SELECT a.attname
          FROM pg_index i
          JOIN pg_attribute a ON a.attrelid = i.indrelid AND a.attnum = ANY(i.indkey)
          WHERE i.indrelid = $1::regclass AND i.indisprimary
        `, [fullTableName]);

        if (pkResult.rows.length > 0) {
          const pkColumns = pkResult.rows.map(r => r.attname).join(', ');
          writeLine(`ALTER TABLE ${fullTableName} ADD PRIMARY KEY (${pkColumns});`);
        }

        // Add indexes
        const indexesResult = await client.query(`
          SELECT indexname, indexdef
          FROM pg_indexes
          WHERE schemaname = $1 AND tablename = $2
          AND indexname NOT LIKE '%_pkey'
        `, [schema, tableName]);

        for (const idx of indexesResult.rows) {
          writeLine(`${idx.indexdef};`);
        }

        // Collect foreign keys for later (aggregate by constraint_name)
        const fkResult = await client.query(`
          SELECT
            tc.constraint_name,
            array_agg(kcu.column_name ORDER BY kcu.ordinal_position) as columns,
            ccu.table_schema AS foreign_table_schema,
            ccu.table_name AS foreign_table_name,
            array_agg(ccu.column_name ORDER BY kcu.ordinal_position) as foreign_columns,
            rc.update_rule,
            rc.delete_rule
          FROM information_schema.table_constraints AS tc
          JOIN information_schema.key_column_usage AS kcu
            ON tc.constraint_name = kcu.constraint_name
            AND tc.table_schema = kcu.table_schema
          JOIN information_schema.constraint_column_usage AS ccu
            ON ccu.constraint_name = tc.constraint_name
            AND ccu.table_schema = tc.table_schema
          JOIN information_schema.referential_constraints AS rc
            ON rc.constraint_name = tc.constraint_name
          WHERE tc.constraint_type = 'FOREIGN KEY'
            AND tc.table_schema = $1
            AND tc.table_name = $2
          GROUP BY tc.constraint_name, ccu.table_schema, ccu.table_name, rc.update_rule, rc.delete_rule
        `, [schema, tableName]);

        for (const fk of fkResult.rows) {
          // Ensure columns are arrays (pg driver may return string or array)
          const columns = Array.isArray(fk.columns) ? fk.columns : [fk.columns];
          const refColumns = Array.isArray(fk.foreign_columns) ? fk.foreign_columns : [fk.foreign_columns];
          
          allForeignKeys.push({
            schema,
            tableName,
            constraintName: fk.constraint_name,
            columns,
            refSchema: fk.foreign_table_schema,
            refTable: fk.foreign_table_name,
            refColumns,
            updateRule: fk.update_rule,
            deleteRule: fk.delete_rule
          });
        }
      }
    }

    // Set sequence ownerships (after tables are created)
    if (allSequenceOwnerships.length > 0) {
      writeLine('\n-- Sequence ownerships');
      for (const own of allSequenceOwnerships) {
        writeLine(`ALTER SEQUENCE ${own.schemaname}.${own.sequencename} OWNED BY ${own.ownedSchema}.${own.ownedTable}.${own.ownedColumn};`);
      }
    }

    // Set sequence values (after ownership)
    if (allSequenceSetvals.length > 0) {
      writeLine('\n-- Sequence values');
      for (const sv of allSequenceSetvals) {
        writeLine(`SELECT setval('${sv.schemaname}.${sv.sequencename}', ${sv.lastValue}, true);`);
      }
    }

    // Insert data for all tables
    writeLine('\n-- Data insertion');
    for (const schemaRow of schemasResult.rows) {
      const schema = schemaRow.schema_name;
      const tablesResult = await client.query(`
        SELECT table_name 
        FROM information_schema.tables 
        WHERE table_schema = $1 
        AND table_type = 'BASE TABLE'
        ORDER BY table_name
      `, [schema]);

      for (const tableRow of tablesResult.rows) {
        const tableName = tableRow.table_name;
        const fullTableName = `${schema}.${tableName}`;

        const dataResult = await client.query(`SELECT * FROM ${fullTableName}`);
        
        if (dataResult.rows.length > 0) {
          writeLine(`\n-- Data for ${fullTableName}`);
          
          for (const row of dataResult.rows) {
            const columns = Object.keys(row).join(', ');
            const values = Object.values(row).map(val => {
              if (val === null) return 'NULL';
              if (typeof val === 'string') return `'${val.replace(/'/g, "''")}'`;
              if (val instanceof Date) return `'${val.toISOString()}'`;
              if (Array.isArray(val)) return `ARRAY[${val.map(v => typeof v === 'string' ? `'${v.replace(/'/g, "''")}'` : v).join(', ')}]`;
              if (typeof val === 'object') return `'${JSON.stringify(val).replace(/'/g, "''")}'`;
              return val;
            }).join(', ');
            
            writeLine(`INSERT INTO ${fullTableName} (${columns}) VALUES (${values});`);
          }
        }
      }
    }

    // Add foreign keys after data insertion
    writeLine('\n-- Foreign key constraints');
    for (const fk of allForeignKeys) {
      const fullTableName = `${fk.schema}.${fk.tableName}`;
      const columnsStr = fk.columns.join(', ');
      const refColumnsStr = fk.refColumns.join(', ');
      writeLine(`ALTER TABLE ${fullTableName} ADD CONSTRAINT ${fk.constraintName} FOREIGN KEY (${columnsStr}) REFERENCES ${fk.refSchema}.${fk.refTable}(${refColumnsStr}) ON UPDATE ${fk.updateRule} ON DELETE ${fk.deleteRule};`);
    }

    writeLine('');
    writeLine('COMMIT;');
    writeLine('');
    writeLine(`-- Backup completed: ${new Date().toISOString()}`);

    // Close the write stream
    await new Promise<void>((resolve, reject) => {
      writeStream.end(() => resolve());
      writeStream.on('error', reject);
    });

    // Upload to SFTP if configured
    await uploadBackupToSFTP(filepath);

    return filepath;
  } catch (error) {
    writeLine('');
    writeLine('ROLLBACK;');
    // Close the write stream on error
    await new Promise<void>((resolve) => {
      writeStream.end(() => resolve());
    });
    throw error;
  } finally {
    client.release();
  }
}

/**
 * Create ZIP archive of project files
 */
export async function createFilesBackup(excludePatterns: string[] = []): Promise<string> {
  const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
  const filename = `files-backup-${timestamp}.zip`;
  const filepath = path.join(BACKUP_DIR, filename);

  await ensureBackupDir();

  console.log(`🔄 [BACKUP-ZIP] Starting archive creation: ${filename}`);

  try {
    const output = createWriteStream(filepath);
    const archive = archiver('zip', {
      zlib: { level: 9 } // Maximum compression
    });

    // Capture errors from stream event listeners
    let archiveError: Error | null = null;
    let outputError: Error | null = null;

    archive.on('error', (err) => {
      console.error(`❌ [BACKUP-ZIP] Archive error:`, err);
      archiveError = err;
    });

    archive.on('warning', (warn) => {
      console.warn(`⚠️  [BACKUP-ZIP] Archive warning:`, warn);
    });

    archive.on('finish', () => {
      console.log(`✅ [BACKUP-ZIP] Archive finalized, waiting for stream close...`);
    });

    output.on('error', (err) => {
      console.error(`❌ [BACKUP-ZIP] Output stream error:`, err);
      outputError = err;
    });

    // Pipe archive to output stream
    archive.pipe(output);

    // Default exclude patterns
    const defaultExcludes = [
      'node_modules/**',
      '.git/**',
      'backups/**',
      '.replit',
      'replit.nix',
      '.cache/**',
      '.upm/**',
      'uploads/**', // Usually too large
      'dist/**',
      'build/**',
      '*.log',
      ...excludePatterns
    ];

    // Add project files
    archive.glob('**/*', {
      cwd: process.cwd(),
      ignore: defaultExcludes,
      dot: true
    });

    // Finalize the archive (this will trigger stream closure)
    await archive.finalize();

    // Wait for output stream to actually finish writing
    await finished(output);
    
    // Check for errors captured during stream processing
    if (archiveError) throw archiveError;
    if (outputError) throw outputError;
    
    console.log(`✅ [BACKUP-ZIP] Stream closed: ${archive.pointer()} bytes written`);
    
    // Upload to SFTP if configured
    try {
      await uploadBackupToSFTP(filepath);
      console.log(`✅ [BACKUP-ZIP] SFTP upload completed`);
    } catch (sftpError) {
      console.error(`⚠️  [BACKUP-ZIP] SFTP upload failed:`, sftpError);
      // Don't throw - local file is still valid
    }

    return filepath;
  } catch (error) {
    console.error(`❌ [BACKUP-ZIP] Fatal error during backup:`, error);
    throw error;
  }
}

/**
 * Create full backup (database + files)
 */
export async function createFullBackup(): Promise<{database: string, files: string}> {
  console.log('🔄 Creating full backup...');
  
  const [databasePath, filesPath] = await Promise.all([
    exportDatabaseToSQL(),
    createFilesBackup()
  ]);

  console.log('✅ Full backup completed');
  
  return {
    database: databasePath,
    files: filesPath
  };
}

/**
 * List all backups
 */
export async function listBackups(): Promise<BackupMetadata[]> {
  await ensureBackupDir();
  
  const files = await fs.readdir(BACKUP_DIR);
  const backups: BackupMetadata[] = [];

  for (const file of files) {
    const filepath = path.join(BACKUP_DIR, file);
    const stats = await fs.stat(filepath);
    
    if (stats.isFile()) {
      let type: 'database' | 'files' | 'full' = 'database';
      if (file.startsWith('database-')) type = 'database';
      else if (file.startsWith('files-')) type = 'files';
      
      backups.push({
        id: file,
        type,
        timestamp: stats.mtime.toISOString(),
        filename: file,
        size: stats.size,
        status: 'completed'
      });
    }
  }

  return backups.sort((a, b) => 
    new Date(b.timestamp).getTime() - new Date(a.timestamp).getTime()
  );
}

/**
 * Delete old backups (keep last N backups)
 */
export async function cleanupOldBackups(keepCount: number = 10): Promise<number> {
  const backups = await listBackups();
  
  if (backups.length <= keepCount) {
    return 0;
  }

  const toDelete = backups.slice(keepCount);
  let deletedCount = 0;

  for (const backup of toDelete) {
    try {
      await fs.unlink(path.join(BACKUP_DIR, backup.filename));
      deletedCount++;
    } catch (error) {
      console.error(`Error deleting backup ${backup.filename}:`, error);
    }
  }

  return deletedCount;
}

/**
 * Get backup file path for download (with path traversal protection)
 */
export function getBackupPath(filename: string): string {
  // Security: prevent path traversal attacks
  const normalized = path.normalize(filename).replace(/^(\.\.(\/|\\|$))+/, '');
  const fullPath = path.join(BACKUP_DIR, normalized);
  
  // Ensure the resolved path is still within BACKUP_DIR
  const resolvedPath = path.resolve(fullPath);
  const resolvedBackupDir = path.resolve(BACKUP_DIR);
  
  if (!resolvedPath.startsWith(resolvedBackupDir)) {
    throw new Error('Invalid backup path: path traversal detected');
  }
  
  return fullPath;
}

/**
 * Upload backup file to SFTP
 */
export async function uploadBackupToSFTP(localFilePath: string): Promise<string | null> {
  try {
    // Check if SFTP is configured
    const result = await pool.query(
      'SELECT * FROM file_storage_settings WHERE provider = $1 AND is_active = true LIMIT 1',
      ['sftp']
    );

    if (result.rows.length === 0) {
      console.log('ℹ️  SFTP not configured - backup stored locally only');
      return null;
    }

    const config = result.rows[0];
    const password = process.env.FILE_STORAGE_PASSWORD || '';

    if (!config.host || !config.username || !password) {
      console.log('⚠️  SFTP configuration incomplete - backup stored locally only');
      return null;
    }

    const filename = path.basename(localFilePath);
    const remotePath = `backups/${filename}`;
    const baseUrl = config.base_url || 'https://files.alpsys.pl';
    const pathPrefix = (config.path_prefix || 'OMS').replace(/^\/+/, '');

    console.log(`📤 Uploading backup to SFTP: ${filename}`);

    const sftp = new SftpClient();
    await sftp.connect({
      host: config.host,
      port: config.port || 22,
      username: config.username,
      password: password,
    });

    try {
      // Ensure backups directory exists
      await sftp.mkdir('backups', true);

      // Upload file
      await sftp.put(localFilePath, remotePath);

      // Build public URL
      const publicUrl = `${baseUrl}/${pathPrefix}/${remotePath}`.replace(/([^:]\/)\/+/g, '$1');
      console.log(`✅ Backup uploaded to SFTP: ${publicUrl}`);
      
      return publicUrl;
    } finally {
      await sftp.end();
    }
  } catch (error) {
    console.error('❌ Error uploading backup to SFTP:', error);
    return null;
  }
}

/**
 * List backups from SFTP
 */
export async function listBackupsFromSFTP(): Promise<BackupMetadata[]> {
  try {
    // Check if SFTP is configured
    const result = await pool.query(
      'SELECT * FROM file_storage_settings WHERE provider = $1 AND is_active = true LIMIT 1',
      ['sftp']
    );

    if (result.rows.length === 0) {
      return [];
    }

    const config = result.rows[0];
    const password = process.env.FILE_STORAGE_PASSWORD || '';

    if (!config.host || !config.username || !password) {
      return [];
    }

    const sftp = new SftpClient();
    await sftp.connect({
      host: config.host,
      port: config.port || 22,
      username: config.username,
      password: password,
    });

    try {
      // List files in backups directory
      const files = await sftp.list('backups');
      const backups: BackupMetadata[] = [];

      for (const file of files) {
        if (file.type === '-') { // Regular file
          let type: 'database' | 'files' | 'full' = 'database';
          if (file.name.startsWith('database-')) type = 'database';
          else if (file.name.startsWith('files-')) type = 'files';

          backups.push({
            id: file.name,
            type,
            timestamp: new Date(file.modifyTime).toISOString(),
            filename: file.name,
            size: file.size,
            status: 'completed'
          });
        }
      }

      return backups.sort((a, b) => 
        new Date(b.timestamp).getTime() - new Date(a.timestamp).getTime()
      );
    } finally {
      await sftp.end();
    }
  } catch (error) {
    console.error('Error listing backups from SFTP:', error);
    return [];
  }
}

/**
 * Download backup from SFTP to local stream
 */
export async function downloadBackupFromSFTP(filename: string): Promise<Buffer | null> {
  try {
    // Check if SFTP is configured
    const result = await pool.query(
      'SELECT * FROM file_storage_settings WHERE provider = $1 AND is_active = true LIMIT 1',
      ['sftp']
    );

    if (result.rows.length === 0) {
      return null;
    }

    const config = result.rows[0];
    const password = process.env.FILE_STORAGE_PASSWORD || '';

    if (!config.host || !config.username || !password) {
      return null;
    }

    const remotePath = `backups/${filename}`;

    const sftp = new SftpClient();
    await sftp.connect({
      host: config.host,
      port: config.port || 22,
      username: config.username,
      password: password,
    });

    try {
      const buffer = await sftp.get(remotePath);
      return buffer as Buffer;
    } finally {
      await sftp.end();
    }
  } catch (error) {
    console.error('Error downloading backup from SFTP:', error);
    return null;
  }
}

/**
 * List all backups (local + SFTP, merged and deduplicated)
 */
export async function listAllBackups(): Promise<BackupMetadata[]> {
  const [localBackups, sftpBackups] = await Promise.all([
    listBackups(),
    listBackupsFromSFTP()
  ]);

  // Merge and deduplicate by filename
  const backupsMap = new Map<string, BackupMetadata>();
  
  // Add local backups
  for (const backup of localBackups) {
    backupsMap.set(backup.filename, { ...backup, id: `local:${backup.filename}` });
  }
  
  // Add SFTP backups (or update if newer)
  for (const backup of sftpBackups) {
    const existing = backupsMap.get(backup.filename);
    if (!existing || new Date(backup.timestamp) > new Date(existing.timestamp)) {
      backupsMap.set(backup.filename, { ...backup, id: `sftp:${backup.filename}` });
    }
  }

  return Array.from(backupsMap.values()).sort((a, b) => 
    new Date(b.timestamp).getTime() - new Date(a.timestamp).getTime()
  );
}
