import OpenAI from 'openai';
import { pool } from './postgres';

const openai = new OpenAI({
  baseURL: process.env.AI_INTEGRATIONS_OPENAI_BASE_URL,
  apiKey: process.env.AI_INTEGRATIONS_OPENAI_API_KEY,
});

interface AISettings {
  systemInstructions: string;
  productMaterialInfo: string;
  edgeBandInfo: string;
  includeProductTable: boolean;
  productTableHeader: string;
  topBannerImageId: number | null;
  bottomBannerImageId: number | null;
  customInstructions: string;
  promptIntro: string;
  promptFeatures: string;
  promptSafety: string;
  promptCare: string;
  promptWarranty: string;
}

export interface ProductData {
  sku?: string;
  title?: string;
  shortDescription?: string;
  longDescriptionHtml?: string;
  price?: number;
  dimensions?: { length?: number; width?: number; height?: number };
  color?: string;
  material?: string;
  images?: string[]; // Array of image URLs
  hasWarranty?: boolean; // Whether product has warranty (default: true)
  addons?: Array<{ name: string; price?: number; addonType?: string }>;
}

export interface GenerateDescriptionOptions {
  productData: ProductData;
  templateHtml?: string;
  tone?: 'professional' | 'friendly' | 'technical';
  language?: 'pl' | 'en';
  maxLength?: number;
  accessories?: {
    groups: Array<{
      groupCode: string;
      groupName: string;
      description?: string;
      items: Array<{
        name: string;
        code?: string;
        description?: string;
      }>;
    }>;
  };
  customPrompts?: {
    intro?: { enabled?: boolean; prompt?: string };
    features?: { enabled?: boolean; prompt?: string };
    safety?: { enabled?: boolean; prompt?: string };
    care?: { enabled?: boolean; prompt?: string };
    warranty?: { enabled?: boolean; prompt?: string };
  };
}

export interface GenerationResult {
  description: string;
  promptTokens: number;
  completionTokens: number;
  totalTokens: number;
  model: string;
  cost: number;
}

const PRICING = {
  'gpt-4o': { prompt: 0.0025, completion: 0.01 },
  'gpt-4o-mini': { prompt: 0.00015, completion: 0.0006 },
  'gpt-3.5-turbo': { prompt: 0.0005, completion: 0.0015 },
};

function calculateCost(model: string, promptTokens: number, completionTokens: number): number {
  const pricing = PRICING[model as keyof typeof PRICING] || PRICING['gpt-4o-mini'];
  const promptCost = (promptTokens / 1000) * pricing.prompt;
  const completionCost = (completionTokens / 1000) * pricing.completion;
  return promptCost + completionCost;
}

async function getAISettings(): Promise<AISettings> {
  try {
    const result = await pool.query(`
      SELECT 
        system_instructions AS "systemInstructions",
        product_material_info AS "productMaterialInfo",
        edge_band_info AS "edgeBandInfo",
        include_product_table AS "includeProductTable",
        product_table_header AS "productTableHeader",
        top_banner_image_id AS "topBannerImageId",
        bottom_banner_image_id AS "bottomBannerImageId",
        custom_instructions AS "customInstructions",
        prompt_intro AS "promptIntro",
        prompt_features AS "promptFeatures",
        prompt_safety AS "promptSafety",
        prompt_care AS "promptCare",
        prompt_warranty AS "promptWarranty"
      FROM ai_prompt_settings
      LIMIT 1
    `);
    
    if (result.rows.length === 0) {
      // Return defaults if no settings exist
      return {
        systemInstructions: 'Jesteś ekspertem w tworzeniu profesjonalnych opisów produktów meblarskich.',
        productMaterialInfo: 'płyta meblowa 18mm renomowanych polskich firm',
        edgeBandInfo: 'obrzeże ABS 0.8mm',
        includeProductTable: true,
        productTableHeader: 'Wymiary produktu',
        topBannerImageId: null,
        bottomBannerImageId: null,
        customInstructions: 'Pamiętaj: DŁUGOŚĆ to wymiar poziomy, SZEROKOŚĆ to wymiar głębokości, WYSOKOŚĆ to wymiar pionowy.',
        promptIntro: '',
        promptFeatures: '',
        promptSafety: '',
        promptCare: '',
        promptWarranty: '',
      };
    }
    
    return result.rows[0];
  } catch (error) {
    console.error('Error fetching AI settings:', error);
    // Return defaults on error
    return {
      systemInstructions: 'Jesteś ekspertem w tworzeniu profesjonalnych opisów produktów meblarskich.',
      productMaterialInfo: 'płyta meblowa 18mm renomowanych polskich firm',
      edgeBandInfo: 'obrzeże ABS 0.8mm',
      includeProductTable: true,
      productTableHeader: 'Wymiary produktu',
      topBannerImageId: null,
      bottomBannerImageId: null,
      customInstructions: 'Pamiętaj: DŁUGOŚĆ to wymiar poziomy, SZEROKOŚĆ to wymiar głębokości, WYSOKOŚĆ to wymiar pionowy.',
      promptIntro: '',
      promptFeatures: '',
      promptSafety: '',
      promptCare: '',
      promptWarranty: '',
    };
  }
}

export async function generateProductDescription(
  options: GenerateDescriptionOptions
): Promise<GenerationResult> {
  const {
    productData,
    templateHtml = '',
    tone = 'professional',
    language = 'pl',
    maxLength = 500,
    accessories,
    customPrompts,
  } = options;

  // Fetch AI settings from database
  const settings = await getAISettings();

  const prompt = buildPrompt(productData, templateHtml, tone, language, maxLength, settings, accessories, customPrompts);

  // Build system prompt with custom instructions
  let systemContent = settings.systemInstructions || 'Jesteś ekspertem od pisania opisów produktów dla e-commerce.';
  systemContent += `\n\nTwoje opisy są atrakcyjne, przekonujące i optymalizowane pod kątem SEO.`;
  systemContent += `\nZawsze piszesz w języku ${language === 'pl' ? 'polskim' : 'angielskim'}.`;
  systemContent += `\nTon ${tone === 'professional' ? 'profesjonalny i merytoryczny' : tone === 'friendly' ? 'przyjazny i ciepły' : 'techniczny i szczegółowy'}.`;
  
  if (settings.customInstructions) {
    systemContent += `\n\n${settings.customInstructions}`;
  }

  const response = await openai.chat.completions.create({
    model: 'gpt-4o-mini',
    messages: [
      {
        role: 'system',
        content: systemContent,
      },
      {
        role: 'user',
        content: prompt,
      },
    ],
    temperature: 0.7,
    max_tokens: 1000,
  });

  let description = response.choices[0]?.message?.content || '';
  
  // Remove markdown code blocks (```html and ```)
  description = description.replace(/```html\s*/g, '').replace(/```\s*/g, '').trim();
  
  // Add product dimensions table if enabled
  if (settings.includeProductTable && productData.dimensions) {
    const dimensionsTable = buildDimensionsTable(productData.dimensions, settings.productTableHeader);
    if (dimensionsTable) {
      description = description + '\n\n' + dimensionsTable;
    }
  }
  
  // Add banners if configured
  description = await addBanners(description, settings);
  
  const usage = response.usage!;
  const model = response.model;

  const cost = calculateCost(model, usage.prompt_tokens, usage.completion_tokens);

  return {
    description,
    promptTokens: usage.prompt_tokens,
    completionTokens: usage.completion_tokens,
    totalTokens: usage.total_tokens,
    model,
    cost,
  };
}

function buildPrompt(
  productData: ProductData,
  templateHtml: string,
  tone: string,
  language: string,
  maxLength: number,
  settings: AISettings,
  accessories?: {
    groups: Array<{
      groupCode: string;
      groupName: string;
      description?: string;
      items: Array<{
        name: string;
        code?: string;
        description?: string;
      }>;
    }>;
  },
  customPrompts?: {
    intro?: { enabled?: boolean; prompt?: string };
    features?: { enabled?: boolean; prompt?: string };
    safety?: { enabled?: boolean; prompt?: string };
    care?: { enabled?: boolean; prompt?: string };
    warranty?: { enabled?: boolean; prompt?: string };
  }
): string {
  let prompt = `Wygeneruj profesjonalny opis produktu w formacie HTML.\n\n`;
  prompt += `WAŻNE: NIE generuj tabeli wymiarów produktu - zostanie ona dodana automatycznie!\n`;
  prompt += `WAŻNE: Zwróć tylko czysty kod HTML bez znaczników \`\`\`html i \`\`\`\n\n`;

  prompt += `**Dane produktu:**\n`;
  if (productData.sku) prompt += `- SKU: ${productData.sku}\n`;
  if (productData.title) prompt += `- Nazwa: ${productData.title}\n`;
  if (productData.shortDescription) prompt += `- Krótki opis: ${productData.shortDescription}\n`;
  if (productData.price) prompt += `- Cena: ${productData.price} PLN\n`;
  if (productData.color) prompt += `- Kolor: ${productData.color}\n`;
  
  // Add material and edge band info from settings
  if (settings.productMaterialInfo) {
    prompt += `- Materiał: ${settings.productMaterialInfo}\n`;
  } else if (productData.material) {
    prompt += `- Materiał: ${productData.material}\n`;
  }
  
  if (settings.edgeBandInfo) {
    prompt += `- Obrzeże: ${settings.edgeBandInfo}\n`;
  }
  
  if (productData.dimensions) {
    const { length, width, height } = productData.dimensions;
    if (length || width || height) {
      // Convert mm to cm for display
      const lengthCm = length ? (length / 10).toFixed(1) : '?';
      const widthCm = width ? (width / 10).toFixed(1) : '?';
      const heightCm = height ? (height / 10).toFixed(1) : '?';
      prompt += `- Wymiary: Długość ${lengthCm}cm × Szerokość ${widthCm}cm × Wysokość ${heightCm}cm\n`;
    }
  }

  if (productData.images && productData.images.length > 0) {
    prompt += `\n**Dostępne zdjęcia produktu:**\n`;
    prompt += `Produkt posiada ${productData.images.length} zdjęć. W szablonie możesz użyć tagów:\n`;
    productData.images.forEach((_, idx) => {
      prompt += `{img${idx + 1} w=100} - zdjęcie ${idx + 1} (dostępne szerokości: 100, 75, 50, 25)\n`;
    });
    prompt += `\n`;
  }

  if (productData.addons && productData.addons.length > 0) {
    prompt += `\n**Dodatki/Komponenty:**\n`;
    productData.addons.forEach((addon, idx) => {
      prompt += `${idx + 1}. ${addon.name}`;
      if (addon.price) prompt += ` (${addon.price} PLN)`;
      if (addon.addonType) prompt += ` [${addon.addonType}]`;
      prompt += `\n`;
    });
  }

  // Add accessories information (hardware, fabrics, fittings)
  if (accessories && accessories.groups && accessories.groups.length > 0) {
    prompt += `\n**Dostępne akcesoria i okucia:**\n`;
    accessories.groups.forEach((group) => {
      prompt += `\n**${group.groupName}** (kod: ${group.groupCode}):\n`;
      if (group.description) {
        prompt += `Opis grupy: ${group.description}\n`;
      }
      if (group.items && group.items.length > 0) {
        prompt += `Dostępne opcje:\n`;
        group.items.forEach((item) => {
          prompt += `  - ${item.name}`;
          if (item.code) prompt += ` (${item.code})`;
          if (item.description) prompt += `: ${item.description}`;
          prompt += `\n`;
        });
      }
    });
    prompt += `\n`;
  }

  // Use AI prompts for content sections - PRIORITY: customPrompts > settings
  // Merge custom prompts with global settings (custom prompts override)
  const effectivePrompts = {
    intro: (customPrompts?.intro?.enabled && customPrompts?.intro?.prompt) 
      ? customPrompts.intro.prompt 
      : settings.promptIntro,
    features: (customPrompts?.features?.enabled && customPrompts?.features?.prompt) 
      ? customPrompts.features.prompt 
      : settings.promptFeatures,
    safety: (customPrompts?.safety?.enabled && customPrompts?.safety?.prompt) 
      ? customPrompts.safety.prompt 
      : settings.promptSafety,
    care: (customPrompts?.care?.enabled && customPrompts?.care?.prompt) 
      ? customPrompts.care.prompt 
      : settings.promptCare,
    warranty: (customPrompts?.warranty?.enabled && customPrompts?.warranty?.prompt) 
      ? customPrompts.warranty.prompt 
      : settings.promptWarranty,
  };
  
  const hasPrompts = effectivePrompts.intro || effectivePrompts.features || effectivePrompts.safety || effectivePrompts.care || effectivePrompts.warranty;
  
  if (hasPrompts) {
    prompt += `\n**STRUKTURA OPISU - generuj następujące sekcje:**\n\n`;
    
    if (effectivePrompts.intro && effectivePrompts.intro.trim()) {
      prompt += `1. **Wprowadzenie produktu ({{ai.intro}}):**\n`;
      prompt += `   Bazując na danych produktu, wygeneruj wprowadzenie zgodnie z wytycznymi:\n`;
      prompt += `   ${effectivePrompts.intro}\n\n`;
    }
    
    if (effectivePrompts.features && effectivePrompts.features.trim()) {
      prompt += `2. **Cechy i zalety ({{ai.features}}):**\n`;
      prompt += `   Opisz kluczowe cechy i zalety produktu:\n`;
      prompt += `   ${effectivePrompts.features}\n`;
      // Add accessories instruction if available
      if (accessories && accessories.groups && accessories.groups.length > 0) {
        prompt += `   \n   **KRYTYCZNIE WAŻNE - AKCESORIA I OKUCIA:**\n`;
        prompt += `   W tej sekcji BEZWZGLĘDNIE MUSISZ wypunktować KAŻDE POJEDYNCZE AKCESORIUM z KAŻDEJ grupy wymienionej w sekcji "Dostępne akcesoria i okucia" powyżej.\n`;
        prompt += `   \n`;
        prompt += `   **JAK TO ZROBIĆ:**\n`;
        prompt += `   1. Przejrzyj KAŻDĄ grupę akcesoriów (np. "VB", "Tkanina Hugo", "Eko skóra Cayenne Alpma")\n`;
        prompt += `   2. Z KAŻDEJ grupy wypunktuj WSZYSTKIE pozycje z listy "Dostępne opcje:"\n`;
        prompt += `   3. NIE wymieniaj samej nazwy grupy - wymień konkretne elementy z tej grupy\n`;
        prompt += `   \n`;
        prompt += `   **PRZYKŁAD POPRAWNEGO FORMATOWANIA:**\n`;
        prompt += `   Jeśli w danych masz grupę "VB" z opcjami: "Siłowniki gazowe", "Słupki metalowe", "Zawiasy z hamulcem", to MUSISZ utworzyć:\n`;
        prompt += `   - **Siłowniki gazowe**: [opis funkcji i zalet]\n`;
        prompt += `   - **Słupki metalowe**: [opis funkcji i zalet]\n`;
        prompt += `   - **Zawiasy z hamulcem**: [opis funkcji i zalet]\n`;
        prompt += `   \n`;
        prompt += `   **NIE rób tak:** "VB: Zestaw akcesoriów meblowych"\n`;
        prompt += `   **TAK rób:** Wypunktuj każdy element osobno jak w przykładzie powyżej.\n`;
      }
      prompt += `\n`;
    }
    
    if (effectivePrompts.safety && effectivePrompts.safety.trim()) {
      prompt += `3. **Bezpieczeństwo użytkowania ({{ai.safety}}):**\n`;
      prompt += `   Wygeneruj informacje o bezpieczeństwie:\n`;
      prompt += `   ${effectivePrompts.safety}\n\n`;
    }
    
    if (effectivePrompts.care && effectivePrompts.care.trim()) {
      prompt += `4. **Pielęgnacja i konserwacja ({{ai.care}}):**\n`;
      prompt += `   Opisz jak dbać o produkt:\n`;
      prompt += `   ${effectivePrompts.care}\n\n`;
    }
    
    // Only include warranty section if product has warranty enabled
    if (effectivePrompts.warranty && effectivePrompts.warranty.trim() && productData.hasWarranty !== false) {
      prompt += `5. **Gwarancja i warunki ({{ai.warranty}}):**\n`;
      prompt += `   Przedstaw informacje o gwarancji:\n`;
      prompt += `   ${effectivePrompts.warranty}\n\n`;
    }
    
    prompt += `\nWygeneruj HTML z powyższymi sekcjami. Każda sekcja powinna być w osobnym <div> lub akapicie.\n`;
    prompt += `Użyj odpowiednich nagłówków (<h2>, <h3>) i formatowania HTML.\n`;
  } else if (templateHtml && templateHtml.trim()) {
    prompt += `\n**Szablon do wykorzystania:**\n${templateHtml}\n`;
    prompt += `\nZmodyfikuj powyższy szablon, uzupełniając go danymi produktu. Zachowaj strukturę HTML.\n`;
  } else {
    prompt += buildFallbackPrompt(templateHtml, maxLength);
  }

  prompt += `\n**Zwróć tylko kod HTML bez dodatkowych komentarzy.**`;

  return prompt;
}

function buildFallbackPrompt(templateHtml: string, maxLength: number): string {
  let fallback = `\n**Wymagania:**\n`;
  fallback += `- Wygeneruj atrakcyjny opis produktu w formacie HTML\n`;
  fallback += `- Użyj tagów <h2>, <p>, <ul>, <li> do strukturyzacji\n`;
  fallback += `- Podkreśl zalety i unikalne cechy produktu\n`;
  fallback += `- Długość: około ${maxLength} słów\n`;
  return fallback;
}

function buildDimensionsTable(
  dimensions: { length?: number; width?: number; height?: number },
  tableHeader: string
): string | null {
  const { length, width, height } = dimensions;
  
  if (!length && !width && !height) {
    return null;
  }
  
  // Convert mm to cm by dividing by 10
  const lengthCm = length ? (length / 10).toFixed(1) : null;
  const widthCm = width ? (width / 10).toFixed(1) : null;
  const heightCm = height ? (height / 10).toFixed(1) : null;
  
  return `
<table style="width: 100%; border-collapse: collapse; margin: 20px 0;">
  <thead>
    <tr>
      <th colspan="2" style="background-color: #f5f5f5; padding: 10px; text-align: left; border: 1px solid #ddd;">
        ${tableHeader}
      </th>
    </tr>
  </thead>
  <tbody>
    ${lengthCm ? `<tr><td style="padding: 8px; border: 1px solid #ddd; width: 50%;">Długość</td><td style="padding: 8px; border: 1px solid #ddd;">${lengthCm} cm</td></tr>` : ''}
    ${widthCm ? `<tr><td style="padding: 8px; border: 1px solid #ddd; width: 50%;">Szerokość</td><td style="padding: 8px; border: 1px solid #ddd;">${widthCm} cm</td></tr>` : ''}
    ${heightCm ? `<tr><td style="padding: 8px; border: 1px solid #ddd; width: 50%;">Wysokość</td><td style="padding: 8px; border: 1px solid #ddd;">${heightCm} cm</td></tr>` : ''}
  </tbody>
</table>
  `.trim();
}

async function addBanners(description: string, settings: AISettings): Promise<string> {
  let result = description;
  
  // Fetch banner images from database if IDs are provided
  if (settings.topBannerImageId || settings.bottomBannerImageId) {
    try {
      const imageIds = [settings.topBannerImageId, settings.bottomBannerImageId].filter(id => id !== null);
      
      if (imageIds.length > 0) {
        const imageResult = await pool.query(`
          SELECT id, url, alt_text AS "altText"
          FROM catalog.product_images
          WHERE id = ANY($1)
        `, [imageIds]);
        
        if (imageResult.rows.length === 0) {
          console.warn('No banner images found for selected IDs:', imageIds);
          return result;
        }
        
        const imagesMap = new Map(imageResult.rows.map((img: any) => [img.id, img]));
        
        // Add top banner (floated right, 50% width, text wraps around it)
        if (settings.topBannerImageId && imagesMap.has(settings.topBannerImageId)) {
          const topImage = imagesMap.get(settings.topBannerImageId);
          const topBannerHtml = `<img src="${topImage.url}" alt="${topImage.altText || 'Top Banner'}" style="float: right; width: 50%; height: auto; margin: 0 0 20px 20px;" />`;
          result = topBannerHtml + '\n\n' + result;
        } else if (settings.topBannerImageId) {
          console.warn('Top banner image not found:', settings.topBannerImageId);
        }
        
        // Add bottom banner (floated left, 50% width, text wraps around it)
        if (settings.bottomBannerImageId && imagesMap.has(settings.bottomBannerImageId)) {
          const bottomImage = imagesMap.get(settings.bottomBannerImageId);
          const bottomBannerHtml = `<img src="${bottomImage.url}" alt="${bottomImage.altText || 'Bottom Banner'}" style="float: left; width: 50%; height: auto; margin: 0 20px 20px 0;" />`;
          result = result + '\n\n' + bottomBannerHtml;
        } else if (settings.bottomBannerImageId) {
          console.warn('Bottom banner image not found:', settings.bottomBannerImageId);
        }
      }
    } catch (error) {
      console.error('Error adding banners:', error);
      // Continue without banners on error
    }
  }
  
  return result;
}

/**
 * Convert HTML to Tiptap JSON document format
 * This is a simplified conversion that wraps HTML in a code block
 * The Tiptap editor can then render/edit the HTML properly
 */
export function htmlToTiptapJson(html: string): any {
  if (!html || !html.trim()) {
    return null;
  }

  // Parse HTML into basic Tiptap structure
  // For now, we'll split by common block elements and create paragraphs
  const lines = html
    .replace(/<\/p>/gi, '\n')
    .replace(/<br\s*\/?>/gi, '\n')
    .replace(/<\/h[1-6]>/gi, '\n')
    .replace(/<[^>]+>/g, '') // Remove all HTML tags
    .split('\n')
    .map(line => line.trim())
    .filter(line => line.length > 0);

  if (lines.length === 0) {
    return {
      type: 'doc',
      content: [
        {
          type: 'paragraph',
          content: []
        }
      ]
    };
  }

  // Create Tiptap document with paragraphs
  const content = lines.map(line => ({
    type: 'paragraph',
    content: [
      {
        type: 'text',
        text: line
      }
    ]
  }));

  return {
    type: 'doc',
    content
  };
}
