import { useQuery } from "@tanstack/react-query";
import { useAuth } from "@/hooks/use-auth";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { ArrowLeft, FileText, CheckCircle, XCircle, AlertCircle } from "lucide-react";
import { useLocation } from "wouter";
import { format } from "date-fns";
import { pl } from "date-fns/locale";

interface WebhookLog {
  id: number;
  webhook_id: number;
  webhook_name: string;
  event_type: string;
  status: "success" | "failed" | "retrying";
  attempt_number: number;
  response_status: number | null;
  response_body: string | null;
  error_message: string | null;
  created_at: string;
}

export default function WebhookLogsPage() {
  const { user } = useAuth();
  const [, setLocation] = useLocation();

  const { data: logs, isLoading } = useQuery<WebhookLog[]>({
    queryKey: ["/api/webhooks/logs"],
    enabled: user?.role === 'admin',
  });

  if (user?.role !== 'admin') {
    return (
      <div className="flex items-center justify-center h-full">
        <Card>
          <CardHeader>
            <CardTitle>Brak dostępu</CardTitle>
            <CardDescription>
              Tylko administratorzy mają dostęp do logów webhooków.
            </CardDescription>
          </CardHeader>
        </Card>
      </div>
    );
  }

  const getStatusBadge = (status: string, responseStatus: number | null) => {
    if (status === "success") {
      return (
        <Badge variant="default" className="gap-1" data-testid="badge-status-success">
          <CheckCircle className="w-3 h-3" />
          Sukces
        </Badge>
      );
    } else if (status === "retrying") {
      return (
        <Badge variant="secondary" className="gap-1" data-testid="badge-status-retrying">
          <AlertCircle className="w-3 h-3" />
          Ponowienie
        </Badge>
      );
    } else {
      return (
        <Badge variant="destructive" className="gap-1" data-testid="badge-status-failed">
          <XCircle className="w-3 h-3" />
          Błąd
        </Badge>
      );
    }
  };

  const getEventBadge = (eventType: string) => {
    const eventLabels: Record<string, string> = {
      "order.created": "Nowe zamówienie",
      "order.updated": "Zaktualizowano",
      "order.paid": "Opłacono",
      "order.shipped": "Wysłano",
      "production.started": "Produkcja rozpoczęta",
      "production.completed": "Produkcja zakończona",
      "production.paused": "Produkcja wstrzymana",
      "production.resumed": "Produkcja wznowiona",
      "inventory.created": "Nowa inwentaryzacja",
      "inventory.finalized": "Inwentaryzacja sfinalizowana",
    };
    return (
      <Badge variant="outline" data-testid={`badge-event-${eventType}`}>
        {eventLabels[eventType] || eventType}
      </Badge>
    );
  };

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Logi Webhooków</h1>
          <p className="text-muted-foreground">Historia dostaw powiadomień webhook</p>
        </div>
        <Button
          variant="outline"
          onClick={() => setLocation("/webhooks")}
          data-testid="button-back"
        >
          <ArrowLeft className="w-4 h-4 mr-2" />
          Powrót do webhooków
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Ostatnie dostawy</CardTitle>
          <CardDescription>
            Audyt wszystkich prób dostarczenia webhooków z informacjami o statusach i błędach
          </CardDescription>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="text-center py-8 text-muted-foreground">Ładowanie...</div>
          ) : logs && logs.length > 0 ? (
            <div className="overflow-x-auto">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Data</TableHead>
                    <TableHead>Webhook</TableHead>
                    <TableHead>Wydarzenie</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead>Próba</TableHead>
                    <TableHead>HTTP</TableHead>
                    <TableHead>Szczegóły</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {logs.map((log) => (
                    <TableRow key={log.id} data-testid={`row-log-${log.id}`}>
                      <TableCell className="text-muted-foreground whitespace-nowrap">
                        {format(new Date(log.created_at), "dd.MM.yyyy HH:mm:ss", { locale: pl })}
                      </TableCell>
                      <TableCell>
                        <span className="text-sm font-medium" data-testid={`text-webhook-${log.id}`}>
                          {log.webhook_name}
                        </span>
                      </TableCell>
                      <TableCell>{getEventBadge(log.event_type)}</TableCell>
                      <TableCell>{getStatusBadge(log.status, log.response_status)}</TableCell>
                      <TableCell>
                        <span className="text-sm text-muted-foreground" data-testid={`text-attempt-${log.id}`}>
                          #{log.attempt_number}
                        </span>
                      </TableCell>
                      <TableCell>
                        {log.response_status ? (
                          <Badge
                            variant={log.response_status >= 200 && log.response_status < 300 ? "default" : "destructive"}
                            data-testid={`badge-http-${log.id}`}
                          >
                            {log.response_status}
                          </Badge>
                        ) : (
                          <span className="text-muted-foreground">—</span>
                        )}
                      </TableCell>
                      <TableCell>
                        {log.error_message ? (
                          <span className="text-sm text-destructive" data-testid={`text-error-${log.id}`}>
                            {log.error_message.substring(0, 50)}
                            {log.error_message.length > 50 ? "..." : ""}
                          </span>
                        ) : log.response_body ? (
                          <span className="text-sm text-muted-foreground" data-testid={`text-response-${log.id}`}>
                            {log.response_body.substring(0, 50)}
                            {log.response_body.length > 50 ? "..." : ""}
                          </span>
                        ) : (
                          <span className="text-muted-foreground">—</span>
                        )}
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          ) : (
            <div className="text-center py-8 text-muted-foreground">
              <FileText className="w-12 h-12 mx-auto mb-4 opacity-50" />
              <p>Brak logów webhooków</p>
              <p className="text-sm">Logi pojawią się po pierwszej próbie dostarczenia webhooka</p>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
