import { useState, useEffect, useMemo, useRef } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Checkbox } from "@/components/ui/checkbox";
import { Badge } from "@/components/ui/badge";
import { 
  Select, 
  SelectContent, 
  SelectItem, 
  SelectTrigger, 
  SelectValue 
} from "@/components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { ScrollArea } from "@/components/ui/scroll-area";
import { 
  FileCheck, 
  Package, 
  MapPin, 
  Building2,
  Truck,
  ArrowLeft,
  Upload,
  FileSpreadsheet,
  AlertTriangle,
  Check,
  X,
  Loader2
} from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";

interface Material {
  id: number;
  name: string;
  internalCode: string;
  quantity: string;
  unitOfMeasure: string;
  categoryName: string;
  groupName?: string;
  locationName?: string;
  carrierName?: string;
  locationId?: number;
  carrierId?: number;
  itemType: 'material' | 'stock_panel' | 'packaging_material';
  dimensions?: string;
  colorCode?: string;
}

interface Location {
  id: number;
  name: string;
  level: number;
  parentId: number | null;
}

interface Carrier {
  id: number;
  name: string;
  type: string;
}

interface CsvRow {
  [key: string]: string;
}

interface CsvMapping {
  nameColumn: string;
  quantityColumn: string;
}

interface ParsedItem {
  originalName: string;
  length: number | null;
  width: number | null;
  colorCode: string | null;
  quantity: number;
  matchedItem: Material | null;
  isNew: boolean;
  matchType?: 'formatka' | 'internalCode' | 'name' | 'none';
}

export default function WarehouseInventoryCountNewPage() {
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const fileInputRef = useRef<HTMLInputElement>(null);

  const [countName, setCountName] = useState(() => {
    const today = new Date();
    const day = String(today.getDate()).padStart(2, '0');
    const month = String(today.getMonth() + 1).padStart(2, '0');
    const year = today.getFullYear();
    return `Spis inwentaryzacyjny ${day}-${month}-${year}`;
  });
  const [isCountNameDirty, setIsCountNameDirty] = useState(false);
  const [notes, setNotes] = useState("");
  const [selectedCategory, setSelectedCategory] = useState<string>("all");
  const [selectedLocationId, setSelectedLocationId] = useState<string>("all");
  const [selectedCarrierId, setSelectedCarrierId] = useState<string>("all");
  const [selectedMaterialIds, setSelectedMaterialIds] = useState<Set<number>>(new Set());

  const [showCsvDialog, setShowCsvDialog] = useState(false);
  const [csvData, setCsvData] = useState<CsvRow[]>([]);
  const [csvHeaders, setCsvHeaders] = useState<string[]>([]);
  const [csvMapping, setCsvMapping] = useState<CsvMapping>({ nameColumn: '', quantityColumn: '' });
  const [parsedItems, setParsedItems] = useState<ParsedItem[]>([]);
  const [showConfirmNewItems, setShowConfirmNewItems] = useState(false);
  const [newItemsToCreate, setNewItemsToCreate] = useState<ParsedItem[]>([]);
  const [isProcessingCsv, setIsProcessingCsv] = useState(false);
  const [csvQuantities, setCsvQuantities] = useState<Map<number, number>>(new Map());

  const categories = [
    { value: "all", label: "Wszystkie kategorie" },
    { value: "tkaniny", label: "Tkaniny" },
    { value: "pianki", label: "Wypełnienia" },
    { value: "plyty", label: "Płyty meblowe" },
    { value: "obrzeza", label: "Obrzeża" },
    { value: "okucia", label: "Okucia" },
    { value: "sruby", label: "Śruby i łączniki" },
    { value: "formatki", label: "Formatki" },
    { value: "opakowania", label: "Opakowania" },
  ];

  const { data: locations = [] } = useQuery<Location[]>({
    queryKey: ["/api/production/locations"],
  });

  const { data: carriers = [] } = useQuery<Carrier[]>({
    queryKey: ["/api/production/carriers"],
  });

  const { data: materials = [], isLoading } = useQuery<Material[]>({
    queryKey: [
      "/api/warehouse/inventory-items",
      { 
        category: selectedCategory === "all" ? undefined : selectedCategory,
        locationId: selectedLocationId === "all" ? undefined : selectedLocationId,
        carrierId: selectedCarrierId === "all" ? undefined : selectedCarrierId,
      }
    ],
  });

  const { data: allFormatki = [] } = useQuery<Material[]>({
    queryKey: ["/api/warehouse/inventory-items", { category: "formatki" }],
  });

  useEffect(() => {
    console.log('🔍 [FRONTEND] Materials data:', { 
      selectedCategory, 
      materialsCount: materials.length, 
      isLoading,
      materials: materials.slice(0, 3)
    });
  }, [materials, selectedCategory, isLoading]);

  const selectedMaterialIdsArray = useMemo(() => 
    Array.from(selectedMaterialIds).sort((a, b) => a - b),
    [selectedMaterialIds]
  );

  useEffect(() => {
    if (isCountNameDirty) return;
    
    const today = new Date();
    const day = String(today.getDate()).padStart(2, '0');
    const month = String(today.getMonth() + 1).padStart(2, '0');
    const year = today.getFullYear();
    const dateStr = `${day}-${month}-${year}`;
    
    const selectedMaterialsData = materials.filter(m => selectedMaterialIds.has(m.id));
    const uniqueGroups = Array.from(new Set(selectedMaterialsData.map(m => m.groupName).filter((g): g is string => Boolean(g))));
    const groupsStr = uniqueGroups.join('-');
    
    const generatedName = `Spis inwentaryzacyjny ${dateStr}${groupsStr ? ' - ' + groupsStr : ''}`;
    setCountName(generatedName);
  }, [selectedCategory, selectedLocationId, selectedCarrierId, selectedMaterialIdsArray, materials, isCountNameDirty]);

  const handleCsvUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    const reader = new FileReader();
    reader.onload = (e) => {
      const text = e.target?.result as string;
      const lines = text.split('\n').filter(line => line.trim());
      
      if (lines.length < 2) {
        toast({ title: "Błąd", description: "Plik CSV jest pusty lub ma nieprawidłowy format", variant: "destructive" });
        return;
      }

      const delimiter = lines[0].includes(';') ? ';' : ',';
      const headers = lines[0].split(delimiter).map(h => h.trim().replace(/^"|"$/g, ''));
      
      const rows: CsvRow[] = [];
      for (let i = 1; i < lines.length; i++) {
        const values = lines[i].split(delimiter).map(v => v.trim().replace(/^"|"$/g, ''));
        if (values.some(v => v)) {
          const row: CsvRow = {};
          headers.forEach((header, index) => {
            row[header] = values[index] || '';
          });
          rows.push(row);
        }
      }

      setCsvHeaders(headers);
      setCsvData(rows);
      
      const nameCol = headers.find(h => 
        h.toLowerCase().includes('nazwa') || 
        h.toLowerCase().includes('name') ||
        h.toLowerCase().includes('krotka')
      ) || headers[0];
      const qtyCol = headers.find(h => 
        h.toLowerCase().includes('ilosc') || 
        h.toLowerCase().includes('qty') || 
        h.toLowerCase().includes('quantity') ||
        h.toLowerCase().includes('mag')
      ) || '';
      setCsvMapping({ nameColumn: nameCol, quantityColumn: qtyCol });
      
      setShowCsvDialog(true);
    };
    reader.readAsText(file, 'UTF-8');
    
    event.target.value = '';
  };

  const parseQuantity = (input: string): number => {
    if (!input || typeof input !== 'string') return 0;
    
    let normalized = input.trim().replace(/\s+/g, '');
    if (!normalized) return 0;
    
    const hasComma = normalized.includes(',');
    const hasDot = normalized.includes('.');
    
    if (hasComma && !hasDot) {
      const parts = normalized.split(',');
      const lastPart = parts[parts.length - 1];
      if (parts.length === 2 && lastPart.length >= 1 && lastPart.length <= 2 && /^\d+$/.test(lastPart)) {
        normalized = normalized.replace(',', '.');
      } else {
        normalized = normalized.replace(/,/g, '');
      }
    } else if (hasDot && !hasComma) {
      const parts = normalized.split('.');
      const lastPart = parts[parts.length - 1];
      if (parts.length === 2 && lastPart.length >= 1 && lastPart.length <= 2 && /^\d+$/.test(lastPart)) {
      } else {
        normalized = normalized.replace(/\./g, '');
      }
    } else if (hasComma && hasDot) {
      const lastComma = normalized.lastIndexOf(',');
      const lastDot = normalized.lastIndexOf('.');
      if (lastComma > lastDot) {
        normalized = normalized.replace(/\./g, '').replace(',', '.');
      } else {
        normalized = normalized.replace(/,/g, '');
      }
    }
    
    normalized = normalized.replace(/[^\d.]/g, '');
    const parsed = parseFloat(normalized);
    return isNaN(parsed) ? 0 : Math.round(parsed);
  };

  const parseItemName = (name: string): { length: number | null; width: number | null; colorCode: string | null } => {
    if (!name) return { length: null, width: null, colorCode: null };
    
    const match = name.match(/^(\d+)x(\d+)-(.+)$/i);
    if (match) {
      return {
        length: parseInt(match[1], 10),
        width: parseInt(match[2], 10),
        colorCode: match[3].toUpperCase()
      };
    }
    
    return { length: null, width: null, colorCode: null };
  };

  const findMatchingItem = (length: number | null, width: number | null, colorCode: string | null): Material | null => {
    if (!length || !width || !colorCode) return null;
    
    const searchIn = allFormatki.length > 0 ? allFormatki : materials;
    
    const match = searchIn.find(m => {
      if (m.itemType !== 'stock_panel') return false;
      
      const dims = m.dimensions?.split('x').map(d => parseInt(d, 10)) || [];
      const matLength = dims[0];
      const matWidth = dims[1];
      const matColor = m.colorCode?.toUpperCase();
      
      return matLength === length && matWidth === width && matColor === colorCode;
    });
    
    return match || null;
  };

  const findMatchingMaterialUniversal = (
    searchName: string, 
    searchMaterials: Material[]
  ): { match: Material | null; matchType: 'formatka' | 'internalCode' | 'name' | 'none' } => {
    if (!searchName || !searchName.trim()) {
      return { match: null, matchType: 'none' };
    }
    
    const normalizedSearch = searchName.trim().toLowerCase();
    if (normalizedSearch.length < 2) {
      return { match: null, matchType: 'none' };
    }
    
    const exactCodeMatch = searchMaterials.find(m => 
      m.internalCode && m.internalCode.trim().toLowerCase() === normalizedSearch
    );
    if (exactCodeMatch) {
      return { match: exactCodeMatch, matchType: 'internalCode' };
    }
    
    const exactNameMatch = searchMaterials.find(m => 
      m.name && m.name.trim().toLowerCase() === normalizedSearch
    );
    if (exactNameMatch) {
      return { match: exactNameMatch, matchType: 'name' };
    }
    
    const codeContainsMatch = searchMaterials.find(m => {
      const matCode = m.internalCode?.trim().toLowerCase();
      if (!matCode || matCode.length < 2) return false;
      return matCode.includes(normalizedSearch) || normalizedSearch.includes(matCode);
    });
    if (codeContainsMatch) {
      return { match: codeContainsMatch, matchType: 'internalCode' };
    }
    
    const nameContainsMatch = searchMaterials.find(m => {
      const matName = m.name?.trim().toLowerCase();
      if (!matName || matName.length < 3) return false;
      return matName.includes(normalizedSearch) || normalizedSearch.includes(matName);
    });
    if (nameContainsMatch) {
      return { match: nameContainsMatch, matchType: 'name' };
    }
    
    return { match: null, matchType: 'none' };
  };

  const processCsvMapping = async () => {
    if (!csvMapping.nameColumn) {
      toast({ title: "Błąd", description: "Wybierz kolumnę z nazwą", variant: "destructive" });
      return;
    }

    setIsProcessingCsv(true);

    const parsed: ParsedItem[] = [];
    const isFormatki = selectedCategory === 'formatki';
    
    for (const row of csvData) {
      const originalName = row[csvMapping.nameColumn] || '';
      if (!originalName.trim()) continue;
      
      const quantityStr = csvMapping.quantityColumn ? row[csvMapping.quantityColumn] || '0' : '0';
      const quantity = parseQuantity(quantityStr);
      
      if (isFormatki) {
        const { length, width, colorCode } = parseItemName(originalName);
        const matchedItem = findMatchingItem(length, width, colorCode);
        
        parsed.push({
          originalName,
          length,
          width,
          colorCode,
          quantity,
          matchedItem,
          isNew: !matchedItem && length !== null && width !== null && colorCode !== null,
          matchType: matchedItem ? 'formatka' : 'none'
        });
      } else {
        const { match, matchType } = findMatchingMaterialUniversal(originalName, materials);
        
        parsed.push({
          originalName,
          length: null,
          width: null,
          colorCode: null,
          quantity,
          matchedItem: match,
          isNew: false,
          matchType
        });
      }
    }

    setParsedItems(parsed);
    setIsProcessingCsv(false);
    
    if (isFormatki) {
      const newItems = parsed.filter(p => p.isNew);
      if (newItems.length > 0) {
        setNewItemsToCreate(newItems);
        setShowCsvDialog(false);
        setShowConfirmNewItems(true);
      } else {
        applyParsedItems(parsed);
        setShowCsvDialog(false);
      }
    } else {
      applyParsedItemsUniversal(parsed);
      setShowCsvDialog(false);
    }
  };

  const applyParsedItems = (items: ParsedItem[], autoCreate: boolean = true) => {
    const matched = items.filter(p => p.matchedItem);
    const matchedIds = matched.map(p => p.matchedItem!.id);
    
    const quantities = new Map<number, number>();
    matched.forEach(p => {
      if (p.matchedItem) {
        quantities.set(p.matchedItem.id, p.quantity);
      }
    });
    setCsvQuantities(quantities);
    
    setSelectedMaterialIds(new Set(matchedIds));
    setSelectedCategory("formatki");
    
    if (autoCreate && matchedIds.length > 0) {
      createCountWithQuantities(matchedIds, quantities);
    } else {
      toast({
        title: "Import zakończony",
        description: `Dopasowano ${matchedIds.length} pozycji z ${items.length} wierszy CSV`,
      });
    }
  };

  const applyParsedItemsUniversal = (items: ParsedItem[]) => {
    const matched = items.filter(p => p.matchedItem);
    const notMatched = items.filter(p => !p.matchedItem);
    const matchedIds = matched.map(p => p.matchedItem!.id);
    
    const quantities = new Map<number, number>();
    matched.forEach(p => {
      if (p.matchedItem) {
        quantities.set(p.matchedItem.id, p.quantity);
      }
    });
    setCsvQuantities(quantities);
    
    setSelectedMaterialIds(new Set(matchedIds));
    
    const byCodeCount = matched.filter(p => p.matchType === 'internalCode').length;
    const byNameCount = matched.filter(p => p.matchType === 'name').length;
    
    if (matchedIds.length > 0) {
      createCountWithQuantitiesUniversal(matchedIds, quantities);
    }
    
    let description = `Dopasowano ${matchedIds.length} z ${items.length} wierszy CSV`;
    if (byCodeCount > 0 || byNameCount > 0) {
      description += ` (po kodzie: ${byCodeCount}, po nazwie: ${byNameCount})`;
    }
    if (notMatched.length > 0) {
      description += `. Nie znaleziono: ${notMatched.length} pozycji`;
    }
    
    toast({
      title: matchedIds.length > 0 ? "Import zakończony" : "Brak dopasowań",
      description,
      variant: matchedIds.length > 0 ? "default" : "destructive",
    });
  };

  const createCountWithQuantitiesUniversal = async (itemIds: number[], quantities: Map<number, number>) => {
    try {
      const selectedItems = materials.filter(m => itemIds.includes(m.id));
      const data: any = {
        name: countName,
        notes: notes || undefined,
      };
      
      const materialTypeIds: number[] = [];
      const panelTypeIds: number[] = [];
      const packagingTypeIds: number[] = [];
      
      selectedItems.forEach(m => {
        if (m.itemType === 'stock_panel') {
          panelTypeIds.push(m.id);
        } else if (m.itemType === 'packaging_material') {
          packagingTypeIds.push(m.id);
        } else {
          materialTypeIds.push(m.id);
        }
      });
      
      if (materialTypeIds.length > 0) {
        data.materialIds = materialTypeIds;
      }
      if (panelTypeIds.length > 0) {
        data.panelIds = panelTypeIds;
      }
      if (packagingTypeIds.length > 0) {
        data.packagingMaterialIds = packagingTypeIds;
      }
      
      const res = await apiRequest("POST", "/api/warehouse/inventory-counts", data);
      const responseData = await res.json();
      const countId = responseData.inventoryCount.id;
      
      if (quantities.size > 0) {
        const items = Array.from(quantities.entries()).map(([itemId, countedQuantity]) => {
          const mat = selectedItems.find(m => m.id === itemId);
          if (mat?.itemType === 'stock_panel') {
            return { panelId: itemId, countedQuantity };
          } else if (mat?.itemType === 'packaging_material') {
            return { packagingMaterialId: itemId, countedQuantity };
          } else {
            return { materialId: itemId, countedQuantity };
          }
        });
        await apiRequest("POST", `/api/warehouse/inventory-counts/${countId}/bulk-update-counted`, { items });
      }
      
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/inventory-counts"] });
      toast({
        title: "Sukces",
        description: `Utworzono spis z ${itemIds.length} pozycjami i ilościami z CSV`,
      });
      setLocation(`/warehouse/inventory-counts/${countId}`);
    } catch (error: any) {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć spisu",
        variant: "destructive",
      });
    }
  };

  const createCountWithQuantities = async (panelIds: number[], quantities: Map<number, number>) => {
    try {
      const res = await apiRequest("POST", "/api/warehouse/inventory-counts", {
        name: countName,
        panelIds,
        notes: notes || undefined,
      });
      const data = await res.json();
      const countId = data.inventoryCount.id;
      
      if (quantities.size > 0) {
        const items = Array.from(quantities.entries()).map(([panelId, countedQuantity]) => ({
          panelId,
          countedQuantity,
        }));
        await apiRequest("POST", `/api/warehouse/inventory-counts/${countId}/bulk-update-counted`, { items });
      }
      
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/inventory-counts"] });
      toast({
        title: "Sukces",
        description: `Utworzono spis z ${panelIds.length} pozycjami i ilościami z CSV`,
      });
      setLocation(`/warehouse/inventory-counts/${countId}`);
    } catch (error: any) {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć spisu",
        variant: "destructive",
      });
    }
  };

  const createNewItemsMutation = useMutation({
    mutationFn: async (items: ParsedItem[]) => {
      const createdItems: Material[] = [];
      
      for (const item of items) {
        if (!item.isNew || !item.length || !item.width || !item.colorCode) continue;
        
        const res = await apiRequest("POST", "/api/warehouse/stock-panels/quick-create", {
          length: item.length,
          width: item.width,
          thickness: 18,
          colorCode: item.colorCode,
          quantity: item.quantity,
          generatedName: item.originalName
        });
        
        const created = await res.json();
        createdItems.push({
          id: created.id,
          name: created.generatedName || item.originalName,
          internalCode: created.generatedName || item.originalName,
          quantity: String(item.quantity),
          unitOfMeasure: 'szt',
          categoryName: 'Formatki',
          groupName: 'Formatki',
          itemType: 'stock_panel',
          dimensions: `${item.length}x${item.width}x18`,
          colorCode: item.colorCode
        });
      }
      
      return createdItems;
    },
    onSuccess: (createdItems) => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/inventory-items"] });
      
      const matched = parsedItems.filter(p => p.matchedItem);
      const allMatchedIds = matched.map(p => p.matchedItem!.id);
      const newIds = createdItems.map(c => c.id);
      const allIds = [...allMatchedIds, ...newIds];
      
      const quantities = new Map<number, number>();
      matched.forEach(p => {
        if (p.matchedItem) {
          quantities.set(p.matchedItem.id, p.quantity);
        }
      });
      newItemsToCreate.forEach((item, idx) => {
        if (createdItems[idx]) {
          quantities.set(createdItems[idx].id, item.quantity);
        }
      });
      
      setSelectedMaterialIds(new Set(allIds));
      setSelectedCategory("formatki");
      setShowConfirmNewItems(false);
      
      if (allIds.length > 0) {
        createCountWithQuantities(allIds, quantities);
      } else {
        toast({
          title: "Sukces",
          description: `Utworzono ${createdItems.length} nowych formatek`,
        });
      }
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć nowych pozycji",
        variant: "destructive",
      });
    }
  });

  const handleConfirmNewItems = () => {
    createNewItemsMutation.mutate(newItemsToCreate);
  };

  const handleSkipNewItems = () => {
    setShowConfirmNewItems(false);
    applyParsedItems(parsedItems);
  };

  const createCountMutation = useMutation({
    mutationFn: async (data: {
      name: string;
      materialIds?: number[];
      panelIds?: number[];
      packagingMaterialIds?: number[];
      notes?: string;
    }) => {
      const res = await apiRequest("POST", "/api/warehouse/inventory-counts", data);
      return await res.json() as {
        inventoryCount: { id: number };
        items: any[];
      };
    },
    onSuccess: (response) => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/inventory-counts"] });
      toast({
        title: "Sukces",
        description: "Spis inwentaryzacyjny został utworzony",
      });
      setLocation(`/warehouse/inventory-counts/${response.inventoryCount.id}`);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć spisu",
        variant: "destructive",
      });
    },
  });

  const handleToggleMaterial = (materialId: number) => {
    const newSelected = new Set(selectedMaterialIds);
    if (newSelected.has(materialId)) {
      newSelected.delete(materialId);
    } else {
      newSelected.add(materialId);
    }
    setSelectedMaterialIds(newSelected);
  };

  const handleSelectAll = () => {
    if (selectedMaterialIds.size === materials.length) {
      setSelectedMaterialIds(new Set());
    } else {
      setSelectedMaterialIds(new Set(materials.map(m => m.id)));
    }
  };

  const handleCreateCount = () => {
    if (!countName.trim()) {
      toast({
        title: "Błąd",
        description: "Podaj nazwę spisu",
        variant: "destructive",
      });
      return;
    }

    if (selectedMaterialIds.size === 0) {
      toast({
        title: "Błąd",
        description: "Zaznacz przynajmniej jeden materiał",
        variant: "destructive",
      });
      return;
    }

    const data: any = {
      name: countName,
      notes: notes || undefined,
    };

    if (selectedCategory === 'formatki') {
      data.panelIds = Array.from(selectedMaterialIds);
    } else if (selectedCategory === 'opakowania') {
      data.packagingMaterialIds = Array.from(selectedMaterialIds);
    } else {
      const selectedItems = materials.filter(m => selectedMaterialIds.has(m.id));
      if (selectedItems.length > 0) {
        const itemType = selectedItems[0].itemType;
        if (itemType === 'stock_panel') {
          data.panelIds = Array.from(selectedMaterialIds);
        } else if (itemType === 'packaging_material') {
          data.packagingMaterialIds = Array.from(selectedMaterialIds);
        } else {
          data.materialIds = Array.from(selectedMaterialIds);
        }
      } else {
        data.materialIds = Array.from(selectedMaterialIds);
      }
    }

    createCountMutation.mutate(data);
  };

  const hallLocations = locations.filter(loc => loc.level === 1);
  
  const isProcessing = createCountMutation.isPending || createNewItemsMutation.isPending;

  return (
    <WarehouseLayout category="all">
      {isProcessing && (
        <div className="fixed inset-0 bg-background/80 backdrop-blur-sm z-50 flex items-center justify-center">
          <div className="flex flex-col items-center gap-4 p-6 bg-card rounded-lg shadow-lg border">
            <Loader2 className="h-12 w-12 animate-spin text-primary" />
            <div className="text-center">
              <p className="text-lg font-semibold">Tworzenie spisu inwentaryzacyjnego...</p>
              <p className="text-sm text-muted-foreground mt-1">Proszę czekać</p>
            </div>
          </div>
        </div>
      )}
      <div className="space-y-6 p-6">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button
              variant="ghost"
              size="icon"
              onClick={() => setLocation("/warehouse/inventory-counts")}
              data-testid="button-back"
            >
              <ArrowLeft className="h-4 w-4" />
            </Button>
            <div>
              <h1 className="text-3xl font-bold">Nowy spis inwentaryzacyjny</h1>
              <p className="text-sm text-muted-foreground mt-1">
                Wybierz materiały do spisania, filtruj po lokalizacji
              </p>
            </div>
          </div>
          <div>
            <input
              ref={fileInputRef}
              type="file"
              accept=".csv"
              onChange={handleCsvUpload}
              className="hidden"
            />
            <Button
              variant="outline"
              onClick={() => fileInputRef.current?.click()}
              data-testid="button-import-csv"
            >
              <Upload className="h-4 w-4 mr-2" />
              Import CSV
            </Button>
          </div>
        </div>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileCheck className="h-5 w-5" />
              Konfiguracja spisu
            </CardTitle>
            <CardDescription>
              Zdefiniuj nazwę i zakres spisu
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid gap-4 md:grid-cols-2">
              <div>
                <Label htmlFor="count-name">Nazwa spisu *</Label>
                <Input
                  id="count-name"
                  placeholder="np. Spis Okuć Hala1 - Grudzień 2024"
                  value={countName}
                  onChange={(e) => {
                    setCountName(e.target.value);
                    setIsCountNameDirty(true);
                  }}
                  data-testid="input-count-name"
                />
              </div>
              <div>
                <Label htmlFor="notes">Notatki</Label>
                <Input
                  id="notes"
                  placeholder="Dodatkowe informacje"
                  value={notes}
                  onChange={(e) => setNotes(e.target.value)}
                  data-testid="input-notes"
                />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <MapPin className="h-5 w-5" />
              Filtrowanie materiałów
            </CardTitle>
            <CardDescription>
              Użyj filtrów aby wybrać konkretną lokalizację lub kategorię
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid gap-4 md:grid-cols-3">
              <div>
                <Label htmlFor="category-filter">Kategoria</Label>
                <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                  <SelectTrigger id="category-filter" data-testid="select-category">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {categories.map((cat) => (
                      <SelectItem key={cat.value} value={cat.value}>
                        {cat.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label htmlFor="location-filter" className="flex items-center gap-1">
                  <Building2 className="h-4 w-4" />
                  Hala
                </Label>
                <Select value={selectedLocationId} onValueChange={setSelectedLocationId}>
                  <SelectTrigger id="location-filter" data-testid="select-location">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie lokalizacje</SelectItem>
                    {hallLocations.map((loc) => (
                      <SelectItem key={loc.id} value={loc.id.toString()}>
                        {loc.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label htmlFor="carrier-filter" className="flex items-center gap-1">
                  <Truck className="h-4 w-4" />
                  Nośnik
                </Label>
                <Select value={selectedCarrierId} onValueChange={setSelectedCarrierId}>
                  <SelectTrigger id="carrier-filter" data-testid="select-carrier">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie nośniki</SelectItem>
                    {carriers.map((carrier) => (
                      <SelectItem key={carrier.id} value={carrier.id.toString()}>
                        {carrier.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <div className="flex items-center justify-between">
              <div>
                <CardTitle className="flex items-center gap-2">
                  <Package className="h-5 w-5" />
                  Materiały do spisania
                </CardTitle>
                <CardDescription className="mt-1">
                  Zaznacz materiały które mają zostać uwzględnione w spisie
                </CardDescription>
              </div>
              <div className="flex items-center gap-3">
                <Badge variant="outline">
                  Zaznaczono: {selectedMaterialIds.size} / {materials.length}
                </Badge>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={handleSelectAll}
                  data-testid="button-select-all"
                >
                  {selectedMaterialIds.size === materials.length ? "Odznacz wszystko" : "Zaznacz wszystko"}
                </Button>
              </div>
            </div>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="text-center py-8 text-muted-foreground">Ładowanie materiałów...</div>
            ) : materials.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground">
                Brak materiałów dla wybranych filtrów
              </div>
            ) : (
              <div className="space-y-2 max-h-96 overflow-y-auto">
                {materials.map((material) => (
                  <div
                    key={material.id}
                    className="flex items-center gap-3 p-3 border rounded-md hover-elevate"
                    data-testid={`material-item-${material.id}`}
                  >
                    <Checkbox
                      checked={selectedMaterialIds.has(material.id)}
                      onCheckedChange={() => handleToggleMaterial(material.id)}
                      data-testid={`checkbox-material-${material.id}`}
                    />
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center gap-2 flex-wrap">
                        <p className="font-medium truncate">{material.name}</p>
                        <Badge variant="secondary" className="text-xs">
                          {material.internalCode}
                        </Badge>
                        {material.locationName && (
                          <Badge variant="outline" className="text-xs">
                            <Building2 className="h-3 w-3 mr-1" />
                            {material.locationName}
                          </Badge>
                        )}
                        {material.carrierName && (
                          <Badge variant="outline" className="text-xs">
                            <Truck className="h-3 w-3 mr-1" />
                            {material.carrierName}
                          </Badge>
                        )}
                      </div>
                      <p className="text-sm text-muted-foreground">
                        Stan: {material.quantity} {material.unitOfMeasure}
                      </p>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>

        <div className="flex items-center justify-between">
          <Button
            variant="outline"
            onClick={() => setLocation("/warehouse/inventory-counts")}
            data-testid="button-cancel"
          >
            Anuluj
          </Button>
          <Button
            onClick={handleCreateCount}
            disabled={createCountMutation.isPending || selectedMaterialIds.size === 0 || !countName.trim()}
            data-testid="button-create-count"
          >
            <FileCheck className="h-4 w-4 mr-2" />
            {createCountMutation.isPending ? "Tworzenie..." : `Utwórz spis (${selectedMaterialIds.size} pozycji)`}
          </Button>
        </div>
      </div>

      <Dialog open={showCsvDialog} onOpenChange={setShowCsvDialog}>
        <DialogContent className="max-w-3xl">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <FileSpreadsheet className="h-5 w-5" />
              Import CSV - Mapowanie pól
            </DialogTitle>
            <DialogDescription>
              Przypisz kolumny z pliku CSV do odpowiednich pól
            </DialogDescription>
          </DialogHeader>

          <div className="space-y-4">
            <div className="grid gap-4 md:grid-cols-2">
              <div>
                <Label>
                  {selectedCategory === 'formatki' 
                    ? 'Kolumna z nazwą (wymiary-kolor) *' 
                    : 'Kolumna z nazwą/kodem materiału *'}
                </Label>
                <Select value={csvMapping.nameColumn} onValueChange={(v) => setCsvMapping(prev => ({ ...prev, nameColumn: v }))}>
                  <SelectTrigger data-testid="select-csv-name-column">
                    <SelectValue placeholder="Wybierz kolumnę" />
                  </SelectTrigger>
                  <SelectContent>
                    {csvHeaders.map((header) => (
                      <SelectItem key={header} value={header}>
                        {header}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                {selectedCategory !== 'formatki' && (
                  <p className="text-xs text-muted-foreground mt-1">
                    Dopasowywanie po kodzie wewnętrznym lub nazwie materiału
                  </p>
                )}
              </div>
              <div>
                <Label>Kolumna z ilością</Label>
                <Select value={csvMapping.quantityColumn || "__none__"} onValueChange={(v) => setCsvMapping(prev => ({ ...prev, quantityColumn: v === "__none__" ? "" : v }))}>
                  <SelectTrigger data-testid="select-csv-quantity-column">
                    <SelectValue placeholder="Wybierz kolumnę (opcjonalne)" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="__none__">Brak</SelectItem>
                    {csvHeaders.map((header) => (
                      <SelectItem key={header} value={header}>
                        {header}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div>
              <Label>Podgląd danych ({csvData.length} wierszy)</Label>
              <ScrollArea className="h-48 border rounded-md mt-2">
                <Table>
                  <TableHeader>
                    <TableRow>
                      {csvHeaders.slice(0, 4).map((header) => (
                        <TableHead key={header} className="text-xs">{header}</TableHead>
                      ))}
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {csvData.slice(0, 10).map((row, idx) => (
                      <TableRow key={idx}>
                        {csvHeaders.slice(0, 4).map((header) => (
                          <TableCell key={header} className="text-xs">{row[header]}</TableCell>
                        ))}
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </ScrollArea>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setShowCsvDialog(false)}>
              Anuluj
            </Button>
            <Button 
              onClick={processCsvMapping} 
              disabled={!csvMapping.nameColumn || isProcessingCsv}
            >
              {isProcessingCsv ? "Przetwarzanie..." : "Importuj"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={showConfirmNewItems} onOpenChange={setShowConfirmNewItems}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2 text-yellow-600">
              <AlertTriangle className="h-5 w-5" />
              Nieznalezione pozycje
            </DialogTitle>
            <DialogDescription>
              Następujące pozycje nie zostały znalezione w magazynie. Czy chcesz je utworzyć?
            </DialogDescription>
          </DialogHeader>

          <ScrollArea className="h-64 border rounded-md">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Nazwa</TableHead>
                  <TableHead>Wymiary</TableHead>
                  <TableHead>Kolor</TableHead>
                  <TableHead>Ilość</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {newItemsToCreate.map((item, idx) => (
                  <TableRow key={idx}>
                    <TableCell className="font-medium">{item.originalName}</TableCell>
                    <TableCell>{item.length}x{item.width}</TableCell>
                    <TableCell>{item.colorCode}</TableCell>
                    <TableCell>{item.quantity}</TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </ScrollArea>

          <div className="bg-muted/50 p-3 rounded-md text-sm">
            <p><strong>Dopasowane:</strong> {parsedItems.filter(p => p.matchedItem).length} pozycji</p>
            <p><strong>Do utworzenia:</strong> {newItemsToCreate.length} pozycji</p>
          </div>

          <DialogFooter className="gap-2">
            <Button variant="outline" onClick={handleSkipNewItems}>
              <X className="h-4 w-4 mr-2" />
              Pomiń nowe, importuj tylko dopasowane
            </Button>
            <Button 
              onClick={handleConfirmNewItems}
              disabled={createNewItemsMutation.isPending}
            >
              <Check className="h-4 w-4 mr-2" />
              {createNewItemsMutation.isPending ? "Tworzenie..." : "Utwórz nowe i importuj"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </WarehouseLayout>
  );
}
