import { useState } from "react";
import { useParams, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import { Plus, Pencil, Trash2, ArrowLeft, LayoutGrid } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { safeGoBack } from "@/lib/navigation";
import { CreateGroupDialog } from "@/components/create-group-dialog";

interface MaterialGroup {
  id: number;
  name: string;
  code: string;
  category: string | null;
  description: string | null;
  displayOrder: number;
  isActive: boolean;
}

// Category display names
const CATEGORY_NAMES: Record<string, string> = {
  plyty: "Płyty meblowe",
  okucia: "Okucia",
  obrzeza: "Obrzęża",
  tkaniny: "Tkaniny",
  sruby: "Śruby",
  pianki: "Pianki",
  opakowania: "Opakowania",
  formatki: "Formatki",
};

export default function WarehouseGroups() {
  const { category } = useParams<{ category: string }>();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const [createDialogOpen, setCreateDialogOpen] = useState(false);
  const [editDialogOpen, setEditDialogOpen] = useState(false);
  const [editingGroup, setEditingGroup] = useState<MaterialGroup | null>(null);
  const [formData, setFormData] = useState({
    name: "",
    description: "",
  });

  // Fetch all groups and filter by category
  const { data: allGroups = [], isLoading } = useQuery<MaterialGroup[]>({
    queryKey: ["/api/warehouse/material-groups"],
  });

  // Filter groups for this category
  const groups = allGroups
    .filter(g => g.category === category && g.isActive)
    .sort((a, b) => {
      if (a.displayOrder !== b.displayOrder) {
        return a.displayOrder - b.displayOrder;
      }
      return a.name.localeCompare(b.name);
    });

  // Update group mutation
  const updateGroupMutation = useMutation({
    mutationFn: async (data: { id: number; name: string; description: string }) => {
      return apiRequest("PATCH", `/api/warehouse/material-groups/${data.id}`, {
        name: data.name,
        description: data.description,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/material-groups"] });
      toast({ title: "Grupa zaktualizowana pomyślnie" });
      setEditDialogOpen(false);
      resetForm();
    },
    onError: () => {
      toast({ title: "Błąd", description: "Nie udało się zaktualizować grupy", variant: "destructive" });
    },
  });

  // Delete group mutation
  const deleteGroupMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/warehouse/material-groups/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/material-groups"] });
      toast({ title: "Grupa usunięta pomyślnie" });
    },
    onError: () => {
      toast({ title: "Błąd", description: "Nie udało się usunąć grupy", variant: "destructive" });
    },
  });

  const resetForm = () => {
    setFormData({ name: "", description: "" });
    setEditingGroup(null);
  };

  const openEditDialog = (group: MaterialGroup) => {
    setEditingGroup(group);
    setFormData({
      name: group.name,
      description: group.description || "",
    });
    setEditDialogOpen(true);
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    updateGroupMutation.mutate({
      id: editingGroup!.id,
      name: formData.name,
      description: formData.description,
    });
  };

  const handleDelete = (id: number) => {
    if (confirm("Czy na pewno chcesz usunąć tę grupę?")) {
      deleteGroupMutation.mutate(id);
    }
  };

  const categoryName = CATEGORY_NAMES[category] || category;

  return (
    <div className="container mx-auto p-6 max-w-7xl">
      <div className="mb-6">
        <Button
          variant="ghost"
          onClick={() => safeGoBack(`/warehouse/${category}`, navigate)}
          className="mb-4"
          data-testid="button-back"
        >
          <ArrowLeft className="w-4 h-4 mr-2" />
          Powrót
        </Button>

        <div className="flex items-center justify-between">
          <div className="flex items-center gap-3">
            <LayoutGrid className="w-8 h-8 text-primary" />
            <div>
              <h1 className="text-3xl font-bold" data-testid="text-page-title">
                Grupy {categoryName}
              </h1>
              <p className="text-muted-foreground">
                Zarządzaj grupami materiałów kategorii "{categoryName}"
              </p>
            </div>
          </div>
          <Button onClick={() => setCreateDialogOpen(true)} data-testid="button-create-group">
            <Plus className="w-4 h-4 mr-2" />
            Dodaj grupę
          </Button>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Lista grup</CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="text-center py-8">Ładowanie...</div>
          ) : groups.length === 0 ? (
            <div className="text-center py-8 text-muted-foreground">
              Brak grup. Dodaj pierwszą grupę używając przycisku powyżej.
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>ID</TableHead>
                  <TableHead>Nazwa</TableHead>
                  <TableHead>Opis</TableHead>
                  <TableHead className="w-[100px]">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {groups.map((group) => (
                  <TableRow key={group.id} data-testid={`row-group-${group.id}`}>
                    <TableCell data-testid={`text-group-id-${group.id}`}>{group.id}</TableCell>
                    <TableCell className="font-medium" data-testid={`text-group-name-${group.id}`}>
                      {group.name}
                    </TableCell>
                    <TableCell className="text-muted-foreground" data-testid={`text-group-description-${group.id}`}>
                      {group.description || "-"}
                    </TableCell>
                    <TableCell>
                      <div className="flex gap-2">
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => openEditDialog(group)}
                          data-testid={`button-edit-group-${group.id}`}
                        >
                          <Pencil className="w-4 h-4" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => handleDelete(group.id)}
                          data-testid={`button-delete-group-${group.id}`}
                        >
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      {/* Create Group Dialog */}
      <CreateGroupDialog
        open={createDialogOpen}
        onOpenChange={setCreateDialogOpen}
        categoryCode={category}
      />

      {/* Edit Group Dialog */}
      <Dialog open={editDialogOpen} onOpenChange={setEditDialogOpen}>
        <DialogContent data-testid="dialog-edit-group">
          <DialogHeader>
            <DialogTitle>Edytuj grupę</DialogTitle>
          </DialogHeader>
          <form onSubmit={handleSubmit}>
            <div className="space-y-4 py-4">
              <div>
                <Label htmlFor="name">Nazwa grupy *</Label>
                <Input
                  id="name"
                  value={formData.name}
                  onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                  required
                  data-testid="input-group-name"
                />
              </div>
              <div>
                <Label htmlFor="description">Opis</Label>
                <Input
                  id="description"
                  value={formData.description}
                  onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                  data-testid="input-group-description"
                />
              </div>
            </div>
            <DialogFooter>
              <Button
                type="button"
                variant="outline"
                onClick={() => {
                  setEditDialogOpen(false);
                  resetForm();
                }}
                data-testid="button-cancel"
              >
                Anuluj
              </Button>
              <Button
                type="submit"
                disabled={updateGroupMutation.isPending}
                data-testid="button-submit"
              >
                Zapisz zmiany
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
