import { useState } from "react";
import { useMutation, useQuery } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { ArrowLeft, Save, FileText, Package, Scissors, Wrench, Box } from "lucide-react";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { Combobox, ComboboxOption } from "@/components/ui/combobox";
import { safeGoBack } from "@/lib/navigation";

const DOCUMENT_TYPES = [
  { value: 'WZ-SPAK', label: 'Produkty spakowane', icon: Package, description: 'Wydanie produktów spakowanych' },
  { value: 'WZ-TAP', label: 'Tapicernia', icon: Scissors, description: 'Wydanie materiałów tapicerskich' },
  { value: 'WZ-FORM', label: 'Formatki', icon: Box, description: 'Wydanie formatek' },
  { value: 'WZ-OKUC', label: 'Okucia', icon: Wrench, description: 'Wydanie okuć' },
  { value: 'WZ-OPAK', label: 'Opakowania', icon: Box, description: 'Wydanie opakowań' },
];

interface ProductionPlan {
  id: number;
  planNumber: string;
  name: string;
}

interface ProductionLocation {
  id: number;
  name: string;
  code: string;
}

export default function WarehouseDocumentNewPage() {
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  
  const [selectedType, setSelectedType] = useState<string>('');
  const [planId, setPlanId] = useState<string>('');
  const [targetDepartment, setTargetDepartment] = useState<string>('');
  const [targetLocationId, setTargetLocationId] = useState<string>('');
  const [remarks, setRemarks] = useState<string>('');

  const { data: plans } = useQuery<ProductionPlan[]>({
    queryKey: ['/api/production/plans'],
    queryFn: async () => {
      const response = await fetch('/api/production/plans?limit=100', { credentials: 'include' });
      if (!response.ok) return [];
      const data = await response.json();
      return data.plans || [];
    },
  });

  const { data: locations } = useQuery<ProductionLocation[]>({
    queryKey: ['/api/production/locations'],
  });

  const createMutation = useMutation({
    mutationFn: async (data: {
      docType: string;
      planId?: number;
      targetDepartment?: string;
      targetLocationId?: number;
      remarks?: string;
    }) => {
      return await apiRequest('POST', '/api/warehouse/documents', data);
    },
    onSuccess: async (response) => {
      const result = await response.json();
      toast({ title: "Sukces", description: "Dokument został utworzony" });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/documents'] });
      setLocation(`/warehouse/documents/${result.id}`);
    },
    onError: (error: Error) => {
      toast({ 
        title: "Błąd", 
        description: error.message || "Nie udało się utworzyć dokumentu", 
        variant: "destructive" 
      });
    },
  });

  const handleSubmit = () => {
    if (!selectedType) {
      toast({ title: "Błąd", description: "Wybierz typ dokumentu", variant: "destructive" });
      return;
    }

    createMutation.mutate({
      docType: selectedType,
      planId: planId ? parseInt(planId) : undefined,
      targetDepartment: targetDepartment || undefined,
      targetLocationId: targetLocationId ? parseInt(targetLocationId) : undefined,
      remarks: remarks || undefined,
    });
  };

  const planOptions: ComboboxOption[] = [
    { value: '', label: 'Brak powiązania' },
    ...(plans || []).map(p => ({
      value: String(p.id),
      label: `${p.planNumber} - ${p.name}`,
    })),
  ];

  const locationOptions: ComboboxOption[] = [
    { value: '', label: 'Brak lokalizacji' },
    ...(locations || []).map(l => ({
      value: String(l.id),
      label: `${l.code} - ${l.name}`,
    })),
  ];

  return (
    <WarehouseLayout category="all" hideSubmenu={true}>
      <div className="p-6 space-y-6">
        <div className="flex items-center justify-between gap-4">
          <div className="flex items-center gap-3">
            <Button 
              variant="ghost" 
              size="icon" 
              onClick={() => safeGoBack('/warehouse/documents', setLocation)} 
              data-testid="button-back"
            >
              <ArrowLeft className="w-5 h-5" />
            </Button>
            <div>
              <h1 className="text-2xl font-bold">Nowy dokument magazynowy</h1>
              <p className="text-sm text-muted-foreground">Utwórz nowy dokument wydania</p>
            </div>
          </div>
        </div>

        <div className="grid gap-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <FileText className="w-5 h-5" />
                Typ dokumentu
              </CardTitle>
              <CardDescription>Wybierz rodzaj dokumentu magazynowego</CardDescription>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-3">
                {DOCUMENT_TYPES.map((type) => {
                  const Icon = type.icon;
                  const isSelected = selectedType === type.value;
                  return (
                    <Card
                      key={type.value}
                      className={`cursor-pointer transition-all ${
                        isSelected 
                          ? 'ring-2 ring-primary bg-primary/5' 
                          : 'hover-elevate'
                      }`}
                      onClick={() => setSelectedType(type.value)}
                      data-testid={`card-type-${type.value}`}
                    >
                      <CardContent className="p-4 text-center">
                        <Icon className={`w-8 h-8 mx-auto mb-2 ${isSelected ? 'text-primary' : 'text-muted-foreground'}`} />
                        <div className="font-mono text-sm font-semibold">{type.value}</div>
                        <div className="text-xs text-muted-foreground mt-1">{type.label}</div>
                      </CardContent>
                    </Card>
                  );
                })}
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Szczegóły dokumentu</CardTitle>
              <CardDescription>Opcjonalne informacje o dokumencie</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="plan">Plan produkcyjny (ZLP)</Label>
                  <Combobox
                    options={planOptions}
                    value={planId}
                    onChange={setPlanId}
                    placeholder="Wybierz plan..."
                    searchPlaceholder="Szukaj planu..."
                    emptyText="Brak planów"
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="targetLocation">Lokalizacja docelowa</Label>
                  <Combobox
                    options={locationOptions}
                    value={targetLocationId}
                    onChange={setTargetLocationId}
                    placeholder="Wybierz lokalizację..."
                    searchPlaceholder="Szukaj lokalizacji..."
                    emptyText="Brak lokalizacji"
                  />
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="department">Dział docelowy</Label>
                <Input
                  id="department"
                  value={targetDepartment}
                  onChange={(e) => setTargetDepartment(e.target.value)}
                  placeholder="np. Produkcja, Montaż, Tapicernia..."
                  data-testid="input-department"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="remarks">Uwagi</Label>
                <Textarea
                  id="remarks"
                  value={remarks}
                  onChange={(e) => setRemarks(e.target.value)}
                  placeholder="Dodatkowe informacje..."
                  rows={3}
                  data-testid="input-remarks"
                />
              </div>
            </CardContent>
          </Card>

          <div className="flex justify-end gap-3">
            <Button 
              variant="outline" 
              onClick={() => safeGoBack('/warehouse/documents', setLocation)}
              data-testid="button-cancel"
            >
              Anuluj
            </Button>
            <Button 
              onClick={handleSubmit}
              disabled={!selectedType || createMutation.isPending}
              data-testid="button-create"
            >
              <Save className="w-4 h-4 mr-2" />
              {createMutation.isPending ? 'Tworzenie...' : 'Utwórz dokument'}
            </Button>
          </div>
        </div>
      </div>
    </WarehouseLayout>
  );
}
