import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Link2, CheckCircle2, AlertCircle, Info, Download, Database, FolderArchive, HardDrive, Trash2 } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Label } from "@/components/ui/label";
import {
  Card,
  CardContent,
  CardDescription,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { FileStorageSettings } from "@/components/file-storage-settings";

export default function Settings() {
  const { toast } = useToast();
  const [autoRefresh, setAutoRefresh] = useState(true);
  const [refreshInterval, setRefreshInterval] = useState("3");
  const [fullSyncEnabled, setFullSyncEnabled] = useState(true);
  const [fullSyncWindowHours, setFullSyncWindowHours] = useState("2");
  const [shoperAutoRefresh, setShoperAutoRefresh] = useState(true);
  const [shoperRefreshInterval, setShoperRefreshInterval] = useState("5");
  const [shoperFullSyncEnabled, setShoperFullSyncEnabled] = useState(true);
  const [shoperFullSyncWindowHours, setShoperFullSyncWindowHours] = useState("2");
  const [erliAutoRefresh, setErliAutoRefresh] = useState(false);
  const [erliRefreshInterval, setErliRefreshInterval] = useState("5");
  const [erliFullSyncEnabled, setErliFullSyncEnabled] = useState(true);
  const [erliFullSyncWindowHours, setErliFullSyncWindowHours] = useState("2");

  const { data: connection, isLoading: connectionLoading } = useQuery<{
    isActive: boolean;
    hasCredentials: boolean;
    fromEnv: boolean;
    clientId?: string;
    requiresReauth?: boolean;
    error?: string | null;
  }>({
    queryKey: ["/api/allegro/connection"],
  });

  const { data: syncSettings } = useQuery<{
    autoRefreshEnabled: boolean;
    refreshIntervalMinutes: string;
    fullSyncEnabled?: boolean;
    fullSyncWindowHours?: string;
  }>({
    queryKey: ["/api/sync/settings"],
  });

  const { data: shoperSyncSettings } = useQuery<{
    shoperAutoRefreshEnabled: boolean;
    shoperRefreshIntervalMinutes: string;
    shoperFullSyncEnabled?: boolean;
    shoperFullSyncWindowHours?: string;
  }>({
    queryKey: ["/api/shoper/sync/settings"],
  });

  const { data: erliConnection } = useQuery<{
    isActive: boolean;
    hasApiKey: boolean;
    shopName?: string;
    connectionError?: string;
    lastSuccessfulSyncAt?: string;
  }>({
    queryKey: ["/api/erli/connection"],
  });

  const { data: erliSyncSettings } = useQuery<{
    erliAutoRefreshEnabled: boolean;
    erliRefreshIntervalMinutes: string;
    erliFullSyncEnabled?: boolean;
    erliFullSyncWindowHours?: string;
    lastErliSyncAt?: string;
  }>({
    queryKey: ["/api/erli/sync/settings"],
  });

  useEffect(() => {
    if (syncSettings) {
      setAutoRefresh(syncSettings.autoRefreshEnabled ?? true);
      setRefreshInterval(syncSettings.refreshIntervalMinutes ?? "3");
      setFullSyncEnabled(syncSettings.fullSyncEnabled ?? true);
      setFullSyncWindowHours(syncSettings.fullSyncWindowHours ?? "2");
    }
  }, [syncSettings]);

  useEffect(() => {
    if (shoperSyncSettings) {
      setShoperAutoRefresh(shoperSyncSettings.shoperAutoRefreshEnabled ?? true);
      setShoperRefreshInterval(shoperSyncSettings.shoperRefreshIntervalMinutes ?? "5");
      setShoperFullSyncEnabled(shoperSyncSettings.shoperFullSyncEnabled ?? true);
      setShoperFullSyncWindowHours(shoperSyncSettings.shoperFullSyncWindowHours ?? "2");
    }
  }, [shoperSyncSettings]);

  useEffect(() => {
    if (erliSyncSettings) {
      setErliAutoRefresh(erliSyncSettings.erliAutoRefreshEnabled ?? false);
      setErliRefreshInterval(erliSyncSettings.erliRefreshIntervalMinutes ?? "5");
      setErliFullSyncEnabled(erliSyncSettings.erliFullSyncEnabled ?? true);
      setErliFullSyncWindowHours(erliSyncSettings.erliFullSyncWindowHours ?? "2");
    }
  }, [erliSyncSettings]);

  const saveSyncSettingsMutation = useMutation({
    mutationFn: async (data: {
      autoRefreshEnabled: boolean;
      refreshIntervalMinutes: string;
      fullSyncEnabled: boolean;
      fullSyncWindowHours: string;
    }) => {
      return await apiRequest("POST", "/api/sync/settings", data);
    },
    onSuccess: () => {
      toast({
        title: "Ustawienia zapisane",
        description: "Ustawienia synchronizacji zostały zaktualizowane.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/sync/settings"] });
    },
  });

  const saveShoperSyncSettingsMutation = useMutation({
    mutationFn: async (data: {
      shoperAutoRefreshEnabled: boolean;
      shoperRefreshIntervalMinutes: string;
      shoperFullSyncEnabled: boolean;
      shoperFullSyncWindowHours: string;
    }) => {
      return await apiRequest("POST", "/api/shoper/sync/settings", data);
    },
    onSuccess: () => {
      toast({
        title: "Ustawienia Shoper zapisane",
        description: "Ustawienia automatycznej synchronizacji Shoper zostały zaktualizowane.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/shoper/sync/settings"] });
    },
  });

  const saveErliSyncSettingsMutation = useMutation({
    mutationFn: async (data: {
      erliAutoRefreshEnabled: boolean;
      erliRefreshIntervalMinutes: string;
      erliFullSyncEnabled: boolean;
      erliFullSyncWindowHours: string;
    }) => {
      return await apiRequest("POST", "/api/erli/sync/settings", data);
    },
    onSuccess: () => {
      toast({
        title: "Ustawienia Erli zapisane",
        description: "Ustawienia automatycznej synchronizacji Erli zostały zaktualizowane.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/erli/sync/settings"] });
    },
  });

  const handleOAuthConnect = async () => {
    if (!connection?.hasCredentials) {
      toast({
        title: "Brak danych uwierzytelniających",
        description: "Dodaj ALLEGRO_CLIENT_ID i ALLEGRO_CLIENT_SECRET jako sekrety w Replit.",
        variant: "destructive",
      });
      return;
    }
      
    window.location.href = "/api/allegro/auth";
  };

  const handleSyncSettingsChange = () => {
    saveSyncSettingsMutation.mutate({
      autoRefreshEnabled: autoRefresh,
      refreshIntervalMinutes: refreshInterval,
      fullSyncEnabled,
      fullSyncWindowHours,
    });
  };

  const handleShoperSyncSettingsChange = () => {
    saveShoperSyncSettingsMutation.mutate({
      shoperAutoRefreshEnabled: shoperAutoRefresh,
      shoperRefreshIntervalMinutes: shoperRefreshInterval,
      shoperFullSyncEnabled,
      shoperFullSyncWindowHours,
    });
  };

  const handleErliSyncSettingsChange = () => {
    saveErliSyncSettingsMutation.mutate({
      erliAutoRefreshEnabled: erliAutoRefresh,
      erliRefreshIntervalMinutes: erliRefreshInterval,
      erliFullSyncEnabled,
      erliFullSyncWindowHours,
    });
  };

  return (
    <div className="flex-1 overflow-auto">
      <div className="max-w-4xl mx-auto px-6 py-8">
        <div className="mb-8">
          <h1 className="text-3xl font-semibold tracking-tight">Konfiguracja</h1>
          <p className="text-sm text-muted-foreground mt-1">
            Zarządzaj połączeniem z Allegro i ustawieniami synchronizacji
          </p>
        </div>

        <div className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>Połączenie z Allegro</CardTitle>
              <CardDescription>
                Skonfiguruj dane OAuth do połączenia z API Allegro
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              {connection?.isActive ? (
                <div className="flex items-center gap-3 p-4 rounded-lg border bg-chart-3/5 border-chart-3/20">
                  <CheckCircle2 className="h-5 w-5 text-chart-3" />
                  <div className="flex-1">
                    <p className="font-medium text-sm">Połączenie aktywne</p>
                    <p className="text-xs text-muted-foreground mt-0.5">
                      Twoje konto jest pomyślnie połączone z Allegro
                    </p>
                  </div>
                </div>
              ) : connection?.requiresReauth ? (
                <div className="flex items-center gap-3 p-4 rounded-lg border bg-destructive/10 border-destructive/20">
                  <AlertCircle className="h-5 w-5 text-destructive" />
                  <div className="flex-1">
                    <p className="font-medium text-sm">Wymagane ponowne połączenie</p>
                    <p className="text-xs text-muted-foreground mt-0.5">
                      {connection.error || 'Token dostępu wygasł - połącz się ponownie'}
                    </p>
                  </div>
                </div>
              ) : (
                <div className="flex items-center gap-3 p-4 rounded-lg border bg-chart-4/5 border-chart-4/20">
                  <AlertCircle className="h-5 w-5 text-chart-4" />
                  <div className="flex-1">
                    <p className="font-medium text-sm">Brak połączenia</p>
                    <p className="text-xs text-muted-foreground mt-0.5">
                      Połącz swoje konto Allegro, aby rozpocząć
                    </p>
                  </div>
                </div>
              )}

              {!connection?.hasCredentials && (
                <div className="flex items-start gap-3 p-4 rounded-lg border bg-muted/50">
                  <Info className="h-5 w-5 text-muted-foreground mt-0.5" />
                  <div className="flex-1 space-y-2">
                    <p className="font-medium text-sm">Wymagane sekrety</p>
                    <p className="text-xs text-muted-foreground">
                      Dodaj następujące sekrety w ustawieniach Replit (Tools → Secrets):
                    </p>
                    <ul className="text-xs text-muted-foreground list-disc list-inside space-y-1">
                      <li><code className="bg-muted px-1 py-0.5 rounded">ALLEGRO_CLIENT_ID</code></li>
                      <li><code className="bg-muted px-1 py-0.5 rounded">ALLEGRO_CLIENT_SECRET</code></li>
                    </ul>
                  </div>
                </div>
              )}

              {connection?.hasCredentials && (
                <div className="flex items-start gap-3 p-4 rounded-lg border bg-muted/50">
                  <CheckCircle2 className="h-5 w-5 text-chart-3 mt-0.5" />
                  <div className="flex-1">
                    <p className="font-medium text-sm">Dane uwierzytelniające skonfigurowane</p>
                    <p className="text-xs text-muted-foreground mt-0.5">
                      {connection.fromEnv 
                        ? "Client ID i Secret pobrane ze zmiennych środowiskowych"
                        : "Client ID i Secret dostępne w systemie"}
                    </p>
                  </div>
                </div>
              )}

              <Button
                variant="default"
                onClick={handleOAuthConnect}
                disabled={connectionLoading || !connection?.hasCredentials}
                data-testid="button-connect-allegro"
                className="w-full"
              >
                <Link2 className="h-4 w-4" />
                Połącz z Allegro
              </Button>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Połączenie z Shoper</CardTitle>
              <CardDescription>
                Skonfiguruj dane API do połączenia z platformą Shoper
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="flex items-start gap-3 p-4 rounded-lg border bg-muted/50">
                <Info className="h-5 w-5 text-muted-foreground mt-0.5" />
                <div className="flex-1 space-y-2">
                  <p className="font-medium text-sm">Wymagane sekrety</p>
                  <p className="text-xs text-muted-foreground">
                    Dodaj następujące sekrety w ustawieniach Replit (Tools → Secrets):
                  </p>
                  <ul className="text-xs text-muted-foreground list-disc list-inside space-y-1">
                    <li><code className="bg-muted px-1 py-0.5 rounded">SHOPER_SHOP_URL</code> - URL sklepu (np. https://sklep.myshoper.com)</li>
                    <li><code className="bg-muted px-1 py-0.5 rounded">SHOPER_USERNAME</code> - Nazwa użytkownika WebAPI</li>
                    <li><code className="bg-muted px-1 py-0.5 rounded">SHOPER_PASSWORD</code> - Hasło użytkownika WebAPI</li>
                  </ul>
                </div>
              </div>

              <ShoperSyncButton />
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Automatyczna synchronizacja Shoper</CardTitle>
              <CardDescription>
                Skonfiguruj cykliczne pobieranie zamówień z Shoper
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label htmlFor="shoper-auto-refresh">Automatyczne odświeżanie</Label>
                  <p className="text-sm text-muted-foreground">
                    Pobieraj zaktualizowane zamówienia automatycznie z Shoper
                  </p>
                </div>
                <Switch
                  id="shoper-auto-refresh"
                  checked={shoperAutoRefresh}
                  onCheckedChange={setShoperAutoRefresh}
                  data-testid="switch-shoper-auto-refresh"
                />
              </div>

              {shoperAutoRefresh && (
                <div className="space-y-2">
                  <Label htmlFor="shoper-refresh-interval">Częstotliwość odświeżania</Label>
                  <Select
                    value={shoperRefreshInterval}
                    onValueChange={setShoperRefreshInterval}
                  >
                    <SelectTrigger
                      id="shoper-refresh-interval"
                      data-testid="select-shoper-refresh-interval"
                    >
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="1">Co 1 minutę</SelectItem>
                      <SelectItem value="3">Co 3 minuty</SelectItem>
                      <SelectItem value="5">Co 5 minut</SelectItem>
                      <SelectItem value="10">Co 10 minut</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              )}

              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label htmlFor="shoper-full-sync-enabled">Pełna synchronizacja okna czasowego</Label>
                  <p className="text-sm text-muted-foreground">
                    Pobieraj wszystkie zamówienia z ostatnich X godzin (zapobiega pomijaniu zmian statusu płatności)
                  </p>
                </div>
                <Switch
                  id="shoper-full-sync-enabled"
                  checked={shoperFullSyncEnabled}
                  onCheckedChange={setShoperFullSyncEnabled}
                  data-testid="switch-shoper-full-sync-enabled"
                />
              </div>

              {shoperFullSyncEnabled && (
                <div className="space-y-2">
                  <Label htmlFor="shoper-full-sync-window">Okno czasowe synchronizacji</Label>
                  <Select
                    value={shoperFullSyncWindowHours}
                    onValueChange={setShoperFullSyncWindowHours}
                  >
                    <SelectTrigger
                      id="shoper-full-sync-window"
                      data-testid="select-shoper-full-sync-window"
                    >
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="1">Ostatnia 1 godzina</SelectItem>
                      <SelectItem value="2">Ostatnie 2 godziny</SelectItem>
                      <SelectItem value="3">Ostatnie 3 godziny</SelectItem>
                      <SelectItem value="6">Ostatnich 6 godzin</SelectItem>
                      <SelectItem value="12">Ostatnich 12 godzin</SelectItem>
                      <SelectItem value="24">Ostatnie 24 godziny</SelectItem>
                    </SelectContent>
                  </Select>
                  <p className="text-xs text-muted-foreground">
                    Synchronizacja pobierze wszystkie zamówienia zaktualizowane w tym okresie, niezależnie od ostatniej synchronizacji
                  </p>
                </div>
              )}

              <Button
                onClick={handleShoperSyncSettingsChange}
                disabled={saveShoperSyncSettingsMutation.isPending}
                data-testid="button-save-shoper-sync-settings"
              >
                Zapisz ustawienia Shoper
              </Button>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Ustawienia synchronizacji Erli</CardTitle>
              <CardDescription>
                Skonfiguruj automatyczne odświeżanie zamówień z Erli
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label>Status połączenia Erli</Label>
                  <p className="text-sm text-muted-foreground">
                    {erliConnection?.hasApiKey ? (
                      erliConnection?.isActive ? (
                        <span className="text-green-600">Połączono</span>
                      ) : (
                        <span className="text-yellow-600">Klucz API skonfigurowany, ale nieaktywny</span>
                      )
                    ) : (
                      <span className="text-red-600">Brak klucza API (ERLI_API_KEY)</span>
                    )}
                  </p>
                </div>
                <ErliTestConnectionButton />
              </div>

              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label htmlFor="erli-auto-refresh">Automatyczne odświeżanie</Label>
                  <p className="text-sm text-muted-foreground">
                    Pobieraj nowe zamówienia automatycznie
                  </p>
                </div>
                <Switch
                  id="erli-auto-refresh"
                  checked={erliAutoRefresh}
                  onCheckedChange={setErliAutoRefresh}
                  disabled={!erliConnection?.hasApiKey}
                  data-testid="switch-erli-auto-refresh"
                />
              </div>

              {erliAutoRefresh && (
                <div className="space-y-2">
                  <Label htmlFor="erli-refresh-interval">Częstotliwość odświeżania</Label>
                  <Select
                    value={erliRefreshInterval}
                    onValueChange={setErliRefreshInterval}
                  >
                    <SelectTrigger id="erli-refresh-interval" data-testid="select-erli-refresh-interval">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="3">Co 3 minuty</SelectItem>
                      <SelectItem value="5">Co 5 minut</SelectItem>
                      <SelectItem value="10">Co 10 minut</SelectItem>
                      <SelectItem value="15">Co 15 minut</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              )}

              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label htmlFor="erli-full-sync-enabled">Pełna synchronizacja okna czasowego</Label>
                  <p className="text-sm text-muted-foreground">
                    Pobieraj wszystkie zamówienia z ostatnich X godzin
                  </p>
                </div>
                <Switch
                  id="erli-full-sync-enabled"
                  checked={erliFullSyncEnabled}
                  onCheckedChange={setErliFullSyncEnabled}
                  disabled={!erliConnection?.hasApiKey}
                  data-testid="switch-erli-full-sync-enabled"
                />
              </div>

              {erliFullSyncEnabled && (
                <div className="space-y-2">
                  <Label htmlFor="erli-full-sync-window">Okno czasowe synchronizacji</Label>
                  <Select
                    value={erliFullSyncWindowHours}
                    onValueChange={setErliFullSyncWindowHours}
                  >
                    <SelectTrigger id="erli-full-sync-window" data-testid="select-erli-full-sync-window">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="1">Ostatnia 1 godzina</SelectItem>
                      <SelectItem value="2">Ostatnie 2 godziny</SelectItem>
                      <SelectItem value="4">Ostatnie 4 godziny</SelectItem>
                      <SelectItem value="8">Ostatnie 8 godzin</SelectItem>
                      <SelectItem value="24">Ostatnie 24 godziny</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              )}

              <Button
                onClick={handleErliSyncSettingsChange}
                disabled={saveErliSyncSettingsMutation.isPending || !erliConnection?.hasApiKey}
                data-testid="button-save-erli-sync-settings"
              >
                Zapisz ustawienia Erli
              </Button>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Ustawienia synchronizacji Allegro</CardTitle>
              <CardDescription>
                Skonfiguruj automatyczne odświeżanie zamówień z Allegro
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label htmlFor="auto-refresh">Automatyczne odświeżanie</Label>
                  <p className="text-sm text-muted-foreground">
                    Pobieraj nowe zamówienia automatycznie
                  </p>
                </div>
                <Switch
                  id="auto-refresh"
                  checked={autoRefresh}
                  onCheckedChange={setAutoRefresh}
                  data-testid="switch-auto-refresh"
                />
              </div>

              {autoRefresh && (
                <div className="space-y-2">
                  <Label htmlFor="refresh-interval">Częstotliwość odświeżania</Label>
                  <Select
                    value={refreshInterval}
                    onValueChange={setRefreshInterval}
                  >
                    <SelectTrigger
                      id="refresh-interval"
                      data-testid="select-refresh-interval"
                    >
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="1">Co 1 minutę</SelectItem>
                      <SelectItem value="3">Co 3 minuty</SelectItem>
                      <SelectItem value="5">Co 5 minut</SelectItem>
                      <SelectItem value="10">Co 10 minut</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              )}

              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label htmlFor="full-sync-enabled">Pełna synchronizacja okna czasowego</Label>
                  <p className="text-sm text-muted-foreground">
                    Pobieraj wszystkie zamówienia z ostatnich X godzin (zapobiega pomijaniu zmian statusu płatności)
                  </p>
                </div>
                <Switch
                  id="full-sync-enabled"
                  checked={fullSyncEnabled}
                  onCheckedChange={setFullSyncEnabled}
                  data-testid="switch-full-sync-enabled"
                />
              </div>

              {fullSyncEnabled && (
                <div className="space-y-2">
                  <Label htmlFor="full-sync-window">Okno czasowe synchronizacji</Label>
                  <Select
                    value={fullSyncWindowHours}
                    onValueChange={setFullSyncWindowHours}
                  >
                    <SelectTrigger
                      id="full-sync-window"
                      data-testid="select-full-sync-window"
                    >
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="1">Ostatnia 1 godzina</SelectItem>
                      <SelectItem value="2">Ostatnie 2 godziny</SelectItem>
                      <SelectItem value="3">Ostatnie 3 godziny</SelectItem>
                      <SelectItem value="6">Ostatnich 6 godzin</SelectItem>
                      <SelectItem value="12">Ostatnich 12 godzin</SelectItem>
                      <SelectItem value="24">Ostatnie 24 godziny</SelectItem>
                    </SelectContent>
                  </Select>
                  <p className="text-xs text-muted-foreground">
                    Synchronizacja pobierze wszystkie zamówienia zaktualizowane w tym okresie, niezależnie od ostatniej synchronizacji
                  </p>
                </div>
              )}

              <Button
                onClick={handleSyncSettingsChange}
                disabled={saveSyncSettingsMutation.isPending}
                data-testid="button-save-sync-settings"
              >
                Zapisz ustawienia
              </Button>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Zdjęcia produktów</CardTitle>
              <CardDescription>
                Pobierz zdjęcia produktów z Allegro API
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <p className="text-sm text-muted-foreground">
                Kliknij poniżej, aby pobrać zdjęcia wszystkich produktów z zamówień.
                Zdjęcia zostaną zapisane w katalogu products/ z nazwami według SKU.
              </p>
              <DownloadImagesButton />
            </CardContent>
          </Card>

          {/* Backup Management */}
          <Card>
            <CardHeader>
              <CardTitle>Zarządzanie backupami</CardTitle>
              <CardDescription>
                Twórz i zarządzaj kopiami zapasowymi bazy danych i plików
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <BackupManagement />
            </CardContent>
          </Card>

          {/* File Storage Settings */}
          <FileStorageSettings />
        </div>
      </div>
    </div>
  );
}

function ShoperSyncButton() {
  const { toast } = useToast();
  
  const syncMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/shoper/sync", {});
    },
    onSuccess: (data: any) => {
      toast({
        title: "Synchronizacja zakończona",
        description: `Zsynchronizowano ${data.ordersFetched} zamówień z Shoper (${data.ordersNew} nowych, ${data.ordersUpdated} zaktualizowanych)`,
      });
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd synchronizacji",
        description: error.message || "Nie udało się zsynchronizować zamówień z Shoper",
        variant: "destructive",
      });
    },
  });

  return (
    <Button
      onClick={() => syncMutation.mutate()}
      disabled={syncMutation.isPending}
      data-testid="button-sync-shoper"
      className="w-full"
      variant="default"
    >
      <Link2 className="h-4 w-4" />
      {syncMutation.isPending ? "Synchronizacja..." : "Synchronizuj zamówienia Shoper"}
    </Button>
  );
}

function ErliTestConnectionButton() {
  const { toast } = useToast();
  
  const testMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/erli/connection/test", {});
    },
    onSuccess: (data: any) => {
      if (data.success) {
        toast({
          title: "Test połączenia zakończony",
          description: "Połączenie z Erli działa poprawnie",
        });
      } else {
        toast({
          title: "Test połączenia nieudany",
          description: data.error || "Nie udało się połączyć z Erli",
          variant: "destructive",
        });
      }
      queryClient.invalidateQueries({ queryKey: ["/api/erli/connection"] });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd połączenia",
        description: error.message || "Nie udało się przetestować połączenia z Erli",
        variant: "destructive",
      });
    },
  });

  return (
    <Button
      onClick={() => testMutation.mutate()}
      disabled={testMutation.isPending}
      variant="outline"
      size="sm"
      data-testid="button-test-erli-connection"
    >
      {testMutation.isPending ? "Testowanie..." : "Testuj połączenie"}
    </Button>
  );
}

function ErliSyncButton() {
  const { toast } = useToast();
  
  const syncMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/erli/sync", {});
    },
    onSuccess: (data: any) => {
      toast({
        title: "Synchronizacja zakończona",
        description: `Zsynchronizowano ${data.ordersFetched || 0} zamówień z Erli (${data.ordersNew || 0} nowych, ${data.ordersUpdated || 0} zaktualizowanych)`,
      });
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd synchronizacji",
        description: error.message || "Nie udało się zsynchronizować zamówień z Erli",
        variant: "destructive",
      });
    },
  });

  return (
    <Button
      onClick={() => syncMutation.mutate()}
      disabled={syncMutation.isPending}
      data-testid="button-sync-erli"
      className="w-full"
      variant="default"
    >
      <Link2 className="h-4 w-4" />
      {syncMutation.isPending ? "Synchronizacja..." : "Synchronizuj zamówienia Erli"}
    </Button>
  );
}

function DownloadImagesButton() {
  const { toast } = useToast();
  
  const downloadMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/products/download-images", {});
    },
    onSuccess: (data: any) => {
      toast({
        title: "Zdjęcia pobrane",
        description: `Pobrano ${data.success}/${data.total} zdjęć produktów`,
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd pobierania",
        description: error.message || "Nie udało się pobrać zdjęć produktów",
        variant: "destructive",
      });
    },
  });

  return (
    <Button
      onClick={() => downloadMutation.mutate()}
      disabled={downloadMutation.isPending}
      data-testid="button-download-images"
    >
      <Download className="h-4 w-4" />
      {downloadMutation.isPending ? "Pobieranie..." : "Pobierz zdjęcia produktów"}
    </Button>
  );
}

interface BackupMetadata {
  id: string;
  type: 'database' | 'files' | 'full';
  timestamp: string;
  filename: string;
  size: number;
  status: 'pending' | 'completed' | 'failed';
  error?: string;
}

function BackupManagement() {
  const { toast } = useToast();

  const { data: backups, isLoading, refetch } = useQuery<BackupMetadata[]>({
    queryKey: ["/api/backup/list"],
  });

  const createDatabaseBackupMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/backup/database", {});
    },
    onSuccess: (data: any) => {
      toast({
        title: "Backup bazy utworzony",
        description: `Utworzono backup: ${data.filename}`,
      });
      refetch();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd tworzenia backupu",
        description: error.message || "Nie udało się utworzyć backupu bazy danych",
        variant: "destructive",
      });
    },
  });

  const createFilesBackupMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/backup/files", {});
    },
    onSuccess: (data: any) => {
      toast({
        title: "Backup plików utworzony",
        description: `Utworzono backup: ${data.filename}`,
      });
      refetch();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd tworzenia backupu",
        description: error.message || "Nie udało się utworzyć backupu plików",
        variant: "destructive",
      });
    },
  });

  const createFullBackupMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/backup/full", {});
    },
    onSuccess: (data: any) => {
      toast({
        title: "Pełny backup utworzony",
        description: `Utworzono backup bazy i plików`,
      });
      refetch();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd tworzenia backupu",
        description: error.message || "Nie udało się utworzyć pełnego backupu",
        variant: "destructive",
      });
    },
  });

  const deleteBackupMutation = useMutation({
    mutationFn: async (filename: string) => {
      return await apiRequest("DELETE", `/api/backup/${filename}`, {});
    },
    onSuccess: () => {
      toast({
        title: "Backup usunięty",
        description: "Backup został pomyślnie usunięty",
      });
      refetch();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć backupu",
        variant: "destructive",
      });
    },
  });

  const cleanupBackupsMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest("POST", "/api/backup/cleanup", { keepCount: 10 });
    },
    onSuccess: (data: any) => {
      toast({
        title: "Czyszczenie zakończone",
        description: `Usunięto ${data.deletedCount} starych backupów`,
      });
      refetch();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd czyszczenia",
        description: error.message || "Nie udało się wyczyścić backupów",
        variant: "destructive",
      });
    },
  });

  const formatBytes = (bytes: number) => {
    if (bytes === 0) return '0 B';
    const k = 1024;
    const sizes = ['B', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return `${(bytes / Math.pow(k, i)).toFixed(2)} ${sizes[i]}`;
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleString('pl-PL');
  };

  const getBackupTypeIcon = (type: string) => {
    switch (type) {
      case 'database': return <Database className="h-4 w-4" />;
      case 'files': return <FolderArchive className="h-4 w-4" />;
      case 'full': return <HardDrive className="h-4 w-4" />;
      default: return <Database className="h-4 w-4" />;
    }
  };

  const getBackupTypeLabel = (type: string) => {
    switch (type) {
      case 'database': return 'Baza danych';
      case 'files': return 'Pliki';
      case 'full': return 'Pełny';
      default: return type;
    }
  };

  return (
    <div className="space-y-4">
      <div className="flex flex-wrap gap-2">
        <Button
          onClick={() => createDatabaseBackupMutation.mutate()}
          disabled={createDatabaseBackupMutation.isPending}
          data-testid="button-backup-database"
          size="sm"
        >
          <Database className="h-4 w-4" />
          {createDatabaseBackupMutation.isPending ? "Tworzenie..." : "Backup bazy"}
        </Button>

        <Button
          onClick={() => createFilesBackupMutation.mutate()}
          disabled={createFilesBackupMutation.isPending}
          data-testid="button-backup-files"
          size="sm"
        >
          <FolderArchive className="h-4 w-4" />
          {createFilesBackupMutation.isPending ? "Tworzenie..." : "Backup plików"}
        </Button>

        <Button
          onClick={() => createFullBackupMutation.mutate()}
          disabled={createFullBackupMutation.isPending}
          data-testid="button-backup-full"
          size="sm"
        >
          <HardDrive className="h-4 w-4" />
          {createFullBackupMutation.isPending ? "Tworzenie..." : "Pełny backup"}
        </Button>

        <Button
          onClick={() => cleanupBackupsMutation.mutate()}
          disabled={cleanupBackupsMutation.isPending}
          data-testid="button-backup-cleanup"
          size="sm"
          variant="outline"
        >
          <Trash2 className="h-4 w-4" />
          {cleanupBackupsMutation.isPending ? "Czyszczenie..." : "Usuń stare (zachowaj 10)"}
        </Button>
      </div>

      {isLoading && <div className="text-sm text-muted-foreground">Ładowanie backupów...</div>}

      {backups && backups.length === 0 && (
        <div className="text-sm text-muted-foreground">Brak backupów. Utwórz pierwszy backup powyżej.</div>
      )}

      {backups && backups.length > 0 && (
        <div className="space-y-2">
          <div className="text-sm font-medium">Dostępne backupy ({backups.length})</div>
          <div className="space-y-1">
            {backups.map((backup) => (
              <div
                key={backup.id}
                className="flex items-center justify-between p-2 border hover-elevate"
                data-testid={`backup-item-${backup.id}`}
              >
                <div className="flex items-center gap-2 flex-1 min-w-0">
                  {getBackupTypeIcon(backup.type)}
                  <div className="flex-1 min-w-0">
                    <div className="text-sm font-medium truncate">{backup.filename}</div>
                    <div className="text-xs text-muted-foreground">
                      {getBackupTypeLabel(backup.type)} • {formatBytes(backup.size)} • {formatDate(backup.timestamp)}
                    </div>
                  </div>
                </div>
                <div className="flex items-center gap-1">
                  <Button
                    size="sm"
                    variant="ghost"
                    asChild
                    data-testid={`button-download-${backup.id}`}
                  >
                    <a href={`/api/backup/download/${backup.filename}`} download>
                      <Download className="h-3 w-3" />
                    </a>
                  </Button>
                  <Button
                    size="sm"
                    variant="ghost"
                    onClick={() => deleteBackupMutation.mutate(backup.filename)}
                    disabled={deleteBackupMutation.isPending}
                    data-testid={`button-delete-${backup.id}`}
                  >
                    <Trash2 className="h-3 w-3" />
                  </Button>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}
    </div>
  );
}
