import { useState, useMemo, useEffect, useRef } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useRoute, useLocation } from "wouter";
import { Loader2, Package } from "lucide-react";
import {
  DndContext,
  DragEndEvent,
  DragStartEvent,
  DragOverlay,
  closestCorners,
  PointerSensor,
  useSensor,
  useSensors,
} from "@dnd-kit/core";
import { restrictToWindowEdges } from "@dnd-kit/modifiers";
import { useAvailableOrdersFilters } from "@/features/production-plans/use-available-orders-filters";
import { PlanHeader } from "@/features/production-plans/PlanHeader";
import { AvailableSourcesPanel } from "@/features/production-plans/AvailableSourcesPanel";
import { PlanItemsPanel } from "@/features/production-plans/PlanItemsPanel";
import { BomFormatkiModal } from "@/features/production-plans/BomFormatkiModal";
import { PlanRoutingSidePanel } from "@/features/production-plans/PlanRoutingSidePanel";
import { MergePointsPanel } from "@/features/production-plans/MergePointsPanel";
import { ZlpDashboard } from "@/features/production-plans/ZlpDashboard";
import { ZlpDashboardInline } from "@/features/production-plans/ZlpDashboardInline";
import { MarketplaceLinkDialog } from "@/components/marketplace-catalog-link-dialog";
import { AddCatalogProductDialog } from "@/features/production-plans/AddCatalogProductDialog";
import { apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";

interface ProductionPlan {
  id: number;
  planNumber: string;
  name: string;
  description?: string;
  plannedStartDate?: string;
  plannedEndDate?: string;
  status: string;
  priority: string;
  notes?: string;
  createdAt: string;
  zlpLocked?: boolean;
  zlpLockedBy?: number;
  zlpLockedAt?: string;
  autoAssignRoutings?: boolean;
}

interface ProductionPlanLine {
  id: number;
  planId: number;
  productId: number;
  quantity: number;
  routingId?: number;
  bomId?: number;
  status: string;
  sequence?: number;
  notes?: string;
  createdAt: string;
  
  // Source metadata
  source_type?: string; // 'marketplace_order', 'catalog_internal', 'cutting_pattern', 'order_demand'
  source_id?: number;
  source_reference?: string;
  metadata?: Record<string, any>;
  
  // Enriched fields from /items endpoint
  product_image?: string | null;
  product_sku?: string;
  product_title?: string;
  product_color?: string | null;
  product_color_options?: string[];
  product_length?: number | null;
  product_width?: number | null;
  product_height?: number | null;
  product_type?: string;
  product_group?: string;
  doors?: string;
  legs?: string;
  material?: string;
  base_price?: string;
  currency?: string;
  
  // Order information (if source_type = 'order_demand')
  order_number?: string;
  order_date?: string;
  buyer_name?: string;
  marketplace?: string;
  
  // Reservation data
  reservedQuantity?: number;
  bom_formatki_count?: number;
  reserved_formatki_count?: number;
}

interface SetComponent {
  component_id: number;
  component_sku: string;
  component_title: string;
  component_color: string | null;
  component_length: string | null;
  component_width: string | null;
  component_height: string | null;
  component_product_type: string | null;
  component_doors: string | null;
  component_legs: string | null;
  quantity: number;
  primary_image_url: string | null;
  parent_set_image_url: string | null;
  is_in_current_plan: boolean;
  is_in_any_plan: boolean;
  in_plan_number: string | null;
  in_plan_id: number | null;
}

interface MarketplaceOrderItem {
  item_id: number;
  offer_external_id: string;
  name: string;
  quantity: number;
  unit_price: string;
  price: string;
  image_url: string | null;
  product_length: string | null;
  product_width: string | null;
  product_height: string | null;
  product_color: string | null;
  product_color_options: string[] | null;
  product_sku: string | null;
  product_type: string | null;
  product_doors: string | null;
  product_legs: string | null;
  marketplace_product_id: number | null;
  link_type: string | null;
  catalog_product_id: number | null;
  catalog_product_sku: string | null;
  catalog_product_title: string | null;
  catalog_set_id: number | null;
  catalog_set_sku: string | null;
  catalog_set_title: string | null;
  platform_link_id: number | null;
  bom_component_count: number | null;
  is_in_plan: boolean;
  in_current_plan: boolean;
  in_plan_number: string | null;
  in_plan_id: number | null;
  set_components: SetComponent[] | null;
}

interface MarketplaceOrder {
  order_id: number;
  order_number: string;
  marketplace: string;
  buyer_first_name: string;
  buyer_last_name: string;
  buyer_email: string;
  order_date: string;
  payment_status: string;
  total_to_pay_amount: string;
  currency: string;
  items: MarketplaceOrderItem[];
}

interface DictionaryItem {
  id: number;
  code: string;
  name: string | null;
  shortName: string | null;
  color: string | null;
  isActive: boolean;
}

interface CatalogProduct {
  id: number;
  sku: string;
  title: string;
  color: string | null;
  color_options: string[] | null;
  length: string | null;
  width: string | null;
  height: string | null;
  product_type: string | null;
  doors: string | null;
  legs: string | null;
  created_at: string;
  image_url: string | null;
  bom_component_count: number;
  product_group_id: number | null;
  product_group_name: string | null;
  product_group_color_hex: string | null;
  is_in_plan: boolean;
  in_plan_number: string | null;
  in_current_plan: boolean;
}

interface CuttingPattern {
  id: number;
  code: string;
  name: string;
  description: string | null;
  status: string;
  board_length: string;
  board_width: string;
  board_thickness: string;
  kerf: string;
  total_quantity: number;
  items_count: number;
  created_at: string;
  is_in_plan: boolean;
  in_plan_number: string | null;
  in_current_plan: boolean;
}

export default function ProductionPlanDetailPage() {
  const [, params] = useRoute("/production/plans/:id");
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  
  const planId = params?.id ? parseInt(params.id) : null;
  
  // Available orders filters hook
  const { filters: orderFilters, updateFilter, resetFilters, getActiveFilters, clearFilter } = useAvailableOrdersFilters();
  
  // Plan items search filter (for tabs integration)
  const [planItemsSearchFilter, setPlanItemsSearchFilter] = useState('');
  
  // Orders search filter (for tabs integration on left panel) - synced with orderFilters.search
  const [ordersSearchFilter, setOrdersSearchFilter] = useState('');
  
  // Linking modal state
  const [linkingItem, setLinkingItem] = useState<{ item: MarketplaceOrderItem; marketplace: string } | null>(null);

  // Drag & Drop state
  const [activeId, setActiveId] = useState<string | null>(null);
  const [draggedItem, setDraggedItem] = useState<MarketplaceOrderItem | null>(null);
  
  // Track recently added items for visual feedback
  const [recentlyAddedItems, setRecentlyAddedItems] = useState<Set<number>>(new Set());
  const [recentlyAddedProducts, setRecentlyAddedProducts] = useState<Set<number>>(new Set());
  const [recentlyAddedPatterns, setRecentlyAddedPatterns] = useState<Set<number>>(new Set());
  
  // Catalog product dialog state
  const [catalogDialogOpen, setCatalogDialogOpen] = useState(false);
  const [selectedCatalogProduct, setSelectedCatalogProduct] = useState<CatalogProduct | null>(null);
  const [isCatalogMutationPending, setIsCatalogMutationPending] = useState(false);
  
  // Routing side panel state
  const [routingPanelOpen, setRoutingPanelOpen] = useState(false);
  
  // ZLP Dashboard state
  const [zlpDashboardOpen, setZlpDashboardOpen] = useState(false);
  
  // Track refetch state after adding items
  const [isRefetchingAfterAdd, setIsRefetchingAfterAdd] = useState(false);
  
  // Track number of pending add mutations (for sets with multiple components)
  const [pendingAddCount, setPendingAddCount] = useState(0);
  
  // Track if ANY bulk operation is in progress (set before first mutation, cleared after refetch)
  const [isBulkOperationActive, setIsBulkOperationActive] = useState(false);
  
  // Track items being added (for detailed loader display with bulk support)
  const [addingItems, setAddingItems] = useState<Array<{
    id: string; // unique identifier for this add operation
    productSku: string;
    productName: string;
    quantity: number;
    orderNumber?: string;
    status: 'pending' | 'processing' | 'completed';
    reservationInfo?: {
      packedProductReserved: boolean;
      packedProductSku?: string;
      packedQtyReserved?: number;
      formatkaSearched: boolean;
      formatkiFound: string[];
      formatkiReserved: string[];
    };
  }>>([]);
  
  // Track bulk removal state
  const [isRemovingBulk, setIsRemovingBulk] = useState(false);
  
  // Highlight product in plan when clicked from orders panel
  const [highlightedProductId, setHighlightedProductId] = useState<number | null>(null);
  const [highlightedOrderNumber, setHighlightedOrderNumber] = useState<string | null>(null);
  
  // Check sessionStorage for highlight data on page load (from modal navigation)
  useEffect(() => {
    const storedProductId = sessionStorage.getItem('highlightProductId');
    const storedOrderNumber = sessionStorage.getItem('highlightOrderNumber');
    
    if (storedProductId && storedOrderNumber) {
      setHighlightedProductId(parseInt(storedProductId, 10));
      setHighlightedOrderNumber(storedOrderNumber);
      
      // Clear sessionStorage
      sessionStorage.removeItem('highlightProductId');
      sessionStorage.removeItem('highlightOrderNumber');
      
      // Clear highlight after 3 seconds
      setTimeout(() => {
        setHighlightedProductId(null);
        setHighlightedOrderNumber(null);
      }, 3000);
    }
  }, [planId]);
  
  // Check URL query param for panel=routing to auto-open routing panel
  useEffect(() => {
    const urlParams = new URLSearchParams(window.location.search);
    const panelParam = urlParams.get('panel');
    
    if (panelParam === 'routing') {
      setRoutingPanelOpen(true);
      // Clear the query param from URL to avoid re-opening on refresh
      const newUrl = window.location.pathname;
      window.history.replaceState({}, '', newUrl);
    }
  }, [planId]);
  
  // Track previous pendingAddCount for refetch detection
  const previousPendingCount = useRef(0);
  const refetchTimeoutRef = useRef<NodeJS.Timeout | null>(null);
  const isRefetchScheduledRef = useRef(false); // Track if refetch was scheduled (for fallback cleanup)
  
  // Track desktop/mobile breakpoint
  const [isDesktop, setIsDesktop] = useState(typeof window !== 'undefined' && window.innerWidth >= 1024);
  
  // Resizable panels state - persist in localStorage
  const [leftPanelWidth, setLeftPanelWidth] = useState<number>(() => {
    try {
      const saved = localStorage.getItem(`plan-${planId}-panel-width`);
      return saved ? parseFloat(saved) : 50; // Default 50%
    } catch {
      return 50;
    }
  });
  
  const [isResizing, setIsResizing] = useState(false);
  
  // Track window resize for responsive layout
  useEffect(() => {
    const handleResize = () => {
      setIsDesktop(window.innerWidth >= 1024);
    };
    
    window.addEventListener('resize', handleResize);
    return () => window.removeEventListener('resize', handleResize);
  }, []);
  
  // Save panel width to localStorage
  useEffect(() => {
    if (planId) {
      localStorage.setItem(`plan-${planId}-panel-width`, leftPanelWidth.toString());
    }
  }, [planId, leftPanelWidth]);
  
  // Handle divider resize
  const handleMouseDown = (e: React.MouseEvent) => {
    e.preventDefault();
    setIsResizing(true);
  };
  
  useEffect(() => {
    const handleMouseMove = (e: MouseEvent) => {
      if (!isResizing) return;
      
      const container = document.getElementById('resizable-container');
      if (!container) return;
      
      const containerRect = container.getBoundingClientRect();
      const newLeftWidth = ((e.clientX - containerRect.left) / containerRect.width) * 100;
      
      // Clamp between 20% and 80%
      const clampedWidth = Math.min(Math.max(newLeftWidth, 20), 80);
      setLeftPanelWidth(clampedWidth);
    };
    
    const handleMouseUp = () => {
      setIsResizing(false);
    };
    
    if (isResizing) {
      document.addEventListener('mousemove', handleMouseMove);
      document.addEventListener('mouseup', handleMouseUp);
      document.body.style.cursor = 'col-resize';
      document.body.style.userSelect = 'none';
    }
    
    return () => {
      document.removeEventListener('mousemove', handleMouseMove);
      document.removeEventListener('mouseup', handleMouseUp);
      document.body.style.cursor = '';
      document.body.style.userSelect = '';
    };
  }, [isResizing]);

  // DnD sensors
  const sensors = useSensors(
    useSensor(PointerSensor, {
      activationConstraint: {
        distance: 8,
      },
    })
  );

  // Fetch plan
  const { data: plan, isLoading: planLoading} = useQuery<ProductionPlan>({
    queryKey: [`/api/production/planning/plans/${planId}`],
    enabled: !!planId,
  });

  // Fetch colors dictionary
  const { data: colors = [] } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries?type=color"],
  });

  // Fetch product types dictionary
  const { data: productTypes = [] } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries?type=product_type"],
  });

  // Fetch dimension_length dictionary
  const { data: dimensionLengths = [] } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries?type=dimension_length"],
  });

  // Fetch dimension_width dictionary
  const { data: dimensionWidths = [] } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries?type=dimension_width"],
  });

  // Fetch plan lines with full details (including product images)
  const { data: lines = [], isLoading: linesLoading } = useQuery<ProductionPlanLine[]>({
    queryKey: [`/api/production/planning/plans/${planId}/items`],
    enabled: !!planId,
  });

  // Fetch available marketplace orders for production planning
  const { data: availableOrdersData, isLoading: ordersLoading } = useQuery<{ orders: MarketplaceOrder[]; total: number }>({
    queryKey: [`/api/production/planning/plans/${planId}/available-orders`, orderFilters],
    queryFn: async () => {
      const params = new URLSearchParams({
        ...(orderFilters.search && { search: orderFilters.search }),
        ...(orderFilters.color && { color: orderFilters.color }),
        ...(orderFilters.sku && { sku: orderFilters.sku }),
        ...(orderFilters.minLength && { minLength: orderFilters.minLength }),
        ...(orderFilters.maxLength && { maxLength: orderFilters.maxLength }),
        ...(orderFilters.minWidth && { minWidth: orderFilters.minWidth }),
        ...(orderFilters.maxWidth && { maxWidth: orderFilters.maxWidth }),
        ...(orderFilters.orderNumber && { orderNumber: orderFilters.orderNumber }),
        ...(orderFilters.customerName && { customerName: orderFilters.customerName }),
        ...(orderFilters.marketplace !== 'all' && { marketplace: orderFilters.marketplace }),
        // Server-side filters
        ...(orderFilters.showSetsOnly && { showSetsOnly: orderFilters.showSetsOnly }),
        ...(orderFilters.showCatalogLinked && { showCatalogLinked: orderFilters.showCatalogLinked }),
        ...(orderFilters.showInPlans && { showInPlans: orderFilters.showInPlans }),
        // Date filters
        ...(orderFilters.dateFilter && { dateFilter: orderFilters.dateFilter }),
        ...(orderFilters.customDays && { customDays: orderFilters.customDays }),
        // Pagination and sorting
        limit: orderFilters.limit.toString(),
        offset: orderFilters.offset.toString(),
        sortBy: orderFilters.sortBy,
        sortOrder: orderFilters.sortOrder,
      });
      const response = await fetch(`/api/production/planning/plans/${planId}/available-orders?${params}`, {
        credentials: "include",
      });
      if (!response.ok) throw new Error("Failed to fetch available orders");
      return response.json();
    },
    enabled: !!planId,
  });

  const availableOrders = availableOrdersData?.orders || [];

  // Fetch available catalog products
  const { data: catalogProductsData, isLoading: catalogLoading } = useQuery<{ products: CatalogProduct[]; total: number }>({
    queryKey: [`/api/production/planning/plans/${planId}/available-catalog-products`],
    enabled: !!planId,
  });

  const catalogProducts = catalogProductsData?.products || [];

  // Fetch available cutting patterns
  const { data: cuttingPatternsData, isLoading: patternsLoading } = useQuery<{ patterns: CuttingPattern[]; total: number }>({
    queryKey: [`/api/production/planning/plans/${planId}/available-cutting-patterns`],
    enabled: !!planId,
  });

  const cuttingPatterns = cuttingPatternsData?.patterns || [];

  // Check if there are ZLPs for this plan (used to show ZLP Dashboard button)
  const { data: zlpDashboardData, refetch: refetchZlpDashboard } = useQuery<{ totalZlps: number }>({
    queryKey: [`/api/production/planning/plans/${planId}/zlp-dashboard`],
    enabled: !!planId,
    refetchOnMount: 'always',
    refetchOnWindowFocus: true,
    refetchInterval: 10000,
    staleTime: 0,
    gcTime: 0,
    retry: 2,
  });
  
  const hasZlps = (zlpDashboardData?.totalZlps ?? 0) > 0;

  // Derived statistics from lines and orders
  const statistics = useMemo(() => {
    const ordersCount = availableOrders.length;
    const productsCount = availableOrders.reduce((sum, o) => sum + o.items.length, 0);
    const planLinesCount = lines.length;
    const totalQuantity = lines.reduce((sum, l) => sum + l.quantity, 0);
    
    // Packed products statistics
    const packedProductsCount = lines.filter(l => (l.reservedQuantity || 0) > 0).length;
    const packedProductsPercent = planLinesCount > 0 ? Math.round((packedProductsCount / planLinesCount) * 100) : 0;
    
    // Formatki statistics - parse as integers since PostgreSQL returns strings
    const totalFormatkiNeeded = lines.reduce((sum, l) => sum + (parseInt(String(l.bom_formatki_count)) || 0), 0);
    const totalFormatkiReserved = lines.reduce((sum, l) => sum + (parseInt(String(l.reserved_formatki_count)) || 0), 0);
    const formatkiFromWarehousePercent = totalFormatkiNeeded > 0 ? Math.round((totalFormatkiReserved / totalFormatkiNeeded) * 100) : 0;
    
    return { 
      ordersCount, 
      productsCount, 
      planLinesCount, 
      totalQuantity,
      packedProductsCount,
      packedProductsPercent,
      totalFormatkiNeeded,
      totalFormatkiReserved,
      formatkiFromWarehousePercent
    };
  }, [availableOrders, lines]);

  // Create line mutation - simplified for adding from orders panel
  const createLineMutation = useMutation({
    mutationFn: async (data: { 
      productId: number; 
      quantity: number; 
      notes?: string; 
      itemId?: number;
      orderNumber?: string;
      buyerName?: string;
      marketplace?: string;
      imageUrl?: string | null;
      setId?: number;
      componentId?: number;
      catalogProductSku?: string;
      catalogProductName?: string;
    }) => {
      const metadata: any = {
        order_number: data.orderNumber,
        buyer_name: data.buyerName,
        marketplace: data.marketplace,
        image_url: data.imageUrl,
      };
      
      // For components: use set_id + component_id instead of order_item_id
      if (data.setId && data.componentId) {
        metadata.set_id = data.setId;
        metadata.component_id = data.componentId;
      } else {
        metadata.order_item_id = data.itemId;
      }
      
      const payload = {
        productId: data.productId,
        quantity: data.quantity,
        notes: data.notes,
        planId,
        sourceType: "order_demand",
        sourceReference: data.orderNumber, // Link to order for JOIN in query
        metadata,
      };
      
      const response = await apiRequest("POST", `/api/production/planning/lines`, payload);
      return response.json(); // Parse JSON and return data with reservationInfo
    },
    onMutate: async (variables) => {
      // Set bulk operation flag (ensures window stays visible throughout entire operation)
      setIsBulkOperationActive(true);
      
      // Increment pending count for each mutation
      setPendingAddCount(prev => prev + 1);
      
      // Generate unique ID for this add operation
      const addItemId = `${variables.productId}-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
      
      // Add item to the list with 'processing' status - use catalog SKU if available
      setAddingItems(prev => [...prev, {
        id: addItemId,
        productSku: variables.catalogProductSku || (variables.orderNumber ? `#${variables.orderNumber}` : `ID:${variables.productId}`),
        productName: variables.catalogProductName || '',
        quantity: variables.quantity,
        orderNumber: variables.orderNumber,
        status: 'processing',
      }]);
      
      const itemsQueryKey = [`/api/production/planning/plans/${planId}/items`];
      
      // Mark item as recently added for visual feedback (no optimistic update)
      if (variables.itemId) {
        setRecentlyAddedItems(prev => new Set(prev).add(variables.itemId!));
        
        // Clear after 700ms
        const timeoutId = setTimeout(() => {
          setRecentlyAddedItems(prev => {
            const newSet = new Set(prev);
            newSet.delete(variables.itemId!);
            return newSet;
          });
        }, 700);
        
        return { timeoutId, addItemId };
      }
      
      return { addItemId };
    },
    onError: (error: any, variables, context: any) => {
      // Clear highlight on error
      if (variables.itemId && context?.timeoutId) {
        clearTimeout(context.timeoutId);
        setRecentlyAddedItems(prev => {
          const newSet = new Set(prev);
          newSet.delete(variables.itemId!);
          return newSet;
        });
      }
      
      // Handle duplicate error (409 Conflict) - check both Polish and English messages
      if (error.message?.includes("already in the plan") || 
          error.message?.includes("already in plan") ||
          error.message?.includes("już istnieje w planie")) {
        toast({ 
          title: "Produkt już w planie", 
          description: error.message || "Ten produkt został już dodany do planu produkcyjnego.",
          variant: "destructive" 
        });
      } else {
        toast({ title: "Błąd", description: error.message, variant: "destructive" });
      }
    },
    onSettled: () => {
      // Decrement pending count
      setPendingAddCount(prev => {
        const newCount = Math.max(0, prev - 1);
        
        // Fallback: clear bulk operation flag if this was the last mutation
        // (in case refetch doesn't fire due to error)
        if (newCount === 0 && prev > 0) {
          setTimeout(() => {
            // Only clear if refetch was NOT scheduled (failsafe for errors)
            if (!isRefetchScheduledRef.current) {
              setIsBulkOperationActive(false);
            }
          }, 150); // Slightly longer than refetch setTimeout (100ms)
        }
        
        return newCount;
      });
    },
    onSuccess: async (data: any, variables: any, context: any) => {
      // Update item with reservation info and mark as completed
      if (context?.addItemId) {
        setAddingItems(prev => prev.map(item => 
          item.id === context.addItemId 
            ? {
                ...item,
                status: 'completed' as const,
                productSku: data?.product_sku || item.productSku,
                productName: data?.product_title || '',
                reservationInfo: data?.reservationInfo
              }
            : item
        ));
      }
    },
  });

  // Delete line mutation - for removing items from plan
  const deleteLineMutation = useMutation({
    mutationFn: async (lineId: number) => {
      return apiRequest("DELETE", `/api/production/planning/lines/${lineId}`);
    },
    onSuccess: () => {
      // Odśwież wszystkie dane planu (włącznie z rezerwacjami magazynowymi)
      // Używamy refetchType: 'all' aby wymusić odświeżenie wszystkich wariantów query (z różnymi filtrami)
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}`], refetchType: 'all' });
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/items`], refetchType: 'all' });
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/available-orders`], refetchType: 'all' });
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/available-catalog-products`], refetchType: 'all' });
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/available-cutting-patterns`], refetchType: 'all' });
      toast({ title: "Produkt został usunięty z planu" });
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });
  
  // Lock ZLP mutation - prevents further editing
  const lockZlpMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", `/api/production/planning/plans/${planId}/lock-zlp`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}`] });
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/zlp-dashboard`] });
      toast({ title: "ZLP zostało zatwierdzone", description: "Modyfikacje są teraz zablokowane" });
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });
  
  // Unlock ZLP mutation - allows editing again
  const unlockZlpMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", `/api/production/planning/plans/${planId}/unlock-zlp`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}`] });
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/zlp-dashboard`] });
      toast({ title: "ZLP zostało odblokowane", description: "Modyfikacje są teraz dozwolone" });
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  // Sync local search state with orderFilters.search (for tabs integration + API query)
  useEffect(() => {
    setOrdersSearchFilter(orderFilters.search || '');
  }, [orderFilters.search]);
  
  // Refetch queries when all pending mutations complete
  useEffect(() => {
    // When pendingAddCount transitions TO 0 (from any non-zero value), execute refetch
    if (pendingAddCount === 0 && previousPendingCount.current !== 0) {
      // Mark refetch as scheduled (for fallback cleanup check)
      isRefetchScheduledRef.current = true;
      
      // Set refetching flag IMMEDIATELY (before setTimeout)
      setIsRefetchingAfterAdd(true);
      
      // Clear any existing timeout
      if (refetchTimeoutRef.current) {
        clearTimeout(refetchTimeoutRef.current);
      }
      
      // Use setTimeout to ensure all onSettled handlers have completed
      refetchTimeoutRef.current = setTimeout(async () => {
        try {
          // Refetch both queries to update UI
          await queryClient.refetchQueries({ queryKey: [`/api/production/planning/plans/${planId}/items`] });
          await queryClient.refetchQueries({ queryKey: [`/api/production/planning/plans/${planId}/available-orders`] });
          
          toast({ title: "Produkty zostały dodane do planu" });
        } finally {
          setIsRefetchingAfterAdd(false);
          setIsBulkOperationActive(false); // Clear bulk operation flag
          // Keep addingItems visible for 2 seconds before clearing
          setTimeout(() => {
            setAddingItems([]);
          }, 2000);
          isRefetchScheduledRef.current = false; // Reset ref
          refetchTimeoutRef.current = null;
        }
      }, 100);
    }
    
    // Always update previous count to prevent infinite loops
    previousPendingCount.current = pendingAddCount;
  }, [pendingAddCount, planId, toast]);
  
  // Cleanup timeout on unmount
  useEffect(() => {
    return () => {
      if (refetchTimeoutRef.current) {
        clearTimeout(refetchTimeoutRef.current);
      }
    };
  }, []);

  // Wrapper function to update BOTH local state (for tabs) AND API query state
  const handleOrdersSearchChange = (value: string) => {
    setOrdersSearchFilter(value); // Update local state for tabs bar
    updateFilter('search', value); // Update API query state
  };

  // Unified handlers for all source types
  const handleAddItem = (item: MarketplaceOrderItem, orderNumber: string, buyerName: string, marketplace: string, setId?: number, componentId?: number) => {
    // Block adding when ZLPs have been approved/locked
    if (plan?.zlpLocked) {
      toast({
        title: "Plan jest zatwierdzony",
        description: "Nie można dodawać produktów po zatwierdzeniu zleceń produkcyjnych (ZLP). Odblokuj plan, aby edytować.",
        variant: "destructive"
      });
      return;
    }
    
    if (!item.catalog_product_id) {
      toast({
        title: "Produkt nie zmapowany",
        description: "Najpierw połącz produkt z katalogiem.",
        variant: "destructive"
      });
      return;
    }
    
    createLineMutation.mutate({
      productId: item.catalog_product_id,
      quantity: item.quantity,
      notes: `Zamówienie: ${orderNumber}, Klient: ${buyerName}`,
      itemId: item.item_id,
      orderNumber,
      buyerName,
      marketplace,
      imageUrl: item.image_url,
      setId,
      componentId,
      catalogProductSku: item.catalog_product_sku || item.product_sku || undefined,
      catalogProductName: item.catalog_product_title || item.name || undefined,
    });
    
    // Visual feedback
    setRecentlyAddedItems(prev => new Set(Array.from(prev).concat(item.item_id)));
    setTimeout(() => {
      setRecentlyAddedItems(prev => {
        const newSet = new Set(prev);
        newSet.delete(item.item_id);
        return newSet;
      });
    }, 2000);
  };

  const handleAddCatalogProduct = (product: CatalogProduct) => {
    // Block adding when ZLPs have been approved/locked
    if (plan?.zlpLocked) {
      toast({
        title: "Plan jest zatwierdzony",
        description: "Nie można dodawać produktów po zatwierdzeniu zleceń produkcyjnych (ZLP). Odblokuj plan, aby edytować.",
        variant: "destructive"
      });
      return;
    }
    
    if (product.in_current_plan) {
      toast({
        title: "Produkt już w planie",
        description: "Ten produkt jest już w tym planie produkcyjnym.",
        variant: "destructive"
      });
      return;
    }
    
    // Open dialog to input quantity
    setSelectedCatalogProduct(product);
    setCatalogDialogOpen(true);
  };
  
  const handleConfirmCatalogProduct = async (quantity: number) => {
    if (!selectedCatalogProduct) return;
    
    const product = selectedCatalogProduct;
    setIsCatalogMutationPending(true);

    const catalogLineMutation = {
      mutationFn: async (data: { productId: number; quantity: number; notes?: string }) => {
        const payload = {
          productId: data.productId,
          quantity: data.quantity,
          notes: data.notes,
          planId,
          sourceType: "catalog_internal",
          metadata: {
            product_sku: product.sku,
            product_title: product.title,
            order_number: `WEW-${Date.now()}`,
            added_at: new Date().toISOString(),
          },
        };
        
        return apiRequest("POST", `/api/production/planning/lines`, payload);
      },
      onSuccess: async () => {
        // Immediately refetch to get full product details
        await Promise.all([
          queryClient.refetchQueries({ queryKey: [`/api/production/planning/plans/${planId}/items`] }),
          queryClient.refetchQueries({ queryKey: [`/api/production/planning/plans/${planId}/available-catalog-products`] }),
        ]);
        
        toast({
          title: "Dodano do planu",
          description: `Produkt ${product.sku} (${quantity} szt.) został dodany do planu produkcyjnego`,
        });
        
        // Close dialog
        setCatalogDialogOpen(false);
        setSelectedCatalogProduct(null);
        setIsCatalogMutationPending(false);
        
        // Visual feedback
        setRecentlyAddedProducts(prev => new Set(Array.from(prev).concat(product.id)));
        setTimeout(() => {
          setRecentlyAddedProducts(prev => {
            const newSet = new Set(prev);
            newSet.delete(product.id);
            return newSet;
          });
        }, 2000);
      },
      onError: (error: any) => {
        toast({
          title: "Błąd",
          description: error.message || "Nie udało się dodać produktu do planu",
          variant: "destructive",
        });
        setIsCatalogMutationPending(false);
      },
    };

    // Execute mutation
    try {
      await queryClient.getMutationCache().build(queryClient, catalogLineMutation).execute({
        productId: product.id,
        quantity,
        notes: `Produkt katalogowy: ${product.title}`,
      });
    } catch (error) {
      // Error handled in onError
    }
  };

  const handleAddCuttingPattern = (pattern: CuttingPattern) => {
    // Block adding when ZLPs have been approved/locked
    if (plan?.zlpLocked) {
      toast({
        title: "Plan jest zatwierdzony",
        description: "Nie można dodawać rozkrojów po zatwierdzeniu zleceń produkcyjnych (ZLP). Odblokuj plan, aby edytować.",
        variant: "destructive"
      });
      return;
    }
    
    if (pattern.in_current_plan) {
      toast({
        title: "Rozkrój już w planie",
        description: "Ten rozkrój jest już w tym planie produkcyjnym.",
        variant: "destructive"
      });
      return;
    }

    const patternLineMutation = {
      mutationFn: async (data: { productId: number; quantity: number; notes?: string; patternId: number }) => {
        const payload = {
          productId: data.productId || 1,
          quantity: data.quantity,
          notes: data.notes,
          planId,
          sourceType: "cutting_pattern",
          metadata: {
            cutting_pattern_id: data.patternId,
            cutting_pattern_code: pattern.code,
            cutting_pattern_name: pattern.name,
            board_dimensions: `${pattern.board_length}×${pattern.board_width}×${pattern.board_thickness}`,
            total_items: pattern.items_count,
            total_quantity: pattern.total_quantity,
          },
        };
        
        return apiRequest("POST", `/api/production/planning/lines`, payload);
      },
      onSuccess: async () => {
        // Immediately refetch to get full product details
        await Promise.all([
          queryClient.refetchQueries({ queryKey: [`/api/production/planning/plans/${planId}/items`] }),
          queryClient.refetchQueries({ queryKey: [`/api/production/planning/plans/${planId}/available-cutting-patterns`] }),
        ]);
        
        toast({
          title: "Dodano do planu",
          description: `Rozkrój ${pattern.code} został dodany do planu produkcyjnego`,
        });
      },
      onError: (error: any) => {
        toast({
          title: "Błąd",
          description: error.message || "Nie udało się dodać rozkroju do planu",
          variant: "destructive",
        });
      },
    };

    // Execute mutation
    queryClient.getMutationCache().build(queryClient, patternLineMutation).execute({
      productId: 1,
      quantity: pattern.total_quantity,
      notes: `Rozkrój: ${pattern.name} (${pattern.code})`,
      patternId: pattern.id,
    });
    
    // Visual feedback
    setRecentlyAddedPatterns(prev => new Set(Array.from(prev).concat(pattern.id)));
    setTimeout(() => {
      setRecentlyAddedPatterns(prev => {
        const newSet = new Set(prev);
        newSet.delete(pattern.id);
        return newSet;
      });
    }, 2000);
  };

  const handleLinkItem = (item: MarketplaceOrderItem, marketplace: string) => {
    setLinkingItem({ item, marketplace });
  };

  // Handle highlight product in plan (from left orders panel)
  const handleHighlightInPlan = (productId: number, orderNumber: string) => {
    setHighlightedProductId(productId);
    setHighlightedOrderNumber(orderNumber);
    
    // Clear highlight after 3 seconds
    setTimeout(() => {
      setHighlightedProductId(null);
      setHighlightedOrderNumber(null);
    }, 3000);
  };

  const handleRemoveItem = (lineId: number) => {
    // Block removing when ZLPs have been approved/locked
    if (plan?.zlpLocked) {
      toast({
        title: "Plan jest zatwierdzony",
        description: "Nie można usuwać produktów po zatwierdzeniu zleceń produkcyjnych (ZLP). Odblokuj plan, aby edytować.",
        variant: "destructive"
      });
      return;
    }
    deleteLineMutation.mutate(lineId);
  };

  const handleRemoveItems = async (lineIds: number[]) => {
    // Block removing when ZLPs have been approved/locked
    if (plan?.zlpLocked) {
      toast({
        title: "Plan jest zatwierdzony",
        description: "Nie można usuwać produktów po zatwierdzeniu zleceń produkcyjnych (ZLP). Odblokuj plan, aby edytować.",
        variant: "destructive"
      });
      return;
    }
    setIsRemovingBulk(true);
    
    try {
      let successCount = 0;
      
      // Remove ALL items first (don't refetch between deletions)
      const deletePromises = lineIds.map(async (lineId) => {
        try {
          await apiRequest("DELETE", `/api/production/planning/lines/${lineId}`);
          return true;
        } catch (error: any) {
          // Ignore 404 errors (item already deleted)
          if (error.message?.includes('404')) {
            return true;
          }
          console.error(`Failed to delete line ${lineId}:`, error);
          return false;
        }
      });
      
      const results = await Promise.all(deletePromises);
      successCount = results.filter(r => r).length;
      
      // NOW refetch everything ONCE after all deletions (włącznie z rezerwacjami magazynowymi)
      // Używamy refetchType: 'all' aby wymusić odświeżenie wszystkich wariantów query (z różnymi filtrami)
      await Promise.all([
        queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}`], refetchType: 'all' }),
        queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/items`], refetchType: 'all' }),
        queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/available-orders`], refetchType: 'all' }),
        queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/available-catalog-products`], refetchType: 'all' }),
        queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/available-cutting-patterns`], refetchType: 'all' }),
      ]);
      
      if (successCount > 0) {
        toast({ title: `Usunięto ${successCount} produktów z planu` });
      }
    } catch (error: any) {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się usunąć produktów",
        variant: "destructive",
      });
    } finally {
      setIsRemovingBulk(false);
    }
  };

  // Drag & Drop handlers
  const handleDragStart = (event: DragStartEvent) => {
    const { active } = event;
    setActiveId(active.id as string);
    
    if (active.data.current?.type === 'order-item') {
      setDraggedItem(active.data.current.item);
    }
  };

  const handleDragEnd = (event: DragEndEvent) => {
    const { active, over } = event;
    
    setActiveId(null);
    setDraggedItem(null);

    if (!over || over.id !== 'plan-items-dropzone') {
      return;
    }

    // Block drag & drop when ZLPs have been approved/locked
    if (plan?.zlpLocked) {
      toast({
        title: "Plan jest zatwierdzony",
        description: "Nie można dodawać produktów po zatwierdzeniu zleceń produkcyjnych (ZLP). Odblokuj plan, aby edytować.",
        variant: "destructive"
      });
      return;
    }

    const dragData = active.data.current;
    if (dragData?.type === 'order-item') {
      const item = dragData.item as MarketplaceOrderItem;
      
      // Check if product has set components - cannot drag entire sets
      if (item.set_components && item.set_components.length > 0) {
        toast({
          title: "Nie można przeciągnąć całego zestawu",
          description: "Dodaj komponenty indywidualnie klikając przycisk przy każdym komponencie.",
          variant: "destructive"
        });
        return;
      }
      
      if (!item.catalog_product_id) {
        toast({
          title: "Nie można dodać",
          description: "Produkt nie jest zmapowany do katalogu.",
          variant: "destructive"
        });
        return;
      }

      createLineMutation.mutate({
        productId: item.catalog_product_id,
        quantity: item.quantity,
        notes: `Zamówienie: ${dragData.orderNumber}, Klient: ${dragData.buyerName}`,
        itemId: item.item_id,
        orderNumber: dragData.orderNumber,
        buyerName: dragData.buyerName,
        marketplace: dragData.marketplace,
        imageUrl: item.image_url,
      });

      toast({
        title: "Dodano do planu",
        description: `${item.name} × ${item.quantity}`,
      });
    }
  };

  if (planLoading) {
    return (
      <div className="flex items-center justify-center h-full">
        <Loader2 className="h-8 w-8 animate-spin" />
      </div>
    );
  }

  if (!plan) {
    return (
      <div className="p-8">
        <p>Plan produkcji nie został znaleziony</p>
      </div>
    );
  }

  return (
    <div className="flex flex-col h-full">
      {/* Dual-Panel Layout with DnD Context */}
      <div className="flex-1 min-h-0 p-6 overflow-hidden">
        <DndContext
          sensors={sensors}
          collisionDetection={closestCorners}
          onDragStart={handleDragStart}
          onDragEnd={handleDragEnd}
          modifiers={[restrictToWindowEdges]}
        >
          {/* Resizable Container - Flex on desktop, Stack on mobile */}
          <div 
            id="resizable-container"
            className="h-full flex flex-col lg:flex-row gap-0"
          >
            {/* Left Panel: Available Orders OR ZLP Dashboard when locked */}
            <div 
              className="h-1/2 lg:h-full overflow-hidden"
              style={{ 
                width: isDesktop ? `${leftPanelWidth}%` : '100%',
              }}
            >
              {plan.zlpLocked ? (
                <ZlpDashboardInline 
                  planId={Number(planId)} 
                  onUnlock={() => unlockZlpMutation.mutate()}
                  isUnlocking={unlockZlpMutation.isPending}
                />
              ) : (
                <AvailableSourcesPanel
                  planId={planId!.toString()}
                  // Customer Orders
                  orders={availableOrders}
                  ordersLoading={ordersLoading}
                  ordersStatistics={{
                    ordersCount: statistics.ordersCount,
                    productsCount: statistics.productsCount
                  }}
                  ordersFilters={orderFilters}
                  ordersSearchFilter={ordersSearchFilter}
                  onOrdersSearchChange={handleOrdersSearchChange}
                  onOrdersFilterChange={(key: string, value: string) => updateFilter(key as any, value as any)}
                  onOrdersClearFilter={(key: string) => clearFilter(key as any)}
                  onOrdersResetFilters={resetFilters}
                  onAddOrderItem={handleAddItem}
                  onLinkOrderItem={handleLinkItem}
                  ordersActiveFiltersCount={getActiveFilters().length}
                  isAddingItem={createLineMutation.isPending || pendingAddCount > 0 || isRefetchingAfterAdd || isBulkOperationActive}
                  addingItems={addingItems}
                  recentlyAddedItems={recentlyAddedItems}
                  onHighlightInPlan={handleHighlightInPlan}
                  // Catalog Products
                  catalogProducts={catalogProducts}
                  catalogLoading={catalogLoading}
                  onAddCatalogProduct={handleAddCatalogProduct}
                  recentlyAddedProducts={recentlyAddedProducts}
                  // Cutting Patterns
                  cuttingPatterns={cuttingPatterns}
                  patternsLoading={patternsLoading}
                  onAddCuttingPattern={handleAddCuttingPattern}
                  recentlyAddedPatterns={recentlyAddedPatterns}
                  // Shared
                  colors={colors}
                  productTypes={productTypes}
                  dimensionLengths={dimensionLengths}
                  dimensionWidths={dimensionWidths}
                  // Plan locked state (ZLPs generated)
                  isLocked={plan.zlpLocked || false}
                />
              )}
            </div>

            {/* Resizable Divider - Only on desktop */}
            <div 
              className="hidden lg:block w-1 bg-border hover:bg-purple-500 transition-colors cursor-col-resize relative group"
              onMouseDown={handleMouseDown}
              data-testid="resize-divider"
            >
              {/* Visual indicator */}
              <div className="absolute inset-y-0 left-1/2 -translate-x-1/2 w-1 bg-purple-500/0 group-hover:bg-purple-500/50 transition-colors" />
              {/* Wider hover area for easier grabbing */}
              <div className="absolute inset-y-0 -left-2 -right-2" />
            </div>

            {/* Right Panel: Plan Items - Flexible width */}
            <div 
              className="h-1/2 lg:h-full overflow-hidden"
              style={{ 
                width: isDesktop ? `${100 - leftPanelWidth}%` : '100%',
              }}
            >
              <PlanItemsPanel
                plan={plan}
                onBack={() => setLocation("/production/plans")}
                planId={planId!.toString()}
                planNumber={plan?.planNumber}
                statistics={statistics}
                searchFilter={planItemsSearchFilter}
                onSearchChange={setPlanItemsSearchFilter}
                isLoading={linesLoading}
                onRemoveItem={handleRemoveItem}
                onRemoveItems={handleRemoveItems}
                isRemoving={isRemovingBulk || deleteLineMutation.isPending}
                highlightedProductId={highlightedProductId}
                highlightedOrderNumber={highlightedOrderNumber}
                onOpenRoutingPanel={() => setRoutingPanelOpen(true)}
                onOpenZlpDashboard={() => setZlpDashboardOpen(true)}
                hasZlps={hasZlps}
                zlpLocked={plan.zlpLocked || false}
                onLockZlp={() => lockZlpMutation.mutate()}
                onUnlockZlp={() => unlockZlpMutation.mutate()}
                isLockPending={lockZlpMutation.isPending}
                isUnlockPending={unlockZlpMutation.isPending}
              />
            </div>
          </div>

          {/* Drag Overlay for visual feedback */}
          <DragOverlay>
            {activeId && draggedItem ? (
              <div className="bg-primary text-primary-foreground px-3 py-1 rounded shadow-lg text-sm flex items-center gap-2">
                <Package className="h-4 w-4" />
                <span>{draggedItem.name} × {draggedItem.quantity}</span>
              </div>
            ) : null}
          </DragOverlay>
        </DndContext>
      </div>

      {/* Marketplace Link Dialog */}
      {linkingItem && (
        <MarketplaceLinkDialog
          open={true}
          onOpenChange={(open) => {
            if (!open) setLinkingItem(null);
          }}
          mode="marketplace-to-catalog"
          marketplaceProduct={{
            id: linkingItem.item.marketplace_product_id || 0,
            externalId: linkingItem.item.offer_external_id,
            platform: linkingItem.marketplace.toLowerCase() as 'allegro' | 'shoper',
            sku: linkingItem.item.product_sku || '',
            name: linkingItem.item.name,
          }}
          onSuccess={() => {
            queryClient.invalidateQueries({
              queryKey: [`/api/production/planning/plans/${planId}/available-orders`]
            });
            setLinkingItem(null);
          }}
        />
      )}
      
      {/* Add Catalog Product Dialog */}
      <AddCatalogProductDialog
        open={catalogDialogOpen}
        onOpenChange={(open) => {
          setCatalogDialogOpen(open);
          if (!open) {
            setSelectedCatalogProduct(null);
            setIsCatalogMutationPending(false);
          }
        }}
        product={selectedCatalogProduct}
        onConfirm={handleConfirmCatalogProduct}
        isLoading={isCatalogMutationPending}
      />

      {/* BOM Formatki Modal */}
      <BomFormatkiModal />

      {/* Routing Side Panel */}
      <PlanRoutingSidePanel
        planId={plan.id}
        isOpen={routingPanelOpen}
        onOpenChange={setRoutingPanelOpen}
        hasZlps={hasZlps}
        zlpLocked={plan.zlpLocked || false}
        planLinesCount={lines.length}
      />

      {/* ZLP Dashboard Panel */}
      <ZlpDashboard
        planId={plan.id}
        isOpen={zlpDashboardOpen}
        onOpenChange={setZlpDashboardOpen}
      />
    </div>
  );
}
