import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { CheckCircle2, XCircle, RefreshCw, Upload, AlertCircle, ExternalLink, Calendar as CalendarIcon, Filter, ArrowUpDown } from "lucide-react";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { Link } from "wouter";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Calendar } from "@/components/ui/calendar";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { cn } from "@/lib/utils";

type Platform = 'allegro' | 'shoper' | 'odoo' | 'erli';
type DateRangeType = 'today' | 'yesterday' | 'day-before-yesterday' | 'last-7-days' | 'custom';
type ErrorFilter = 'all' | 'payment_mismatch' | 'fulfillment_mismatch' | 'missing_in_oms' | 'missing_in_odoo';
type SortBy = 'errors_first' | 'errors_last' | 'date_asc' | 'date_desc';

interface OrderMatch {
  sourceOrderId: string;
  platform: string;
  orderDate: string;
  inOMS: boolean;
  inOdoo: boolean;
  omsOrderId?: string;
  omsOrderNumber?: number;
  odooOrderId?: number;
  originalPlatform?: string;
  platformPaymentStatus?: string;
  omsPaymentStatus?: string;
  paymentMismatch?: boolean;
  platformFulfillmentStatus?: string;
  omsFulfillmentStatus?: string;
  fulfillmentMismatch?: boolean;
  platformPaymentAmount?: string;
  omsPaymentAmount?: string;
}

interface MatchResponse {
  platform: string;
  dateRange: string;
  startDate: string;
  endDate: string;
  summary: {
    total: number;
    inOMS: number;
    notInOMS: number;
    inOdoo: number;
    notInOdoo: number;
    mismatches: number;
    paymentMismatches?: number;
    fulfillmentMismatches?: number;
    filtered: number;
    truncated: boolean;
  };
  orders: OrderMatch[];
  filterApplied?: string;
  sortApplied?: string;
}

const getOdooOrderUrl = (odooOrderId: number, odooUserUrl: string): string => {
  const baseUrl = odooUserUrl.replace(/\/$/, '');
  return `${baseUrl}/sales/${odooOrderId}`;
};

export default function OrderMatching() {
  const [platform, setPlatform] = useState<Platform>('allegro');
  const [dateRangeType, setDateRangeType] = useState<DateRangeType>('last-7-days');
  const [customDateFrom, setCustomDateFrom] = useState<Date>();
  const [customDateTo, setCustomDateTo] = useState<Date>();
  const [errorFilter, setErrorFilter] = useState<ErrorFilter>('all');
  const [sortBy, setSortBy] = useState<SortBy>('date_desc');
  const { toast } = useToast();

  const { data: odooConfig } = useQuery<{ url: string; user_url?: string }>({
    queryKey: ['/api/odoo/config'],
  });

  const { data: shoperConfig } = useQuery<{ orders_url?: string }>({
    queryKey: ['/api/shoper/config'],
  });

  const getPlatformOrderUrl = (platform: string, orderId: string): string => {
    switch (platform.toLowerCase()) {
      case 'allegro':
        return `https://allegro.pl/moje-allegro/sprzedaz/zamowienia/${orderId}`;
      case 'shoper':
        const baseUrl = shoperConfig?.orders_url || 'https://alpmeb.pl/admin/orders/';
        return `${baseUrl}${orderId}`;
      default:
        return '#';
    }
  };

  const buildQueryParams = () => {
    const params = new URLSearchParams();
    params.append('platform', platform);
    params.append('errorFilter', errorFilter);
    params.append('sortBy', sortBy);
    
    if (dateRangeType === 'custom' && customDateFrom && customDateTo) {
      params.append('dateRange', 'custom');
      params.append('startDate', format(customDateFrom, 'yyyy-MM-dd'));
      params.append('endDate', format(customDateTo, 'yyyy-MM-dd'));
    } else {
      params.append('dateRange', dateRangeType);
    }
    
    return params.toString();
  };

  const { data, isLoading, refetch } = useQuery<MatchResponse>({
    queryKey: ['/api/orders/match', platform, dateRangeType, customDateFrom, customDateTo, errorFilter, sortBy],
    queryFn: async () => {
      const queryParams = buildQueryParams();
      const response = await fetch(`/api/orders/match?${queryParams}`);
      if (!response.ok) {
        throw new Error('Failed to fetch order matches');
      }
      return response.json();
    },
    enabled: dateRangeType !== 'custom' || !!(customDateFrom && customDateTo),
    refetchInterval: 60 * 60 * 1000, // Auto-refresh co godzinę
  });

  const handleSyncToOdoo = async (orderNumber: number) => {
    try {
      const response = await fetch(`/api/orders/by-number/${orderNumber}`);
      if (!response.ok) {
        throw new Error('Nie znaleziono zamówienia');
      }
      
      const order = await response.json();
      await apiRequest('POST', `/api/orders/${order.orderNumber}/sync-to-odoo`);
      
      toast({
        title: "Synchronizacja z Odoo rozpoczęta",
        description: `Zamówienie #${orderNumber} zostało dodane do kolejki synchronizacji Odoo`,
      });
      
      setTimeout(() => refetch(), 2000);
    } catch (error: any) {
      toast({
        title: "Błąd synchronizacji",
        description: error.message,
        variant: "destructive",
      });
    }
  };

  const handleFetchToOMS = async (sourceOrderId: string, platform: string) => {
    try {
      await apiRequest('POST', '/api/orders/fetch-from-platform', {
        sourceOrderId,
        platform
      });
      
      toast({
        title: "Pobieranie rozpoczęte",
        description: `Zamówienie z ${platform} zostało pobrane do OMS`,
      });
      
      setTimeout(() => refetch(), 2000);
    } catch (error: any) {
      toast({
        title: "Błąd pobierania",
        description: error.message,
        variant: "destructive",
      });
    }
  };

  const handleForceUpdate = async (sourceOrderId: string, platform: string) => {
    try {
      await apiRequest('POST', '/api/orders/force-update', {
        sourceOrderId,
        platform
      });
      
      toast({
        title: "Aktualizacja zakończona",
        description: `Status płatności został zaktualizowany z platformy ${platform}`,
      });
      
      setTimeout(() => refetch(), 1500);
    } catch (error: any) {
      toast({
        title: "Błąd aktualizacji",
        description: error.message,
        variant: "destructive",
      });
    }
  };

  const StatusIcon = ({ status }: { status: boolean }) => {
    return status ? (
      <CheckCircle2 className="w-4 h-4 text-green-500" data-testid="icon-check" />
    ) : (
      <XCircle className="w-4 h-4 text-red-500" data-testid="icon-x" />
    );
  };

  const formatAmount = (amount: string | undefined): string => {
    if (!amount) return '-';
    const num = parseFloat(amount);
    return new Intl.NumberFormat('pl-PL', {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(num);
  };

  const getStatusLabel = (status: string | undefined): string => {
    if (!status) return '-';
    const labels: Record<string, string> = {
      'READY_FOR_PROCESSING': 'Do realizacji',
      'SENT': 'Wysłano',
      'DELIVERED': 'Dostarczono',
      'CANCELLED': 'Anulowano',
      'paid': 'Opłacono',
      'unpaid': 'Nieopłacono',
      'PAID': 'Opłacono',
      'UNPAID': 'Nieopłacono',
      '1': 'Nowe',
      '2': 'W realizacji',
      '3': 'Wysłane',
      '4': 'Zrealizowane',
      '5': 'Anulowane',
      'NEW': 'Nowe',
    };
    return labels[status] || status;
  };

  const summary = data?.summary;

  // Oblicz całkowitą liczbę problemów (wszystkie rodzaje błędów)
  const totalIssues = summary 
    ? (summary.notInOMS || 0) + (summary.mismatches || 0) + (summary.paymentMismatches || 0) + (summary.fulfillmentMismatches || 0)
    : 0;

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold" data-testid="text-page-title">Weryfikacja Zamówień</h1>
          <p className="text-muted-foreground" data-testid="text-page-description">
            Sprawdź zgodność zamówień między platformami
          </p>
        </div>
        <div className="flex items-center gap-3">
          <div 
            className={cn(
              "flex items-center gap-2 px-3 py-2 rounded-md border",
              !summary 
                ? "bg-muted/50 border-muted"
                : totalIssues > 0 
                  ? "bg-red-500/10 border-red-500/30" 
                  : "bg-green-500/10 border-green-500/30"
            )}
            data-testid={!summary ? "alert-loading" : totalIssues > 0 ? "alert-issues" : "alert-ok"}
          >
            <div className={cn(
              "w-2 h-2 rounded-full",
              !summary 
                ? "bg-muted-foreground" 
                : totalIssues > 0 ? "bg-red-500" : "bg-green-500"
            )} />
            <div className="flex flex-col">
              <span className={cn(
                "text-sm font-semibold",
                !summary 
                  ? "text-muted-foreground"
                  : totalIssues > 0 ? "text-red-600" : "text-green-600"
              )}>
                {!summary 
                  ? 'Ładowanie...'
                  : totalIssues > 0 
                    ? `Wykryto ${totalIssues} ${totalIssues === 1 ? 'problem' : totalIssues < 5 ? 'problemy' : 'problemów'}`
                    : 'Brak problemów'
                }
              </span>
              <span className="text-xs text-muted-foreground">
                Odświeża się automatycznie co godzinę
              </span>
            </div>
          </div>
          <Button 
            onClick={() => refetch()} 
            variant="outline"
            data-testid="button-refresh"
          >
            <RefreshCw className="w-4 h-4 mr-2" />
            Odśwież
          </Button>
        </div>
      </div>

      {summary && (
        <div className="grid gap-4 md:grid-cols-6">
          <Card data-testid="card-total">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Wszystkie</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold" data-testid="text-total">{summary.total}</div>
            </CardContent>
          </Card>

          <Card data-testid="card-in-oms">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">W OMS</CardTitle>
              <CheckCircle2 className="w-4 h-4 text-green-500" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-green-600" data-testid="text-in-oms">{summary.inOMS}</div>
            </CardContent>
          </Card>

          <Card data-testid="card-not-in-oms">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Brak w OMS</CardTitle>
              <XCircle className="w-4 h-4 text-red-500" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-red-600" data-testid="text-not-in-oms">{summary.notInOMS}</div>
            </CardContent>
          </Card>

          <Card data-testid="card-in-odoo">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">W Odoo</CardTitle>
              <CheckCircle2 className="w-4 h-4 text-purple-500" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-purple-600" data-testid="text-in-odoo">{summary.inOdoo}</div>
            </CardContent>
          </Card>

          <Card data-testid="card-mismatches">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Niezgodności</CardTitle>
              <AlertCircle className="w-4 h-4 text-orange-500" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-orange-600" data-testid="text-mismatches">{summary.mismatches}</div>
            </CardContent>
          </Card>

          <Card data-testid="card-payment-mismatches">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Błędy płatności</CardTitle>
              <AlertCircle className="w-4 h-4 text-yellow-500" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold text-yellow-600" data-testid="text-payment-mismatches">
                {summary.paymentMismatches || 0}
              </div>
            </CardContent>
          </Card>
        </div>
      )}

      <Card>
        <CardHeader>
          <CardTitle data-testid="text-card-title">Porównanie Zamówień</CardTitle>
          <CardDescription data-testid="text-card-description">
            Wybierz platformę i zakres dat aby sprawdzić zgodność
          </CardDescription>
        </CardHeader>
        <CardContent>
          <Tabs value={platform} onValueChange={(v) => setPlatform(v as Platform)}>
            <div className="flex items-center justify-between mb-4 flex-wrap gap-3">
              <TabsList data-testid="tabs-platform">
                <TabsTrigger value="allegro" data-testid="tab-allegro">Allegro</TabsTrigger>
                <TabsTrigger value="shoper" data-testid="tab-shoper">Shoper</TabsTrigger>
                <TabsTrigger value="odoo" data-testid="tab-odoo">Odoo</TabsTrigger>
                <TabsTrigger value="erli" data-testid="tab-erli">Erli</TabsTrigger>
              </TabsList>

              <div className="flex gap-2 flex-wrap">
                <Button
                  size="sm"
                  variant={dateRangeType === 'today' ? 'default' : 'outline'}
                  onClick={() => setDateRangeType('today')}
                  data-testid="button-today"
                >
                  Dzisiaj
                </Button>
                <Button
                  size="sm"
                  variant={dateRangeType === 'yesterday' ? 'default' : 'outline'}
                  onClick={() => setDateRangeType('yesterday')}
                  data-testid="button-yesterday"
                >
                  Wczoraj
                </Button>
                <Button
                  size="sm"
                  variant={dateRangeType === 'day-before-yesterday' ? 'default' : 'outline'}
                  onClick={() => setDateRangeType('day-before-yesterday')}
                  data-testid="button-day-before-yesterday"
                >
                  Przedwczoraj
                </Button>
                <Button
                  size="sm"
                  variant={dateRangeType === 'last-7-days' ? 'default' : 'outline'}
                  onClick={() => setDateRangeType('last-7-days')}
                  data-testid="button-last-7-days"
                >
                  Ostatnie 7 dni
                </Button>
                
                <Popover>
                  <PopoverTrigger asChild>
                    <Button
                      size="sm"
                      variant={dateRangeType === 'custom' ? 'default' : 'outline'}
                      data-testid="button-custom-range"
                      className={cn(
                        "justify-start text-left font-normal",
                        !customDateFrom && !customDateTo && "text-muted-foreground"
                      )}
                    >
                      <CalendarIcon className="mr-2 h-4 w-4" />
                      {customDateFrom && customDateTo ? (
                        <>
                          {format(customDateFrom, "dd.MM.yy", { locale: pl })} - {format(customDateTo, "dd.MM.yy", { locale: pl })}
                        </>
                      ) : (
                        <span>Zakres dat</span>
                      )}
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-auto p-0" align="end">
                    <div className="p-3 space-y-3">
                      <div className="space-y-2">
                        <div className="text-sm font-medium">Data od:</div>
                        <Calendar
                          mode="single"
                          selected={customDateFrom}
                          onSelect={(date) => {
                            setCustomDateFrom(date);
                            setDateRangeType('custom');
                          }}
                          locale={pl}
                          disabled={(date) => date > new Date()}
                        />
                      </div>
                      <div className="space-y-2">
                        <div className="text-sm font-medium">Data do:</div>
                        <Calendar
                          mode="single"
                          selected={customDateTo}
                          onSelect={(date) => {
                            setCustomDateTo(date);
                            setDateRangeType('custom');
                          }}
                          locale={pl}
                          disabled={(date) => date > new Date() || (customDateFrom ? date < customDateFrom : false)}
                        />
                      </div>
                    </div>
                  </PopoverContent>
                </Popover>
              </div>
            </div>

            {/* Filtry i sortowanie */}
            <div className="flex items-center gap-3 mb-4 flex-wrap">
              <div className="flex items-center gap-2">
                <Filter className="w-4 h-4 text-muted-foreground" />
                <Select value={errorFilter} onValueChange={(v) => setErrorFilter(v as ErrorFilter)}>
                  <SelectTrigger className="w-[220px]" data-testid="select-error-filter">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all" data-testid="filter-all">Wszystkie zamówienia</SelectItem>
                    <SelectItem value="missing_in_oms" data-testid="filter-missing-oms">Brak w OMS</SelectItem>
                    <SelectItem value="missing_in_odoo" data-testid="filter-missing-odoo">Brak w Odoo</SelectItem>
                    <SelectItem value="payment_mismatch" data-testid="filter-payment">Błędy płatności</SelectItem>
                    <SelectItem value="fulfillment_mismatch" data-testid="filter-fulfillment">Błędy realizacji</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="flex items-center gap-2">
                <ArrowUpDown className="w-4 h-4 text-muted-foreground" />
                <Select value={sortBy} onValueChange={(v) => setSortBy(v as SortBy)}>
                  <SelectTrigger className="w-[200px]" data-testid="select-sort">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="errors_first" data-testid="sort-errors-first">Błędy najpierw</SelectItem>
                    <SelectItem value="errors_last" data-testid="sort-errors-last">Błędy na końcu</SelectItem>
                    <SelectItem value="date_desc" data-testid="sort-date-desc">Data: najnowsze</SelectItem>
                    <SelectItem value="date_asc" data-testid="sort-date-asc">Data: najstarsze</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              {summary && summary.truncated && (
                <Badge variant="secondary" className="ml-auto" data-testid="badge-truncated">
                  <AlertCircle className="w-3 h-3 mr-1" />
                  Pokazano {summary.filtered} z wielu wyników (limit: 100)
                </Badge>
              )}
            </div>

            <TabsContent value={platform} className="mt-0">
              {isLoading ? (
                <div className="text-center py-8 text-muted-foreground" data-testid="text-loading">
                  Ładowanie...
                </div>
              ) : data?.orders && data.orders.length > 0 ? (
                <div className="border rounded-md">
                  <Table>
                    <TableHeader>
                      <TableRow className="bg-muted/50">
                        <TableHead className="h-9 px-2 text-xs font-semibold" data-testid="header-order-id">ID Zamówienia</TableHead>
                        <TableHead className="h-9 px-2 text-xs font-semibold" data-testid="header-date">Data</TableHead>
                        <TableHead className="h-9 px-2 text-xs font-semibold" data-testid="header-oms-code">Kod OMS</TableHead>
                        <TableHead className="h-9 px-2 text-xs font-semibold" data-testid="header-in-oms">W OMS</TableHead>
                        <TableHead className="h-9 px-2 text-xs font-semibold" data-testid="header-in-odoo">W Odoo</TableHead>
                        <TableHead className="h-9 px-2 text-xs font-semibold" data-testid="header-payment">Płatność</TableHead>
                        <TableHead className="h-9 px-2 text-xs font-semibold" data-testid="header-odoo-id">ID Odoo</TableHead>
                        <TableHead className="h-9 px-2 text-xs font-semibold text-right" data-testid="header-actions">Akcje</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {data.orders.map((order, idx) => {
                        const platformUrl = getPlatformOrderUrl(order.platform, order.sourceOrderId);
                        const odooUserUrl = odooConfig?.user_url || odooConfig?.url;
                        const odooUrl = order.odooOrderId && odooUserUrl 
                          ? getOdooOrderUrl(order.odooOrderId, odooUserUrl)
                          : null;

                        return (
                          <TableRow key={idx} data-testid={`row-order-${idx}`} className="h-10 hover:bg-muted/30">
                            <TableCell className="py-1 px-2 text-xs font-mono" data-testid={`cell-order-id-${idx}`}>
                              <a
                                href={platformUrl}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="text-primary hover:underline inline-flex items-center gap-1"
                                data-testid={`link-platform-${idx}`}
                              >
                                {order.sourceOrderId}
                                <ExternalLink className="w-3 h-3" />
                              </a>
                            </TableCell>
                            <TableCell className="py-1 px-2 text-xs whitespace-nowrap" data-testid={`cell-date-${idx}`}>
                              {order.orderDate && !isNaN(new Date(order.orderDate).getTime()) 
                                ? format(new Date(order.orderDate), "dd.MM.yy HH:mm", { locale: pl })
                                : '-'
                              }
                            </TableCell>
                            <TableCell className="py-1 px-2 text-xs" data-testid={`cell-oms-code-${idx}`}>
                              {order.omsOrderNumber && order.omsOrderId ? (
                                <Link href={`/order/${order.omsOrderNumber}`} data-testid={`link-oms-${idx}`}>
                                  <Badge 
                                    variant="outline" 
                                    className="hover:bg-accent cursor-pointer h-5 text-xs px-1.5"
                                    data-testid={`badge-oms-${idx}`}
                                  >
                                    {order.omsOrderNumber}
                                  </Badge>
                                </Link>
                              ) : (
                                <span className="text-muted-foreground">-</span>
                              )}
                            </TableCell>
                            <TableCell className="py-1 px-2 text-xs" data-testid={`cell-in-oms-${idx}`}>
                              <div className="flex items-center gap-1.5">
                                <StatusIcon status={order.inOMS} />
                                {order.fulfillmentMismatch && order.platformFulfillmentStatus && order.omsFulfillmentStatus && (
                                  <div className="text-xs text-muted-foreground" title={`Platforma: ${order.platformFulfillmentStatus}, OMS: ${order.omsFulfillmentStatus}`}>
                                    <span className="text-orange-600">{getStatusLabel(order.platformFulfillmentStatus)}</span>
                                    {' → '}
                                    <span className="text-blue-600">{getStatusLabel(order.omsFulfillmentStatus)}</span>
                                  </div>
                                )}
                              </div>
                            </TableCell>
                            <TableCell className="py-1 px-2 text-xs" data-testid={`cell-in-odoo-${idx}`}>
                              <div className="flex items-center gap-1.5">
                                <StatusIcon status={order.inOdoo} />
                                {order.inOdoo && order.omsFulfillmentStatus && (
                                  <span className="text-xs text-muted-foreground">
                                    {getStatusLabel(order.omsFulfillmentStatus)}
                                  </span>
                                )}
                              </div>
                            </TableCell>
                            <TableCell className="py-1 px-2 text-xs" data-testid={`cell-payment-${idx}`}>
                              {order.paymentMismatch ? (
                                <div className="flex items-center gap-1.5" title={`Platforma: ${order.platformPaymentStatus}, OMS: ${order.omsPaymentStatus}`}>
                                  <AlertCircle className="w-4 h-4 text-yellow-500" data-testid={`icon-payment-mismatch-${idx}`} />
                                  <div className="flex flex-col">
                                    <span className="text-xs text-yellow-600 font-medium">Niezgodność</span>
                                    {(order.platformPaymentAmount || order.omsPaymentAmount) && (
                                      <span className="text-xs text-muted-foreground">
                                        {formatAmount(order.platformPaymentAmount || order.omsPaymentAmount)} zł
                                      </span>
                                    )}
                                  </div>
                                </div>
                              ) : order.inOMS ? (
                                <div className="flex items-center gap-1.5">
                                  <CheckCircle2 className="w-4 h-4 text-green-500" data-testid={`icon-payment-ok-${idx}`} />
                                  <div className="flex flex-col">
                                    <span className="text-xs text-green-600 font-medium">{getStatusLabel(order.omsPaymentStatus)}</span>
                                    {(order.omsPaymentAmount || order.platformPaymentAmount) && (
                                      <span className="text-xs text-muted-foreground">
                                        {formatAmount(order.omsPaymentAmount || order.platformPaymentAmount)} zł
                                      </span>
                                    )}
                                  </div>
                                </div>
                              ) : (
                                <span className="text-muted-foreground">-</span>
                              )}
                            </TableCell>
                            <TableCell className="py-1 px-2 text-xs" data-testid={`cell-odoo-id-${idx}`}>
                              {order.odooOrderId ? (
                                odooUrl ? (
                                  <a
                                    href={odooUrl}
                                    target="_blank"
                                    rel="noopener noreferrer"
                                    className="inline-flex items-center gap-1"
                                    data-testid={`link-odoo-${idx}`}
                                  >
                                    <Badge variant="secondary" className="h-5 text-xs px-1.5" data-testid={`badge-odoo-${idx}`}>
                                      {order.odooOrderId}
                                    </Badge>
                                    <ExternalLink className="w-3 h-3 text-muted-foreground" />
                                  </a>
                                ) : (
                                  <Badge variant="secondary" className="h-5 text-xs px-1.5" data-testid={`badge-odoo-${idx}`}>
                                    {order.odooOrderId}
                                  </Badge>
                                )
                              ) : (
                                <span className="text-muted-foreground">-</span>
                              )}
                            </TableCell>
                            <TableCell className="py-1 px-2 text-right" data-testid={`cell-actions-${idx}`}>
                              <div className="flex justify-end gap-1">
                                {!order.inOMS ? (
                                  <Button
                                    size="sm"
                                    variant="outline"
                                    className="h-7 px-2 text-xs"
                                    onClick={() => handleFetchToOMS(order.sourceOrderId, order.platform)}
                                    data-testid={`button-fetch-${idx}`}
                                  >
                                    <Upload className="w-3 h-3 mr-1" />
                                    Pobierz
                                  </Button>
                                ) : (
                                  <>
                                    {order.paymentMismatch && (
                                      <Button
                                        size="sm"
                                        variant="outline"
                                        className="h-7 px-2 text-xs text-yellow-600 hover:text-yellow-700"
                                        onClick={() => handleForceUpdate(order.sourceOrderId, order.platform)}
                                        data-testid={`button-update-payment-${idx}`}
                                      >
                                        <RefreshCw className="w-3 h-3 mr-1" />
                                        Aktualizuj
                                      </Button>
                                    )}
                                    {order.inOMS && !order.inOdoo && order.omsOrderNumber && (
                                      <Button
                                        size="sm"
                                        variant="outline"
                                        className="h-7 px-2 text-xs"
                                        onClick={() => handleSyncToOdoo(order.omsOrderNumber!)}
                                        data-testid={`button-sync-${idx}`}
                                      >
                                        <Upload className="w-3 h-3 mr-1" />
                                        Synchronizuj
                                      </Button>
                                    )}
                                  </>
                                )}
                              </div>
                            </TableCell>
                          </TableRow>
                        );
                      })}
                    </TableBody>
                  </Table>
                </div>
              ) : (
                <div className="text-center py-8 text-muted-foreground" data-testid="text-no-orders">
                  Brak zamówień dla wybranych kryteriów
                </div>
              )}
            </TabsContent>
          </Tabs>
        </CardContent>
      </Card>
    </div>
  );
}
