import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Skeleton } from "@/components/ui/skeleton";
import { Button } from "@/components/ui/button";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Calendar } from "@/components/ui/calendar";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, Calendar as CalendarIcon, ArrowUpDown } from "lucide-react";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { cn } from "@/lib/utils";

type DateRangeType = 'all' | 'today' | 'yesterday' | 'day-before-yesterday' | 'custom';
type SortField = 'occurred_at' | 'category' | 'fee_type' | 'fee_name' | 'amount';

export default function FeesTable() {
  const [page, setPage] = useState(0);
  const [pageSize, setPageSize] = useState(50);
  const [category, setCategory] = useState<string>("all");
  const [dateRange, setDateRange] = useState<DateRangeType>('all');
  const [customDateFrom, setCustomDateFrom] = useState<Date>();
  const [customDateTo, setCustomDateTo] = useState<Date>();
  const [sortBy, setSortBy] = useState<SortField>('occurred_at');
  const [sortOrder, setSortOrder] = useState<'ASC' | 'DESC'>('DESC');

  const buildQueryParams = () => {
    const params: Record<string, any> = {
      offset: page * pageSize,
      limit: pageSize,
      sortBy,
      sortOrder,
    };

    if (category !== 'all') {
      params.category = category;
    }

    if (dateRange === 'custom' && customDateFrom && customDateTo) {
      params.dateRange = 'custom';
      params.startDate = format(customDateFrom, 'yyyy-MM-dd');
      params.endDate = format(customDateTo, 'yyyy-MM-dd');
    } else if (dateRange !== 'all') {
      params.dateRange = dateRange;
    }

    return params;
  };

  const { data, isLoading } = useQuery<any>({
    queryKey: ['/api/allegro/fees/history', buildQueryParams()],
    enabled: dateRange !== 'custom' || !!(customDateFrom && customDateTo),
  });

  const handleSort = (field: SortField) => {
    if (sortBy === field) {
      setSortOrder(sortOrder === 'ASC' ? 'DESC' : 'ASC');
    } else {
      setSortBy(field);
      setSortOrder('DESC');
    }
    setPage(0);
  };

  const formatAmount = (amount: string | number) => {
    const num = typeof amount === 'string' ? parseFloat(amount) : amount;
    return new Intl.NumberFormat('pl-PL', {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(num);
  };

  const formatDateTime = (dateStr: string) => {
    return new Date(dateStr).toLocaleString('pl-PL', {
      year: 'numeric',
      month: '2-digit',
      day: '2-digit',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const getCategoryLabel = (cat: string) => {
    const labels: Record<string, string> = {
      commission: 'Prowizja',
      ads: 'Reklama',
      delivery: 'Dostawa',
      refund: 'Zwrot',
      listing: 'Wystawienie',
      other: 'Inne',
    };
    return labels[cat] || cat;
  };

  const getCategoryColor = (cat: string) => {
    const colors: Record<string, string> = {
      commission: 'text-red-600 dark:text-red-400',
      ads: 'text-purple-600 dark:text-purple-400',
      delivery: 'text-blue-600 dark:text-blue-400',
      refund: 'text-orange-600 dark:text-orange-400',
      listing: 'text-green-600 dark:text-green-400',
      other: 'text-gray-600 dark:text-gray-400',
    };
    return colors[cat] || 'text-gray-600';
  };

  const totalPages = data?.total ? Math.ceil(data.total / pageSize) : 0;

  const SortableHeader = ({ field, children }: { field: SortField; children: React.ReactNode }) => (
    <TableHead className="h-8 px-2 text-xs font-semibold cursor-pointer hover:bg-muted/50" onClick={() => handleSort(field)}>
      <div className="flex items-center gap-1">
        {children}
        {sortBy === field && (
          <ArrowUpDown className={cn("w-3 h-3", sortOrder === 'DESC' ? 'rotate-180' : '')} />
        )}
      </div>
    </TableHead>
  );

  const PaginationControls = () => (
    <div className="flex items-center justify-between gap-4">
      <div className="flex items-center gap-2">
        <span className="text-xs text-muted-foreground whitespace-nowrap">Wyników na stronę:</span>
        <Select value={pageSize.toString()} onValueChange={(v) => { setPageSize(parseInt(v)); setPage(0); }}>
          <SelectTrigger className="h-7 w-20 text-xs" data-testid="select-page-size">
            <SelectValue />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="25">25</SelectItem>
            <SelectItem value="50">50</SelectItem>
            <SelectItem value="250">250</SelectItem>
            <SelectItem value="500">500</SelectItem>
          </SelectContent>
        </Select>
      </div>

      <div className="flex items-center gap-2">
        <span className="text-xs text-muted-foreground whitespace-nowrap">
          Strona {page + 1} z {totalPages || 1}
        </span>
        <div className="flex gap-1">
          <Button
            variant="outline"
            size="sm"
            className="h-7 w-7 p-0"
            onClick={() => setPage(0)}
            disabled={page === 0}
            data-testid="button-first-page"
          >
            <ChevronsLeft className="h-3 w-3" />
          </Button>
          <Button
            variant="outline"
            size="sm"
            className="h-7 w-7 p-0"
            onClick={() => setPage(p => Math.max(0, p - 1))}
            disabled={page === 0}
            data-testid="button-prev-page"
          >
            <ChevronLeft className="h-3 w-3" />
          </Button>
          <Button
            variant="outline"
            size="sm"
            className="h-7 w-7 p-0"
            onClick={() => setPage(p => Math.min(totalPages - 1, p + 1))}
            disabled={page >= totalPages - 1 || totalPages === 0}
            data-testid="button-next-page"
          >
            <ChevronRight className="h-3 w-3" />
          </Button>
          <Button
            variant="outline"
            size="sm"
            className="h-7 w-7 p-0"
            onClick={() => setPage(totalPages - 1)}
            disabled={page >= totalPages - 1 || totalPages === 0}
            data-testid="button-last-page"
          >
            <ChevronsRight className="h-3 w-3" />
          </Button>
        </div>
      </div>
    </div>
  );

  return (
    <div className="p-6 space-y-4">
      <div>
        <h1 className="text-3xl font-bold" data-testid="heading-fees-table">Tabela opłat</h1>
        <p className="text-sm text-muted-foreground">Szczegółowa lista wszystkich opłat</p>
      </div>

      <Card className="p-4">
        <div className="space-y-3">
          {/* Filtry dat */}
          <div className="flex items-center gap-2 flex-wrap">
            <span className="text-xs font-medium text-muted-foreground">Zakres dat:</span>
            <Button
              size="sm"
              variant={dateRange === 'all' ? 'default' : 'outline'}
              onClick={() => setDateRange('all')}
              className="h-7 px-3 text-xs"
              data-testid="button-date-all"
            >
              Wszystkie
            </Button>
            <Button
              size="sm"
              variant={dateRange === 'today' ? 'default' : 'outline'}
              onClick={() => setDateRange('today')}
              className="h-7 px-3 text-xs"
              data-testid="button-date-today"
            >
              Dzisiaj
            </Button>
            <Button
              size="sm"
              variant={dateRange === 'yesterday' ? 'default' : 'outline'}
              onClick={() => setDateRange('yesterday')}
              className="h-7 px-3 text-xs"
              data-testid="button-date-yesterday"
            >
              Wczoraj
            </Button>
            <Button
              size="sm"
              variant={dateRange === 'day-before-yesterday' ? 'default' : 'outline'}
              onClick={() => setDateRange('day-before-yesterday')}
              className="h-7 px-3 text-xs"
              data-testid="button-date-day-before-yesterday"
            >
              Przedwczoraj
            </Button>

            <Popover>
              <PopoverTrigger asChild>
                <Button
                  size="sm"
                  variant={dateRange === 'custom' ? 'default' : 'outline'}
                  className="h-7 px-3 text-xs"
                  data-testid="button-date-custom"
                >
                  <CalendarIcon className="mr-1.5 h-3 w-3" />
                  {customDateFrom && customDateTo ? (
                    <>
                      {format(customDateFrom, "dd.MM.yy", { locale: pl })} - {format(customDateTo, "dd.MM.yy", { locale: pl })}
                    </>
                  ) : (
                    <span>Zakres dat</span>
                  )}
                </Button>
              </PopoverTrigger>
              <PopoverContent className="w-auto p-0" align="start">
                <div className="p-3 space-y-3">
                  <div className="space-y-2">
                    <div className="text-sm font-medium">Data od:</div>
                    <Calendar
                      mode="single"
                      selected={customDateFrom}
                      onSelect={(date) => {
                        setCustomDateFrom(date);
                        setDateRange('custom');
                      }}
                      locale={pl}
                      disabled={(date) => date > new Date()}
                    />
                  </div>
                  <div className="space-y-2">
                    <div className="text-sm font-medium">Data do:</div>
                    <Calendar
                      mode="single"
                      selected={customDateTo}
                      onSelect={(date) => {
                        setCustomDateTo(date);
                        setDateRange('custom');
                      }}
                      locale={pl}
                      disabled={(date) => date > new Date() || (customDateFrom ? date < customDateFrom : false)}
                    />
                  </div>
                </div>
              </PopoverContent>
            </Popover>
          </div>

          {/* Filtry kategorii */}
          <div className="flex items-center gap-2 flex-wrap">
            <span className="text-xs font-medium text-muted-foreground">Kategoria:</span>
            <Button
              size="sm"
              variant={category === 'all' ? 'default' : 'outline'}
              onClick={() => { setCategory('all'); setPage(0); }}
              className="h-7 px-3 text-xs"
              data-testid="button-category-all"
            >
              Wszystkie
            </Button>
            <Button
              size="sm"
              variant={category === 'commission' ? 'default' : 'outline'}
              onClick={() => { setCategory('commission'); setPage(0); }}
              className="h-7 px-3 text-xs"
              data-testid="button-category-commission"
            >
              🔴 Prowizje
            </Button>
            <Button
              size="sm"
              variant={category === 'ads' ? 'default' : 'outline'}
              onClick={() => { setCategory('ads'); setPage(0); }}
              className="h-7 px-3 text-xs"
              data-testid="button-category-ads"
            >
              🟣 Reklamy
            </Button>
            <Button
              size="sm"
              variant={category === 'delivery' ? 'default' : 'outline'}
              onClick={() => { setCategory('delivery'); setPage(0); }}
              className="h-7 px-3 text-xs"
              data-testid="button-category-delivery"
            >
              🔵 Dostawa
            </Button>
            <Button
              size="sm"
              variant={category === 'refund' ? 'default' : 'outline'}
              onClick={() => { setCategory('refund'); setPage(0); }}
              className="h-7 px-3 text-xs"
              data-testid="button-category-refund"
            >
              🟠 Zwroty
            </Button>
            <Button
              size="sm"
              variant={category === 'listing' ? 'default' : 'outline'}
              onClick={() => { setCategory('listing'); setPage(0); }}
              className="h-7 px-3 text-xs"
              data-testid="button-category-listing"
            >
              🟢 Wystawienie
            </Button>
            <Button
              size="sm"
              variant={category === 'other' ? 'default' : 'outline'}
              onClick={() => { setCategory('other'); setPage(0); }}
              className="h-7 px-3 text-xs"
              data-testid="button-category-other"
            >
              ⚪ Inne
            </Button>
          </div>

          {/* Paginacja górna */}
          <div className="pt-2 border-t">
            <PaginationControls />
          </div>
        </div>
      </Card>

      {/* Podsumowanie */}
      {!isLoading && data && (
        <div className="flex items-center gap-4 text-sm">
          <span className="text-muted-foreground">
            Wyświetlono <span className="font-semibold text-foreground">{data.fees?.length || 0}</span> z <span className="font-semibold text-foreground">{data.total || 0}</span> opłat
          </span>
          <span className="text-muted-foreground">
            Suma: <span className="font-bold text-foreground">{formatAmount(data.totalAmount || 0)} PLN</span>
          </span>
        </div>
      )}

      {/* Tabela */}
      <div className="border rounded-md">
        {isLoading ? (
          <div className="p-4 space-y-2">
            {[...Array(10)].map((_, i) => (
              <Skeleton key={i} className="h-8 w-full" />
            ))}
          </div>
        ) : (
          <Table>
            <TableHeader>
              <TableRow className="bg-muted/50">
                <SortableHeader field="occurred_at">Data</SortableHeader>
                <SortableHeader field="category">Kategoria</SortableHeader>
                <SortableHeader field="fee_type">Typ</SortableHeader>
                <SortableHeader field="fee_name">Nazwa</SortableHeader>
                <SortableHeader field="amount">Kwota</SortableHeader>
              </TableRow>
            </TableHeader>
            <TableBody>
              {data?.fees?.map((fee: any) => (
                <TableRow key={fee.id} data-testid={`row-fee-${fee.id}`} className="h-9 hover:bg-muted/30">
                  <TableCell className="py-1 px-2 text-xs whitespace-nowrap">
                    {formatDateTime(fee.occurred_at)}
                  </TableCell>
                  <TableCell className="py-1 px-2 text-xs">
                    <span className={`font-medium ${getCategoryColor(fee.category)}`}>
                      {getCategoryLabel(fee.category)}
                    </span>
                  </TableCell>
                  <TableCell className="py-1 px-2 font-mono text-xs">{fee.type_id}</TableCell>
                  <TableCell className="py-1 px-2 text-xs max-w-md truncate" title={fee.type_name}>
                    {fee.type_name}
                  </TableCell>
                  <TableCell className="py-1 px-2 text-xs font-semibold text-right whitespace-nowrap">
                    {formatAmount(fee.amount)} {fee.currency}
                  </TableCell>
                </TableRow>
              ))}
              {data?.fees?.length === 0 && (
                <TableRow>
                  <TableCell colSpan={5} className="text-center text-muted-foreground py-8 text-sm">
                    Brak opłat do wyświetlenia
                  </TableCell>
                </TableRow>
              )}
            </TableBody>
          </Table>
        )}
      </div>

      {/* Paginacja dolna */}
      {!isLoading && totalPages > 0 && (
        <Card className="p-3">
          <PaginationControls />
        </Card>
      )}
    </div>
  );
}
