import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";
import { RefreshCw, TrendingDown, Megaphone, Truck, RotateCcw, DollarSign } from "lucide-react";
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer, LineChart, Line } from "recharts";
import { DateRangeFilter, type DateRangePreset, getDateRangeFromPreset, presetToDays } from "@/components/date-range-filter";

export default function FeesSummary() {
  const { toast } = useToast();
  const [dateRange, setDateRange] = useState<DateRangePreset>("last30days");

  const range = getDateRangeFromPreset(dateRange);
  const days = presetToDays(dateRange);

  const { data: summaries, isLoading: summariesLoading } = useQuery<any[]>({
    queryKey: [`/api/allegro/fees/summaries?from=${range.from}&to=${range.to}`],
  });

  const syncMutation = useMutation({
    mutationFn: async () => {
      const response = await fetch('/api/allegro/fees/sync', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'include',
        body: JSON.stringify({ days: Math.max(days, 90) })
      });
      if (!response.ok) throw new Error('Sync failed');
      return response.json();
    },
    onSuccess: (data: any) => {
      queryClient.invalidateQueries({ queryKey: ['/api/allegro/fees/summaries'] });
      queryClient.invalidateQueries({ queryKey: ['/api/allegro/fees/history'] });
      queryClient.invalidateQueries({ queryKey: ['/api/allegro/fees/grouped'] });
      toast({
        title: "Synchronizacja zakończona",
        description: `Pobrano ${data.totalFetched} wpisów, zapisano ${data.totalSaved} opłat`,
      });
    },
    onError: () => {
      toast({
        title: "Błąd synchronizacji",
        description: "Nie udało się zsynchronizować opłat",
        variant: "destructive",
      });
    },
  });

  const formatAmount = (amount: string | number) => {
    const num = typeof amount === 'string' ? parseFloat(amount) : amount;
    return new Intl.NumberFormat('pl-PL', {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(num);
  };

  const formatDate = (dateStr: string) => {
    return new Date(dateStr).toLocaleDateString('pl-PL', { 
      day: '2-digit', 
      month: 'short' 
    });
  };

  const totalStats = summaries?.reduce((acc, summary) => ({
    commission: acc.commission + parseFloat(summary.commission_total || '0'),
    ads: acc.ads + parseFloat(summary.ads_total || '0'),
    delivery: acc.delivery + parseFloat(summary.delivery_total || '0'),
    refunds: acc.refunds + parseFloat(summary.refunds_total || '0'),
    other: acc.other + parseFloat(summary.other_total || '0'),
    grand: acc.grand + parseFloat(summary.grand_total || '0'),
  }), {
    commission: 0,
    ads: 0,
    delivery: 0,
    refunds: 0,
    other: 0,
    grand: 0,
  });

  const chartData = summaries?.slice().reverse().map(s => ({
    date: formatDate(s.summary_date),
    Prowizje: parseFloat(s.commission_total || '0'),
    Reklamy: parseFloat(s.ads_total || '0'),
    Dostawa: parseFloat(s.delivery_total || '0'),
    Zwroty: parseFloat(s.refunds_total || '0'),
  }));

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold" data-testid="heading-fees-summary">Zestawienia opłat</h1>
          <p className="text-muted-foreground">Podsumowanie opłat Allegro</p>
        </div>
        <Button
          onClick={() => syncMutation.mutate()}
          disabled={syncMutation.isPending}
          data-testid="button-sync-fees"
        >
          {syncMutation.isPending ? (
            <>
              <RefreshCw className="h-4 w-4 mr-2 animate-spin" />
              Synchronizacja...
            </>
          ) : (
            <>
              <RefreshCw className="h-4 w-4 mr-2" />
              Synchronizuj
            </>
          )}
        </Button>
      </div>

      {/* Date range filter */}
      <Card className="bg-muted/30">
        <CardContent className="pt-6">
          <DateRangeFilter value={dateRange} onChange={setDateRange} />
        </CardContent>
      </Card>

      {summariesLoading ? (
        <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-4">
          {[...Array(6)].map((_, i) => (
            <Card key={i}>
              <CardHeader>
                <Skeleton className="h-4 w-24" />
              </CardHeader>
              <CardContent>
                <Skeleton className="h-8 w-32" />
              </CardContent>
            </Card>
          ))}
        </div>
      ) : (
        <>
          <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Prowizje</CardTitle>
                <TrendingDown className="h-4 w-4 text-red-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{formatAmount(totalStats?.commission || 0)} zł</div>
                <p className="text-xs text-muted-foreground">SUC - Prowizja od sprzedaży</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Reklamy</CardTitle>
                <Megaphone className="h-4 w-4 text-purple-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{formatAmount(totalStats?.ads || 0)} zł</div>
                <p className="text-xs text-muted-foreground">NSP, ADS, PRO, FHB, FSF</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Dostawa</CardTitle>
                <Truck className="h-4 w-4 text-blue-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{formatAmount(totalStats?.delivery || 0)} zł</div>
                <p className="text-xs text-muted-foreground">HB1, DEL, DPD, SHP, DTR</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Zwroty</CardTitle>
                <RotateCcw className="h-4 w-4 text-orange-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{formatAmount(totalStats?.refunds || 0)} zł</div>
                <p className="text-xs text-muted-foreground">REF, RET - Zwroty płatności</p>
              </CardContent>
            </Card>

            <Card>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Inne</CardTitle>
                <DollarSign className="h-4 w-4 text-gray-600" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{formatAmount(totalStats?.other || 0)} zł</div>
                <p className="text-xs text-muted-foreground">LIS, PAD i pozostałe</p>
              </CardContent>
            </Card>

            <Card className="bg-primary/5 border-primary">
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium">Suma</CardTitle>
                <TrendingDown className="h-4 w-4 text-primary" />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold text-primary">{formatAmount(totalStats?.grand || 0)} zł</div>
                <p className="text-xs text-muted-foreground">Wszystkie opłaty razem</p>
              </CardContent>
            </Card>
          </div>

          <div className="grid gap-6 lg:grid-cols-2">
            <Card>
              <CardHeader>
                <CardTitle>Opłaty dzienne - słupki</CardTitle>
                <CardDescription>Podział na kategorie</CardDescription>
              </CardHeader>
              <CardContent>
                <ResponsiveContainer width="100%" height={300}>
                  <BarChart data={chartData}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="date" />
                    <YAxis />
                    <Tooltip formatter={(value: any) => `${formatAmount(value)} zł`} />
                    <Legend />
                    <Bar dataKey="Prowizje" fill="#ef4444" />
                    <Bar dataKey="Reklamy" fill="#a855f7" />
                    <Bar dataKey="Dostawa" fill="#3b82f6" />
                    <Bar dataKey="Zwroty" fill="#f97316" />
                  </BarChart>
                </ResponsiveContainer>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Trend opłat - linie</CardTitle>
                <CardDescription>Zmiana w czasie</CardDescription>
              </CardHeader>
              <CardContent>
                <ResponsiveContainer width="100%" height={300}>
                  <LineChart data={chartData}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="date" />
                    <YAxis />
                    <Tooltip formatter={(value: any) => `${formatAmount(value)} zł`} />
                    <Legend />
                    <Line type="monotone" dataKey="Prowizje" stroke="#ef4444" strokeWidth={2} />
                    <Line type="monotone" dataKey="Reklamy" stroke="#a855f7" strokeWidth={2} />
                    <Line type="monotone" dataKey="Dostawa" stroke="#3b82f6" strokeWidth={2} />
                    <Line type="monotone" dataKey="Zwroty" stroke="#f97316" strokeWidth={2} />
                  </LineChart>
                </ResponsiveContainer>
              </CardContent>
            </Card>
          </div>
        </>
      )}
    </div>
  );
}
