import { QueryClient, QueryFunction } from "@tanstack/react-query";

async function throwIfResNotOk(res: Response) {
  if (!res.ok) {
    const text = (await res.text()) || res.statusText;
    
    // Try to parse JSON response and extract message
    try {
      const json = JSON.parse(text);
      if (json.message) {
        throw new Error(json.message);
      }
    } catch (parseError) {
      // Not JSON or no message field, use raw text
    }
    
    throw new Error(`${res.status}: ${text}`);
  }
}

export async function apiRequest(
  method: string,
  url: string,
  data?: unknown | undefined,
): Promise<Response> {
  const res = await fetch(url, {
    method,
    headers: data ? { "Content-Type": "application/json" } : {},
    body: data ? JSON.stringify(data) : undefined,
    credentials: "include",
  });

  await throwIfResNotOk(res);
  return res;
}

type UnauthorizedBehavior = "returnNull" | "throw";
export const getQueryFn: <T>(options: {
  on401: UnauthorizedBehavior;
}) => QueryFunction<T> =
  ({ on401: unauthorizedBehavior }) =>
  async ({ queryKey }) => {
    // Buduj URL z queryKey
    let url = queryKey[0] as string;
    
    // Iteruj przez pozostałe elementy queryKey
    for (let i = 1; i < queryKey.length; i++) {
      const element = queryKey[i];
      
      // Jeśli element to string lub number, dołącz jako path segment
      if (typeof element === 'string' || typeof element === 'number') {
        url = `${url}/${element}`;
      }
      // Jeśli element to obiekt z parametrami, dodaj je do URL jako query params
      else if (typeof element === 'object' && element !== null) {
        const params = new URLSearchParams();
        const queryParams = element as Record<string, any>;
        Object.entries(queryParams).forEach(([key, value]) => {
          if (value !== undefined && value !== null) {
            const stringValue = String(value);
            params.append(key, stringValue);
            
            // Debug logging for search parameter
            if (key === 'search' && stringValue.includes(';')) {
              console.log('🔍 [QUERY-CLIENT] Search with semicolon:', {
                original: stringValue,
                encoded: params.toString(),
                hasSemicolon: stringValue.includes(';')
              });
            }
          }
        });
        url = `${url}?${params.toString()}`;
      }
    }
    
    const res = await fetch(url, {
      credentials: "include",
    });

    if (unauthorizedBehavior === "returnNull" && res.status === 401) {
      return null;
    }

    await throwIfResNotOk(res);
    return await res.json();
  };

export const queryClient = new QueryClient({
  defaultOptions: {
    queries: {
      queryFn: getQueryFn({ on401: "throw" }),
      refetchInterval: false,
      refetchOnWindowFocus: true,  // ✅ Refetch when returning to tab
      staleTime: 0,  // ✅ Data is always considered stale, will refetch on invalidation
      retry: false,
    },
    mutations: {
      retry: false,
    },
  },
});
