import { useState, useEffect, useCallback, useRef } from 'react';
import { useQuery, useMutation } from '@tanstack/react-query';
import { apiRequest, queryClient } from '@/lib/queryClient';

export interface FilterTab {
  id: string;
  label: string;
  token: string;
  isActive: boolean;
}

export interface UseFilterTabsConfig {
  planId: string;
  storageKeyPrefix: string;
}

export interface UseFilterTabsReturn {
  tabs: FilterTab[];
  activeTabs: FilterTab[];
  activeTokens: string[];
  addTab: (label: string) => void;
  toggleTab: (tabId: string) => void;
  removeTab: (tabId: string) => void;
  reorderTabs: (newOrder: FilterTab[]) => void;
  setTabsFromTokens: (tokens: string[]) => void;
  isLoading: boolean;
}

function getPageSlug(prefix: string, _planId: string): string {
  // Global filter tabs - shared across all plans
  return `${prefix}-global`;
}

function sanitizeToken(input: string): string {
  return input.trim().toUpperCase();
}

function validateFilterTabs(data: unknown): FilterTab[] {
  if (!Array.isArray(data)) return [];
  
  return data.filter(
    (item): item is FilterTab =>
      typeof item === 'object' &&
      item !== null &&
      typeof item.id === 'string' &&
      typeof item.label === 'string' &&
      typeof item.token === 'string' &&
      typeof item.isActive === 'boolean'
  );
}

function getLocalStorageKey(prefix: string, _planId: string): string {
  // Global filter tabs - shared across all plans
  return `${prefix}:global`;
}

interface LocalStorageData {
  tabs: FilterTab[];
  needsSync: boolean;
}

function loadFromLocalStorage(storageKey: string): LocalStorageData {
  if (typeof window === 'undefined') return { tabs: [], needsSync: false };
  
  try {
    const stored = localStorage.getItem(storageKey);
    if (!stored) return { tabs: [], needsSync: false };
    const parsed = JSON.parse(stored);
    
    if (Array.isArray(parsed)) {
      return { tabs: validateFilterTabs(parsed), needsSync: false };
    }
    
    if (typeof parsed === 'object' && parsed !== null) {
      return {
        tabs: validateFilterTabs(parsed.tabs),
        needsSync: Boolean(parsed.needsSync)
      };
    }
    
    return { tabs: [], needsSync: false };
  } catch {
    return { tabs: [], needsSync: false };
  }
}

function saveToLocalStorage(storageKey: string, tabs: FilterTab[], needsSync: boolean = true): void {
  if (typeof window === 'undefined') return;
  
  try {
    localStorage.setItem(storageKey, JSON.stringify({ tabs, needsSync }));
  } catch (error) {
    console.error('Error saving filter tabs to localStorage:', error);
  }
}

type PendingEdit = 
  | { type: 'add'; token: string; id: string }
  | { type: 'toggle'; token: string }
  | { type: 'remove'; token: string }
  | { type: 'reorder'; tokens: string[] }
  | { type: 'setFromTokens'; tokens: string[] };

function applyPendingEditsToTabs(baseTabs: FilterTab[], edits: PendingEdit[]): FilterTab[] {
  let result = [...baseTabs];
  
  for (const edit of edits) {
    switch (edit.type) {
      case 'add': {
        if (!result.some(tab => tab.token === edit.token)) {
          result.push({
            id: edit.id,
            label: edit.token,
            token: edit.token,
            isActive: true
          });
        }
        break;
      }
      case 'toggle': {
        result = result.map(tab => 
          tab.token === edit.token ? { ...tab, isActive: !tab.isActive } : tab
        );
        break;
      }
      case 'remove': {
        result = result.filter(tab => tab.token !== edit.token);
        break;
      }
      case 'reorder': {
        const tokenOrder = edit.tokens;
        const orderedTabs: FilterTab[] = [];
        
        for (const token of tokenOrder) {
          const tab = result.find(t => t.token === token);
          if (tab) orderedTabs.push(tab);
        }
        
        for (const tab of result) {
          if (!tokenOrder.includes(tab.token)) {
            orderedTabs.push(tab);
          }
        }
        
        result = orderedTabs;
        break;
      }
      case 'setFromTokens': {
        const tokens = edit.tokens;
        const existingTokens = result.map(t => t.token);
        const missingTokens = tokens.filter(token => !existingTokens.includes(token));
        
        const newTabs = missingTokens.map(token => ({
          id: `tab-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
          label: token,
          token: token,
          isActive: true
        }));
        
        result = result.map(tab => ({
          ...tab,
          isActive: tokens.includes(tab.token)
        }));
        
        result = [...result, ...newTabs];
        break;
      }
    }
  }
  
  return result;
}

export function useFilterTabs({ planId, storageKeyPrefix }: UseFilterTabsConfig): UseFilterTabsReturn {
  const pageSlug = getPageSlug(storageKeyPrefix, planId);
  const localStorageKey = getLocalStorageKey(storageKeyPrefix, planId);
  const previousPlanRef = useRef<string>(`${storageKeyPrefix}:${planId}`);
  
  const [tabs, setTabs] = useState<FilterTab[]>(() => loadFromLocalStorage(localStorageKey).tabs);
  const [serverSyncComplete, setServerSyncComplete] = useState(false);
  const saveTimeoutRef = useRef<NodeJS.Timeout | null>(null);
  const lastSavedToServerRef = useRef<string>('');
  const retryTimeoutRef = useRef<NodeJS.Timeout | null>(null);
  const pendingEditsRef = useRef<PendingEdit[]>([]);
  
  useEffect(() => {
    const currentKey = `${storageKeyPrefix}:${planId}`;
    
    if (previousPlanRef.current !== currentKey) {
      const [prevPrefix, prevId] = previousPlanRef.current.split(':');
      if (prevPrefix && prevId) {
        const prevLocalStorageKey = getLocalStorageKey(prevPrefix, prevId);
        const isAlreadySynced = JSON.stringify(tabs) === lastSavedToServerRef.current;
        saveToLocalStorage(prevLocalStorageKey, tabs, !isAlreadySynced);
      }
      
      if (saveTimeoutRef.current) {
        clearTimeout(saveTimeoutRef.current);
        saveTimeoutRef.current = null;
      }
      if (retryTimeoutRef.current) {
        clearTimeout(retryTimeoutRef.current);
        retryTimeoutRef.current = null;
      }
      
      pendingEditsRef.current = [];
      lastSavedToServerRef.current = '';
      
      const newLocalData = loadFromLocalStorage(getLocalStorageKey(storageKeyPrefix, planId));
      setTabs(newLocalData.tabs);
      setServerSyncComplete(false);
      
      previousPlanRef.current = currentKey;
    }
  }, [planId, storageKeyPrefix, tabs]);
  
  const { data: serverSettings, isLoading: isQueryLoading, isSuccess } = useQuery<any>({
    queryKey: [`/api/user-settings/${pageSlug}`],
    staleTime: 0,
    refetchOnMount: true,
    refetchOnWindowFocus: false,
    retry: 3,
    retryDelay: 1000,
  });
  
  interface MutationVariables {
    filterTabs: FilterTab[];
    targetSlug: string;
    targetPlanId: string;
    targetPrefix: string;
  }
  
  const saveSettingsMutation = useMutation({
    mutationFn: ({ filterTabs, targetSlug }: MutationVariables) => 
      apiRequest('PUT', `/api/user-settings/${targetSlug}`, {
        filters: { persistentFilterTabs: filterTabs }
      }),
    onSuccess: (_data, { filterTabs, targetSlug, targetPlanId, targetPrefix }) => {
      const currentSlug = getPageSlug(storageKeyPrefix, planId);
      const isCurrentPlan = targetSlug === currentSlug;
      
      const targetLocalStorageKey = getLocalStorageKey(targetPrefix, targetPlanId);
      
      saveToLocalStorage(targetLocalStorageKey, filterTabs, false);
      
      queryClient.setQueryData([`/api/user-settings/${targetSlug}`], (old: any) => ({
        ...(old || {}),
        filters: { ...(old?.filters || {}), persistentFilterTabs: filterTabs }
      }));
      
      if (!isCurrentPlan) {
        return;
      }
      
      const savedJson = JSON.stringify(filterTabs);
      lastSavedToServerRef.current = savedJson;
      
      if (retryTimeoutRef.current) {
        clearTimeout(retryTimeoutRef.current);
        retryTimeoutRef.current = null;
      }
    },
    onError: (_error, { targetSlug }) => {
      const currentSlug = getPageSlug(storageKeyPrefix, planId);
      
      if (targetSlug !== currentSlug) {
        return;
      }
      
      retryTimeoutRef.current = setTimeout(() => {
        setTabs(prev => [...prev]);
      }, 3000);
    }
  });
  
  const applyPendingEdits = useCallback((baseTabs: FilterTab[]): FilterTab[] => {
    return applyPendingEditsToTabs(baseTabs, pendingEditsRef.current);
  }, []);
  
  useEffect(() => {
    if (isSuccess && !serverSyncComplete) {
      const serverTabs = validateFilterTabs(serverSettings?.filters?.persistentFilterTabs);
      const localData = loadFromLocalStorage(localStorageKey);
      
      let baseTabs: FilterTab[];
      let needsServerSave = false;
      
      if (localData.needsSync) {
        baseTabs = localData.tabs;
        needsServerSave = true;
      } else if (serverTabs.length > 0) {
        baseTabs = serverTabs;
        needsServerSave = false;
      } else if (localData.tabs.length > 0) {
        baseTabs = localData.tabs;
        needsServerSave = true;
      } else {
        baseTabs = [];
        needsServerSave = false;
      }
      
      const finalTabs = applyPendingEdits(baseTabs);
      pendingEditsRef.current = [];
      
      setTabs(finalTabs);
      
      if (needsServerSave || pendingEditsRef.current.length > 0) {
        saveToLocalStorage(localStorageKey, finalTabs, true);
      } else {
        saveToLocalStorage(localStorageKey, finalTabs, false);
        lastSavedToServerRef.current = JSON.stringify(serverTabs);
      }
      
      setServerSyncComplete(true);
    }
  }, [isSuccess, serverSettings, serverSyncComplete, localStorageKey, applyPendingEdits]);
  
  useEffect(() => {
    if (!serverSyncComplete) return;
    
    const currentTabsJson = JSON.stringify(tabs);
    const isInSync = currentTabsJson === lastSavedToServerRef.current;
    
    saveToLocalStorage(localStorageKey, tabs, !isInSync);
    
    if (isInSync) {
      return;
    }
    
    if (saveTimeoutRef.current) {
      clearTimeout(saveTimeoutRef.current);
    }
    
    const slugToSave = pageSlug;
    const planIdToSave = planId;
    const prefixToSave = storageKeyPrefix;
    const tabsToSave = [...tabs];
    
    saveTimeoutRef.current = setTimeout(() => {
      saveSettingsMutation.mutate({ 
        filterTabs: tabsToSave, 
        targetSlug: slugToSave, 
        targetPlanId: planIdToSave, 
        targetPrefix: prefixToSave 
      });
    }, 500);
    
    return () => {
      if (saveTimeoutRef.current) {
        clearTimeout(saveTimeoutRef.current);
      }
    };
  }, [tabs, serverSyncComplete, localStorageKey]);
  
  useEffect(() => {
    return () => {
      if (retryTimeoutRef.current) {
        clearTimeout(retryTimeoutRef.current);
      }
    };
  }, []);

  const addTab = useCallback((label: string) => {
    const token = sanitizeToken(label);
    
    if (!token) return;
    
    const newId = `tab-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
    
    if (!serverSyncComplete) {
      pendingEditsRef.current.push({ type: 'add', token, id: newId });
    }
    
    setTabs(prev => {
      const exists = prev.some(tab => tab.token === token);
      if (exists) return prev;
      
      const newTab: FilterTab = {
        id: newId,
        label: token,
        token: token,
        isActive: true
      };
      
      return [...prev, newTab];
    });
  }, [serverSyncComplete]);

  const toggleTab = useCallback((tabId: string) => {
    setTabs(prev => {
      const tab = prev.find(t => t.id === tabId);
      if (!tab) return prev;
      
      if (!serverSyncComplete) {
        pendingEditsRef.current.push({ type: 'toggle', token: tab.token });
      }
      
      return prev.map(t => 
        t.id === tabId ? { ...t, isActive: !t.isActive } : t
      );
    });
  }, [serverSyncComplete]);

  const removeTab = useCallback((tabId: string) => {
    setTabs(prev => {
      const tab = prev.find(t => t.id === tabId);
      if (!tab) return prev;
      
      if (!serverSyncComplete) {
        pendingEditsRef.current.push({ type: 'remove', token: tab.token });
      }
      
      return prev.filter(t => t.id !== tabId);
    });
  }, [serverSyncComplete]);

  const reorderTabs = useCallback((newOrder: FilterTab[]) => {
    if (!serverSyncComplete) {
      pendingEditsRef.current.push({ type: 'reorder', tokens: newOrder.map(t => t.token) });
    }
    
    setTabs(newOrder);
  }, [serverSyncComplete]);

  const setTabsFromTokens = useCallback((tokens: string[]) => {
    const sanitizedTokens = tokens.map(t => sanitizeToken(t)).filter(Boolean);
    
    if (!serverSyncComplete) {
      pendingEditsRef.current.push({ type: 'setFromTokens', tokens: sanitizedTokens });
    }
    
    setTabs(prev => {
      const existingTokens = prev.map(t => t.token);
      const missingTokens = sanitizedTokens.filter(token => !existingTokens.includes(token));
      
      const newTabs = missingTokens.map(token => ({
        id: `tab-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
        label: token,
        token: token,
        isActive: true
      }));
      
      const updatedTabs = prev.map(tab => ({
        ...tab,
        isActive: sanitizedTokens.includes(tab.token)
      }));
      
      const hasChanges = 
        newTabs.length > 0 || 
        updatedTabs.some((tab, idx) => tab.isActive !== prev[idx].isActive);
      
      if (!hasChanges) return prev;
      
      return [...updatedTabs, ...newTabs];
    });
  }, [serverSyncComplete]);

  const activeTabs = tabs.filter(t => t.isActive);
  const activeTokens = activeTabs.map(t => t.token);

  return {
    tabs,
    activeTabs,
    activeTokens,
    addTab,
    toggleTab,
    removeTab,
    reorderTabs,
    setTabsFromTokens,
    isLoading: isQueryLoading && !serverSyncComplete
  };
}
