import { useState, useEffect, useCallback } from 'react';

const STORAGE_KEY = 'production-plan-available-orders-filters';

export interface AvailableOrdersFilters {
  search: string;
  color: string;
  sku: string;
  minLength: string;
  maxLength: string;
  minWidth: string;
  maxWidth: string;
  orderNumber: string;
  customerName: string;
  marketplace: string;
  limit: number;
  offset: number;
  sortBy: 'order_date' | 'order_number' | 'buyer_name' | 'total_amount' | 'product_sku';
  sortOrder: 'asc' | 'desc';
  showSetsOnly: string;
  showCatalogLinked: string;
  showInPlans: string;
  dateFilter: 'all' | 'today' | 'yesterday' | 'day-before' | 'custom-days';
  customDays: string;
}

const defaultFilters: AvailableOrdersFilters = {
  search: '',
  color: '',
  sku: '',
  minLength: '',
  maxLength: '',
  minWidth: '',
  maxWidth: '',
  orderNumber: '',
  customerName: '',
  marketplace: 'all',
  limit: 100,
  offset: 0,
  sortBy: 'order_date',
  sortOrder: 'desc',
  showSetsOnly: 'false',
  showCatalogLinked: 'false',
  showInPlans: 'false',
  dateFilter: 'all',
  customDays: '',
};

export function useAvailableOrdersFilters() {
  const [filters, setFilters] = useState<AvailableOrdersFilters>(() => {
    if (typeof window === 'undefined') return defaultFilters;
    
    try {
      const saved = localStorage.getItem(STORAGE_KEY);
      if (saved) {
        return { ...defaultFilters, ...JSON.parse(saved) };
      }
    } catch (error) {
      console.error('Error loading filters from localStorage:', error);
    }
    return defaultFilters;
  });

  useEffect(() => {
    if (typeof window === 'undefined') return;
    
    try {
      localStorage.setItem(STORAGE_KEY, JSON.stringify(filters));
    } catch (error) {
      console.error('Error saving filters to localStorage:', error);
    }
  }, [filters]);

  const updateFilter = useCallback(<K extends keyof AvailableOrdersFilters>(
    key: K,
    value: AvailableOrdersFilters[K]
  ) => {
    setFilters(prev => ({ ...prev, [key]: value }));
  }, []);

  const updateFilters = useCallback((updates: Partial<AvailableOrdersFilters>) => {
    setFilters(prev => ({ ...prev, ...updates }));
  }, []);

  const resetFilters = useCallback(() => {
    setFilters(defaultFilters);
  }, []);

  const getActiveFilters = useCallback(() => {
    return Object.entries(filters)
      .filter(([key, value]) => {
        if (key === 'marketplace' && value === 'all') return false;
        return value !== '' && value !== defaultFilters[key as keyof AvailableOrdersFilters];
      })
      .map(([key, value]) => ({ key, value: value as string }));
  }, [filters]);

  const clearFilter = useCallback((key: string) => {
    updateFilter(key as keyof AvailableOrdersFilters, defaultFilters[key as keyof AvailableOrdersFilters]);
  }, [updateFilter]);

  return {
    filters,
    updateFilter,
    updateFilters,
    resetFilters,
    getActiveFilters,
    clearFilter,
  };
}
