import { useMemo } from "react";
import { useQuery } from "@tanstack/react-query";
import { Link } from "wouter";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Progress } from "@/components/ui/progress";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip";
import { 
  Factory, 
  Package, 
  CheckCircle, 
  Clock, 
  AlertTriangle,
  Play,
  ExternalLink,
  Loader2,
  Palette,
  Layers,
  BarChart3,
  Lock,
  Unlock
} from "lucide-react";
import { Button } from "@/components/ui/button";

interface ZlpWorkOrder {
  id: number;
  workOrderNumber: string;
  sequence: number;
  status: string;
  operationName: string | null;
  operationCode: string | null;
  workCenterName: string | null;
  quantityPlanned: number;
  quantityProduced: number;
  quantityScrap: number;
  actualStartTime: string | null;
  actualEndTime: string | null;
}

interface ZlpSummary {
  id: number;
  orderNumber: string;
  status: string;
  colorCode: string | null;
  routingId: number | null;
  routingName: string | null;
  quantityPlanned: number;
  quantityProduced: number;
  quantityScrap: number;
  progress: number;
  workOrders: ZlpWorkOrder[];
  activeOperationIndex: number | null;
  createdAt: string;
  updatedAt: string;
}

interface ZlpDashboardComponent {
  id: number;
  componentType: string;
  name: string;
  sku: string | null;
  quantity: number;
  reservedQuantity: number;
  zlpCount: number;
}

interface ZlpDashboardData {
  planId: number;
  planNumber: string;
  planName: string;
  totalZlps: number;
  completedZlps: number;
  inProgressZlps: number;
  pendingZlps: number;
  overallProgress: number;
  zlps: ZlpSummary[];
  components: ZlpDashboardComponent[];
}

interface ZlpDashboardInlineProps {
  planId: number;
  onUnlock?: () => void;
  isUnlocking?: boolean;
}

const statusConfig: Record<string, { color: string; bgColor: string; label: string; icon: any }> = {
  draft: { color: "text-gray-500", bgColor: "bg-gray-100 dark:bg-gray-800", label: "Szkic", icon: Clock },
  confirmed: { color: "text-blue-500", bgColor: "bg-blue-100 dark:bg-blue-900/30", label: "Potwierdzone", icon: CheckCircle },
  pending: { color: "text-gray-500", bgColor: "bg-gray-100 dark:bg-gray-800", label: "Oczekuje", icon: Clock },
  in_progress: { color: "text-yellow-500", bgColor: "bg-yellow-100 dark:bg-yellow-900/30", label: "W trakcie", icon: Play },
  done: { color: "text-green-500", bgColor: "bg-green-100 dark:bg-green-900/30", label: "Zakonczone", icon: CheckCircle },
  cancelled: { color: "text-red-500", bgColor: "bg-red-100 dark:bg-red-900/30", label: "Anulowane", icon: AlertTriangle },
};

const workOrderStatusConfig: Record<string, { color: string; bgClass: string }> = {
  pending: { color: "bg-gray-400", bgClass: "bg-gray-100 dark:bg-gray-800" },
  ready: { color: "bg-blue-400", bgClass: "bg-blue-100 dark:bg-blue-900/30" },
  in_progress: { color: "bg-yellow-400", bgClass: "bg-yellow-100 dark:bg-yellow-900/30" },
  done: { color: "bg-green-400", bgClass: "bg-green-100 dark:bg-green-900/30" },
  cancelled: { color: "bg-red-400", bgClass: "bg-red-100 dark:bg-red-900/30" },
};

function MiniWorkOrderFlow({ workOrders }: { workOrders: ZlpWorkOrder[] }) {
  if (workOrders.length === 0) {
    return <span className="text-xs text-muted-foreground">Brak operacji</span>;
  }

  return (
    <div className="flex items-center gap-0.5 overflow-x-auto pb-1">
      {workOrders.map((wo, index) => {
        const statusCfg = workOrderStatusConfig[wo.status] || workOrderStatusConfig.pending;
        return (
          <Tooltip key={wo.id}>
            <TooltipTrigger asChild>
              <div 
                className={`w-5 h-5 rounded-sm flex items-center justify-center text-[9px] font-bold ${statusCfg.bgClass} cursor-default`}
              >
                <div className={`w-2.5 h-2.5 rounded-full ${statusCfg.color}`} />
              </div>
            </TooltipTrigger>
            <TooltipContent side="top" className="text-xs">
              <div className="font-medium">{wo.operationName || wo.operationCode || `Op ${wo.sequence}`}</div>
              <div className="text-muted-foreground">{wo.workCenterName || 'Brak stanowiska'}</div>
              <div className="text-muted-foreground">
                {wo.quantityProduced}/{wo.quantityPlanned} szt.
              </div>
            </TooltipContent>
          </Tooltip>
        );
      })}
    </div>
  );
}

function ComponentsTable({ components }: { components: ZlpDashboardComponent[] }) {
  const grouped = useMemo(() => {
    const byType: Record<string, ZlpDashboardComponent[]> = {};
    components.forEach(comp => {
      const type = comp.componentType || 'other';
      if (!byType[type]) byType[type] = [];
      byType[type].push(comp);
    });
    return byType;
  }, [components]);

  const typeLabels: Record<string, string> = {
    packed_product: "Produkty spakowane",
    cutting_pattern: "Rozkroje",
    formatka: "Formatki",
    other: "Inne"
  };

  return (
    <div className="space-y-3">
      {Object.entries(grouped).map(([type, comps]) => (
        <div key={type} className="space-y-1">
          <div className="text-xs font-medium text-muted-foreground flex items-center gap-1.5">
            <Package className="h-3 w-3" />
            {typeLabels[type] || type} ({comps.length})
          </div>
          <div className="space-y-1">
            {comps.slice(0, 5).map(comp => (
              <div 
                key={comp.id} 
                className="flex items-center justify-between text-xs p-1.5 rounded bg-muted/50"
              >
                <div className="flex-1 min-w-0">
                  <div className="font-medium truncate">{comp.name}</div>
                  {comp.sku && <div className="text-muted-foreground text-[10px] truncate">{comp.sku}</div>}
                </div>
                <div className="flex items-center gap-2 ml-2">
                  <Badge variant="outline" className="text-[10px]">
                    {comp.quantity} szt.
                  </Badge>
                  {comp.reservedQuantity > 0 && (
                    <Badge variant="secondary" className="text-[10px] bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-400">
                      Rez: {comp.reservedQuantity}
                    </Badge>
                  )}
                </div>
              </div>
            ))}
            {comps.length > 5 && (
              <div className="text-xs text-muted-foreground text-center py-1">
                ... i {comps.length - 5} wiecej
              </div>
            )}
          </div>
        </div>
      ))}
    </div>
  );
}

export function ZlpDashboardInline({ planId, onUnlock, isUnlocking }: ZlpDashboardInlineProps) {
  const { data: dashboard, isLoading, error } = useQuery<ZlpDashboardData>({
    queryKey: [`/api/production/planning/plans/${planId}/zlp-dashboard`],
    enabled: !!planId,
  });

  return (
    <Card className="h-full flex flex-col" data-testid="zlp-dashboard-inline">
      <CardHeader className="flex-shrink-0 py-3 px-4 border-b">
        <div className="flex items-center justify-between gap-2 flex-wrap">
          <div className="flex items-center gap-2">
            <div className="flex items-center gap-2 text-orange-600 dark:text-orange-400">
              <Lock className="h-4 w-4" />
              <span className="text-sm font-medium">Plan zablokowany</span>
            </div>
            <span className="text-xs text-muted-foreground">- wygenerowano ZLP</span>
          </div>
          {onUnlock && (
            <Button
              variant="outline"
              size="sm"
              onClick={onUnlock}
              disabled={isUnlocking}
              className="text-orange-600 border-orange-300 hover:bg-orange-50 dark:text-orange-400 dark:border-orange-700 dark:hover:bg-orange-900/20"
              data-testid="button-unlock-plan"
            >
              {isUnlocking ? (
                <Loader2 className="h-3.5 w-3.5 mr-1.5 animate-spin" />
              ) : (
                <Unlock className="h-3.5 w-3.5 mr-1.5" />
              )}
              Odblokuj do regeneracji
            </Button>
          )}
        </div>
      </CardHeader>

      <ScrollArea className="flex-1">
        <div className="p-4 space-y-4">
          {isLoading && (
            <div className="flex items-center justify-center py-8">
              <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
            </div>
          )}

          {error && (
            <div className="text-center text-destructive py-8">
              Blad ladowania danych ZLP
            </div>
          )}

          {dashboard && (
            <>
              <Card>
                <CardHeader className="pb-2">
                  <CardTitle className="text-sm flex items-center gap-2">
                    <BarChart3 className="h-4 w-4" />
                    Podsumowanie ZLP
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-3">
                  <div className="grid grid-cols-4 gap-2 text-center">
                    <div className="space-y-1">
                      <div className="text-2xl font-bold">{dashboard.totalZlps}</div>
                      <div className="text-xs text-muted-foreground">Lacznie</div>
                    </div>
                    <div className="space-y-1">
                      <div className="text-2xl font-bold text-green-600">{dashboard.completedZlps}</div>
                      <div className="text-xs text-muted-foreground">Zakonczone</div>
                    </div>
                    <div className="space-y-1">
                      <div className="text-2xl font-bold text-yellow-600">{dashboard.inProgressZlps}</div>
                      <div className="text-xs text-muted-foreground">W trakcie</div>
                    </div>
                    <div className="space-y-1">
                      <div className="text-2xl font-bold text-gray-500">{dashboard.pendingZlps}</div>
                      <div className="text-xs text-muted-foreground">Oczekuje</div>
                    </div>
                  </div>

                  <div className="space-y-1">
                    <div className="flex items-center justify-between text-xs">
                      <span className="text-muted-foreground">Postep ogolny</span>
                      <span className="font-medium">{dashboard.overallProgress}%</span>
                    </div>
                    <Progress value={dashboard.overallProgress} className="h-2" />
                  </div>
                </CardContent>
              </Card>

              {dashboard.zlps.length > 0 && (
                <div className="space-y-2">
                  <h3 className="text-sm font-medium flex items-center gap-2">
                    <Factory className="h-4 w-4" />
                    Zlecenia produkcyjne ({dashboard.zlps.length})
                  </h3>
                  <div className="space-y-2">
                    {dashboard.zlps.map(zlp => {
                      const statusCfg = statusConfig[zlp.status] || statusConfig.pending;
                      const StatusIcon = statusCfg.icon;
                      
                      return (
                        <Card key={zlp.id} className="overflow-hidden">
                          <div className="p-3 space-y-2">
                            <div className="flex items-center justify-between">
                              <Link 
                                href={`/production/orders/${zlp.id}`}
                                className="font-medium text-sm hover:underline flex items-center gap-1.5"
                                data-testid={`link-zlp-${zlp.id}`}
                              >
                                {zlp.orderNumber}
                                <ExternalLink className="h-3 w-3 opacity-50" />
                              </Link>
                              <Badge className={`${statusCfg.bgColor} ${statusCfg.color} text-[10px]`}>
                                <StatusIcon className="h-3 w-3 mr-1" />
                                {statusCfg.label}
                              </Badge>
                            </div>

                            <div className="flex items-center gap-3 text-xs text-muted-foreground">
                              {zlp.colorCode && (
                                <div className="flex items-center gap-1">
                                  <Palette className="h-3 w-3" />
                                  {zlp.colorCode}
                                </div>
                              )}
                              <div className="flex items-center gap-1">
                                <Package className="h-3 w-3" />
                                {zlp.quantityProduced}/{zlp.quantityPlanned} szt.
                              </div>
                            </div>

                            <div className="space-y-1">
                              <div className="flex items-center justify-between text-xs">
                                <span className="text-muted-foreground">Postep</span>
                                <span>{zlp.progress}%</span>
                              </div>
                              <Progress value={zlp.progress} className="h-1.5" />
                            </div>

                            <div className="pt-1">
                              <MiniWorkOrderFlow workOrders={zlp.workOrders} />
                            </div>
                          </div>
                        </Card>
                      );
                    })}
                  </div>
                </div>
              )}

              {dashboard.components.length > 0 && (
                <div className="space-y-2">
                  <h3 className="text-sm font-medium flex items-center gap-2">
                    <Layers className="h-4 w-4" />
                    Komponenty ({dashboard.components.length})
                  </h3>
                  <ComponentsTable components={dashboard.components} />
                </div>
              )}
            </>
          )}
        </div>
      </ScrollArea>
    </Card>
  );
}
