import { useMemo, useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Link } from "wouter";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Progress } from "@/components/ui/progress";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Separator } from "@/components/ui/separator";
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip";
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible";
import { 
  Factory, 
  Package, 
  CheckCircle, 
  Clock, 
  AlertTriangle,
  Play,
  ExternalLink,
  Loader2,
  Palette,
  Layers,
  BarChart3,
  ChevronDown,
  ChevronUp,
  Box,
  Ruler,
  Route,
  List,
  Grid
} from "lucide-react";

interface ZlpWorkOrder {
  id: number;
  workOrderNumber: string;
  sequence: number;
  status: string;
  operationName: string | null;
  operationCode: string | null;
  workCenterName: string | null;
  quantityPlanned: number;
  quantityProduced: number;
  quantityScrap: number;
  actualStartTime: string | null;
  actualEndTime: string | null;
}

interface ZlpSummary {
  id: number;
  orderNumber: string;
  status: string;
  colorCode: string | null;
  routingId: number | null;
  routingName: string | null;
  quantityPlanned: number;
  quantityProduced: number;
  quantityScrap: number;
  progress: number;
  workOrders: ZlpWorkOrder[];
  activeOperationIndex: number | null;
  createdAt: string;
  updatedAt: string;
}

interface ZlpDashboardComponent {
  id: number;
  componentType: string;
  name: string;
  sku: string | null;
  quantity: number;
  reservedQuantity: number;
  zlpCount: number;
}

interface ZlpBomItem {
  id: number;
  componentName: string;
  componentType: string | null;
  quantity: number;
  unitOfMeasure: string;
  colorCode: string | null;
  length: number | null;
  width: number | null;
  thickness: number | null;
  isDamaged: boolean;
  itemStatus: string;
}

interface ZlpDashboardData {
  planId: number;
  planNumber: string;
  planName: string;
  totalZlps: number;
  completedZlps: number;
  inProgressZlps: number;
  pendingZlps: number;
  overallProgress: number;
  zlps: ZlpSummary[];
  components: ZlpDashboardComponent[];
}

interface ZlpDashboardProps {
  planId: number;
  isOpen: boolean;
  onOpenChange: (open: boolean) => void;
}

const statusConfig: Record<string, { color: string; bgColor: string; label: string; icon: any }> = {
  draft: { color: "text-gray-500", bgColor: "bg-gray-100 dark:bg-gray-800", label: "Szkic", icon: Clock },
  confirmed: { color: "text-blue-500", bgColor: "bg-blue-100 dark:bg-blue-900/30", label: "Potwierdzone", icon: CheckCircle },
  pending: { color: "text-gray-500", bgColor: "bg-gray-100 dark:bg-gray-800", label: "Oczekuje", icon: Clock },
  in_progress: { color: "text-yellow-500", bgColor: "bg-yellow-100 dark:bg-yellow-900/30", label: "W trakcie", icon: Play },
  done: { color: "text-green-500", bgColor: "bg-green-100 dark:bg-green-900/30", label: "Zakonczone", icon: CheckCircle },
  cancelled: { color: "text-red-500", bgColor: "bg-red-100 dark:bg-red-900/30", label: "Anulowane", icon: AlertTriangle },
};

const workOrderStatusConfig: Record<string, { color: string; bgClass: string }> = {
  pending: { color: "bg-gray-400", bgClass: "bg-gray-100 dark:bg-gray-800" },
  ready: { color: "bg-blue-400", bgClass: "bg-blue-100 dark:bg-blue-900/30" },
  in_progress: { color: "bg-yellow-400", bgClass: "bg-yellow-100 dark:bg-yellow-900/30" },
  done: { color: "bg-green-400", bgClass: "bg-green-100 dark:bg-green-900/30" },
  cancelled: { color: "bg-red-400", bgClass: "bg-red-100 dark:bg-red-900/30" },
};

const operationIcons: Record<string, string> = {
  cutting: "Rozkroj",
  edging: "Okleinowanie",
  drilling: "Wiercenie",
  drilling_holes: "Wiercenie otworow",
  drilling_mount: "Wiercenie montazowe",
  upholstering: "Tapicerowanie",
  assembly: "Montaz",
  packing: "Pakowanie",
  strapping: "Spinanie",
};

function MiniWorkOrderFlow({ workOrders }: { workOrders: ZlpWorkOrder[] }) {
  if (workOrders.length === 0) {
    return <span className="text-xs text-muted-foreground">Brak operacji</span>;
  }

  return (
    <div className="flex items-center gap-0.5 overflow-x-auto pb-1">
      {workOrders.map((wo, index) => {
        const statusCfg = workOrderStatusConfig[wo.status] || workOrderStatusConfig.pending;
        return (
          <Tooltip key={wo.id}>
            <TooltipTrigger asChild>
              <div 
                className={`w-5 h-5 rounded flex items-center justify-center text-[10px] font-medium cursor-default ${statusCfg.bgClass} border`}
              >
                <div className={`w-2 h-2 rounded-full ${statusCfg.color}`} />
              </div>
            </TooltipTrigger>
            <TooltipContent side="top" className="text-xs max-w-[200px]">
              <div className="font-medium">{wo.operationName || wo.operationCode || `Op. ${wo.sequence}`}</div>
              <div className="text-muted-foreground">
                {workOrderStatusConfig[wo.status]?.color === "bg-green-400" ? "Zakonczone" : 
                 workOrderStatusConfig[wo.status]?.color === "bg-yellow-400" ? "W trakcie" : "Oczekuje"}
              </div>
              {wo.quantityScrap > 0 && (
                <div className="text-red-400">Brak: {wo.quantityScrap} szt.</div>
              )}
            </TooltipContent>
          </Tooltip>
        );
      })}
    </div>
  );
}

function ZlpCard({ zlp, showMissingLabels = false }: { zlp: ZlpSummary; showMissingLabels?: boolean }) {
  const [isExpanded, setIsExpanded] = useState(false);
  const statusCfg = statusConfig[zlp.status] || statusConfig.pending;
  const StatusIcon = statusCfg.icon;

  // Fetch BOM items when expanded
  const { data: bomItems = [], isLoading: isLoadingItems } = useQuery<ZlpBomItem[]>({
    queryKey: [`/api/production/orders/${zlp.id}/items`],
    enabled: isExpanded,
  });

  const componentTypeLabels: Record<string, string> = {
    formatka: "Formatka",
    packed_product: "Produkt spakowany",
    accessory: "Akcesorium",
    other: "Inny",
  };

  return (
    <Card className="hover-elevate transition-all">
      <CardContent className="p-3 space-y-2">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-2 flex-wrap">
            <Link href={`/production/orders/${zlp.id}`}>
              <Button variant="ghost" size="sm" className="h-auto p-0 text-sm font-medium text-primary hover:underline" data-testid={`link-zlp-${zlp.id}`}>
                {zlp.orderNumber}
                <ExternalLink className="ml-1 h-3 w-3" />
              </Button>
            </Link>
            <Badge variant="outline" className="text-xs">
              <Palette className="mr-1 h-3 w-3" />
              {zlp.colorCode || 'Bez koloru'}
            </Badge>
            <Badge variant="outline" className="text-xs">
              <Route className="mr-1 h-3 w-3" />
              {zlp.routingName || 'Bez marszruty'}
            </Badge>
          </div>
          <Badge variant="secondary" className={`${statusCfg.bgColor} ${statusCfg.color} text-xs`}>
            <StatusIcon className="mr-1 h-3 w-3" />
            {statusCfg.label}
          </Badge>
        </div>

        <div className="flex items-center gap-4 text-xs text-muted-foreground">
          <span className="flex items-center gap-1">
            <Package className="h-3 w-3" />
            Plan: {zlp.quantityPlanned}
          </span>
          <span className="flex items-center gap-1">
            <CheckCircle className="h-3 w-3 text-green-500" />
            Wyk: {zlp.quantityProduced}
          </span>
          {zlp.quantityScrap > 0 && (
            <span className="flex items-center gap-1 text-red-500">
              <AlertTriangle className="h-3 w-3" />
              Brak: {zlp.quantityScrap}
            </span>
          )}
        </div>

        <div className="space-y-1">
          <div className="flex items-center justify-between text-xs">
            <span className="text-muted-foreground">Postep operacji</span>
            <span className="font-medium">{zlp.progress}%</span>
          </div>
          <Progress value={zlp.progress} className="h-1.5" />
        </div>

        <MiniWorkOrderFlow workOrders={zlp.workOrders} />

        {/* Expandable items section */}
        <Collapsible open={isExpanded} onOpenChange={setIsExpanded}>
          <CollapsibleTrigger asChild>
            <Button 
              variant="ghost" 
              size="sm" 
              className="w-full h-7 text-xs gap-1 mt-1"
              data-testid={`button-expand-zlp-items-${zlp.id}`}
            >
              <Box className="h-3 w-3" />
              Elementy (formatki/komponenty)
              {isExpanded ? <ChevronUp className="h-3 w-3 ml-auto" /> : <ChevronDown className="h-3 w-3 ml-auto" />}
            </Button>
          </CollapsibleTrigger>
          <CollapsibleContent className="mt-2">
            {isLoadingItems ? (
              <div className="flex items-center justify-center py-3">
                <Loader2 className="h-4 w-4 animate-spin text-muted-foreground" />
              </div>
            ) : bomItems.length === 0 ? (
              <div className="text-center text-xs text-muted-foreground py-2">
                Brak elementow
              </div>
            ) : (
              <div className="space-y-1 max-h-[200px] overflow-y-auto">
                {bomItems.map((item) => (
                  <div 
                    key={item.id}
                    className={`flex items-center justify-between p-2 rounded text-xs bg-muted/50 ${item.isDamaged ? 'border-l-2 border-red-500' : ''}`}
                  >
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center gap-1.5">
                        <span className="font-medium truncate">{item.componentName}</span>
                        {item.colorCode && (
                          <Badge variant="outline" className="text-[10px] px-1 py-0 h-4">
                            {item.colorCode}
                          </Badge>
                        )}
                      </div>
                      <div className="flex items-center gap-2 text-muted-foreground mt-0.5">
                        <span>{componentTypeLabels[item.componentType || 'other'] || item.componentType}</span>
                        {item.length && item.width && (
                          <span className="flex items-center gap-0.5">
                            <Ruler className="h-2.5 w-2.5" />
                            {item.length}x{item.width}
                            {item.thickness && `x${item.thickness}`}
                          </span>
                        )}
                      </div>
                    </div>
                    <div className="flex items-center gap-2 ml-2">
                      <span className="font-medium">{item.quantity} {item.unitOfMeasure}</span>
                      {item.isDamaged && (
                        <AlertTriangle className="h-3 w-3 text-red-500" />
                      )}
                    </div>
                  </div>
                ))}
              </div>
            )}
          </CollapsibleContent>
        </Collapsible>
      </CardContent>
    </Card>
  );
}

function ComponentsTable({ components }: { components: ZlpDashboardComponent[] }) {
  const groupedComponents = useMemo(() => {
    const groups: Record<string, ZlpDashboardComponent[]> = {};
    for (const comp of components) {
      const type = comp.componentType || 'other';
      if (!groups[type]) groups[type] = [];
      groups[type].push(comp);
    }
    return groups;
  }, [components]);

  const typeLabels: Record<string, string> = {
    formatka: "Formatki",
    packed_product: "Produkty spakowane",
    accessory: "Akcesoria",
    other: "Inne",
    unknown: "Nieznane",
  };

  if (components.length === 0) {
    return (
      <div className="text-center text-muted-foreground py-4 text-sm">
        Brak komponentow do wyswietlenia
      </div>
    );
  }

  return (
    <div className="space-y-3">
      {Object.entries(groupedComponents).map(([type, comps]) => (
        <div key={type} className="space-y-1">
          <div className="text-xs font-medium text-muted-foreground uppercase tracking-wide flex items-center gap-1">
            <Layers className="h-3 w-3" />
            {typeLabels[type] || type} ({comps.length})
          </div>
          <div className="space-y-1">
            {comps.slice(0, 5).map((comp) => (
              <div 
                key={comp.id} 
                className="flex items-center justify-between text-xs p-1.5 rounded bg-muted/50"
              >
                <div className="flex-1 min-w-0">
                  <div className="font-medium truncate">{comp.name}</div>
                  {comp.sku && <div className="text-muted-foreground text-[10px] truncate">{comp.sku}</div>}
                </div>
                <div className="flex items-center gap-2 ml-2">
                  <Badge variant="outline" className="text-[10px]">
                    {comp.quantity} szt.
                  </Badge>
                  {comp.reservedQuantity > 0 && (
                    <Badge variant="secondary" className="text-[10px] bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-400">
                      Rez: {comp.reservedQuantity}
                    </Badge>
                  )}
                </div>
              </div>
            ))}
            {comps.length > 5 && (
              <div className="text-xs text-muted-foreground text-center py-1">
                ... i {comps.length - 5} wiecej
              </div>
            )}
          </div>
        </div>
      ))}
    </div>
  );
}

type ViewMode = 'list' | 'grouped';
type GroupBy = 'color' | 'routing' | 'color-routing';

interface ZlpGroup {
  key: string;
  label: string;
  colorCode: string | null;
  routingName: string | null;
  zlps: ZlpSummary[];
  progress: number;
  completedCount: number;
}

function GroupedZlpList({ zlps, groupBy }: { zlps: ZlpSummary[]; groupBy: GroupBy }) {
  const [expandedGroups, setExpandedGroups] = useState<Set<string>>(new Set());

  const groups = useMemo(() => {
    const groupMap = new Map<string, ZlpGroup>();
    
    for (const zlp of zlps) {
      let key: string;
      let label: string;
      const colorLabel = zlp.colorCode || 'Bez koloru';
      const routingLabel = zlp.routingName || 'Bez marszruty';
      
      if (groupBy === 'color') {
        key = zlp.colorCode || 'no-color';
        label = colorLabel;
      } else if (groupBy === 'routing') {
        key = zlp.routingId?.toString() || 'no-routing';
        label = routingLabel;
      } else {
        key = `${zlp.colorCode || 'no-color'}-${zlp.routingId || 'no-routing'}`;
        label = `${colorLabel} / ${routingLabel}`;
      }
      
      if (!groupMap.has(key)) {
        groupMap.set(key, {
          key,
          label,
          colorCode: zlp.colorCode,
          routingName: zlp.routingName,
          zlps: [],
          progress: 0,
          completedCount: 0,
        });
      }
      
      groupMap.get(key)!.zlps.push(zlp);
    }
    
    const result = Array.from(groupMap.values()).map(group => {
      const completed = group.zlps.filter(z => z.status === 'done').length;
      const totalProgress = group.zlps.reduce((sum, z) => sum + z.progress, 0);
      return {
        ...group,
        progress: group.zlps.length > 0 ? Math.round(totalProgress / group.zlps.length) : 0,
        completedCount: completed,
      };
    });
    
    return result.sort((a, b) => a.label.localeCompare(b.label));
  }, [zlps, groupBy]);

  const toggleGroup = (key: string) => {
    setExpandedGroups(prev => {
      const next = new Set(prev);
      if (next.has(key)) {
        next.delete(key);
      } else {
        next.add(key);
      }
      return next;
    });
  };

  if (groups.length === 0) {
    return (
      <div className="text-center text-muted-foreground py-4 text-sm">
        Brak zlecen do wyswietlenia
      </div>
    );
  }

  return (
    <div className="space-y-2">
      {groups.map(group => {
        const isExpanded = expandedGroups.has(group.key);
        return (
          <Collapsible key={group.key} open={isExpanded} onOpenChange={() => toggleGroup(group.key)}>
            <CollapsibleTrigger asChild>
              <Card className="hover-elevate cursor-pointer">
                <CardContent className="p-3">
                  <div className="flex items-center justify-between gap-2">
                    <div className="flex items-center gap-2 min-w-0 flex-1">
                      {groupBy === 'color' && (
                        <Palette className="h-4 w-4 text-muted-foreground flex-shrink-0" />
                      )}
                      {groupBy === 'routing' && (
                        <Route className="h-4 w-4 text-muted-foreground flex-shrink-0" />
                      )}
                      {groupBy === 'color-routing' && (
                        <>
                          <Palette className="h-4 w-4 text-muted-foreground flex-shrink-0" />
                          <Route className="h-4 w-4 text-muted-foreground flex-shrink-0" />
                        </>
                      )}
                      <span className="font-medium text-sm truncate">{group.label}</span>
                      <Badge variant="secondary" className="text-xs flex-shrink-0">
                        {group.zlps.length} ZLP
                      </Badge>
                    </div>
                    <div className="flex items-center gap-2 flex-shrink-0">
                      <div className="flex items-center gap-1 text-xs text-muted-foreground">
                        <span>{group.progress}%</span>
                      </div>
                      {isExpanded ? (
                        <ChevronUp className="h-4 w-4 text-muted-foreground" />
                      ) : (
                        <ChevronDown className="h-4 w-4 text-muted-foreground" />
                      )}
                    </div>
                  </div>
                  <Progress value={group.progress} className="h-1.5 mt-2" />
                  <div className="flex items-center gap-4 mt-2 text-xs text-muted-foreground">
                    <span className="flex items-center gap-1">
                      <CheckCircle className="h-3 w-3 text-green-500" />
                      {group.completedCount}/{group.zlps.length} zakonczone
                    </span>
                  </div>
                </CardContent>
              </Card>
            </CollapsibleTrigger>
            <CollapsibleContent>
              <div className="space-y-2 pl-4 pt-2">
                {group.zlps.map(zlp => (
                  <ZlpCard key={zlp.id} zlp={zlp} />
                ))}
              </div>
            </CollapsibleContent>
          </Collapsible>
        );
      })}
    </div>
  );
}

export function ZlpDashboard({ planId, isOpen, onOpenChange }: ZlpDashboardProps) {
  const [viewMode, setViewMode] = useState<ViewMode>('grouped');
  const [groupBy, setGroupBy] = useState<GroupBy>('color-routing');
  
  const { data: dashboard, isLoading, error } = useQuery<ZlpDashboardData>({
    queryKey: [`/api/production/planning/plans/${planId}/zlp-dashboard`],
    enabled: isOpen && !!planId,
  });

  if (!isOpen) return null;

  return (
    <div 
      className={`fixed top-0 right-0 h-full bg-background border-l shadow-xl z-[9999] transition-all duration-300 ${
        isOpen ? 'w-[420px]' : 'w-0'
      }`}
      data-testid="zlp-dashboard-panel"
    >
      <div className="flex flex-col h-full">
        <div className="flex items-center justify-between p-4 border-b">
          <div className="flex items-center gap-2">
            <Factory className="h-5 w-5 text-primary" />
            <h2 className="font-semibold">Dashboard ZLP</h2>
          </div>
          <Button 
            variant="ghost" 
            size="icon" 
            onClick={() => onOpenChange(false)}
            data-testid="button-close-zlp-dashboard"
          >
            <ExternalLink className="h-4 w-4 rotate-180" />
          </Button>
        </div>

        <ScrollArea className="flex-1">
          <div className="p-4 space-y-4">
            {isLoading && (
              <div className="flex items-center justify-center py-8">
                <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
              </div>
            )}

            {error && (
              <div className="text-center text-destructive py-8">
                Blad ladowania danych ZLP
              </div>
            )}

            {dashboard && (
              <>
                <Card>
                  <CardHeader className="pb-2">
                    <CardTitle className="text-sm flex items-center gap-2">
                      <BarChart3 className="h-4 w-4" />
                      Podsumowanie
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-3">
                    <div className="grid grid-cols-4 gap-2 text-center">
                      <div className="space-y-1">
                        <div className="text-2xl font-bold">{dashboard.totalZlps}</div>
                        <div className="text-xs text-muted-foreground">Lacznie</div>
                      </div>
                      <div className="space-y-1">
                        <div className="text-2xl font-bold text-green-600">{dashboard.completedZlps}</div>
                        <div className="text-xs text-muted-foreground">Zakonczone</div>
                      </div>
                      <div className="space-y-1">
                        <div className="text-2xl font-bold text-yellow-600">{dashboard.inProgressZlps}</div>
                        <div className="text-xs text-muted-foreground">W trakcie</div>
                      </div>
                      <div className="space-y-1">
                        <div className="text-2xl font-bold text-gray-500">{dashboard.pendingZlps}</div>
                        <div className="text-xs text-muted-foreground">Oczekuje</div>
                      </div>
                    </div>

                    <div className="space-y-1">
                      <div className="flex items-center justify-between text-xs">
                        <span className="text-muted-foreground">Postep ogolny</span>
                        <span className="font-medium">{dashboard.overallProgress}%</span>
                      </div>
                      <Progress value={dashboard.overallProgress} className="h-2" />
                    </div>
                  </CardContent>
                </Card>

                <Separator />

                <div className="space-y-2">
                  <div className="flex items-center justify-between">
                    <h3 className="text-sm font-medium flex items-center gap-2">
                      <Factory className="h-4 w-4" />
                      Zlecenia produkcyjne ({dashboard.zlps.length})
                    </h3>
                    <div className="flex items-center gap-1">
                      <Tooltip>
                        <TooltipTrigger asChild>
                          <Button
                            variant={viewMode === 'list' ? 'secondary' : 'ghost'}
                            size="icon"
                            className="h-7 w-7"
                            onClick={() => setViewMode('list')}
                            data-testid="button-view-list"
                          >
                            <List className="h-3.5 w-3.5" />
                          </Button>
                        </TooltipTrigger>
                        <TooltipContent>Lista</TooltipContent>
                      </Tooltip>
                      <Tooltip>
                        <TooltipTrigger asChild>
                          <Button
                            variant={viewMode === 'grouped' ? 'secondary' : 'ghost'}
                            size="icon"
                            className="h-7 w-7"
                            onClick={() => setViewMode('grouped')}
                            data-testid="button-view-grouped"
                          >
                            <Grid className="h-3.5 w-3.5" />
                          </Button>
                        </TooltipTrigger>
                        <TooltipContent>Grupuj</TooltipContent>
                      </Tooltip>
                    </div>
                  </div>
                  
                  {viewMode === 'grouped' && (
                    <div className="flex items-center gap-1 text-xs">
                      <span className="text-muted-foreground">Grupuj wg:</span>
                      <Button
                        variant={groupBy === 'color' ? 'secondary' : 'ghost'}
                        size="sm"
                        className="h-6 px-2 text-xs"
                        onClick={() => setGroupBy('color')}
                        data-testid="button-group-by-color"
                      >
                        <Palette className="h-3 w-3 mr-1" />
                        Kolor
                      </Button>
                      <Button
                        variant={groupBy === 'routing' ? 'secondary' : 'ghost'}
                        size="sm"
                        className="h-6 px-2 text-xs"
                        onClick={() => setGroupBy('routing')}
                        data-testid="button-group-by-routing"
                      >
                        <Route className="h-3 w-3 mr-1" />
                        Marszruta
                      </Button>
                      <Button
                        variant={groupBy === 'color-routing' ? 'secondary' : 'ghost'}
                        size="sm"
                        className="h-6 px-2 text-xs"
                        onClick={() => setGroupBy('color-routing')}
                        data-testid="button-group-by-both"
                      >
                        Oba
                      </Button>
                    </div>
                  )}
                  
                  {viewMode === 'list' ? (
                    <div className="space-y-2">
                      {dashboard.zlps.map((zlp) => (
                        <ZlpCard key={zlp.id} zlp={zlp} />
                      ))}
                    </div>
                  ) : (
                    <GroupedZlpList zlps={dashboard.zlps} groupBy={groupBy} />
                  )}
                </div>

                {dashboard.components.length > 0 && (
                  <>
                    <Separator />
                    <div className="space-y-2">
                      <h3 className="text-sm font-medium flex items-center gap-2">
                        <Layers className="h-4 w-4" />
                        Komponenty ({dashboard.components.length})
                      </h3>
                      <ComponentsTable components={dashboard.components} />
                    </div>
                  </>
                )}
              </>
            )}
          </div>
        </ScrollArea>
      </div>
    </div>
  );
}
