import { useState, useRef, useEffect } from "react";
import { Card, CardContent, CardHeader } from "@/components/ui/card";
import { CategorySelector, type ProductSource } from "./CategorySelector";
import { OrdersPanel } from "./OrdersPanel";
import { CatalogProductsList } from "./CatalogProductsList";
import { CuttingPatternsList } from "./CuttingPatternsList";
import { Loader2, Package, Scissors } from "lucide-react";
import { Input } from "@/components/ui/input";
import { Search } from "lucide-react";
import { useScrollCollapse } from "./use-scroll-collapse";

interface SetComponent {
  component_id: number;
  component_sku: string;
  component_title: string;
  component_color: string | null;
  component_length: string | null;
  component_width: string | null;
  component_height: string | null;
  component_product_type: string | null;
  component_doors: string | null;
  component_legs: string | null;
  quantity: number;
  primary_image_url: string | null;
  parent_set_image_url: string | null;
  is_in_current_plan: boolean;
  is_in_any_plan: boolean;
  in_plan_number: string | null;
  in_plan_id: number | null;
}

interface MarketplaceOrderItem {
  item_id: number;
  offer_external_id: string;
  name: string;
  quantity: number;
  unit_price: string;
  price: string;
  image_url: string | null;
  product_length: string | null;
  product_width: string | null;
  product_height: string | null;
  product_color: string | null;
  product_color_options: string[] | null;
  product_sku: string | null;
  product_type: string | null;
  product_doors: string | null;
  product_legs: string | null;
  marketplace_product_id: number | null;
  link_type: string | null;
  catalog_product_id: number | null;
  catalog_product_sku: string | null;
  catalog_product_title: string | null;
  catalog_set_id: number | null;
  catalog_set_sku: string | null;
  catalog_set_title: string | null;
  platform_link_id: number | null;
  bom_component_count: number | null;
  is_in_plan: boolean;
  in_plan_number: string | null;
  in_plan_id: number | null;
  in_current_plan: boolean;
  set_components: SetComponent[] | null;
  warehouse_total_qty: number | null;
  warehouse_reserved_qty: number | null;
  line_reserved_qty: number | null;
}

interface MarketplaceOrder {
  order_id: number;
  order_number: string;
  marketplace: string;
  buyer_first_name: string;
  buyer_last_name: string;
  buyer_email: string;
  order_date: string;
  payment_status: string;
  total_to_pay_amount: string;
  currency: string;
  items: MarketplaceOrderItem[];
}

interface CatalogProduct {
  id: number;
  sku: string;
  title: string;
  color: string | null;
  color_options: string[] | null;
  length: string | null;
  width: string | null;
  height: string | null;
  product_type: string | null;
  doors: string | null;
  legs: string | null;
  created_at: string;
  image_url: string | null;
  bom_component_count: number;
  product_group_id: number | null;
  product_group_name: string | null;
  product_group_color_hex: string | null;
  is_in_plan: boolean;
  in_plan_number: string | null;
  in_current_plan: boolean;
  warehouse_total_qty: number | null;
  warehouse_reserved_qty: number | null;
}

interface CuttingPattern {
  id: number;
  code: string;
  name: string;
  description: string | null;
  status: string;
  board_length: string;
  board_width: string;
  board_thickness: string;
  kerf: string;
  total_quantity: number;
  items_count: number;
  created_at: string;
  is_in_plan: boolean;
  in_plan_number: string | null;
  in_current_plan: boolean;
}

interface DictionaryItem {
  id: number;
  code: string;
  name: string | null;
  shortName: string | null;
  color: string | null;
  isActive: boolean;
}

interface AvailableSourcesPanelProps {
  planId: string;
  
  // Customer Orders
  orders: MarketplaceOrder[];
  ordersLoading: boolean;
  ordersStatistics: {
    ordersCount: number;
    productsCount: number;
  };
  ordersFilters: any;
  ordersSearchFilter: string;
  onOrdersSearchChange: (value: string) => void;
  onOrdersFilterChange: (key: string, value: string) => void;
  onOrdersClearFilter: (key: string) => void;
  onOrdersResetFilters: () => void;
  onAddOrderItem: (item: MarketplaceOrderItem, orderNumber: string, buyerName: string, marketplace: string) => void;
  onLinkOrderItem: (item: MarketplaceOrderItem, marketplace: string) => void;
  ordersActiveFiltersCount: number;
  isAddingItem?: boolean;
  addingItems?: Array<{
    id: string;
    productSku: string;
    productName: string;
    quantity: number;
    orderNumber?: string;
    status: 'pending' | 'processing' | 'completed';
    reservationInfo?: {
      packedProductReserved: boolean;
      packedProductSku?: string;
      packedQtyReserved?: number;
      formatkaSearched: boolean;
      formatkiFound: string[];
      formatkiReserved: string[];
    };
  }>;
  recentlyAddedItems?: Set<number>;
  onHighlightInPlan?: (productId: number, orderNumber: string) => void;
  
  // Catalog Products
  catalogProducts?: CatalogProduct[];
  catalogLoading?: boolean;
  onAddCatalogProduct?: (product: CatalogProduct) => void;
  recentlyAddedProducts?: Set<number>;
  
  // Cutting Patterns
  cuttingPatterns?: CuttingPattern[];
  patternsLoading?: boolean;
  onAddCuttingPattern?: (pattern: CuttingPattern) => void;
  recentlyAddedPatterns?: Set<number>;
  
  // Shared
  colors?: DictionaryItem[];
  productTypes?: DictionaryItem[];
  dimensionLengths?: DictionaryItem[];
  dimensionWidths?: DictionaryItem[];
  
  // Plan locked state (ZLPs have been generated)
  isLocked?: boolean;
}

export function AvailableSourcesPanel(props: AvailableSourcesPanelProps) {
  const [selectedSource, setSelectedSource] = useState<ProductSource>('customer_orders');
  const [catalogSearch, setCatalogSearch] = useState('');
  const [patternsSearch, setPatternsSearch] = useState('');

  // Track collapse state for all three panels
  const [ordersCollapsed, setOrdersCollapsed] = useState(false);
  const [catalogCollapsed, setCatalogCollapsed] = useState(false);
  const [patternsCollapsed, setPatternsCollapsed] = useState(false);

  // Scroll refs for catalog and patterns (OrdersPanel has its own)
  const catalogScrollRef = useRef<HTMLDivElement>(null);
  const patternsScrollRef = useRef<HTMLDivElement>(null);
  
  // Use scroll collapse hooks for catalog and patterns panels
  useScrollCollapse({ 
    scrollRef: catalogScrollRef, 
    threshold: 80,
    onChange: setCatalogCollapsed,
    enabled: selectedSource === 'catalog_products'
  });
  
  useScrollCollapse({ 
    scrollRef: patternsScrollRef, 
    threshold: 80,
    onChange: setPatternsCollapsed,
    enabled: selectedSource === 'cutting_patterns'
  });

  // Filter catalog products by search
  const filteredCatalogProducts = (props.catalogProducts || []).filter(product => {
    if (!catalogSearch) return true;
    const searchLower = catalogSearch.toLowerCase();
    return (
      product.title.toLowerCase().includes(searchLower) ||
      product.sku.toLowerCase().includes(searchLower) ||
      product.color?.toLowerCase().includes(searchLower)
    );
  });

  // Filter cutting patterns by search
  const filteredPatterns = (props.cuttingPatterns || []).filter(pattern => {
    if (!patternsSearch) return true;
    const searchLower = patternsSearch.toLowerCase();
    return (
      pattern.name.toLowerCase().includes(searchLower) ||
      pattern.code.toLowerCase().includes(searchLower) ||
      pattern.description?.toLowerCase().includes(searchLower)
    );
  });

  // Determine if selector should be compact based on scroll state
  const isCompact = selectedSource === 'customer_orders'
    ? ordersCollapsed
    : selectedSource === 'catalog_products' 
    ? catalogCollapsed 
    : patternsCollapsed;

  return (
    <div className="h-full flex flex-col">
      {/* CategorySelector - automatically switches between full and compact modes */}
      <div className="flex-shrink-0 px-4 pt-2">
        <CategorySelector
          selectedSource={selectedSource}
          onSourceChange={setSelectedSource}
          compact={isCompact}
        />
      </div>

      {/* Source-specific content */}
      <div className="flex-1 min-h-0 overflow-hidden">
        {selectedSource === 'customer_orders' && (
          <OrdersPanel
            planId={props.planId}
            orders={props.orders}
            isLoading={props.ordersLoading}
            statistics={props.ordersStatistics}
            filters={props.ordersFilters}
            searchFilter={props.ordersSearchFilter}
            onSearchChange={props.onOrdersSearchChange}
            onFilterChange={props.onOrdersFilterChange}
            onClearFilter={props.onOrdersClearFilter}
            onResetFilters={props.onOrdersResetFilters}
            onAddItem={props.onAddOrderItem}
            onLinkItem={props.onLinkOrderItem}
            activeFiltersCount={props.ordersActiveFiltersCount}
            isAddingItem={props.isAddingItem}
            addingItems={props.addingItems}
            recentlyAddedItems={props.recentlyAddedItems}
            colors={props.colors}
            productTypes={props.productTypes}
            dimensionLengths={props.dimensionLengths}
            dimensionWidths={props.dimensionWidths}
            onCollapseChange={setOrdersCollapsed}
            enabled={selectedSource === 'customer_orders'}
            onHighlightInPlan={props.onHighlightInPlan}
            isLocked={props.isLocked}
          />
        )}

        {selectedSource === 'catalog_products' && (
          <Card className="h-full flex flex-col rounded-none border-0 overflow-hidden">
            {/* Collapsible Header + Search */}
            <div 
              className={`flex-shrink-0 transition-all duration-300 ease-in-out overflow-hidden ${
                catalogCollapsed ? 'max-h-0 opacity-0' : 'max-h-[500px] opacity-100'
              }`}
              style={{ transitionProperty: 'max-height, opacity' }}
            >
              <CardHeader className="flex-shrink-0 pb-3 px-4 border-b">
                <div className="flex items-center gap-2">
                  <Package className="h-5 w-5" />
                  <h3 className="font-semibold">Produkty Katalogowe</h3>
                </div>
              </CardHeader>
              <div className="flex-shrink-0 flex items-center gap-2 px-4 pt-4 pb-2">
                <Search className="h-4 w-4 text-muted-foreground flex-shrink-0" />
                <Input
                  placeholder="Szukaj po nazwie, SKU, kolorze..."
                  value={catalogSearch}
                  onChange={(e) => setCatalogSearch(e.target.value)}
                  className="flex-1"
                  data-testid="input-search-catalog-products"
                />
              </div>
            </div>

            {/* Scrollable Products List */}
            <div ref={catalogScrollRef} className="flex-1 overflow-y-auto px-4 pb-4" data-testid="catalog-products-list-container">
              {props.catalogLoading ? (
                <div className="flex items-center justify-center p-8">
                  <Loader2 className="h-6 w-6 animate-spin" />
                </div>
              ) : (
                <CatalogProductsList
                  products={filteredCatalogProducts}
                  colors={props.colors || []}
                  onAddProduct={props.onAddCatalogProduct || (() => {})}
                  isAddingProduct={props.isAddingItem}
                  recentlyAddedProducts={props.recentlyAddedProducts}
                  isLocked={props.isLocked}
                />
              )}
            </div>
          </Card>
        )}

        {selectedSource === 'cutting_patterns' && (
          <Card className="h-full flex flex-col rounded-none border-0 overflow-hidden">
            {/* Collapsible Header + Search */}
            <div 
              className={`flex-shrink-0 transition-all duration-300 ease-in-out overflow-hidden ${
                patternsCollapsed ? 'max-h-0 opacity-0' : 'max-h-[500px] opacity-100'
              }`}
              style={{ transitionProperty: 'max-height, opacity' }}
            >
              <CardHeader className="flex-shrink-0 pb-3 px-4 border-b">
                <div className="flex items-center gap-2">
                  <Scissors className="h-5 w-5" />
                  <h3 className="font-semibold">Szablony Rozkrojów</h3>
                </div>
              </CardHeader>
              <div className="flex-shrink-0 flex items-center gap-2 px-4 pt-4 pb-2">
                <Search className="h-4 w-4 text-muted-foreground flex-shrink-0" />
                <Input
                  placeholder="Szukaj po nazwie, kodzie..."
                  value={patternsSearch}
                  onChange={(e) => setPatternsSearch(e.target.value)}
                  className="flex-1"
                  data-testid="input-search-cutting-patterns"
                />
              </div>
            </div>

            {/* Scrollable Patterns List */}
            <div ref={patternsScrollRef} className="flex-1 overflow-y-auto px-4 pb-4" data-testid="cutting-patterns-list-container">
              {props.patternsLoading ? (
                <div className="flex items-center justify-center p-8">
                  <Loader2 className="h-6 w-6 animate-spin" />
                </div>
              ) : (
                <CuttingPatternsList
                  patterns={filteredPatterns}
                  onAddPattern={props.onAddCuttingPattern || (() => {})}
                  isAddingPattern={props.isAddingItem}
                  recentlyAddedPatterns={props.recentlyAddedPatterns}
                  isLocked={props.isLocked}
                />
              )}
            </div>
          </Card>
        )}
      </div>
    </div>
  );
}
