import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Combobox, ComboboxOption } from "@/components/ui/combobox";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Card } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Loader2, Search, Package, Check, ArrowRightLeft, Plus } from "lucide-react";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";

interface CatalogItem {
  id: number;
  item_type: 'product' | 'set';
  sku: string;
  name: string;
  color: string | null;
  length: number | null;
  width: number | null;
  height: number | null;
  type: string | null;
  price: string | null;
  image_url: string | null;
}

interface FilterOptions {
  colors: ComboboxOption[];
  productTypes: ComboboxOption[];
  lengths: ComboboxOption[];
  widths: ComboboxOption[];
  doors: ComboboxOption[];
  legs: ComboboxOption[];
  colorOptions: ComboboxOption[];
}

interface ReplaceProductDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  orderId: string;
  orderNumber: string;
  mode: 'replace' | 'add';
  currentItem?: {
    id: string;
    name: string;
    sku: string;
    quantity: number;
    price: { amount: string; currency: string };
    color?: string;
    length?: number;
    width?: number;
    productType?: string;
    doors?: string;
    legs?: string;
  };
  onSuccess?: () => void;
}

export function ReplaceProductDialog({
  open,
  onOpenChange,
  orderId,
  orderNumber,
  mode,
  currentItem,
  onSuccess
}: ReplaceProductDialogProps) {
  const { toast } = useToast();
  const [filters, setFilters] = useState({
    color: '',
    length: '',
    width: '',
    type: '',
    doors: '',
    legs: '',
    colorOption: '',
    sku: ''
  });
  const [selectedItem, setSelectedItem] = useState<CatalogItem | null>(null);
  const [quantity, setQuantity] = useState(1);

  const { data: filterOptions } = useQuery<FilterOptions>({
    queryKey: ["/api/catalog/filter-options"],
    enabled: open,
  });

  useEffect(() => {
    if (open) {
      setFilters({
        color: currentItem?.color || '',
        length: currentItem?.length ? String(currentItem.length) : '',
        width: currentItem?.width ? String(currentItem.width) : '',
        type: currentItem?.productType || '',
        doors: currentItem?.doors || '',
        legs: currentItem?.legs || '',
        colorOption: '',
        sku: ''
      });
      setSelectedItem(null);
      setQuantity(currentItem?.quantity || 1);
    }
  }, [open, currentItem]);

  const searchParams = new URLSearchParams();
  if (filters.color) searchParams.set('color', filters.color);
  if (filters.length) searchParams.set('length', filters.length);
  if (filters.width) searchParams.set('width', filters.width);
  if (filters.type) searchParams.set('type', filters.type);
  if (filters.doors) searchParams.set('doors', filters.doors);
  if (filters.legs) searchParams.set('legs', filters.legs);
  if (filters.colorOption) searchParams.set('colorOption', filters.colorOption);
  if (filters.sku) searchParams.set('sku', filters.sku);
  searchParams.set('limit', '30');

  const hasFilters = Boolean(filters.color || filters.length || filters.width || filters.type || filters.doors || filters.legs || filters.colorOption || filters.sku);

  const { data: searchResults, isLoading: isSearching } = useQuery<{ items: CatalogItem[]; facets: { colors: string[]; types: string[]; lengths: number[]; widths: number[] } }>({
    queryKey: [`/api/catalog/items/search?${searchParams.toString()}`],
    enabled: open && hasFilters,
  });

  const replaceMutation = useMutation({
    mutationFn: async (data: { catalogProductId?: number; catalogSetId?: number; quantity: number }) => {
      if (mode === 'replace' && currentItem) {
        return apiRequest('PATCH', `/api/orders/${orderId}/items/${currentItem.id}/replace`, data);
      } else {
        return apiRequest('POST', `/api/orders/${orderId}/items`, data);
      }
    },
    onSuccess: () => {
      toast({
        title: mode === 'replace' ? "Produkt zamieniony" : "Produkt dodany",
        description: mode === 'replace' 
          ? `Produkt "${currentItem?.name}" został zamieniony na "${selectedItem?.name}"`
          : `Produkt "${selectedItem?.name}" został dodany do zamówienia`,
      });
      queryClient.invalidateQueries({ queryKey: [`/api/orders/by-number/${orderNumber}`] });
      queryClient.invalidateQueries({ queryKey: [`/api/orders/${orderNumber}/products-info`] });
      queryClient.invalidateQueries({ queryKey: [`/api/orders/${orderId}/component-upholstery`] });
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
      queryClient.invalidateQueries({ queryKey: ["/api/orders/statistics"] });
      onOpenChange(false);
      onSuccess?.();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error?.message || "Nie udało się wykonać operacji",
        variant: "destructive",
      });
    },
  });

  const handleSubmit = () => {
    if (!selectedItem) return;
    
    const data: { catalogProductId?: number; catalogSetId?: number; quantity: number } = {
      quantity,
    };
    
    if (selectedItem.item_type === 'product') {
      data.catalogProductId = selectedItem.id;
    } else {
      data.catalogSetId = selectedItem.id;
    }
    
    replaceMutation.mutate(data);
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-3xl max-h-[90vh] flex flex-col">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            {mode === 'replace' ? (
              <>
                <ArrowRightLeft className="h-5 w-5" />
                Zamień produkt
              </>
            ) : (
              <>
                <Plus className="h-5 w-5" />
                Dodaj produkt
              </>
            )}
          </DialogTitle>
          <DialogDescription>
            {mode === 'replace' && currentItem ? (
              <>Zamień "{currentItem.name}" ({currentItem.sku}) na inny produkt z katalogu</>
            ) : (
              <>Wyszukaj i dodaj produkt z katalogu do zamówienia</>
            )}
          </DialogDescription>
        </DialogHeader>

        <div className="grid gap-4 py-4">
          <div className="grid grid-cols-2 sm:grid-cols-3 gap-3">
            <div className="space-y-1.5">
              <Label className="text-xs">Kolor</Label>
              <Combobox
                options={filterOptions?.colors || []}
                value={filters.color}
                onChange={(val) => setFilters(f => ({ ...f, color: val }))}
                placeholder="Wybierz kolor..."
                searchPlaceholder="Szukaj koloru..."
                emptyText="Brak kolorów"
                className="w-full"
              />
            </div>
            <div className="space-y-1.5">
              <Label className="text-xs">Długość (cm)</Label>
              <Combobox
                options={filterOptions?.lengths || []}
                value={filters.length}
                onChange={(val) => setFilters(f => ({ ...f, length: val }))}
                placeholder="Wybierz długość..."
                searchPlaceholder="Szukaj długości..."
                emptyText="Brak długości"
                className="w-full"
              />
            </div>
            <div className="space-y-1.5">
              <Label className="text-xs">Szerokość (cm)</Label>
              <Combobox
                options={filterOptions?.widths || []}
                value={filters.width}
                onChange={(val) => setFilters(f => ({ ...f, width: val }))}
                placeholder="Wybierz szerokość..."
                searchPlaceholder="Szukaj szerokości..."
                emptyText="Brak szerokości"
                className="w-full"
              />
            </div>
            <div className="space-y-1.5">
              <Label className="text-xs">Rodzaj</Label>
              <Combobox
                options={filterOptions?.productTypes || []}
                value={filters.type}
                onChange={(val) => setFilters(f => ({ ...f, type: val }))}
                placeholder="Wybierz rodzaj..."
                searchPlaceholder="Szukaj rodzaju..."
                emptyText="Brak rodzajów"
                className="w-full"
              />
            </div>
            <div className="space-y-1.5">
              <Label className="text-xs">Drzwi</Label>
              <Combobox
                options={filterOptions?.doors || []}
                value={filters.doors}
                onChange={(val) => setFilters(f => ({ ...f, doors: val }))}
                placeholder="Wybierz drzwi..."
                searchPlaceholder="Szukaj drzwi..."
                emptyText="Brak drzwi"
                className="w-full"
              />
            </div>
            <div className="space-y-1.5">
              <Label className="text-xs">Nogi</Label>
              <Combobox
                options={filterOptions?.legs || []}
                value={filters.legs}
                onChange={(val) => setFilters(f => ({ ...f, legs: val }))}
                placeholder="Wybierz nogi..."
                searchPlaceholder="Szukaj nóg..."
                emptyText="Brak nóg"
                className="w-full"
              />
            </div>
            <div className="space-y-1.5">
              <Label className="text-xs">Opcja kolorystyczna</Label>
              <Combobox
                options={filterOptions?.colorOptions || []}
                value={filters.colorOption}
                onChange={(val) => setFilters(f => ({ ...f, colorOption: val }))}
                placeholder="Zmiana koloru drzwi..."
                searchPlaceholder="Szukaj opcji..."
                emptyText="Brak opcji"
                className="w-full"
              />
            </div>
            <div className="space-y-1.5 col-span-2 sm:col-span-2">
              <Label htmlFor="filter-sku" className="text-xs">SKU</Label>
              <Input
                id="filter-sku"
                placeholder="Szukaj po SKU..."
                value={filters.sku}
                onChange={(e) => setFilters(f => ({ ...f, sku: e.target.value }))}
                data-testid="input-filter-sku"
              />
            </div>
          </div>

          <ScrollArea className="h-[300px] rounded-md border">
            {!hasFilters ? (
              <div className="flex flex-col items-center justify-center h-full text-muted-foreground py-8">
                <Search className="h-10 w-10 mb-3 opacity-50" />
                <p className="text-sm">Wprowadź kryteria wyszukiwania</p>
                <p className="text-xs">Kolor, długość, szerokość, rodzaj lub SKU</p>
              </div>
            ) : isSearching ? (
              <div className="flex items-center justify-center h-full py-8">
                <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
              </div>
            ) : !searchResults?.items?.length ? (
              <div className="flex flex-col items-center justify-center h-full text-muted-foreground py-8">
                <Package className="h-10 w-10 mb-3 opacity-50" />
                <p className="text-sm">Brak wyników</p>
                <p className="text-xs">Spróbuj zmienić kryteria wyszukiwania</p>
              </div>
            ) : (
              <div className="p-2 space-y-2">
                {searchResults.items.map((item) => (
                  <Card
                    key={`${item.item_type}-${item.id}`}
                    className={`p-3 cursor-pointer transition-colors hover-elevate ${
                      selectedItem?.id === item.id && selectedItem?.item_type === item.item_type
                        ? 'ring-2 ring-primary bg-primary/5'
                        : ''
                    }`}
                    onClick={() => setSelectedItem(item)}
                    data-testid={`card-catalog-item-${item.id}`}
                  >
                    <div className="flex gap-3">
                      <div className="w-16 h-16 flex-shrink-0 rounded overflow-hidden bg-muted flex items-center justify-center">
                        {item.image_url ? (
                          <img
                            src={item.image_url}
                            alt={item.name}
                            className="w-full h-full object-cover"
                          />
                        ) : (
                          <Package className="h-6 w-6 text-muted-foreground" />
                        )}
                      </div>
                      <div className="flex-1 min-w-0">
                        <div className="flex items-start justify-between gap-2">
                          <div className="font-medium text-sm truncate">{item.name}</div>
                          {selectedItem?.id === item.id && selectedItem?.item_type === item.item_type && (
                            <Check className="h-4 w-4 text-primary flex-shrink-0" />
                          )}
                        </div>
                        <div className="text-xs text-muted-foreground font-mono">{item.sku}</div>
                        <div className="flex flex-wrap gap-1 mt-1">
                          <Badge variant="secondary" className="text-xs">
                            {item.item_type === 'product' ? 'Produkt' : 'Zestaw'}
                          </Badge>
                          {item.color && (
                            <Badge variant="outline" className="text-xs">{item.color}</Badge>
                          )}
                          {item.length && (
                            <Badge variant="outline" className="text-xs">{item.length}cm</Badge>
                          )}
                          {item.width && (
                            <Badge variant="outline" className="text-xs">szer. {item.width}cm</Badge>
                          )}
                        </div>
                        {item.price && (
                          <div className="text-sm font-medium mt-1">{parseFloat(item.price).toFixed(2)} PLN</div>
                        )}
                      </div>
                    </div>
                  </Card>
                ))}
              </div>
            )}
          </ScrollArea>

          {selectedItem && (
            <div className="flex items-center gap-4 p-3 bg-muted/50 rounded-md">
              <div className="flex-1">
                <div className="text-sm font-medium">Wybrany: {selectedItem.name}</div>
                <div className="text-xs text-muted-foreground">{selectedItem.sku}</div>
              </div>
              <div className="flex items-center gap-2">
                <Label htmlFor="quantity" className="text-sm whitespace-nowrap">Ilość:</Label>
                <Input
                  id="quantity"
                  type="number"
                  min={1}
                  value={quantity}
                  onChange={(e) => setQuantity(Math.max(1, parseInt(e.target.value) || 1))}
                  className="w-20"
                  data-testid="input-quantity"
                />
              </div>
            </div>
          )}
        </div>

        <DialogFooter>
          <Button variant="outline" onClick={() => onOpenChange(false)} data-testid="button-cancel">
            Anuluj
          </Button>
          <Button
            onClick={handleSubmit}
            disabled={!selectedItem || replaceMutation.isPending}
            data-testid="button-confirm-replace"
          >
            {replaceMutation.isPending && <Loader2 className="h-4 w-4 mr-2 animate-spin" />}
            {mode === 'replace' ? 'Zamień produkt' : 'Dodaj produkt'}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
