import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger, DialogFooter, DialogClose } from "@/components/ui/dialog";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { useToast } from "@/hooks/use-toast";
import { 
  Package, Plus, RefreshCw, Truck, Warehouse, Timer, 
  CheckCircle, AlertCircle, Clock, Play, Square, ChevronDown, ChevronUp
} from "lucide-react";
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible";

interface Pallet {
  id: number;
  production_order_id: number;
  pallet_label: string;
  sequence: number;
  color_code: string | null;
  flow_code: string;
  carrier_id: number | null;
  carrier_code: string | null;
  carrier_code_resolved: string | null;
  carrier_name: string | null;
  status: string;
  is_filled: boolean;
  max_capacity: number;
  current_load: number;
  fill_percentage: string | null;
  current_operation_code: string | null;
  warehouse_id: number | null;
  warehouse_entry_time: string | null;
  notes: string | null;
}

interface Carrier {
  id: number;
  code: string;
  name: string;
  carrier_group_id: number;
}

interface Location {
  id: number;
  name: string;
  code: string;
  type: string;
}

interface WarehouseStay {
  id: number;
  pallet_id: number;
  pallet_label: string;
  color_code: string | null;
  flow_code: string;
  warehouse_id: number;
  warehouse_name_resolved: string | null;
  start_time: string;
  current_duration_seconds: number;
  status: string;
}

interface PalletManagerProps {
  productionOrderId: number;
  colorCode?: string | null;
}

function formatDuration(seconds: number): string {
  const hours = Math.floor(seconds / 3600);
  const mins = Math.floor((seconds % 3600) / 60);
  const secs = seconds % 60;
  if (hours > 0) {
    return `${hours}h ${mins}m ${secs}s`;
  }
  return `${mins}m ${secs}s`;
}

function PalletStatusBadge({ status, isFilled }: { status: string; isFilled: boolean }) {
  const statusConfig: Record<string, { label: string; className: string }> = {
    open: { label: "Otwarta", className: "bg-blue-100 text-blue-700 dark:bg-blue-900/30 dark:text-blue-300" },
    filled: { label: "Zapełniona", className: "bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-300" },
    in_buffer: { label: "W buforze", className: "bg-yellow-100 text-yellow-700 dark:bg-yellow-900/30 dark:text-yellow-300" },
    in_operation: { label: "W operacji", className: "bg-purple-100 text-purple-700 dark:bg-purple-900/30 dark:text-purple-300" },
    in_warehouse: { label: "Na magazynie", className: "bg-orange-100 text-orange-700 dark:bg-orange-900/30 dark:text-orange-300" },
    completed: { label: "Zakończona", className: "bg-gray-100 text-gray-700 dark:bg-gray-900/30 dark:text-gray-300" },
  };

  const config = statusConfig[status] || statusConfig.open;
  
  return (
    <Badge className={`text-[10px] ${config.className}`} data-testid={`badge-pallet-status-${status}`}>
      {config.label}
      {isFilled && status !== 'filled' && " (pełna)"}
    </Badge>
  );
}

function PalletCard({ pallet, carriers, locations, onRefresh }: { 
  pallet: Pallet; 
  carriers: Carrier[];
  locations: Location[];
  onRefresh: () => void;
}) {
  const { toast } = useToast();
  const [showCarrierDialog, setShowCarrierDialog] = useState(false);
  const [showWarehouseDialog, setShowWarehouseDialog] = useState(false);
  const [selectedCarrierId, setSelectedCarrierId] = useState<string>("");
  const [selectedWarehouseId, setSelectedWarehouseId] = useState<string>("");
  const [warehouseReason, setWarehouseReason] = useState("");

  const markFilledMutation = useMutation({
    mutationFn: async (isFilled: boolean) => {
      return apiRequest("POST", `/api/production/pallets/${pallet.id}/mark-filled`, { isFilled });
    },
    onSuccess: () => {
      toast({ title: pallet.is_filled ? "Paleta oznaczona jako otwarta" : "Paleta oznaczona jako zapełniona" });
      onRefresh();
    },
    onError: () => {
      toast({ title: "Błąd", description: "Nie udało się zmienić statusu palety", variant: "destructive" });
    },
  });

  const changeCarrierMutation = useMutation({
    mutationFn: async (data: { toCarrierId: number; toCarrierCode: string }) => {
      return apiRequest("POST", `/api/production/pallets/${pallet.id}/change-carrier`, data);
    },
    onSuccess: () => {
      toast({ title: "Nośnik zmieniony" });
      setShowCarrierDialog(false);
      onRefresh();
    },
    onError: () => {
      toast({ title: "Błąd", description: "Nie udało się zmienić nośnika", variant: "destructive" });
    },
  });

  const startWarehouseStayMutation = useMutation({
    mutationFn: async (data: { warehouseId: number; reason?: string }) => {
      return apiRequest("POST", `/api/production/pallets/${pallet.id}/warehouse-stay/start`, data);
    },
    onSuccess: () => {
      toast({ title: "Rozpoczęto pobyt na magazynie" });
      setShowWarehouseDialog(false);
      onRefresh();
    },
    onError: () => {
      toast({ title: "Błąd", description: "Nie udało się rozpocząć pobytu na magazynie", variant: "destructive" });
    },
  });

  const endWarehouseStayMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", `/api/production/pallets/${pallet.id}/warehouse-stay/end`, {});
    },
    onSuccess: () => {
      toast({ title: "Zakończono pobyt na magazynie" });
      onRefresh();
    },
    onError: () => {
      toast({ title: "Błąd", description: "Nie udało się zakończyć pobytu na magazynie", variant: "destructive" });
    },
  });

  const handleChangeCarrier = () => {
    const carrier = carriers.find(c => c.id === parseInt(selectedCarrierId));
    if (carrier) {
      changeCarrierMutation.mutate({ toCarrierId: carrier.id, toCarrierCode: carrier.code });
    }
  };

  const handleStartWarehouse = () => {
    if (selectedWarehouseId) {
      startWarehouseStayMutation.mutate({ 
        warehouseId: parseInt(selectedWarehouseId), 
        reason: warehouseReason || undefined 
      });
    }
  };

  const fillPercentage = parseFloat(pallet.fill_percentage || "0");
  const warehouseLocations = locations.filter(l => l.type === 'warehouse' || l.type === 'storage');

  return (
    <div 
      className="p-2 rounded-lg border bg-card hover-elevate transition-all"
      data-testid={`pallet-card-${pallet.id}`}
    >
      <div className="flex items-start justify-between gap-2 mb-2">
        <div className="flex-1 min-w-0">
          <div className="flex items-center gap-1.5">
            <Package className="h-3.5 w-3.5 text-muted-foreground flex-shrink-0" />
            <span className="font-medium text-sm truncate">{pallet.pallet_label}</span>
          </div>
          {pallet.color_code && (
            <div className="text-[10px] text-muted-foreground mt-0.5">
              Kolor: {pallet.color_code}
            </div>
          )}
        </div>
        <PalletStatusBadge status={pallet.status} isFilled={pallet.is_filled} />
      </div>

      <div className="grid grid-cols-2 gap-1 text-[10px] text-muted-foreground mb-2">
        <div className="flex items-center gap-1">
          <Truck className="h-3 w-3" />
          <span>{pallet.carrier_code_resolved || pallet.carrier_code || "Brak nośnika"}</span>
        </div>
        <div className="flex items-center gap-1">
          <span>{pallet.current_load}/{pallet.max_capacity}</span>
          <span className="text-muted-foreground">({fillPercentage.toFixed(0)}%)</span>
        </div>
      </div>

      {fillPercentage > 0 && (
        <div className="h-1.5 w-full bg-muted rounded-full mb-2 overflow-hidden">
          <div 
            className={`h-full transition-all ${fillPercentage >= 100 ? 'bg-red-500' : fillPercentage >= 75 ? 'bg-yellow-500' : 'bg-green-500'}`}
            style={{ width: `${Math.min(100, fillPercentage)}%` }}
          />
        </div>
      )}

      <div className="flex flex-wrap gap-1">
        <Button
          size="sm"
          variant={pallet.is_filled ? "outline" : "default"}
          className="h-6 text-[10px] px-2"
          onClick={() => markFilledMutation.mutate(!pallet.is_filled)}
          disabled={markFilledMutation.isPending}
          data-testid={`button-toggle-filled-${pallet.id}`}
        >
          {pallet.is_filled ? (
            <>
              <AlertCircle className="h-3 w-3 mr-1" />
              Otwórz
            </>
          ) : (
            <>
              <CheckCircle className="h-3 w-3 mr-1" />
              Zapełniona
            </>
          )}
        </Button>

        <Dialog open={showCarrierDialog} onOpenChange={setShowCarrierDialog}>
          <DialogTrigger asChild>
            <Button
              size="sm"
              variant="outline"
              className="h-6 text-[10px] px-2"
              data-testid={`button-change-carrier-${pallet.id}`}
            >
              <RefreshCw className="h-3 w-3 mr-1" />
              Zmień nośnik
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-sm">
            <DialogHeader>
              <DialogTitle>Zmień nośnik palety</DialogTitle>
            </DialogHeader>
            <div className="space-y-3">
              <div>
                <Label className="text-xs">Obecny nośnik</Label>
                <div className="text-sm font-medium">{pallet.carrier_name || pallet.carrier_code || "Brak"}</div>
              </div>
              <div>
                <Label className="text-xs">Nowy nośnik</Label>
                <Select value={selectedCarrierId} onValueChange={setSelectedCarrierId}>
                  <SelectTrigger data-testid="select-new-carrier">
                    <SelectValue placeholder="Wybierz nośnik" />
                  </SelectTrigger>
                  <SelectContent>
                    {carriers.map(c => (
                      <SelectItem key={c.id} value={c.id.toString()}>{c.code} - {c.name}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
            <DialogFooter>
              <DialogClose asChild>
                <Button variant="outline" size="sm">Anuluj</Button>
              </DialogClose>
              <Button 
                size="sm" 
                onClick={handleChangeCarrier}
                disabled={!selectedCarrierId || changeCarrierMutation.isPending}
                data-testid="button-confirm-carrier-change"
              >
                Zmień
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        {pallet.status !== 'in_warehouse' ? (
          <Dialog open={showWarehouseDialog} onOpenChange={setShowWarehouseDialog}>
            <DialogTrigger asChild>
              <Button
                size="sm"
                variant="outline"
                className="h-6 text-[10px] px-2"
                data-testid={`button-send-to-warehouse-${pallet.id}`}
              >
                <Warehouse className="h-3 w-3 mr-1" />
                Na magazyn
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-sm">
              <DialogHeader>
                <DialogTitle>Przenieś na magazyn</DialogTitle>
              </DialogHeader>
              <div className="space-y-3">
                <div>
                  <Label className="text-xs">Magazyn</Label>
                  <Select value={selectedWarehouseId} onValueChange={setSelectedWarehouseId}>
                    <SelectTrigger data-testid="select-warehouse">
                      <SelectValue placeholder="Wybierz magazyn" />
                    </SelectTrigger>
                    <SelectContent>
                      {warehouseLocations.length > 0 ? (
                        warehouseLocations.map(l => (
                          <SelectItem key={l.id} value={l.id.toString()}>{l.name}</SelectItem>
                        ))
                      ) : (
                        locations.slice(0, 10).map(l => (
                          <SelectItem key={l.id} value={l.id.toString()}>{l.name}</SelectItem>
                        ))
                      )}
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label className="text-xs">Powód (opcjonalnie)</Label>
                  <Input 
                    value={warehouseReason}
                    onChange={(e) => setWarehouseReason(e.target.value)}
                    placeholder="np. przerwa, oczekiwanie na części"
                    data-testid="input-warehouse-reason"
                  />
                </div>
              </div>
              <DialogFooter>
                <DialogClose asChild>
                  <Button variant="outline" size="sm">Anuluj</Button>
                </DialogClose>
                <Button 
                  size="sm" 
                  onClick={handleStartWarehouse}
                  disabled={!selectedWarehouseId || startWarehouseStayMutation.isPending}
                  data-testid="button-confirm-warehouse"
                >
                  <Play className="h-3 w-3 mr-1" />
                  Rozpocznij
                </Button>
              </DialogFooter>
            </DialogContent>
          </Dialog>
        ) : (
          <Button
            size="sm"
            variant="default"
            className="h-6 text-[10px] px-2 bg-orange-600 hover:bg-orange-700"
            onClick={() => endWarehouseStayMutation.mutate()}
            disabled={endWarehouseStayMutation.isPending}
            data-testid={`button-end-warehouse-stay-${pallet.id}`}
          >
            <Square className="h-3 w-3 mr-1" />
            Zakończ magazynowanie
          </Button>
        )}
      </div>
    </div>
  );
}

function ActiveWarehouseStays({ stays, onRefresh }: { stays: WarehouseStay[]; onRefresh: () => void }) {
  const { toast } = useToast();
  const [tick, setTick] = useState(0);
  
  useEffect(() => {
    if (stays.length === 0) return;
    const interval = setInterval(() => setTick(t => t + 1), 1000);
    return () => clearInterval(interval);
  }, [stays.length]);

  const endWarehouseStayMutation = useMutation({
    mutationFn: async (palletId: number) => {
      return apiRequest("POST", `/api/production/pallets/${palletId}/warehouse-stay/end`, {});
    },
    onSuccess: () => {
      toast({ title: "Zakończono pobyt na magazynie" });
      onRefresh();
    },
    onError: () => {
      toast({ title: "Błąd", description: "Nie udało się zakończyć pobytu", variant: "destructive" });
    },
  });

  if (stays.length === 0) return null;

  const calculateLiveDuration = (stay: WarehouseStay) => {
    const startTime = new Date(stay.start_time).getTime();
    const now = Date.now();
    const diffSeconds = Math.floor((now - startTime) / 1000);
    return Math.max(0, diffSeconds);
  };

  return (
    <div className="mb-4 p-3 rounded-lg bg-orange-50 dark:bg-orange-900/20 border border-orange-200 dark:border-orange-800">
      <div className="flex items-center gap-2 mb-2">
        <Timer className="h-4 w-4 text-orange-600 dark:text-orange-400 animate-pulse" />
        <span className="text-sm font-medium text-orange-700 dark:text-orange-300">
          Palety na magazynie ({stays.length})
        </span>
      </div>
      <div className="space-y-2">
        {stays.map(stay => {
          const liveDuration = calculateLiveDuration(stay);
          return (
            <div 
              key={stay.id}
              className="flex flex-col md:flex-row md:items-center justify-between gap-2 p-2 rounded bg-white dark:bg-gray-800 text-sm"
            >
              <div className="flex items-center gap-2">
                <span className="font-medium">{stay.pallet_label}</span>
                <span className="text-muted-foreground text-xs">
                  w {stay.warehouse_name_resolved || 'magazynie'}
                </span>
                {stay.flow_code && (
                  <Badge variant="outline" className="text-[9px]">{stay.flow_code}</Badge>
                )}
              </div>
              <div className="flex items-center gap-3">
                <div className="flex items-center gap-1.5 px-2 py-1 rounded bg-orange-100 dark:bg-orange-900/40">
                  <Clock className="h-3.5 w-3.5 text-orange-600 dark:text-orange-400" />
                  <span className="font-mono font-bold text-orange-700 dark:text-orange-300 text-sm">
                    {formatDuration(liveDuration)}
                  </span>
                </div>
                <Button
                  size="sm"
                  variant="default"
                  className="h-7 text-xs bg-green-600 hover:bg-green-700"
                  onClick={() => endWarehouseStayMutation.mutate(stay.pallet_id)}
                  disabled={endWarehouseStayMutation.isPending}
                  data-testid={`button-end-stay-${stay.id}`}
                >
                  <Square className="h-3 w-3 mr-1" />
                  Zakończ
                </Button>
              </div>
            </div>
          );
        })}
      </div>
    </div>
  );
}

export function PalletManager({ productionOrderId, colorCode }: PalletManagerProps) {
  const { toast } = useToast();
  const [isOpen, setIsOpen] = useState(true);
  const [showCreateDialog, setShowCreateDialog] = useState(false);
  const [newPalletFlowCode, setNewPalletFlowCode] = useState("COW");
  const [newPalletColorCode, setNewPalletColorCode] = useState(colorCode || "");

  const { data: pallets = [], refetch: refetchPallets } = useQuery<Pallet[]>({
    queryKey: ['/api/production/orders', productionOrderId, 'pallets'],
    queryFn: async () => {
      const res = await fetch(`/api/production/orders/${productionOrderId}/pallets`);
      if (!res.ok) throw new Error('Failed to fetch pallets');
      return res.json();
    },
  });

  const { data: carriers = [] } = useQuery<Carrier[]>({
    queryKey: ['/api/production/carriers'],
    queryFn: async () => {
      const res = await fetch('/api/production/carriers');
      if (!res.ok) throw new Error('Failed to fetch carriers');
      return res.json();
    },
  });

  const { data: locations = [] } = useQuery<Location[]>({
    queryKey: ['/api/production/locations'],
    queryFn: async () => {
      const res = await fetch('/api/production/locations');
      if (!res.ok) throw new Error('Failed to fetch locations');
      return res.json();
    },
  });

  const { data: activeWarehouseStays = [], refetch: refetchStays } = useQuery<WarehouseStay[]>({
    queryKey: ['/api/production/orders', productionOrderId, 'warehouse-stays', 'active'],
    queryFn: async () => {
      const res = await fetch(`/api/production/orders/${productionOrderId}/warehouse-stays/active`);
      if (!res.ok) throw new Error('Failed to fetch warehouse stays');
      return res.json();
    },
    refetchInterval: 5000,
  });

  const createPalletMutation = useMutation({
    mutationFn: async (data: { flowCode: string; colorCode?: string }) => {
      return apiRequest("POST", `/api/production/orders/${productionOrderId}/pallets`, data);
    },
    onSuccess: () => {
      toast({ title: "Paleta utworzona" });
      setShowCreateDialog(false);
      refetchPallets();
    },
    onError: () => {
      toast({ title: "Błąd", description: "Nie udało się utworzyć palety", variant: "destructive" });
    },
  });

  const handleRefresh = () => {
    refetchPallets();
    refetchStays();
    queryClient.invalidateQueries({ queryKey: ['/api/production/orders', productionOrderId] });
  };

  const flowGroups = pallets.reduce((acc, p) => {
    const key = p.flow_code;
    if (!acc[key]) acc[key] = [];
    acc[key].push(p);
    return acc;
  }, {} as Record<string, Pallet[]>);

  return (
    <Collapsible open={isOpen} onOpenChange={setIsOpen}>
      <Card className="mt-4">
        <CardHeader className="py-2 px-3">
          <CollapsibleTrigger asChild>
            <div className="flex items-center justify-between cursor-pointer">
              <CardTitle className="text-sm flex items-center gap-2">
                <Package className="h-4 w-4" />
                Zarządzanie paletami ({pallets.length})
              </CardTitle>
              <div className="flex items-center gap-2">
                <Dialog open={showCreateDialog} onOpenChange={setShowCreateDialog}>
                  <DialogTrigger asChild>
                    <Button 
                      size="sm" 
                      variant="outline" 
                      className="h-7"
                      onClick={(e) => e.stopPropagation()}
                      data-testid="button-create-pallet"
                    >
                      <Plus className="h-3 w-3 mr-1" />
                      Dodaj paletę
                    </Button>
                  </DialogTrigger>
                  <DialogContent className="max-w-sm" onClick={(e) => e.stopPropagation()}>
                    <DialogHeader>
                      <DialogTitle>Utwórz nową paletę</DialogTitle>
                    </DialogHeader>
                    <div className="space-y-3">
                      <div>
                        <Label className="text-xs">Przepływ (Flow)</Label>
                        <Select value={newPalletFlowCode} onValueChange={setNewPalletFlowCode}>
                          <SelectTrigger data-testid="select-flow-code">
                            <SelectValue />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="CO">CO (Cięcie-Oklejanie)</SelectItem>
                            <SelectItem value="COW">COW (Cięcie-Oklejanie-Wiercenie)</SelectItem>
                            <SelectItem value="CW">CW (Cięcie-Wiercenie)</SelectItem>
                            <SelectItem value="C">C (Tylko cięcie)</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>
                      <div>
                        <Label className="text-xs">Kolor (opcjonalnie)</Label>
                        <Input 
                          value={newPalletColorCode}
                          onChange={(e) => setNewPalletColorCode(e.target.value)}
                          placeholder="np. BIALY, WOTAN"
                          data-testid="input-color-code"
                        />
                      </div>
                    </div>
                    <DialogFooter>
                      <DialogClose asChild>
                        <Button variant="outline" size="sm">Anuluj</Button>
                      </DialogClose>
                      <Button 
                        size="sm" 
                        onClick={() => createPalletMutation.mutate({ 
                          flowCode: newPalletFlowCode, 
                          colorCode: newPalletColorCode || undefined 
                        })}
                        disabled={createPalletMutation.isPending}
                        data-testid="button-confirm-create-pallet"
                      >
                        Utwórz
                      </Button>
                    </DialogFooter>
                  </DialogContent>
                </Dialog>
                {isOpen ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
              </div>
            </div>
          </CollapsibleTrigger>
        </CardHeader>
        <CollapsibleContent>
          <CardContent className="px-3 pb-3 pt-0">
            <ActiveWarehouseStays stays={activeWarehouseStays} onRefresh={handleRefresh} />
            
            {Object.entries(flowGroups).length > 0 ? (
              <div className="space-y-3">
                {Object.entries(flowGroups).map(([flowCode, flowPallets]) => (
                  <div key={flowCode}>
                    <div className="flex items-center gap-2 mb-2">
                      <Badge variant="outline" className="text-[10px]">{flowCode}</Badge>
                      <span className="text-xs text-muted-foreground">({flowPallets.length} palet)</span>
                    </div>
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-2">
                      {flowPallets.map(pallet => (
                        <PalletCard 
                          key={pallet.id} 
                          pallet={pallet} 
                          carriers={carriers}
                          locations={locations}
                          onRefresh={handleRefresh}
                        />
                      ))}
                    </div>
                  </div>
                ))}
              </div>
            ) : (
              <div className="text-center py-6 text-muted-foreground text-sm">
                Brak palet. Kliknij "Dodaj paletę" aby utworzyć pierwszą.
              </div>
            )}
          </CardContent>
        </CollapsibleContent>
      </Card>
    </Collapsible>
  );
}
