import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger, DialogFooter, DialogClose } from "@/components/ui/dialog";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { Warehouse, Plus, ArrowRight } from "lucide-react";

interface WorkOrder {
  id: number;
  work_order_number: string;
  sequence: number;
  status: string;
  operation_name: string | null;
  operation_code: string | null;
  location_name: string | null;
}

interface Location {
  id: number;
  name: string;
  code: string;
  type: string;
}

interface InsertWarehouseOperationProps {
  productionOrderId: number;
  workOrders: WorkOrder[];
  onSuccess?: () => void;
}

export function InsertWarehouseOperation({ productionOrderId, workOrders, onSuccess }: InsertWarehouseOperationProps) {
  const { toast } = useToast();
  const [open, setOpen] = useState(false);
  const [selectedWorkOrderId, setSelectedWorkOrderId] = useState<string>("");
  const [selectedWarehouseId, setSelectedWarehouseId] = useState<string>("");
  const [estimatedDuration, setEstimatedDuration] = useState<string>("60");
  const [notes, setNotes] = useState("");

  const { data: locations = [] } = useQuery<Location[]>({
    queryKey: ['/api/production/locations'],
    queryFn: async () => {
      const res = await fetch('/api/production/locations');
      if (!res.ok) throw new Error('Failed to fetch locations');
      return res.json();
    },
  });

  const warehouseLocations = locations.filter(l => 
    l.type === 'warehouse' || l.type === 'storage' || l.name.toLowerCase().includes('magazyn')
  );

  const insertMutation = useMutation({
    mutationFn: async (data: { 
      afterWorkOrderId: number; 
      warehouseId: number; 
      warehouseName?: string;
      estimatedDuration?: number;
      notes?: string;
    }) => {
      return apiRequest("POST", `/api/production/orders/${productionOrderId}/insert-warehouse-operation`, data);
    },
    onSuccess: () => {
      toast({ title: "Operacja magazynowania wstawiona" });
      setOpen(false);
      setSelectedWorkOrderId("");
      setSelectedWarehouseId("");
      setEstimatedDuration("60");
      setNotes("");
      queryClient.invalidateQueries({ queryKey: ['/api/production/orders', productionOrderId] });
      onSuccess?.();
    },
    onError: (error) => {
      toast({ 
        title: "Błąd", 
        description: "Nie udało się wstawić operacji magazynowania", 
        variant: "destructive" 
      });
    },
  });

  const handleInsert = () => {
    if (!selectedWorkOrderId || !selectedWarehouseId) {
      toast({ title: "Błąd", description: "Wybierz operację i magazyn", variant: "destructive" });
      return;
    }

    const warehouse = locations.find(l => l.id === parseInt(selectedWarehouseId));

    insertMutation.mutate({
      afterWorkOrderId: parseInt(selectedWorkOrderId),
      warehouseId: parseInt(selectedWarehouseId),
      warehouseName: warehouse?.name,
      estimatedDuration: parseInt(estimatedDuration) || 60,
      notes: notes || undefined,
    });
  };

  const activeWorkOrders = workOrders.filter(wo => wo.status !== 'done' && wo.status !== 'cancelled');

  return (
    <Dialog open={open} onOpenChange={setOpen}>
      <DialogTrigger asChild>
        <Button 
          size="sm" 
          variant="outline" 
          className="h-7"
          data-testid="button-insert-warehouse-operation"
        >
          <Warehouse className="h-3 w-3 mr-1" />
          <Plus className="h-3 w-3 mr-1" />
          Wstaw magazynowanie
        </Button>
      </DialogTrigger>
      <DialogContent className="max-w-md">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Warehouse className="h-5 w-5" />
            Wstaw operację magazynowania
          </DialogTitle>
        </DialogHeader>
        
        <div className="space-y-4">
          <div className="p-3 rounded-lg bg-muted/50 text-sm text-muted-foreground">
            Ta funkcja pozwala wstawić dodatkową operację magazynowania do przepływu ZLP.
            Wybierz po której operacji chcesz wstawić magazynowanie.
          </div>

          <div>
            <Label className="text-xs">Wstaw po operacji</Label>
            <Select value={selectedWorkOrderId} onValueChange={setSelectedWorkOrderId}>
              <SelectTrigger data-testid="select-after-work-order">
                <SelectValue placeholder="Wybierz operację" />
              </SelectTrigger>
              <SelectContent>
                {workOrders.map(wo => (
                  <SelectItem key={wo.id} value={wo.id.toString()}>
                    #{wo.sequence} - {wo.operation_name || wo.operation_code || wo.work_order_number}
                    {wo.status === 'done' && ' (zakończona)'}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div>
            <Label className="text-xs">Magazyn docelowy</Label>
            <Select value={selectedWarehouseId} onValueChange={setSelectedWarehouseId}>
              <SelectTrigger data-testid="select-target-warehouse">
                <SelectValue placeholder="Wybierz magazyn" />
              </SelectTrigger>
              <SelectContent>
                {(warehouseLocations.length > 0 ? warehouseLocations : locations.slice(0, 15)).map(l => (
                  <SelectItem key={l.id} value={l.id.toString()}>
                    {l.name} {l.code ? `(${l.code})` : ''}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div>
            <Label className="text-xs">Szacowany czas (minuty)</Label>
            <Input 
              type="number"
              value={estimatedDuration}
              onChange={(e) => setEstimatedDuration(e.target.value)}
              placeholder="60"
              data-testid="input-estimated-duration"
            />
          </div>

          <div>
            <Label className="text-xs">Notatki (opcjonalnie)</Label>
            <Input 
              value={notes}
              onChange={(e) => setNotes(e.target.value)}
              placeholder="np. przerwa na weekend, oczekiwanie na części"
              data-testid="input-warehouse-notes"
            />
          </div>

          {selectedWorkOrderId && selectedWarehouseId && (
            <div className="p-3 rounded-lg border bg-blue-50 dark:bg-blue-900/20 text-sm">
              <div className="flex items-center gap-2 text-blue-700 dark:text-blue-300">
                <span>
                  Po: #{workOrders.find(w => w.id === parseInt(selectedWorkOrderId))?.sequence}
                </span>
                <ArrowRight className="h-4 w-4" />
                <span className="font-medium">
                  Magazynowanie ({locations.find(l => l.id === parseInt(selectedWarehouseId))?.name})
                </span>
                <ArrowRight className="h-4 w-4" />
                <span>Następna operacja</span>
              </div>
            </div>
          )}
        </div>

        <DialogFooter>
          <DialogClose asChild>
            <Button variant="outline" size="sm">Anuluj</Button>
          </DialogClose>
          <Button 
            size="sm" 
            onClick={handleInsert}
            disabled={!selectedWorkOrderId || !selectedWarehouseId || insertMutation.isPending}
            data-testid="button-confirm-insert-warehouse"
          >
            {insertMutation.isPending ? "Wstawianie..." : "Wstaw operację"}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
