import { useState, forwardRef } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Plus, Trash2, Calculator, Settings2, X, ArrowDownToLine } from "lucide-react";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip";
import { useToast } from "@/hooks/use-toast";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import type { FormulaPreset } from "@shared/schema";

interface DimensionVariant {
  dimension: number;
  value: number | null;
  formula: string | null;
}

interface DimensionVariantsDialogProps {
  type: "length" | "width";
  variants: DimensionVariant[];
  onChange: (variants: DimensionVariant[]) => void;
  furnitureType: string;
  furnitureLength?: number | null;
  furnitureWidth?: number | null;
  furnitureLengthCondition?: number | null;
  furnitureWidthCondition?: number | null;
  cz1?: string;
  cz2?: string | null;
  onExpandAfterSave?: () => void;
  triggerButton?: React.ReactNode;
}

export function DimensionVariantsDialog({ 
  type, 
  variants, 
  onChange, 
  furnitureType, 
  furnitureLength, 
  furnitureWidth, 
  furnitureLengthCondition, 
  furnitureWidthCondition, 
  cz1, 
  cz2,
  onExpandAfterSave,
  triggerButton
}: DimensionVariantsDialogProps) {
  const { toast } = useToast();
  const [localVariants, setLocalVariants] = useState<DimensionVariant[]>(variants || []);
  const [newDimension, setNewDimension] = useState<string>("");
  const [isOpen, setIsOpen] = useState(false);
  const [newPresetFormula, setNewPresetFormula] = useState<string>("");
  const [showAddPreset, setShowAddPreset] = useState(false);

  const isLength = type === "length";
  const title = isLength ? "Warianty długości" : "Warianty szerokości";
  const elementName = [cz1, cz2].filter(Boolean).join(' - ');
  const suggestedDimensions = isLength 
    ? [30, 50, 60, 80, 100, 120]
    : [30, 36, 46, 55, 65];

  // Fetch formula presets from database
  const { data: allPresets = [] } = useQuery<FormulaPreset[]>({
    queryKey: ['/api/formula-presets'],
  });

  // Filter presets by type (length or width)
  const formulaPresets = allPresets.filter(p => p.type === type);

  // Mutation to add new preset
  const addPresetMutation = useMutation({
    mutationFn: async (formula: string) => {
      return apiRequest('POST', '/api/formula-presets', {
        formula,
        type,
        sort_order: allPresets.filter(p => p.type === type).length
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/formula-presets'] });
      toast({ title: "Preset dodany", description: "Nowy wzór został zapisany" });
      setNewPresetFormula("");
      setShowAddPreset(false);
    },
    onError: () => {
      toast({ title: "Błąd", description: "Nie udało się dodać presetu", variant: "destructive" });
    }
  });

  // Mutation to delete preset
  const deletePresetMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest('DELETE', `/api/formula-presets/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/formula-presets'] });
      toast({ title: "Preset usunięty" });
    },
    onError: () => {
      toast({ title: "Błąd", description: "Nie udało się usunąć presetu", variant: "destructive" });
    }
  });

  const addVariant = () => {
    const dim = parseInt(newDimension);
    if (!dim || dim <= 0) {
      toast({ title: "Błąd", description: "Podaj poprawny wymiar", variant: "destructive" });
      return;
    }
    
    if (localVariants.some(v => v.dimension === dim)) {
      toast({ title: "Błąd", description: "Ten wymiar już istnieje", variant: "destructive" });
      return;
    }

    const newVariant: DimensionVariant = {
      dimension: dim,
      value: null,
      formula: null
    };
    
    const updated = [...localVariants, newVariant].sort((a, b) => a.dimension - b.dimension);
    setLocalVariants(updated);
    setNewDimension("");
  };

  const removeVariant = (dimension: number) => {
    setLocalVariants(localVariants.filter(v => v.dimension !== dimension));
  };

  const updateValue = (dimension: number, value: string) => {
    const numValue = value ? parseFloat(value) : null;
    setLocalVariants(localVariants.map(v => 
      v.dimension === dimension ? { ...v, value: numValue } : v
    ));
  };

  const updateFormula = (dimension: number, formula: string) => {
    setLocalVariants(localVariants.map(v => {
      if (v.dimension !== dimension) return v;
      
      const newVariant = { ...v, formula: formula || null };
      
      // Automatycznie oblicz wartość jeśli wzór jest poprawny
      if (!formula || !formula.trim()) {
        return { ...newVariant, value: null };
      }
      
      try {
        const trimmedFormula = formula.trim();
        
        // Sprawdź czy to prosta liczba (np. "390")
        const numberMatch = trimmedFormula.match(/^(\d+(?:\.\d+)?)$/);
        if (numberMatch) {
          const value = parseFloat(numberMatch[1]);
          return { ...newVariant, value: value };
        }
        
        // Sprawdź czy to proste wyrażenie matematyczne (np. "280 - 2", "300 + 50")
        const simpleExprMatch = trimmedFormula.match(/^(\d+(?:\.\d+)?)\s*([+\-*/])\s*(\d+(?:\.\d+)?)$/);
        if (simpleExprMatch) {
          const [, num1Str, operator, num2Str] = simpleExprMatch;
          const num1 = parseFloat(num1Str);
          const num2 = parseFloat(num2Str);
          let result = num1;
          switch (operator) {
            case '+': result = num1 + num2; break;
            case '-': result = num1 - num2; break;
            case '*': result = num1 * num2; break;
            case '/': result = num1 / num2; break;
          }
          return { ...newVariant, value: Math.round(result) };
        }
        
        // Sprawdź czy to wzór z operatorem (np. "dlMeb - 38")
        const match = trimmedFormula.match(/^(dlMeb|szMeb)\s*([+\-*/])\s*(\d+(?:\.\d+)?)$/);
        
        if (!match) {
          return newVariant; // Niepoprawny wzór - zostaw wartość jak jest
        }

        const [, variable, operator, operandStr] = match;
        const operand = parseFloat(operandStr);
        
        // dlMeb/szMeb to wymiar MEBLA w MM
        // v.dimension to cm (30, 50, 60), konwertujemy na mm
        const baseValueMm = v.dimension * 10;  // 30 cm = 300 mm
        
        // Wykonaj obliczenie w MM (operand też jest w MM)
        let resultMm = baseValueMm;
        switch (operator) {
          case '+': resultMm = baseValueMm + operand; break;  // 300 + 50 = 350 mm
          case '-': resultMm = baseValueMm - operand; break;  // 300 - 38 = 262 mm
          case '*': resultMm = baseValueMm * operand; break;
          case '/': resultMm = baseValueMm / operand; break;
        }

        return { ...newVariant, value: Math.round(resultMm) };
      } catch (error) {
        return newVariant; // Błąd parsowania - zostaw wartość jak jest
      }
    }));
  };

  // Funkcja do zastosowania wzoru do WSZYSTKICH wariantów i NATYCHMIASTOWEGO obliczenia
  const applyFormulaToAll = (formula: string) => {
    let calculated = 0;
    let failed = 0;
    
    const updated = localVariants.map(variant => {
      const newVariant = { ...variant, formula };
      
      // Automatycznie oblicz wartość dla tego wzoru
      try {
        const trimmedFormula = formula.trim();
        
        // Sprawdź czy to prosta liczba (np. "390")
        const numberMatch = trimmedFormula.match(/^(\d+(?:\.\d+)?)$/);
        if (numberMatch) {
          const value = parseFloat(numberMatch[1]);
          calculated++;
          return { ...newVariant, value: value };
        }
        
        // Sprawdź czy to proste wyrażenie matematyczne (np. "280 - 2", "300 + 50")
        const simpleExprMatch = trimmedFormula.match(/^(\d+(?:\.\d+)?)\s*([+\-*/])\s*(\d+(?:\.\d+)?)$/);
        if (simpleExprMatch) {
          const [, num1Str, operator, num2Str] = simpleExprMatch;
          const num1 = parseFloat(num1Str);
          const num2 = parseFloat(num2Str);
          let result = num1;
          switch (operator) {
            case '+': result = num1 + num2; break;
            case '-': result = num1 - num2; break;
            case '*': result = num1 * num2; break;
            case '/': result = num1 / num2; break;
          }
          calculated++;
          return { ...newVariant, value: Math.round(result) };
        }
        
        // Sprawdź czy to wzór z operatorem (np. "dlMeb - 38")
        const match = trimmedFormula.match(/^(dlMeb|szMeb)\s*([+\-*/])\s*(\d+(?:\.\d+)?)$/);
        
        if (!match) {
          failed++;
          return newVariant;
        }

        const [, variable, operator, operandStr] = match;
        const operand = parseFloat(operandStr);
        
        // dlMeb/szMeb to wymiar MEBLA w MM
        // variant.dimension to cm (30, 50, 60), konwertujemy na mm
        const baseValueMm = variant.dimension * 10;  // 30 cm = 300 mm
        
        // Wykonaj obliczenie w MM (operand też jest w MM)
        let resultMm = baseValueMm;
        switch (operator) {
          case '+': resultMm = baseValueMm + operand; break;  // 300 + 50 = 350 mm
          case '-': resultMm = baseValueMm - operand; break;  // 300 - 38 = 262 mm
          case '*': resultMm = baseValueMm * operand; break;
          case '/': resultMm = baseValueMm / operand; break;
        }

        calculated++;
        return { ...newVariant, value: Math.round(resultMm) };
      } catch (error) {
        failed++;
        return newVariant;
      }
    });
    
    setLocalVariants(updated);
    
    if (calculated > 0) {
      toast({ 
        title: "Wzór zastosowany i obliczony", 
        description: `Wzór "${formula}" zastosowano i przeliczono dla ${calculated} wariantów` 
      });
    } else if (failed > 0) {
      toast({ 
        title: "Błąd wzoru", 
        description: `Nie udało się obliczyć wartości. Sprawdź poprawność wzoru.`,
        variant: "destructive"
      });
    }
  };

  const calculateValues = () => {
    // Sprawdź czy mamy wymiary mebla do obliczeń
    const furnitureDimension = isLength ? furnitureLength : furnitureWidth;
    const dimensionName = isLength ? "długość" : "szerokość";
    
    if (!furnitureDimension) {
      toast({ 
        title: "Brak wymiaru bazowego", 
        description: `Najpierw wprowadź ${dimensionName} formatki (pole "Dł." lub "Szer.")`,
        variant: "destructive" 
      });
      return;
    }

    let calculated = 0;
    const updated = localVariants.map(variant => {
      if (!variant.formula) return variant;
      
      try {
        // Parse formula - obsługa prostych operacji
        const formula = variant.formula.trim();
        
        // Sprawdź czy to prosta liczba (np. "390")
        const numberMatch = formula.match(/^(\d+(?:\.\d+)?)$/);
        if (numberMatch) {
          const value = parseFloat(numberMatch[1]);
          calculated++;
          return { ...variant, value: Math.round(value * 100) / 100 };
        }
        
        // Sprawdź czy to wzór z operatorem (np. "dlMeb - 38")
        const match = formula.match(/^(dlMeb|szMeb)\s*([+\-*/])\s*(\d+(?:\.\d+)?)$/);
        
        if (!match) {
          toast({ 
            title: "Błędny wzór", 
            description: `Wzór "${formula}" jest nieprawidłowy. Użyj liczby (np. 390) lub wzoru (np. dlMeb - 38)`,
            variant: "destructive" 
          });
          return variant;
        }

        const [, variable, operator, operandStr] = match;
        const operand = parseFloat(operandStr);
        
        // dlMeb i szMeb to wymiar MEBLA w MM
        // variant.dimension to cm (30, 50, 60), więc konwertujemy na mm
        const baseValueMm = variant.dimension * 10;  // 30 cm = 300 mm
        
        // Wykonaj obliczenie w MM (operand też jest w MM)
        let resultMm = baseValueMm;
        switch (operator) {
          case '+': resultMm = baseValueMm + operand; break;  // 300 + 50 = 350 mm
          case '-': resultMm = baseValueMm - operand; break;  // 300 - 38 = 262 mm
          case '*': resultMm = baseValueMm * operand; break;
          case '/': resultMm = baseValueMm / operand; break;
        }

        calculated++;
        return { ...variant, value: Math.round(resultMm) };
      } catch (error) {
        toast({ 
          title: "Błąd obliczenia", 
          description: `Nie udało się obliczyć wartości dla wymiaru ${variant.dimension}`,
          variant: "destructive" 
        });
        return variant;
      }
    });

    setLocalVariants(updated);
    if (calculated > 0) {
      toast({ title: "Obliczono wartości", description: `Przeliczono ${calculated} wariantów` });
    }
  };

  const handleSave = () => {
    onChange(localVariants);
    setIsOpen(false);
    toast({ title: "Zapisano warianty", description: `Zapisano ${localVariants.length} wariantów` });
    
    // Auto-expand row if variants were added
    if (localVariants.length > 0 && onExpandAfterSave) {
      setTimeout(() => onExpandAfterSave(), 100);
    }
  };

  const addSuggestedDimension = (dim: number) => {
    if (localVariants.some(v => v.dimension === dim)) {
      toast({ title: "Błąd", description: "Ten wymiar już istnieje", variant: "destructive" });
      return;
    }

    const newVariant: DimensionVariant = {
      dimension: dim,
      value: null,
      formula: null
    };
    
    const updated = [...localVariants, newVariant].sort((a, b) => a.dimension - b.dimension);
    setLocalVariants(updated);
  };

  const addAllSuggestedDimensions = () => {
    const newVariants = suggestedDimensions
      .filter(dim => !localVariants.some(v => v.dimension === dim))
      .map(dim => ({
        dimension: dim,
        value: null,
        formula: null
      }));
    
    if (newVariants.length === 0) {
      toast({ title: "Info", description: "Wszystkie sugerowane wymiary są już dodane" });
      return;
    }

    const updated = [...localVariants, ...newVariants].sort((a, b) => a.dimension - b.dimension);
    setLocalVariants(updated);
    toast({ title: "Dodano wymiary", description: `Dodano ${newVariants.length} wymiarów` });
  };

  // Skopiuj wartość z pierwszego wariantu do wszystkich pozostałych
  const copyFirstValueToAll = () => {
    if (localVariants.length === 0) return;
    
    const firstValue = localVariants[0].value;
    if (firstValue === null) {
      toast({ 
        title: "Brak wartości", 
        description: "Najpierw wprowadź wartość w pierwszym polu", 
        variant: "destructive" 
      });
      return;
    }

    const updated = localVariants.map(v => ({ ...v, value: firstValue, formula: null }));
    setLocalVariants(updated);
    toast({ 
      title: "Skopiowano wartość", 
      description: `Wartość ${firstValue} została skopiowana do wszystkich wariantów` 
    });
  };

  return (
    <Dialog open={isOpen} onOpenChange={setIsOpen}>
      <DialogTrigger asChild>
        {triggerButton || (
          <Button 
            variant="ghost" 
            size="sm"
            className="h-8 px-3 bg-blue-500/80 hover:bg-blue-600 text-white rounded-l-none border-l-0"
            data-testid={`button-variants-${type}`}
          >
            <Settings2 className="w-4 h-4" />
          </Button>
        )}
      </DialogTrigger>
      <DialogContent className="max-w-3xl max-h-[90vh] flex flex-col overflow-hidden">
        <DialogHeader className="flex-shrink-0">
          <DialogTitle className="text-xl md:text-2xl font-bold">
            {elementName || furnitureType} {type === 'length' ? 'Długość' : 'Szerokość'}
          </DialogTitle>
          <DialogDescription className="text-sm md:text-base">
            Zdefiniuj wymiary (w cm) i wartości dla różnych wariantów. Możesz podać wartość bezpośrednio lub użyć wzoru. W wzorach dlMeb/szMeb reprezentują wymiar mebla w mm, operatory też w mm (np. "dlMeb + 50" dla 30cm: 300mm + 50mm = 350mm).
          </DialogDescription>
        </DialogHeader>

        <div className="space-y-4 overflow-y-auto flex-1 min-h-0 pr-2">
          {/* Sugerowane wymiary */}
          <div className="space-y-2">
            <span className="text-xs md:text-sm text-muted-foreground">Szybkie dodawanie:</span>
            <div className="flex flex-wrap gap-1.5 md:gap-2 items-center">
              {suggestedDimensions.map(dim => (
                <Button
                  key={dim}
                  variant="outline"
                  size="sm"
                  onClick={() => addSuggestedDimension(dim)}
                  disabled={localVariants.some(v => v.dimension === dim)}
                  data-testid={`button-add-${dim}`}
                  className="text-xs md:text-sm"
                >
                  {dim}
                </Button>
              ))}
              <Button
                variant="secondary"
                size="sm"
                onClick={addAllSuggestedDimensions}
                data-testid="button-add-all-suggested"
                className="ml-0 md:ml-2 text-xs md:text-sm"
              >
                <Plus className="w-3 h-3 mr-1" />
                <span className="hidden sm:inline">Dodaj wszystkie</span>
                <span className="sm:hidden">Wszystkie</span>
              </Button>
            </div>
          </div>

          {/* Presety wzorów z bazy danych */}
          <div className="border rounded-md p-2 md:p-3 bg-muted/20">
            <div className="flex items-center gap-2 mb-2">
              <Calculator className="w-3 h-3 md:w-4 md:h-4 text-muted-foreground" />
              <span className="text-xs md:text-sm font-medium">
                <span className="hidden sm:inline">Kliknij aby zastosować wzór do wszystkich pól:</span>
                <span className="sm:hidden">Zastosuj wzór:</span>
              </span>
            </div>
            <div className="flex flex-wrap gap-1.5 md:gap-2 items-center">
              {formulaPresets.map((preset) => (
                <div key={preset.id} className="relative group">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => applyFormulaToAll(preset.formula)}
                    className="font-mono text-xs pr-6"
                    data-testid={`button-formula-preset-${preset.id}`}
                  >
                    {preset.label || preset.formula}
                  </Button>
                  <Button
                    variant="ghost"
                    size="icon"
                    className="h-5 w-5 absolute -top-1 -right-1 opacity-0 group-hover:opacity-100 transition-opacity bg-destructive text-destructive-foreground hover:bg-destructive/90"
                    onClick={(e) => {
                      e.stopPropagation();
                      deletePresetMutation.mutate(preset.id);
                    }}
                    data-testid={`button-delete-preset-${preset.id}`}
                  >
                    <X className="w-3 h-3" />
                  </Button>
                </div>
              ))}
              
              {/* Add new preset button */}
              {!showAddPreset && (
                <Button
                  variant="secondary"
                  size="sm"
                  onClick={() => setShowAddPreset(true)}
                  data-testid="button-show-add-preset"
                  className="font-mono text-xs"
                >
                  <Plus className="w-3 h-3 mr-1" />
                  Dodaj wzór
                </Button>
              )}
              
              {/* Add preset input */}
              {showAddPreset && (
                <div className="flex gap-1 items-center">
                  <Input
                    type="text"
                    placeholder={isLength ? "dlMeb - 38" : "szMeb * 10"}
                    value={newPresetFormula}
                    onChange={(e) => setNewPresetFormula(e.target.value)}
                    onKeyDown={(e) => {
                      if (e.key === 'Enter' && newPresetFormula.trim()) {
                        addPresetMutation.mutate(newPresetFormula.trim());
                      } else if (e.key === 'Escape') {
                        setShowAddPreset(false);
                        setNewPresetFormula("");
                      }
                    }}
                    className="h-8 w-32 font-mono text-xs"
                    autoFocus
                    data-testid="input-new-preset"
                  />
                  <Button
                    variant="ghost"
                    size="icon"
                    className="h-8 w-8"
                    onClick={() => {
                      if (newPresetFormula.trim()) {
                        addPresetMutation.mutate(newPresetFormula.trim());
                      }
                    }}
                    disabled={!newPresetFormula.trim() || addPresetMutation.isPending}
                    data-testid="button-save-preset"
                  >
                    <Plus className="w-4 h-4" />
                  </Button>
                  <Button
                    variant="ghost"
                    size="icon"
                    className="h-8 w-8"
                    onClick={() => {
                      setShowAddPreset(false);
                      setNewPresetFormula("");
                    }}
                    data-testid="button-cancel-preset"
                  >
                    <X className="w-4 h-4" />
                  </Button>
                </div>
              )}
            </div>
          </div>

          {/* Dodaj własny wymiar */}
          <div className="flex gap-2">
            <Input
              type="number"
              placeholder="Własny wymiar (np. 75)"
              value={newDimension}
              onChange={(e) => setNewDimension(e.target.value)}
              onKeyDown={(e) => e.key === 'Enter' && addVariant()}
              className="flex-1 text-sm md:text-base"
              data-testid="input-custom-dimension"
            />
            <Button onClick={addVariant} data-testid="button-add-custom" className="text-xs md:text-sm">
              <Plus className="w-4 h-4 mr-1 md:mr-2" />
              <span className="hidden sm:inline">Dodaj</span>
              <span className="sm:hidden">+</span>
            </Button>
          </div>

          {/* Lista wariantów */}
          <ScrollArea className="h-[200px] md:h-[300px] lg:h-[400px] border rounded-md p-4">
            <div className="space-y-3">
              {localVariants.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  Brak zdefiniowanych wariantów. Dodaj wymiary używając przycisków powyżej.
                </div>
              ) : (
                localVariants.map((variant, index) => (
                  <div key={variant.dimension} className="flex flex-col sm:flex-row items-start gap-2 p-2 md:p-3 border rounded-md bg-muted/30">
                    {/* Długość/Szerokość mebla - NIE DO EDYCJI */}
                    <div className="flex-none w-full sm:w-24">
                      <Label className="text-xs text-muted-foreground">
                        <span className="hidden sm:inline">{isLength ? "Długość mebla (cm)" : "Szerokość mebla (cm)"}</span>
                        <span className="sm:hidden">{isLength ? "Dł. mebla (cm)" : "Szer. mebla (cm)"}</span>
                      </Label>
                      <div className="font-semibold text-base md:text-lg mt-1">{variant.dimension}</div>
                    </div>
                    
                    <div className="flex-1 w-full grid grid-cols-1 sm:grid-cols-2 gap-2">
                      {/* Wzór - DO EDYCJI */}
                      <div className="flex-1">
                        <Label className="text-xs text-muted-foreground">Wzór</Label>
                        <Input
                          value={variant.formula ?? ''}
                          onChange={(e) => updateFormula(variant.dimension, e.target.value)}
                          placeholder={isLength ? "np. dlMeb - 38" : "np. szMeb - 20"}
                          className="h-8 font-mono text-xs bg-muted/40 text-muted-foreground"
                          data-testid={`input-formula-${variant.dimension}`}
                        />
                      </div>

                      {/* Długość/Szerokość formatki - WYGENEROWANA + DO EDYCJI */}
                      <div className="flex-1">
                        <Label className="text-xs text-muted-foreground">
                          <span className="hidden sm:inline">{isLength ? "Długość formatki (mm)" : "Szerokość formatki (mm)"}</span>
                          <span className="sm:hidden">{isLength ? "Dł. formatki (mm)" : "Szer. formatki (mm)"}</span>
                        </Label>
                        <div className="flex gap-1">
                          <Input
                            type="number"
                            value={variant.value ?? ''}
                            onChange={(e) => updateValue(variant.dimension, e.target.value)}
                            placeholder="--"
                            className="h-8"
                            data-testid={`input-value-${variant.dimension}`}
                          />
                          {index === 0 && localVariants.length > 1 && (
                            <TooltipProvider>
                              <Tooltip>
                                <TooltipTrigger asChild>
                                  <Button
                                    variant="ghost"
                                    size="icon"
                                    onClick={copyFirstValueToAll}
                                    className="h-8 w-8 flex-shrink-0"
                                    data-testid="button-copy-value-to-all"
                                  >
                                    <ArrowDownToLine className="w-4 h-4" />
                                  </Button>
                                </TooltipTrigger>
                                <TooltipContent>
                                  <p>Skopiuj tę wartość do wszystkich pól poniżej</p>
                                </TooltipContent>
                              </Tooltip>
                            </TooltipProvider>
                          )}
                        </div>
                      </div>
                    </div>

                    <Button
                      variant="ghost"
                      size="icon"
                      onClick={() => removeVariant(variant.dimension)}
                      className="mt-0 sm:mt-5 h-8 w-8 text-destructive self-end sm:self-start"
                      data-testid={`button-remove-${variant.dimension}`}
                    >
                      <Trash2 className="w-4 h-4" />
                    </Button>
                  </div>
                ))
              )}
            </div>
          </ScrollArea>

          {/* Akcje */}
          <div className="flex flex-col gap-2 sm:flex-row sm:justify-between pt-2 border-t">
            <Button
              variant="outline"
              onClick={calculateValues}
              disabled={localVariants.every(v => !v.formula)}
              data-testid="button-calculate"
              className="w-full sm:w-auto"
            >
              <Calculator className="w-4 h-4 mr-2" />
              Przelicz wzory
            </Button>
            
            <div className="flex gap-2">
              <Button variant="outline" onClick={() => setIsOpen(false)} data-testid="button-cancel" className="flex-1 sm:flex-none">
                Anuluj
              </Button>
              <Button onClick={handleSave} data-testid="button-save-variants" className="flex-1 sm:flex-none">
                Zapisz warianty
              </Button>
            </div>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}
