import { useState, useRef } from "react";
import { useMutation } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { Upload, Loader2, Download, AlertCircle, CheckCircle2 } from "lucide-react";
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert";
import { ScrollArea } from "@/components/ui/scroll-area";

interface CSVImportDialogProps {
  endpoint: string;
  templateCsv: string;
  templateFilename: string;
  invalidateQueryKey: string[];
  title?: string;
  description?: string;
  children?: React.ReactNode;
}

interface ImportResult {
  success: number;
  failed: number;
  errors: Array<{ row: number; error: string; data?: any }>;
}

export function CSVImportDialog({
  endpoint,
  templateCsv,
  templateFilename,
  invalidateQueryKey,
  title = "Import z CSV",
  description = "Importuj dane z pliku CSV",
  children,
}: CSVImportDialogProps) {
  const [open, setOpen] = useState(false);
  const [file, setFile] = useState<File | null>(null);
  const [result, setResult] = useState<ImportResult | null>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const { toast } = useToast();

  const importMutation = useMutation({
    mutationFn: async (csvData: string) => {
      const response = await apiRequest("POST", endpoint, { csvData });
      return await response.json();
    },
    onSuccess: (data: ImportResult) => {
      setResult(data);
      queryClient.invalidateQueries({ queryKey: invalidateQueryKey });
      
      if (data.failed === 0) {
        toast({
          title: "Import zakończony sukcesem",
          description: `Zaimportowano ${data.success} rekordów`,
        });
      } else {
        toast({
          title: "Import częściowo zakończony",
          description: `Sukces: ${data.success}, Błędy: ${data.failed}`,
          variant: "destructive",
        });
      }
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd importu",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = e.target.files?.[0];
    if (selectedFile) {
      if (!selectedFile.name.endsWith('.csv')) {
        toast({
          title: "Nieprawidłowy format",
          description: "Proszę wybrać plik CSV",
          variant: "destructive",
        });
        return;
      }
      setFile(selectedFile);
      setResult(null);
    }
  };

  const handleImport = async () => {
    if (!file) return;

    const reader = new FileReader();
    reader.onload = async (e) => {
      const csvData = e.target?.result as string;
      importMutation.mutate(csvData);
    };
    reader.readAsText(file);
  };

  const handleDownloadTemplate = () => {
    const blob = new Blob([templateCsv], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = templateFilename;
    link.click();
  };

  const handleReset = () => {
    setFile(null);
    setResult(null);
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  return (
    <Dialog open={open} onOpenChange={(newOpen) => {
      setOpen(newOpen);
      if (!newOpen) {
        handleReset();
      }
    }}>
      <DialogTrigger asChild>
        {children || (
          <Button variant="outline" data-testid="button-import-csv">
            <Upload className="mr-2 h-4 w-4" />
            Importuj CSV
          </Button>
        )}
      </DialogTrigger>
      <DialogContent className="max-w-2xl max-h-[80vh] overflow-hidden flex flex-col">
        <DialogHeader>
          <DialogTitle>{title}</DialogTitle>
          <DialogDescription>{description}</DialogDescription>
        </DialogHeader>

        <div className="space-y-4 flex-1 overflow-hidden flex flex-col">
          <div className="space-y-2">
            <Button
              type="button"
              variant="outline"
              size="sm"
              onClick={handleDownloadTemplate}
              data-testid="button-download-template"
            >
              <Download className="mr-2 h-4 w-4" />
              Pobierz szablon CSV
            </Button>
            <p className="text-sm text-muted-foreground">
              Pobierz przykładowy plik CSV z poprawnymi nagłówkami kolumn
            </p>
          </div>

          <div className="space-y-2">
            <input
              ref={fileInputRef}
              type="file"
              accept=".csv"
              onChange={handleFileChange}
              className="hidden"
              id="csv-file-input"
              data-testid="input-csv-file"
            />
            <Button
              type="button"
              variant="outline"
              onClick={() => fileInputRef.current?.click()}
              className="w-full"
              data-testid="button-select-file"
            >
              <Upload className="mr-2 h-4 w-4" />
              {file ? file.name : "Wybierz plik CSV"}
            </Button>
          </div>

          {file && !result && (
            <Button
              onClick={handleImport}
              disabled={importMutation.isPending}
              className="w-full"
              data-testid="button-start-import"
            >
              {importMutation.isPending ? (
                <>
                  <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                  Importowanie...
                </>
              ) : (
                <>
                  <Upload className="mr-2 h-4 w-4" />
                  Importuj
                </>
              )}
            </Button>
          )}

          {result && (
            <div className="space-y-4 flex-1 overflow-hidden flex flex-col">
              <Alert variant={result.failed === 0 ? "default" : "destructive"}>
                {result.failed === 0 ? (
                  <CheckCircle2 className="h-4 w-4" />
                ) : (
                  <AlertCircle className="h-4 w-4" />
                )}
                <AlertTitle>Wynik importu</AlertTitle>
                <AlertDescription>
                  Pomyślnie zaimportowano: {result.success} <br />
                  Błędy: {result.failed}
                </AlertDescription>
              </Alert>

              {result.errors.length > 0 && (
                <div className="space-y-2 flex-1 overflow-hidden flex flex-col">
                  <h4 className="font-medium text-sm">Szczegóły błędów:</h4>
                  <ScrollArea className="flex-1 border rounded-md p-4">
                    <div className="space-y-2">
                      {result.errors.map((error, index) => (
                        <div
                          key={index}
                          className="text-sm p-2 bg-destructive/10 rounded"
                          data-testid={`error-row-${error.row}`}
                        >
                          <div className="font-medium">Wiersz {error.row}:</div>
                          <div className="text-muted-foreground">{error.error}</div>
                          {error.data && (
                            <div className="text-xs mt-1 font-mono">
                              {JSON.stringify(error.data, null, 2)}
                            </div>
                          )}
                        </div>
                      ))}
                    </div>
                  </ScrollArea>
                </div>
              )}

              <Button
                type="button"
                variant="outline"
                onClick={handleReset}
                className="w-full"
                data-testid="button-reset-import"
              >
                Importuj kolejny plik
              </Button>
            </div>
          )}
        </div>
      </DialogContent>
    </Dialog>
  );
}
