import { ReactNode } from "react";
import { Link } from "wouter";
import { Checkbox } from "@/components/ui/checkbox";
import { Button } from "@/components/ui/button";
import { MoreVertical, ExternalLink } from "lucide-react";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { cn } from "@/lib/utils";

interface CompactListItemProps {
  index: number; // Globalny indeks (z offset)
  imageUrl?: string | null;
  primaryText: string;
  secondaryText: string;
  badge?: ReactNode;
  price?: string | null;
  selectable?: boolean;
  selected?: boolean;
  onSelect?: () => void;
  href?: string;
  menuItems?: ReactNode;
  className?: string;
  testId?: string;
}

export function CompactListItem({
  index,
  imageUrl,
  primaryText,
  secondaryText,
  badge,
  price,
  selectable = false,
  selected = false,
  onSelect,
  href,
  menuItems,
  className,
  testId,
}: CompactListItemProps) {
  const content = (
    <>
      {/* Numeracja - zawsze widoczna */}
      <div className="flex-shrink-0 w-8 text-[10px] text-muted-foreground font-mono text-right pr-1">
        {index + 1}
      </div>

      {/* Checkbox dla selekcji */}
      {selectable && (
        <div className="flex-shrink-0" onClick={(e) => e.stopPropagation()}>
          <Checkbox
            checked={selected}
            onCheckedChange={onSelect}
            data-testid={`${testId}-checkbox`}
            className="h-3 w-3"
          />
        </div>
      )}

      {/* Miniaturka zdjęcia - bardzo mała (24px) */}
      {imageUrl && (
        <div className="w-6 h-6 flex-shrink-0 overflow-hidden border rounded">
          <img
            src={imageUrl}
            alt={primaryText}
            className="w-full h-full object-cover"
          />
        </div>
      )}

      {/* Tekst - Nazwa i kod obok siebie, responsywne */}
      <div className="flex-1 min-w-0">
        <div className="flex items-baseline gap-1.5">
          <span className="text-[11px] font-medium truncate">
            {primaryText}
          </span>
          <span className="text-[10px] text-muted-foreground truncate">{secondaryText}</span>
        </div>
      </div>

      {/* Badge (opcjonalnie) */}
      {badge && <div className="flex-shrink-0">{badge}</div>}

      {/* Cena (opcjonalnie) */}
      {price && (
        <div className="text-[10px] font-medium whitespace-nowrap flex-shrink-0">
          {parseFloat(price).toFixed(2)} zł
        </div>
      )}

      {/* Menu akcji */}
      {menuItems && (
        <div className="flex-shrink-0" onClick={(e) => e.stopPropagation()}>
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button
                variant="ghost"
                size="icon"
                className="h-5 w-5"
                data-testid={`${testId}-menu`}
              >
                <MoreVertical className="w-3 h-3" />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end">{menuItems}</DropdownMenuContent>
          </DropdownMenu>
        </div>
      )}
    </>
  );

  if (href) {
    return (
      <Link href={href} className={cn(
        "flex items-center gap-1.5 sm:gap-2 px-1.5 sm:px-2 py-0.5 border-b hover-elevate cursor-pointer",
        className
      )}
      data-testid={testId}>
        {content}
      </Link>
    );
  }

  return (
    <div
      className={cn(
        "flex items-center gap-1.5 sm:gap-2 px-1.5 sm:px-2 py-0.5 border-b hover-elevate",
        className
      )}
      data-testid={testId}
    >
      {content}
    </div>
  );
}
