import { useState, useEffect, useMemo } from "react";
import { Settings, Package, CalendarRange, LogOut, Users, Key, Webhook, BookOpen, Server, History, GitCompare, Image, FolderTree, FileText, Sparkles, BookMarked, Grid3x3, TrendingDown, AlertCircle, Activity, Clock, Wrench, ChevronDown, ShoppingCart, Box, ExternalLink, Boxes, LayoutGrid, FileStack, CheckCircle2, RefreshCw, DollarSign, Store, Warehouse, FileBox, Cylinder, Hammer, Paintbrush, Ruler, Scissors, Layers, Archive, Container, PackageCheck, Shirt, Clipboard, ListChecks, Link2, Factory, ClipboardList, Building2, GitBranch, Package2, MapPin, Cog, QrCode, ArrowRightLeft, CalendarClock, Plus, LayoutDashboard, BarChart3, type LucideIcon } from "lucide-react";
import { Link, useLocation } from "wouter";
import { useQuery } from "@tanstack/react-query";
import type { WarehouseCategory } from "@shared/schema";
import {
  Sidebar,
  SidebarContent,
  SidebarGroup,
  SidebarGroupContent,
  SidebarGroupLabel,
  SidebarMenu,
  SidebarMenuButton,
  SidebarMenuItem,
  SidebarMenuSub,
  SidebarMenuSubButton,
  SidebarMenuSubItem,
  SidebarHeader,
  SidebarFooter,
  SidebarTrigger,
} from "@/components/ui/sidebar";
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible";
import { useAuth } from "@/hooks/use-auth";
import { Button } from "@/components/ui/button";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { canAccessPage, type UserRole } from "@shared/permissions";
import type { Permission } from "@shared/permissions";
import { CreateGroupDialog } from "@/components/create-group-dialog";

// Icon mapping for warehouse categories
const ICON_MAP: Record<string, LucideIcon> = {
  Wrench,
  FileBox,
  Cylinder,
  Package,
  Box,
  Hammer,
  Paintbrush,
  Ruler,
  Scissors,
  Layers,
  Archive,
  Container,
  Boxes,
  PackageCheck,
  Shirt,
  Warehouse,
};

// Define menu item types
interface MenuItem {
  id: string;
  title: string;
  url: string;
  testId: string;
  permission: Permission;
  disabled?: boolean;
  icon?: any; // Optional icon for menu item
  iconColor?: string; // Optional color for icon
  subItems?: MenuItem[]; // Optional submenu items
}

interface StandardMenuItem {
  title: string;
  url: string;
  icon: any;
  testId: string;
  permission: Permission;
  external?: boolean;
}

interface MenuSection {
  id: string;
  title: string;
  icon: any;
  defaultOpen: boolean;
  items: MenuItem[];
}

// Define menu structure with nested items
const menuStructure: {
  collapsibleSections: MenuSection[];
  standardItems: StandardMenuItem[];
} = {
  collapsibleSections: [
    {
      id: "produkty",
      title: "Produkty",
      icon: Box,
      defaultOpen: true,
      items: [
        {
          id: "catalog-products",
          title: "Katalog produktów",
          url: "/catalog-products",
          testId: "link-catalog-products",
          permission: "view_orders" as Permission,
          icon: Package,
          iconColor: "text-green-600/70",
        },
        {
          id: "catalog-sets",
          title: "Katalog zestawów",
          url: "/catalog-sets",
          testId: "link-catalog-sets",
          permission: "view_orders" as Permission,
          icon: Boxes,
          iconColor: "text-green-600/70",
        },
        {
          id: "airtable-matcher",
          title: "Airtable Matcher",
          url: "/airtable-matcher",
          testId: "link-airtable-matcher",
          permission: "view_orders" as Permission,
          icon: Link2,
          iconColor: "text-purple-600/70",
        },
        {
          id: "product-matrices",
          title: "Matryce produktów",
          url: "/product-matrices",
          testId: "link-product-matrices",
          permission: "view_orders" as Permission,
          icon: Grid3x3,
          iconColor: "text-blue-600/70",
        },
        {
          id: "set-matrices",
          title: "Matryce zestawów",
          url: "/set-matrices",
          testId: "link-set-matrices",
          permission: "view_orders" as Permission,
          icon: LayoutGrid,
          iconColor: "text-blue-600/70",
        },
        {
          id: "templates",
          title: "Szablony opisów",
          url: "/templates",
          testId: "link-templates",
          permission: "view_orders" as Permission,
          icon: FileText,
        },
        {
          id: "dictionaries",
          title: "Słowniki",
          url: "/dictionaries",
          testId: "link-dictionaries",
          permission: "view_orders" as Permission,
          icon: BookOpen,
        },
        {
          id: "accessories",
          title: "Akcesoria",
          url: "/accessories",
          testId: "link-accessories",
          permission: "view_orders" as Permission,
          icon: Wrench,
        },
      ],
    },
    {
      id: "magazyn",
      title: "Magazyn",
      icon: Warehouse,
      defaultOpen: true,
      items: [], // Dynamically populated from API
    },
    {
      id: "bom",
      title: "BOM",
      icon: Clipboard,
      defaultOpen: true,
      items: [
        {
          id: "bom-components",
          title: "Formatki BOM",
          url: "/bom-components",
          testId: "link-bom-components",
          permission: "view_orders" as Permission,
          icon: FileStack,
          iconColor: "text-teal-600/70",
        },
        {
          id: "matrix-templates",
          title: "Matryca formatek",
          url: "/matrix",
          testId: "link-matrix-templates",
          permission: "view_orders" as Permission,
          icon: Layers,
          iconColor: "text-purple-600/70",
        },
      ],
    },
    {
      id: "produkcja",
      title: "Produkcja",
      icon: Factory,
      defaultOpen: true,
      items: [
        {
          id: "production-analytics",
          title: "Analityka Produkcji",
          url: "/production/analytics",
          testId: "link-production-analytics",
          permission: "view_orders" as Permission,
          icon: BarChart3,
          iconColor: "text-emerald-600/70",
        },
        {
          id: "production-plans",
          title: "Plany Produkcji",
          url: "/production/plans",
          testId: "link-production-plans",
          permission: "view_orders" as Permission,
          icon: CalendarClock,
          iconColor: "text-violet-600/70",
        },
        {
          id: "production-orders",
          title: "Zlecenia produkcyjne (ZLP)",
          url: "/production/orders",
          testId: "link-production-orders",
          permission: "view_orders" as Permission,
          icon: ClipboardList,
          iconColor: "text-blue-600/70",
        },
        {
          id: "production-scan",
          title: "Skanowanie QR",
          url: "/production/scan",
          testId: "link-production-scan",
          permission: "view_orders" as Permission,
          icon: QrCode,
          iconColor: "text-cyan-600/70",
        },
        {
          id: "production-movements",
          title: "Ruchy materiałów",
          url: "/production/movements",
          testId: "link-production-movements",
          permission: "view_orders" as Permission,
          icon: ArrowRightLeft,
          iconColor: "text-teal-600/70",
        },
        {
          id: "production-work-centers",
          title: "Gniazda produkcyjne",
          url: "/production/work-centers",
          testId: "link-production-work-centers",
          permission: "view_orders" as Permission,
          icon: Building2,
          iconColor: "text-indigo-600/70",
        },
        {
          id: "production-operators",
          title: "Operatorzy",
          url: "/production/operators",
          testId: "link-production-operators",
          permission: "view_orders" as Permission,
          icon: Users,
          iconColor: "text-emerald-600/70",
        },
        {
          id: "production-routings",
          title: "Marszruty produkcyjne",
          url: "/production/routings",
          testId: "link-production-routings",
          permission: "view_orders" as Permission,
          icon: GitBranch,
          iconColor: "text-purple-600/70",
        },
        {
          id: "production-routing-variants",
          title: "Warianty Marszrut",
          url: "/production/routing-variants",
          testId: "link-production-routing-variants",
          permission: "view_orders" as Permission,
          icon: GitBranch,
          iconColor: "text-fuchsia-600/70",
        },
        {
          id: "production-component-routing-templates",
          title: "Szablony komponentów",
          url: "/production/component-routing-templates",
          testId: "link-component-routing-templates",
          permission: "view_production" as Permission,
          icon: GitBranch,
          iconColor: "text-violet-600/70",
        },
        {
          id: "production-carriers",
          title: "Nośniki",
          url: "/production/carriers",
          testId: "link-production-carriers",
          permission: "view_orders" as Permission,
          icon: Package2,
          iconColor: "text-orange-600/70",
        },
        {
          id: "production-locations",
          title: "Lokalizacje",
          url: "/production/locations",
          testId: "link-production-locations",
          permission: "view_orders" as Permission,
          icon: MapPin,
          iconColor: "text-green-600/70",
        },
        {
          id: "production-operations",
          title: "Operacje",
          url: "/production/operations",
          testId: "link-production-operations",
          permission: "view_orders" as Permission,
          icon: Wrench,
          iconColor: "text-blue-600/70",
        },
        {
          id: "cutting-patterns",
          title: "Rozkroje Formatek",
          url: "/cutting-patterns",
          testId: "link-cutting-patterns",
          permission: "view_orders" as Permission,
          icon: Scissors,
          iconColor: "text-orange-600/70",
        },
        {
          id: "production-settings",
          title: "Ustawienia produkcji",
          url: "/production/settings",
          testId: "link-production-settings",
          permission: "manage_settings" as Permission,
          icon: Cog,
          iconColor: "text-gray-600/70",
        },
      ],
    },
    {
      id: "sprzedaz",
      title: "Sprzedaż",
      icon: ShoppingCart,
      defaultOpen: true,
      items: [
        {
          id: "orders",
          title: "Zamówienia",
          url: "/orders",
          testId: "link-orders",
          permission: "view_orders" as Permission,
          icon: FileStack,
          iconColor: "text-orange-600",
        },
        {
          id: "order-changes",
          title: "Historia zamówień",
          url: "/order-changes",
          testId: "link-order-changes",
          permission: "view_recent_updates" as Permission,
          icon: History,
          iconColor: "text-purple-600",
        },
        {
          id: "order-matching",
          title: "Weryfikacja zamówień",
          url: "/order-matching",
          testId: "link-order-matching",
          permission: "view_orders" as Permission,
          icon: CheckCircle2,
          iconColor: "text-emerald-600",
        },
        {
          id: "recent-updates",
          title: "Ostatnie aktualizacje",
          url: "/recent-updates",
          testId: "link-recent-updates",
          permission: "view_orders" as Permission,
          icon: Clock,
          iconColor: "text-sky-600",
        },
        {
          id: "fees",
          title: "Opłaty",
          url: "/fees/summary",
          testId: "link-fees",
          permission: "view_orders" as Permission,
          icon: DollarSign,
          iconColor: "text-amber-600",
        },
        {
          id: "products-marketplace",
          title: "Marketplace",
          url: "/products",
          testId: "link-products",
          permission: "view_orders" as Permission,
          icon: Store,
          iconColor: "text-indigo-600",
        },
        {
          id: "date-sync",
          title: "Synchronizacja wg dat",
          url: "/date-sync",
          testId: "link-date-sync",
          permission: "manage_sync" as Permission,
          icon: CalendarRange,
          iconColor: "text-cyan-600",
        },
        {
          id: "sync-health",
          title: "Monitor synchronizacji",
          url: "/sync-health",
          testId: "link-sync-health",
          permission: "manage_settings" as Permission,
          icon: Activity,
          iconColor: "text-rose-600",
        },
      ],
    },
  ],
  standardItems: [
    {
      title: "Konfiguracja",
      url: "/settings",
      icon: Settings,
      testId: "link-settings",
      permission: "manage_settings" as Permission,
    },
    {
      title: "Użytkownicy",
      url: "/users",
      icon: Users,
      testId: "link-users",
      permission: "manage_users" as Permission,
    },
    {
      title: "Logi błędów",
      url: "/error-logs",
      icon: AlertCircle,
      testId: "link-error-logs",
      permission: "manage_users" as Permission,
    },
    {
      title: "Tokeny API",
      url: "/api-tokens",
      icon: Key,
      testId: "link-api-tokens",
      permission: "manage_api_tokens" as Permission,
    },
    {
      title: "Webhooks",
      url: "/webhooks",
      icon: Webhook,
      testId: "link-webhooks",
      permission: "manage_webhooks" as Permission,
    },
    {
      title: "Integracja Odoo",
      url: "/odoo",
      icon: Server,
      testId: "link-odoo",
      permission: "manage_settings" as Permission,
    },
    {
      title: "Ustawienia AI",
      url: "/ai-settings",
      icon: Sparkles,
      testId: "link-ai-settings",
      permission: "manage_settings" as Permission,
    },
    {
      title: "Kategorie magazynu",
      url: "/warehouse/categories-manager",
      icon: LayoutGrid,
      testId: "link-warehouse-categories",
      permission: "manage_settings" as Permission,
    },
    {
      title: "Dokumentacja API",
      url: "/api-docs-page",
      icon: BookOpen,
      testId: "link-api-docs",
      permission: "manage_api_tokens" as Permission,
    },
  ],
};

const STORAGE_KEY = "sidebar-menu-state";
const WAREHOUSE_STORAGE_KEY = "sidebar-warehouse-state";

interface StatsResponse {
  products: {
    catalogProducts: number;
    catalogSets: number;
    productMatrices: number;
    setMatrices: number;
    accessories: number;
  };
  warehouse: {
    categories: Record<string, number>;
    groups: Array<{
      id: number;
      code: string;
      name: string;
      category: string;
      count: number;
    }>;
  };
}

export function AppSidebar() {
  const [location] = useLocation();
  const { user, logoutMutation } = useAuth();
  const [createGroupDialog, setCreateGroupDialog] = useState<{
    open: boolean;
    categoryId: number | null;
    categoryCode: string | null;
  }>({ open: false, categoryId: null, categoryCode: null });

  // Fetch warehouse categories
  const { data: warehouseCategories = [] } = useQuery<WarehouseCategory[]>({
    queryKey: ["/api/warehouse/categories"],
    refetchInterval: 60000, // Refresh every 60 seconds
  });

  // Fetch warehouse material groups
  const { data: warehouseMaterialGroups = [] } = useQuery<Array<{
    id: number;
    code: string;
    name: string;
    category: string;
    displayOrder: number;
    isActive: boolean;
  }>>({
    queryKey: ["/api/warehouse/material-groups"],
    refetchInterval: 60000,
  });

  // Fetch stats counts for menu badges
  const { data: stats } = useQuery<StatsResponse>({
    queryKey: ['/api/stats/counts'],
    refetchInterval: 30000, // Refresh every 30 seconds
  });

  // Fetch Allegro connection status
  const { data: allegroStatus } = useQuery<{
    connected: boolean;
    requiresReauth: boolean;
    error: string | null;
    lastErrorAt: string | null;
  }>({
    queryKey: ['/api/allegro/connection-status'],
    refetchInterval: 60000,
    retry: false,
  });

  // Dynamically generate warehouse menu items from categories with groups as subItems
  const warehouseMenuItems = useMemo<MenuItem[]>(() => {
    return warehouseCategories.map((category) => {
      const IconComponent = ICON_MAP[category.icon || 'Box'] || Box;
      
      // Get groups for this category
      const categoryGroups = warehouseMaterialGroups
        .filter(g => g.category === category.code && g.isActive)
        .sort((a, b) => {
          if (a.displayOrder !== b.displayOrder) {
            return a.displayOrder - b.displayOrder;
          }
          return a.name.localeCompare(b.name);
        });

      // Create subItems for groups
      const groupSubItems: MenuItem[] = categoryGroups.map(group => ({
        id: `${category.code}-group-${group.id}`,
        title: group.name,
        url: `/warehouse/${category.code}/group/${group.code}`,
        testId: `link-warehouse-${category.code}-group-${group.code}`,
        permission: "view_orders" as Permission,
        icon: FolderTree,
        iconColor: "text-slate-600/70",
      }));

      // Add "All" and "Groups" items
      const allSubItems: MenuItem[] = [
        {
          id: `${category.code}-all`,
          title: "Wszystkie",
          url: `/warehouse/${category.code}`,
          testId: `link-warehouse-${category.code}-all`,
          permission: "view_orders" as Permission,
          icon: LayoutGrid,
          iconColor: "text-slate-600/70",
        },
        ...groupSubItems,
        {
          id: `${category.code}-groups`,
          title: "Grupy",
          url: `/warehouse/${category.code}/groups`,
          testId: `link-warehouse-${category.code}-groups`,
          permission: "view_orders" as Permission,
          icon: FolderTree,
          iconColor: "text-slate-600/70",
        },
      ];
      
      return {
        id: category.code,
        title: category.name,
        url: `/warehouse/${category.code}`,
        testId: `link-warehouse-${category.code}`,
        permission: "view_orders" as Permission,
        icon: IconComponent,
        iconColor: category.iconColor || "text-gray-600/70",
        subItems: allSubItems,
      };
    });
  }, [warehouseCategories, warehouseMaterialGroups]);

  // Dynamic menu structure with warehouse categories
  const dynamicMenuStructure = useMemo(() => {
    const sections = menuStructure.collapsibleSections.map(section => {
      if (section.id === 'magazyn') {
        const inventoryCountsItem: MenuItem = {
          id: "inventory-counts",
          title: "Spisy inwentaryzacyjne",
          url: "/warehouse/inventory-counts",
          testId: "link-inventory-counts",
          permission: "view_orders" as Permission,
          icon: ClipboardList,
          iconColor: "text-amber-600/70",
        };
        const documentsItem: MenuItem = {
          id: "documents",
          title: "Dokumenty magazynowe",
          url: "/warehouse/documents",
          testId: "link-warehouse-documents",
          permission: "view_orders" as Permission,
          icon: FileText,
          iconColor: "text-blue-600/70",
        };
        const archiveItem: MenuItem = {
          id: "archive",
          title: "Archiwum",
          url: "/warehouse/archiwum",
          testId: "link-warehouse-archive",
          permission: "view_orders" as Permission,
          icon: Archive,
          iconColor: "text-gray-500/70",
        };
        return {
          ...section,
          items: [inventoryCountsItem, documentsItem, archiveItem, ...warehouseMenuItems],
        };
      }
      return section;
    });
    return {
      collapsibleSections: sections,
      standardItems: menuStructure.standardItems,
    };
  }, [warehouseMenuItems]);

  // Load initial state from localStorage
  const getInitialState = () => {
    try {
      const saved = localStorage.getItem(STORAGE_KEY);
      if (saved) {
        return JSON.parse(saved);
      }
    } catch (e) {
      console.error("Failed to load sidebar state:", e);
    }
    // Default state - all sections open
    return dynamicMenuStructure.collapsibleSections.reduce((acc, section) => {
      acc[section.id] = section.defaultOpen;
      return acc;
    }, {} as Record<string, boolean>);
  };

  const [openSections, setOpenSections] = useState<Record<string, boolean>>(getInitialState);

  // Load warehouse items state from localStorage
  const getInitialWarehouseState = () => {
    try {
      const saved = localStorage.getItem(WAREHOUSE_STORAGE_KEY);
      if (saved) {
        return JSON.parse(saved);
      }
    } catch (e) {
      console.error("Failed to load warehouse state:", e);
    }
    // Default state - all warehouse items collapsed to save space
    return {};
  };

  const [openWarehouseItems, setOpenWarehouseItems] = useState<Record<string, boolean>>(getInitialWarehouseState);

  // Save section state to localStorage whenever it changes
  useEffect(() => {
    try {
      localStorage.setItem(STORAGE_KEY, JSON.stringify(openSections));
    } catch (e) {
      console.error("Failed to save sidebar state:", e);
    }
  }, [openSections]);

  // Save warehouse items state to localStorage whenever it changes
  useEffect(() => {
    try {
      localStorage.setItem(WAREHOUSE_STORAGE_KEY, JSON.stringify(openWarehouseItems));
    } catch (e) {
      console.error("Failed to save warehouse state:", e);
    }
  }, [openWarehouseItems]);

  const toggleSection = (sectionId: string) => {
    setOpenSections((prev) => ({
      ...prev,
      [sectionId]: !prev[sectionId],
    }));
  };

  const toggleWarehouseItem = (itemId: string) => {
    setOpenWarehouseItems((prev) => ({
      ...prev,
      [itemId]: !prev[itemId],
    }));
  };

  const handleLogout = () => {
    logoutMutation.mutate();
  };

  const getUserInitials = () => {
    if (user?.firstName && user?.lastName) {
      return `${user.firstName[0]}${user.lastName[0]}`.toUpperCase();
    }
    if (user?.username) {
      return user.username.slice(0, 2).toUpperCase();
    }
    return "U";
  };

  // Check if a path is active (including nested routes)
  const isPathActive = (url: string) => {
    if (url === "/") {
      return location === "/";
    }
    return location.startsWith(url);
  };

  // Helper: Get count for specific menu item
  const getItemCount = (itemId: string): number | null => {
    if (!stats) return null;

    // Produkty
    if (itemId === 'catalog-products') return stats.products.catalogProducts;
    if (itemId === 'catalog-sets') return stats.products.catalogSets;
    if (itemId === 'product-matrices') return stats.products.productMatrices;
    if (itemId === 'set-matrices') return stats.products.setMatrices;
    if (itemId === 'accessories') return stats.products.accessories;

    // Magazyn - dynamic categories from API
    // Check if itemId matches any warehouse category code
    if (stats.warehouse.categories[itemId] !== undefined) {
      return stats.warehouse.categories[itemId];
    }

    return null;
  };

  // Helper: Get total count for section
  const getSectionCount = (sectionId: string): number | null => {
    if (!stats) return null;

    if (sectionId === 'produkty') {
      return (
        stats.products.catalogProducts +
        stats.products.catalogSets +
        stats.products.productMatrices +
        stats.products.setMatrices +
        stats.products.accessories
      );
    }

    if (sectionId === 'magazyn') {
      return Object.values(stats.warehouse.categories).reduce((sum, count) => sum + count, 0);
    }

    return null;
  };

  return (
    <Sidebar>
      <SidebarHeader className="p-4 space-y-3">
        <div className="flex items-center gap-3">
          <img 
            src="/alpma-logo.png" 
            alt="Alpma Logo" 
            className="h-8 w-8 object-contain"
          />
          <div className="flex-1">
            <h1 className="font-semibold text-sm">Alpma OMS</h1>
            <p className="text-xs text-muted-foreground">Zarządzanie zamówieniami</p>
          </div>
          <SidebarTrigger className="h-7 w-7" data-testid="button-sidebar-toggle" />
        </div>
        
        {/* Compact Allegro reconnection banner */}
        {allegroStatus?.requiresReauth && (
          <Link href="/settings">
            <div 
              className="flex items-center gap-2 px-2.5 py-1.5 bg-destructive/10 border border-destructive/30 rounded-md hover-elevate active-elevate-2 cursor-pointer"
              data-testid="banner-allegro-reconnect"
            >
              <AlertCircle className="h-3.5 w-3.5 text-destructive flex-shrink-0" />
              <span className="text-xs text-destructive font-medium truncate">
                Wymagane połączenie z Allegro
              </span>
            </div>
          </Link>
        )}
      </SidebarHeader>
      <SidebarContent>
        <SidebarGroup>
          <SidebarGroupLabel>Menu</SidebarGroupLabel>
          <SidebarGroupContent>
            <SidebarMenu>
              {/* Dashboard - first item */}
              {user?.role && canAccessPage(user.role as UserRole, "/") && (
                <SidebarMenuItem>
                  <SidebarMenuButton
                    asChild
                    isActive={location === "/"}
                    data-testid="link-dashboard"
                  >
                    <Link href="/">
                      <LayoutDashboard className="h-4 w-4" />
                      <span>Dashboard</span>
                    </Link>
                  </SidebarMenuButton>
                </SidebarMenuItem>
              )}
              
              {/* Collapsible sections */}
              {dynamicMenuStructure.collapsibleSections.map((section) => {
                const visibleItems = section.items.filter(
                  (item) => user?.role && canAccessPage(user.role as UserRole, item.url)
                );

                if (visibleItems.length === 0) return null;

                return (
                  <Collapsible
                    key={section.id}
                    open={openSections[section.id]}
                    onOpenChange={() => toggleSection(section.id)}
                  >
                    <SidebarMenuItem>
                      <CollapsibleTrigger asChild>
                        <SidebarMenuButton data-testid={`button-toggle-${section.id}`}>
                          <section.icon className="h-4 w-4" />
                          <span className="flex items-center gap-1.5">
                            {section.title}
                            {getSectionCount(section.id) !== null && (
                              <span className="text-xs text-muted-foreground">
                                ({getSectionCount(section.id)})
                              </span>
                            )}
                          </span>
                          <ChevronDown
                            className={`ml-auto h-4 w-4 transition-transform ${
                              openSections[section.id] ? "rotate-180" : ""
                            }`}
                          />
                        </SidebarMenuButton>
                      </CollapsibleTrigger>
                      <CollapsibleContent>
                        <SidebarMenuSub className="bg-sidebar-accent/30 mx-2 px-2 py-1">
                          {visibleItems.map((item) => {
                            // If item has subItems, render as nested collapsible (with localStorage persistence for warehouse categories)
                            if (item.subItems && item.subItems.length > 0) {
                              const isWarehouseCategory = section.id === 'magazyn';
                              
                              return (
                                <Collapsible 
                                  key={item.id} 
                                  open={isWarehouseCategory ? (openWarehouseItems[item.id] || false) : false}
                                  onOpenChange={isWarehouseCategory ? () => toggleWarehouseItem(item.id) : undefined}
                                >
                                  <SidebarMenuSubItem>
                                    <div className="flex w-full">
                                      <SidebarMenuSubButton
                                        asChild
                                        isActive={isPathActive(item.url)}
                                        data-testid={item.testId}
                                        className="flex-1"
                                      >
                                        <Link href={item.url} className="flex items-center gap-2">
                                          {item.icon && (() => {
                                            const Icon = item.icon;
                                            const iconStyle = item.iconColor ? {
                                              color: item.iconColor.includes('green') ? 'rgb(22 163 74 / 0.7)' :
                                                     item.iconColor.includes('blue') ? 'rgb(37 99 235 / 0.7)' :
                                                     item.iconColor.includes('slate') ? 'rgb(100 116 139 / 0.7)' :
                                                     item.iconColor.includes('amber') ? 'rgb(217 119 6 / 0.7)' :
                                                     item.iconColor.includes('gray') ? 'rgb(75 85 99 / 0.7)' :
                                                     item.iconColor.includes('teal') ? 'rgb(20 184 166 / 0.7)' :
                                                     item.iconColor.includes('orange') ? 'rgb(234 88 12 / 0.7)' :
                                                     item.iconColor.includes('purple') ? 'rgb(147 51 234 / 0.7)' :
                                                     undefined
                                            } : undefined;
                                            return <Icon className="h-4 w-4 shrink-0" style={iconStyle} />;
                                          })()}
                                          <span className="flex items-center gap-1.5">
                                            {item.title}
                                            {getItemCount(item.id) !== null && (
                                              <span className="text-xs text-muted-foreground">
                                                ({getItemCount(item.id)})
                                              </span>
                                            )}
                                          </span>
                                        </Link>
                                      </SidebarMenuSubButton>
                                      <CollapsibleTrigger asChild>
                                        <button 
                                          className="px-2 hover-elevate active-elevate-2 rounded"
                                          data-testid={`button-toggle-${item.id}`}
                                        >
                                          <ChevronDown className={`h-3 w-3 transition-transform ${
                                            (isWarehouseCategory && openWarehouseItems[item.id]) ? "rotate-180" : ""
                                          }`} />
                                        </button>
                                      </CollapsibleTrigger>
                                    </div>
                                    <CollapsibleContent>
                                      <SidebarMenuSub className="ml-4 mt-1">
                                        {item.subItems.map((subItem) => {
                                          const isGroupsItem = subItem.id.endsWith('-groups');
                                          const categoryForGroups = isGroupsItem 
                                            ? warehouseCategories.find(cat => cat.code === subItem.id.replace('-groups', ''))
                                            : null;

                                          return (
                                            <SidebarMenuSubItem key={subItem.id}>
                                              <div className="relative group/groupitem flex items-center">
                                                <SidebarMenuSubButton
                                                  asChild
                                                  isActive={isPathActive(subItem.url)}
                                                  data-testid={subItem.testId}
                                                  className="flex-1"
                                                >
                                                  <Link href={subItem.url} className="flex items-center gap-2">
                                                    {subItem.icon && (() => {
                                                      const Icon = subItem.icon;
                                                      const iconStyle = subItem.iconColor ? {
                                                        color: subItem.iconColor.includes('slate') ? 'rgb(100 116 139 / 0.7)' :
                                                               undefined
                                                      } : undefined;
                                                      return <Icon className="h-3 w-3 shrink-0" style={iconStyle} />;
                                                    })()}
                                                    <span>{subItem.title}</span>
                                                  </Link>
                                                </SidebarMenuSubButton>
                                                {isGroupsItem && categoryForGroups && (
                                                  <button
                                                    onClick={(e) => {
                                                      e.preventDefault();
                                                      e.stopPropagation();
                                                      setCreateGroupDialog({
                                                        open: true,
                                                        categoryId: categoryForGroups.id,
                                                        categoryCode: categoryForGroups.code,
                                                      });
                                                    }}
                                                    className="opacity-0 group-hover/groupitem:opacity-100 transition-opacity p-1 bg-primary/10 hover:bg-primary/20 rounded ml-1"
                                                    data-testid={`button-create-group-${subItem.id}`}
                                                    title="Dodaj nową grupę"
                                                  >
                                                    <Plus className="h-3 w-3 text-muted-foreground" />
                                                  </button>
                                                )}
                                              </div>
                                            </SidebarMenuSubItem>
                                          );
                                        })}
                                      </SidebarMenuSub>
                                    </CollapsibleContent>
                                  </SidebarMenuSubItem>
                                </Collapsible>
                              );
                            }

                            // Regular item without subItems
                            return (
                              <SidebarMenuSubItem key={item.id}>
                                <div className="relative group/menuitem">
                                  <SidebarMenuSubButton
                                    asChild={!item.disabled}
                                    isActive={isPathActive(item.url)}
                                    data-testid={item.testId}
                                  >
                                    {item.disabled ? (
                                      <span className="opacity-50 cursor-not-allowed flex items-center gap-2">
                                        {item.icon && (() => {
                                          const Icon = item.icon;
                                          const iconStyle = item.iconColor ? {
                                            color: item.iconColor.includes('green') ? 'rgb(22 163 74 / 0.7)' :
                                                   item.iconColor.includes('blue') ? 'rgb(37 99 235 / 0.7)' :
                                                   item.iconColor.includes('orange') ? 'rgb(234 88 12)' :
                                                   item.iconColor.includes('purple') ? 'rgb(147 51 234)' :
                                                   item.iconColor.includes('emerald') ? 'rgb(5 150 105)' :
                                                   item.iconColor.includes('sky') ? 'rgb(2 132 199)' :
                                                   item.iconColor.includes('amber') ? 'rgb(217 119 6)' :
                                                   item.iconColor.includes('indigo') ? 'rgb(79 70 229)' :
                                                   item.iconColor.includes('cyan') ? 'rgb(8 145 178)' :
                                                   item.iconColor.includes('rose') ? 'rgb(225 29 72)' :
                                                   undefined
                                          } : undefined;
                                          return <Icon className="h-4 w-4 shrink-0" style={iconStyle} />;
                                        })()}
                                        <span className="flex items-center gap-1.5">
                                          {item.title}
                                          {getItemCount(item.id) !== null && (
                                            <span className="text-xs text-muted-foreground">
                                              ({getItemCount(item.id)})
                                            </span>
                                          )}
                                        </span>
                                      </span>
                                    ) : (
                                      <Link href={item.url} className="flex items-center gap-2">
                                        {item.icon && (() => {
                                          const Icon = item.icon;
                                          const iconStyle = item.iconColor ? {
                                            color: item.iconColor.includes('green') ? 'rgb(22 163 74 / 0.7)' :
                                                   item.iconColor.includes('blue') ? 'rgb(37 99 235 / 0.7)' :
                                                   item.iconColor.includes('orange') ? 'rgb(234 88 12)' :
                                                   item.iconColor.includes('purple') ? 'rgb(147 51 234)' :
                                                   item.iconColor.includes('emerald') ? 'rgb(5 150 105)' :
                                                   item.iconColor.includes('sky') ? 'rgb(2 132 199)' :
                                                   item.iconColor.includes('amber') ? 'rgb(217 119 6)' :
                                                   item.iconColor.includes('indigo') ? 'rgb(79 70 229)' :
                                                   item.iconColor.includes('cyan') ? 'rgb(8 145 178)' :
                                                   item.iconColor.includes('rose') ? 'rgb(225 29 72)' :
                                                   undefined
                                          } : undefined;
                                          return <Icon className="h-4 w-4 shrink-0" style={iconStyle} />;
                                        })()}
                                        <span className="flex items-center gap-1.5">
                                          {item.title}
                                          {getItemCount(item.id) !== null && (
                                            <span className="text-xs text-muted-foreground">
                                              ({getItemCount(item.id)})
                                            </span>
                                          )}
                                        </span>
                                      </Link>
                                    )}
                                  </SidebarMenuSubButton>
                                  {!item.disabled && (
                                    <button
                                      onClick={(e) => {
                                        e.preventDefault();
                                        e.stopPropagation();
                                        window.open(item.url, '_blank');
                                      }}
                                      className="absolute right-2 top-1/2 -translate-y-1/2 opacity-0 group-hover/menuitem:opacity-100 transition-opacity p-1 bg-primary/10 hover:bg-primary/20 rounded"
                                      data-testid={`button-external-${item.testId}`}
                                    >
                                      <ExternalLink className="h-3 w-3 text-muted-foreground" />
                                    </button>
                                  )}
                                </div>
                              </SidebarMenuSubItem>
                            );
                          })}
                        </SidebarMenuSub>
                      </CollapsibleContent>
                    </SidebarMenuItem>
                  </Collapsible>
                );
              })}

              {/* Standard menu items */}
              {dynamicMenuStructure.standardItems
                .filter(
                  (item) => user?.role && canAccessPage(user.role as UserRole, item.url)
                )
                .map((item) => (
                  <SidebarMenuItem key={item.title}>
                    <div className="relative group/menuitem">
                      <SidebarMenuButton
                        asChild
                        isActive={isPathActive(item.url)}
                        data-testid={item.testId}
                      >
                        {(item as any).external ? (
                          <a href={item.url}>
                            <item.icon className="h-4 w-4" />
                            <span>{item.title}</span>
                          </a>
                        ) : (
                          <Link href={item.url}>
                            <item.icon className="h-4 w-4" />
                            <span>{item.title}</span>
                          </Link>
                        )}
                      </SidebarMenuButton>
                      <button
                        onClick={(e) => {
                          e.preventDefault();
                          e.stopPropagation();
                          window.open(item.url, '_blank');
                        }}
                        className="absolute right-2 top-1/2 -translate-y-1/2 opacity-0 group-hover/menuitem:opacity-100 transition-opacity p-1 bg-primary/10 hover:bg-primary/20 rounded"
                        data-testid={`button-external-${item.testId}`}
                      >
                        <ExternalLink className="h-3 w-3 text-muted-foreground" />
                      </button>
                    </div>
                  </SidebarMenuItem>
                ))}
            </SidebarMenu>
          </SidebarGroupContent>
        </SidebarGroup>
      </SidebarContent>
      <SidebarFooter className="p-4 border-t">
        {user && (
          <div className="space-y-3">
            <div className="flex items-center gap-3">
              <Avatar className="h-8 w-8">
                <AvatarFallback className="bg-primary/10 text-primary">
                  {getUserInitials()}
                </AvatarFallback>
              </Avatar>
              <div className="flex-1 min-w-0">
                <p className="text-sm font-medium truncate" data-testid="text-username">
                  {user.firstName && user.lastName 
                    ? `${user.firstName} ${user.lastName}`
                    : user.username
                  }
                </p>
                <p className="text-xs text-muted-foreground truncate" data-testid="text-email">
                  {user.email}
                </p>
              </div>
            </div>
            <Button
              variant="outline"
              size="sm"
              className="w-full"
              onClick={handleLogout}
              disabled={logoutMutation.isPending}
              data-testid="button-logout"
            >
              <LogOut className="h-4 w-4 mr-2" />
              {logoutMutation.isPending ? "Wylogowywanie..." : "Wyloguj się"}
            </Button>
          </div>
        )}
        <p className="text-xs text-muted-foreground text-center mt-3">
          © 2025 Alpma OMS
        </p>
      </SidebarFooter>
      
      {/* Create Group Dialog */}
      {createGroupDialog.categoryCode && (
        <CreateGroupDialog
          open={createGroupDialog.open}
          onOpenChange={(open) => setCreateGroupDialog({ ...createGroupDialog, open })}
          categoryCode={createGroupDialog.categoryCode}
        />
      )}
    </Sidebar>
  );
}
