import { useQuery } from "@tanstack/react-query";
import { TrendingDown, Megaphone, RotateCcw, Truck, Wallet } from "lucide-react";

interface AllegroBalanceData {
  available: { [currency: string]: number };
  pending: { [currency: string]: number };
  holds: { [currency: string]: number };
}

export function AllegroFeesWidget({ days = 30 }: { days?: number }) {
  const { data: feesData, isLoading: feesLoading } = useQuery({
    queryKey: ['/api/allegro/fees', { days, todayOnly: true }],
    refetchInterval: 5 * 60 * 1000,
  });

  const { data: balanceData, isLoading: balanceLoading } = useQuery<AllegroBalanceData>({
    queryKey: ["/api/allegro/balance"],
    refetchInterval: 5 * 60 * 1000,
  });

  if (feesLoading || balanceLoading) {
    return (
      <div className="flex items-center gap-3 py-1.5 px-3 bg-muted/30 rounded-md border text-xs">
        <div className="h-3 w-32 bg-muted rounded animate-pulse" />
      </div>
    );
  }

  const fees = (feesData as any)?.fees || {
    commission: 0,
    listing: 0,
    ads: 0,
    delivery: 0,
    refunds: 0,
    other: 0,
    total: 0
  };

  const counts = (feesData as any)?.counts || {
    commission: 0,
    listing: 0,
    ads: 0,
    adsBreakdown: { NSP: 0, ADS: 0, PRO: 0, FHB: 0, FSF: 0 },
    delivery: 0,
    refunds: 0,
    other: 0
  };

  const availablePLN = balanceData?.available?.PLN || 0;

  const formatAmount = (amount: number) => {
    return new Intl.NumberFormat('pl-PL', {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(amount);
  };

  return (
    <div className="flex items-center gap-3 py-1.5 px-3 bg-muted/30 rounded-md border text-xs">
      <div className="flex items-center gap-1.5">
        <a 
          href="https://salescenter.allegro.com/funds-and-operations-history" 
          target="_blank" 
          rel="noopener noreferrer"
          className="flex items-center gap-1 text-green-600 dark:text-green-400 hover:underline"
          data-testid="allegro-balance-available"
        >
          <Wallet className="h-3.5 w-3.5" />
          <span className="font-medium">
            {formatAmount(availablePLN)} PLN
          </span>
        </a>
        <span className="text-[9px] text-muted-foreground uppercase tracking-wide">Dostępne</span>
      </div>

      <div className="h-3 w-px bg-border" />

      <div className="flex items-center gap-1.5">
        <div className="flex items-center gap-1 text-orange-600 dark:text-orange-400">
          <RotateCcw className="h-3.5 w-3.5" />
          <span className="font-medium" data-testid="allegro-fees-refunds">
            {formatAmount(fees.refunds)} PLN {counts.refunds > 0 && `(${counts.refunds})`}
          </span>
        </div>
        <span className="text-[9px] text-muted-foreground uppercase tracking-wide">Zwroty</span>
      </div>

      <div className="h-3 w-px bg-border" />

      <div className="flex items-center gap-1.5">
        <div className="flex items-center gap-1 text-blue-600 dark:text-blue-400">
          <Truck className="h-3.5 w-3.5" />
          <span className="font-medium" data-testid="allegro-fees-delivery">
            {formatAmount(fees.delivery)} PLN {counts.delivery > 0 && `(${counts.delivery})`}
          </span>
        </div>
        <span className="text-[9px] text-muted-foreground uppercase tracking-wide">Dostawa</span>
      </div>

      <div className="h-3 w-px bg-border" />

      <div className="flex items-center gap-1.5">
        <div className="flex items-center gap-1 text-purple-600 dark:text-purple-400">
          <Megaphone className="h-3.5 w-3.5" />
          <span className="font-medium" data-testid="allegro-fees-ads">
            {formatAmount(fees.ads)} PLN {counts.ads > 0 && `(${counts.ads})`}
          </span>
        </div>
        <span className="text-[9px] text-muted-foreground uppercase tracking-wide">
          Reklamy
          {(counts.adsBreakdown.NSP > 0 || counts.adsBreakdown.FHB > 0 || counts.adsBreakdown.FSF > 0) && (
            <span className="ml-1">
              {counts.adsBreakdown.NSP > 0 && `Ads:${counts.adsBreakdown.NSP}`}
              {counts.adsBreakdown.FHB > 0 && ` Promo:${counts.adsBreakdown.FHB}`}
              {counts.adsBreakdown.FSF > 0 && ` Wyrózn:${counts.adsBreakdown.FSF}`}
            </span>
          )}
        </span>
      </div>

      <div className="h-3 w-px bg-border" />

      <div className="flex items-center gap-1.5">
        <div className="flex items-center gap-1 text-red-600 dark:text-red-400">
          <TrendingDown className="h-3.5 w-3.5" />
          <span className="font-medium" data-testid="allegro-fees-commission">
            {formatAmount(fees.commission)} PLN {counts.commission > 0 && `(${counts.commission})`}
          </span>
        </div>
        <span className="text-[9px] text-muted-foreground uppercase tracking-wide">Prowizje</span>
      </div>
    </div>
  );
}
