import logging

import dateutil
from odoo import _, api, fields, models
from odoo.exceptions import UserError

from .allegro_client import MarketError

_logger = logging.getLogger(__name__)


class AllegroEvent(models.Model):
    _name = 'trilab_allegro.event'
    _description = 'Allegro Event'
    _inherit = 'trilab.market.event'

    event_source = fields.Selection(selection_add=[('order', 'Order'), ('offer', 'Offer')])
    settings_id = fields.Many2one('trilab.market.settings', string='Settings')
    account_id = fields.Many2one(related='settings_id.account_id', store=True)

    @api.model
    def parse_allegro_data(self, data: dict, source: str, settings_id=None, account_id=None) -> dict:
        assert bool(settings_id) ^ bool(account_id), 'settings_id or account_id must be provided'

        settings_id = settings_id or account_id.get_settings()

        return {
            'ref': data['id'],
            'event_date': dateutil.parser.parse(data['occurredAt']).replace(tzinfo=None),
            'event_source': source,
            'event_type': data['type'],
            'data': data,
            'settings_id': settings_id.id,
        }

    def _process(self, client=None):
        super()._process()

        if self.event_type in self.env['ir.config_parameter'].sudo().get_param('allegro_ignored_events', '').split(','):
            _logger.info(f'ignore event {self.ref} {self.event_type} - blacklisted')

        elif self.event_source == 'offer' and (data := self.data.get('offer')):
            ref = self.account_id.get_offer_ref(offer_data=data)
            offer_id = self.account_id.get_offer(ref=ref, offer_data=data)

            _logger.info(f'process event {self.ref} {self.event_type} offer {ref}/{offer_id}')

            if offer_id:
                _logger.debug(f'found market offer {self.ref}/{ref} -> {offer_id}')
                self.offer_id = offer_id

                if self.event_type == 'OFFER_ARCHIVED':
                    _logger.info(f'offer {offer_id} archived, set inactive and disable sync')
                    offer_id.active = False
                    offer_id.sync = False

                elif self.event_type == 'OFFER_STOCK_CHANGED' and not offer_id.account_id.sync_stock:
                    _logger.info(f'offer {offer_id} archived, set inactive and disable sync')
                    # ignore stock change events is stock is not synced
                    pass

                elif self.event_type == 'OFFER_PRICE_CHANGED' and not offer_id.account_id.sync_price:
                    # ignore stock change events is stock is not synced
                    pass

                else:
                    _logger.debug(f'offer {offer_id} mark dirty')
                    offer_id.mark_dirty()

            else:
                _logger.debug('offer not found, force sync')
                self.account_id.do_sync_offer(ref=ref, update=False)

        elif self.event_source == 'order':
            _logger.info(f'process order event {self.event_type} {self.ref}')

            for item in ((self.data or {}).get('order') or {}).get('lineItems', []):
                if order_id := self.account_id.get_offer(order_data=item):
                    _logger.debug(f'found offer {order_id} in order')
                    self.offer_id = order_id
                    break

            client = self.account_id.get_client(client=client)
            checkout_form = (self.data.get('order') or {}).get('checkoutForm', {}).get('id')

            if checkout_form:
                try:
                    order_data = client.get_order_details(checkout_form)
                    order_data['x_event_data'] = self.data
                    self.account_id.process_order(order_data, update=True, client=client)

                except MarketError as err:
                    _logger.exception(f'event processing error: {err}')
                    raise UserError(_('event processing error: %s', str(err))) from err

            else:
                _logger.warning(f'response missing checkout form id: {self.data}')

        else:
            _logger.info(f'ignore event {self.ref} {self.event_type} - no handlers')

        self.mark_done()
