import { Pool } from "pg";
import PDFDocument from "pdfkit";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { execSync } from "child_process";
import fs from "fs";
import path from "path";

function getDejaVuFontPath(): string {
  try {
    const fontPath = execSync('fc-match "DejaVu Sans" -f "%{file}"', { encoding: 'utf-8' }).trim();
    if (fontPath && fs.existsSync(fontPath)) {
      return fontPath;
    }
  } catch (e) {
    console.warn('[Document PDF] Could not find DejaVu Sans font');
  }
  return '';
}

function getDejaVuBoldFontPath(): string {
  try {
    const fontPath = execSync('fc-match "DejaVu Sans:bold" -f "%{file}"', { encoding: 'utf-8' }).trim();
    if (fontPath && fs.existsSync(fontPath)) {
      return fontPath;
    }
  } catch (e) {
    console.warn('[Document PDF] Could not find DejaVu Sans Bold font');
  }
  return '';
}

function formatDatePL(date: Date | string | null | undefined): string {
  if (!date) return '-';
  return format(new Date(date), "dd.MM.yyyy HH:mm", { locale: pl });
}

function formatDateShortPL(date: Date | string | null | undefined): string {
  if (!date) return '-';
  return format(new Date(date), "dd.MM.yyyy", { locale: pl });
}

const DOCUMENT_TYPE_LABELS: Record<string, string> = {
  'WZ-SPAK': 'Wydanie zewnętrzne - Produkty spakowane',
  'WZ-TAP': 'Wydanie zewnętrzne - Tapicernia',
  'WZ-FORM': 'Wydanie zewnętrzne - Formatki',
  'WZ-OKUC': 'Wydanie zewnętrzne - Okucia',
  'WZ-OPAK': 'Wydanie zewnętrzne - Opakowania',
  'WZ-PROD': 'Wydanie na produkcję',
  'PZ-PROD': 'Przyjęcie z produkcji',
  'MM': 'Przesunięcie międzymagazynowe',
  'WZ-HDF': 'Wydanie zewnętrzne - Płyty HDF',
  'WZ-SUR': 'Wydanie zewnętrzne - Surowce',
};

const STATUS_LABELS: Record<string, string> = {
  'draft': 'Szkic',
  'confirmed': 'Potwierdzony',
  'printed': 'Wydrukowany',
  'completed': 'Zrealizowany',
  'cancelled': 'Anulowany',
};

interface DocumentLine {
  id: number;
  lineNumber: number;
  productName: string;
  sku: string | null;
  quantityRequested: number;
  quantityPicked: number;
  unit: string;
  warehouseLocationName: string | null;
  notes: string | null;
}

interface WarehouseDocument {
  id: number;
  docNumber: string;
  docType: string;
  status: string;
  planName: string | null;
  targetDepartment: string | null;
  targetLocationName: string | null;
  sourceLocationName: string | null;
  issuedAt: string | null;
  issuedByName: string | null;
  confirmedAt: string | null;
  confirmedByName: string | null;
  completedAt: string | null;
  completedByName: string | null;
  remarks: string | null;
  totalLines: number;
  totalQuantity: number;
  createdAt: string;
  lines: DocumentLine[];
}

export async function generateWarehouseDocumentPDF(pool: Pool, documentId: number): Promise<Buffer | null> {
  const docResult = await pool.query(`
    SELECT 
      d.id,
      d.doc_number as "docNumber",
      d.doc_type as "docType",
      d.status,
      p.name as "planName",
      d.target_department as "targetDepartment",
      tl.name as "targetLocationName",
      sl.name as "sourceLocationName",
      d.issued_at as "issuedAt",
      iu.username as "issuedByName",
      d.confirmed_at as "confirmedAt",
      cu.username as "confirmedByName",
      d.completed_at as "completedAt",
      cou.username as "completedByName",
      d.remarks,
      d.total_lines as "totalLines",
      d.total_quantity as "totalQuantity",
      d.created_at as "createdAt"
    FROM warehouse.documents d
    LEFT JOIN production.production_plans p ON d.plan_id = p.id
    LEFT JOIN production.production_locations tl ON d.target_location_id = tl.id
    LEFT JOIN production.production_locations sl ON d.source_location_id = sl.id
    LEFT JOIN users iu ON d.issued_by = iu.id
    LEFT JOIN users cu ON d.confirmed_by = cu.id
    LEFT JOIN users cou ON d.completed_by = cou.id
    WHERE d.id = $1
  `, [documentId]);

  if (docResult.rows.length === 0) {
    console.error(`[Document PDF] Document ${documentId} not found`);
    return null;
  }

  const document: WarehouseDocument = docResult.rows[0];

  const linesResult = await pool.query(`
    SELECT 
      id,
      line_number as "lineNumber",
      product_name as "productName",
      sku,
      quantity_requested as "quantityRequested",
      quantity_picked as "quantityPicked",
      unit,
      warehouse_location_name as "warehouseLocationName",
      notes
    FROM warehouse.document_lines
    WHERE document_id = $1
    ORDER BY line_number
  `, [documentId]);

  document.lines = linesResult.rows;

  return new Promise(async (resolve, reject) => {
    try {
      const chunks: Buffer[] = [];
      const doc = new PDFDocument({
        size: 'A4',
        margin: 40,
        bufferPages: true,
      });

      const dejaVuPath = getDejaVuFontPath();
      const dejaVuBoldPath = getDejaVuBoldFontPath();
      
      if (dejaVuPath) {
        doc.registerFont('DejaVu', dejaVuPath);
      }
      if (dejaVuBoldPath) {
        doc.registerFont('DejaVu-Bold', dejaVuBoldPath);
      }
      
      const fontRegular = dejaVuPath ? 'DejaVu' : 'Helvetica';
      const fontBold = dejaVuBoldPath ? 'DejaVu-Bold' : 'Helvetica-Bold';

      doc.on('data', (chunk: Buffer) => chunks.push(chunk));
      doc.on('end', () => resolve(Buffer.concat(chunks)));
      doc.on('error', (err: Error) => {
        console.error(`[Document PDF] Error for document ${documentId}:`, err);
        reject(err);
      });

      doc.font(fontBold).fontSize(18).text(document.docNumber, 50, 40);
      doc.font(fontRegular).fontSize(11).fillColor('#666666').text(
        DOCUMENT_TYPE_LABELS[document.docType] || document.docType, 
        50, 65
      );

      doc.font(fontBold).fontSize(10).fillColor('#000000');
      doc.text(`Status: ${STATUS_LABELS[document.status] || document.status}`, 400, 40);
      doc.font(fontRegular).fontSize(9).fillColor('#666666');
      doc.text(`Utworzono: ${formatDatePL(document.createdAt)}`, 400, 55);

      doc.moveTo(40, 85).lineTo(555, 85).stroke('#cccccc');

      let yPos = 100;

      if (document.planName) {
        doc.font(fontRegular).fontSize(9).fillColor('#333333');
        doc.text(`Plan produkcji: ${document.planName}`, 50, yPos);
        yPos += 15;
      }

      if (document.docType === 'MM' && document.sourceLocationName) {
        doc.font(fontRegular).fontSize(9).fillColor('#333333');
        doc.text(`Lokalizacja źródłowa: ${document.sourceLocationName}`, 50, yPos);
        yPos += 15;
      }

      if (document.targetDepartment) {
        doc.text(`Dział docelowy: ${document.targetDepartment}`, 50, yPos);
        yPos += 15;
      }

      if (document.targetLocationName) {
        doc.text(`Lokalizacja docelowa: ${document.targetLocationName}`, 50, yPos);
        yPos += 15;
      }

      if (document.issuedByName && document.issuedAt) {
        doc.text(`Wystawił: ${document.issuedByName} (${formatDatePL(document.issuedAt)})`, 50, yPos);
        yPos += 15;
      }

      if (document.confirmedByName && document.confirmedAt) {
        doc.text(`Potwierdził: ${document.confirmedByName} (${formatDatePL(document.confirmedAt)})`, 50, yPos);
        yPos += 15;
      }

      if (document.completedByName && document.completedAt) {
        doc.text(`Zrealizował: ${document.completedByName} (${formatDatePL(document.completedAt)})`, 50, yPos);
        yPos += 15;
      }

      doc.moveTo(40, yPos + 5).lineTo(555, yPos + 5).stroke('#cccccc');
      yPos += 20;

      doc.fillColor('#000000').font(fontBold).fontSize(12).text('POZYCJE DOKUMENTU', 50, yPos);
      yPos += 18;
      doc.font(fontRegular).fontSize(9).fillColor('#666666').text(`Łączna ilość pozycji: ${document.totalLines}`, 50, yPos);
      yPos += 18;

      const col1 = 50;
      const col2 = 80;
      const col3 = 320;
      const col4 = 400;
      const col5 = 470;
      const col6 = 520;

      doc.font(fontBold).fontSize(8).fillColor('#000000');
      doc.text('Lp.', col1, yPos);
      doc.text('Nazwa produktu / SKU', col2, yPos);
      doc.text('Lokalizacja', col3, yPos);
      doc.text('Zamówiono', col4, yPos);
      doc.text('Wydano', col5, yPos);
      doc.text('Jm.', col6, yPos);

      doc.moveTo(40, yPos + 12).lineTo(555, yPos + 12).stroke('#dddddd');
      yPos += 16;

      doc.font(fontRegular).fontSize(8).fillColor('#333333');

      for (const line of document.lines) {
        if (yPos > 780) {
          doc.addPage();
          yPos = 50;
          
          doc.font(fontBold).fontSize(8).fillColor('#000000');
          doc.text('Lp.', col1, yPos);
          doc.text('Nazwa produktu / SKU', col2, yPos);
          doc.text('Lokalizacja', col3, yPos);
          doc.text('Zamówiono', col4, yPos);
          doc.text('Wydano', col5, yPos);
          doc.text('Jm.', col6, yPos);
          doc.moveTo(40, yPos + 12).lineTo(555, yPos + 12).stroke('#dddddd');
          yPos += 16;
          doc.font(fontRegular).fontSize(8).fillColor('#333333');
        }

        const productText = line.sku 
          ? `${line.productName}\n${line.sku}` 
          : line.productName;

        doc.text(`${line.lineNumber}.`, col1, yPos);
        doc.text(productText, col2, yPos, { width: 230 });
        doc.text(line.warehouseLocationName || '-', col3, yPos, { width: 70 });
        doc.text(`${line.quantityRequested}`, col4, yPos);
        doc.text(`${line.quantityPicked}`, col5, yPos);
        doc.text(line.unit || 'szt', col6, yPos);

        const lineHeight = line.sku ? 38 : 22;
        yPos += lineHeight;
      }

      doc.moveTo(40, yPos + 5).lineTo(555, yPos + 5).stroke('#cccccc');
      yPos += 15;

      doc.font(fontBold).fontSize(9).fillColor('#000000');
      doc.text(`Suma ilości: ${document.totalQuantity}`, 400, yPos);

      if (document.remarks) {
        yPos += 25;
        if (yPos > 720) {
          doc.addPage();
          yPos = 50;
        }
        doc.moveTo(40, yPos).lineTo(555, yPos).stroke('#cccccc');
        yPos += 15;
        doc.font(fontBold).fontSize(10).fillColor('#000000').text('UWAGI:', 50, yPos);
        yPos += 15;
        doc.font(fontRegular).fontSize(9).fillColor('#333333').text(document.remarks, 50, yPos, { width: 500 });
      }

      yPos += 60;
      if (yPos > 700) {
        doc.addPage();
        yPos = 50;
      }

      doc.moveTo(40, yPos).lineTo(555, yPos).stroke('#cccccc');
      yPos += 30;

      doc.font(fontRegular).fontSize(8).fillColor('#666666');
      doc.text('Wystawił:', 50, yPos);
      doc.text('Zatwierdził:', 200, yPos);
      doc.text('Przyjął:', 350, yPos);

      yPos += 40;
      doc.moveTo(50, yPos).lineTo(150, yPos).stroke('#999999');
      doc.moveTo(200, yPos).lineTo(300, yPos).stroke('#999999');
      doc.moveTo(350, yPos).lineTo(450, yPos).stroke('#999999');

      yPos += 5;
      doc.fontSize(7).fillColor('#999999');
      doc.text('(podpis)', 85, yPos);
      doc.text('(podpis)', 235, yPos);
      doc.text('(podpis)', 385, yPos);

      const logoPath = path.join(process.cwd(), 'server/assets/alpma-logo.png');
      const generatedAt = formatDatePL(new Date());
      const issuedDate = document.issuedAt ? formatDateShortPL(document.issuedAt) : formatDateShortPL(document.createdAt);
      const totalPages = doc.bufferedPageRange().count;
      
      for (let i = 0; i < totalPages; i++) {
        doc.switchToPage(i);
        
        doc.moveTo(40, 800).lineTo(555, 800).stroke('#dddddd');
        
        if (fs.existsSync(logoPath)) {
          try {
            doc.image(logoPath, 40, 803, { width: 22, height: 22 });
          } catch (e) {
          }
        }
        
        doc.font(fontRegular).fontSize(7).fillColor('#999999');
        doc.text(document.docNumber, 68, 810, { continued: false, lineBreak: false });
        doc.text(`Wystawiono: ${issuedDate}`, 180, 810, { continued: false, lineBreak: false });
        doc.text(`Wygenerowano: ${generatedAt}`, 300, 810, { continued: false, lineBreak: false });
        doc.text(`Strona ${i + 1} z ${totalPages}`, 480, 810, { continued: false, lineBreak: false });
      }

      doc.end();
    } catch (err) {
      console.error(`[Document PDF] Unexpected error for document ${documentId}:`, err);
      reject(err);
    }
  });
}
