import { pool } from '../../postgres';

export interface RoutingVariant {
  id: number;
  variantCode: string;
  variantName: string;
  defaultOperations: string[];
  routingIds: number[];
}

export interface PlanRoutingAssignment {
  id: number;
  routingId: number;
  routingCode: string;
  routingName: string;
  locationId: number | null;
  locationCode: string | null;
  locationName: string | null;
  defaultOperations: string[];
}

export interface RoutingResolutionResult {
  variant: RoutingVariant | null;
  matchedRule: {
    ruleId: number;
    formatkaCodes: string[];
    namePattern: string | null;
    colorPattern: string | null;
    priority: number;
    matchType: 'formatkaCode' | 'namePattern' | null;
  } | null;
  planAssignment: PlanRoutingAssignment | null;
  resolutionSource?: 'plan_assignment' | 'template' | 'global_rule' | 'none';
  resolutionDetails?: string;
}

export interface RoutingLogCallback {
  (message: string, level: 'debug' | 'info' | 'match' | 'skip'): void;
}

/**
 * Wyodrębnij kod formatki z nazwy komponentu.
 * Formatka to część nazwy przed wymiarami (np. "100x50" lub "45x45").
 * Myślniki są zachowane, tylko rozmiar jest usuwany.
 * 
 * Np. "SIEDZISKO 45x45" -> "SIEDZISKO"
 *     "WD-100x50" -> "WD"
 *     "KORPUS-LEWY 120x80" -> "KORPUS-LEWY"
 *     "BOCZEK-SZUFLADY-PRAWY-200x150" -> "BOCZEK-SZUFLADY-PRAWY"
 */
function extractFormatkaCode(componentName: string): string {
  // Usuń rozmiar (wzorzec: cyfry x cyfry opcjonalnie na końcu lub przed więcej informacji)
  // oraz wszystko po rozmiarze
  const sizePattern = /[\s-]?\d+x\d+.*$/i;
  const withoutSize = componentName.replace(sizePattern, '').trim();
  return withoutSize.toUpperCase();
}

/**
 * Rozwiązuje routing variant dla komponentu na podstawie cz1 (kodu formatki) i koloru.
 * 
 * Algorytm:
 * 1. Pobiera wszystkie aktywne reguły posortowane według priorytetu (DESC)
 * 2. Dla każdej reguły sprawdza (w kolejności):
 *    a) formatka_codes - czy kod cz1 pasuje (exact match lub starts-with)
 *    b) name_pattern - czy nazwa pasuje do wzorca SQL LIKE (fallback)
 *    c) color_pattern == null LUB color_pattern == color
 * 3. Zwraca pierwszy matching routing variant
 * 
 * @param componentName - Nazwa komponentu (np. "WD-100x50", "SIEDZISKO 45x45", "POLKA 80x30")
 * @param color - Kod koloru (np. "WOTAN", "SUROWA", "BIALY") lub null
 * @param existingClient - Opcjonalny istniejący klient bazy danych (unika zagnieżdżonych połączeń)
 * @param cz1 - Opcjonalny kod formatki (np. "BOK-L", "POLKA"). Jeśli podany, używany zamiast wyciągania z nazwy.
 * @returns RoutingResolutionResult z matched variant i reguła lub null jeśli nie znaleziono
 */
export async function resolveRoutingVariant(
  componentName: string,
  color: string | null = null,
  existingClient?: any,
  cz1?: string | null
): Promise<RoutingResolutionResult> {
  const client = existingClient || await pool.connect();
  const shouldReleaseClient = !existingClient;

  try {
    // Użyj cz1 jeśli podany, w przeciwnym razie wyodrębnij z nazwy (fallback)
    const formatkaCode = cz1 ? cz1.toUpperCase() : extractFormatkaCode(componentName);

    // Pobierz wszystkie aktywne reguły z routing variants, posortowane według priorytetu
    const result = await client.query<{
      rule_id: number;
      variant_id: number;
      variant_code: string;
      variant_name: string;
      default_operations: any;
      routing_ids: number[] | null;
      formatka_codes: string[] | null;
      name_pattern: string | null;
      color_pattern: string | null;
      priority: number;
    }>(
      `
      SELECT 
        r.id as rule_id,
        v.id as variant_id,
        v.variant_code,
        v.variant_name,
        v.default_operations,
        CASE 
          WHEN v.routing_ids IS NOT NULL THEN 
            ARRAY(SELECT jsonb_array_elements_text(v.routing_ids)::integer)
          ELSE ARRAY[]::integer[]
        END as routing_ids,
        r.formatka_codes,
        r.name_pattern,
        r.color_pattern,
        r.priority
      FROM production.production_routing_variant_rules r
      JOIN production.production_routing_variants v ON v.id = r.routing_variant_id
      WHERE r.is_active = true 
        AND v.is_active = true
      ORDER BY r.priority DESC, r.id ASC
      `
    );

    // Iteruj przez reguły i znajdź pierwszy match
    for (const rule of result.rows) {
      // Sprawdź color_pattern (exact match lub null)
      const colorMatches = rule.color_pattern
        ? rule.color_pattern === color
        : true; // null pattern = match wszystko

      if (!colorMatches) continue;

      // 1. Najpierw sprawdź formatka_codes (ma priorytet)
      // Używamy "starts with" - formatkaCode może mieć więcej segmentów niż wzorzec
      // np. "BOK-L-NADST" pasuje do "BOK-L"
      const formatkaCodes = rule.formatka_codes || [];
      if (formatkaCodes.length > 0) {
        const formatkaMatches = formatkaCodes.some((code: string) => 
          formatkaCode === code || formatkaCode.startsWith(code + '-')
        );
        if (formatkaMatches) {
          return {
            variant: {
              id: rule.variant_id,
              variantCode: rule.variant_code,
              variantName: rule.variant_name,
              defaultOperations: rule.default_operations,
              routingIds: rule.routing_ids || [],
            },
            matchedRule: {
              ruleId: rule.rule_id,
              formatkaCodes: formatkaCodes,
              namePattern: rule.name_pattern,
              colorPattern: rule.color_pattern,
              priority: rule.priority,
              matchType: 'formatkaCode',
            },
            planAssignment: null,
          };
        }
      }

      // 2. Fallback do name_pattern (SQL LIKE)
      if (rule.name_pattern) {
        const nameMatches = await checkNamePattern(client, componentName, rule.name_pattern);
        if (nameMatches) {
          return {
            variant: {
              id: rule.variant_id,
              variantCode: rule.variant_code,
              variantName: rule.variant_name,
              defaultOperations: rule.default_operations,
              routingIds: rule.routing_ids || [],
            },
            matchedRule: {
              ruleId: rule.rule_id,
              formatkaCodes: formatkaCodes,
              namePattern: rule.name_pattern,
              colorPattern: rule.color_pattern,
              priority: rule.priority,
              matchType: 'namePattern',
            },
            planAssignment: null,
          };
        }
      }

      // 3. Jeśli brak formatka_codes i name_pattern to match wszystko
      if (formatkaCodes.length === 0 && !rule.name_pattern) {
        return {
          variant: {
            id: rule.variant_id,
            variantCode: rule.variant_code,
            variantName: rule.variant_name,
            defaultOperations: rule.default_operations,
            routingIds: rule.routing_ids || [],
          },
          matchedRule: {
            ruleId: rule.rule_id,
            formatkaCodes: [],
            namePattern: null,
            colorPattern: rule.color_pattern,
            priority: rule.priority,
            matchType: null,
          },
          planAssignment: null,
        };
      }
    }

    // Nie znaleziono żadnego matcha
    return {
      variant: null,
      matchedRule: null,
      planAssignment: null,
    };
  } finally {
    if (shouldReleaseClient) {
      client.release();
    }
  }
}

/**
 * Sprawdza czy componentName pasuje do jednego lub wielu SQL LIKE patterns.
 * Wzorce można rozdzielić znakiem | lub nową linią.
 * 
 * @param client - PostgreSQL client
 * @param componentName - Nazwa komponentu
 * @param pattern - SQL LIKE pattern(s) (np. "WD-%", "SIEDZISKO%|BOCZKI%", "DNO%\nPOLKA%")
 * @returns true jeśli pasuje do któregokolwiek wzorca, false w przeciwnym wypadku
 */
async function checkNamePattern(
  client: any,
  componentName: string,
  pattern: string
): Promise<boolean> {
  // Rozdziel wzorce po | lub nowej linii, usuń białe znaki
  const patterns = pattern
    .split(/[|\n]/)
    .map(p => p.trim())
    .filter(p => p.length > 0);
  
  if (patterns.length === 0) {
    return false;
  }
  
  // Sprawdź każdy wzorzec - jeśli którykolwiek pasuje, zwróć true
  for (const singlePattern of patterns) {
    const result = await client.query(
      'SELECT $1 LIKE $2 as matches',
      [componentName, singlePattern]
    );
    if (result.rows[0]?.matches) {
      return true;
    }
  }
  
  return false;
}

/**
 * Rozwiązuje routing variant z logowaniem dla debugowania.
 */
async function resolveRoutingVariantWithLogger(
  componentName: string,
  color: string | null = null,
  existingClient?: any,
  logger?: RoutingLogCallback,
  cz1?: string | null
): Promise<RoutingResolutionResult> {
  const client = existingClient || await pool.connect();
  const shouldReleaseClient = !existingClient;
  const log = logger || (() => {});

  try {
    // Użyj cz1 jeśli podany, w przeciwnym razie wyodrębnij z nazwy (fallback)
    const formatkaCode = cz1 ? cz1.toUpperCase() : extractFormatkaCode(componentName);
    log(`  [FORMATKA] Kod formatki: "${formatkaCode}" (cz1: ${cz1 || 'brak'}, nazwa: "${componentName}")`, 'debug');

    const result = await client.query<{
      rule_id: number;
      variant_id: number;
      variant_code: string;
      variant_name: string;
      default_operations: any;
      routing_ids: number[] | null;
      formatka_codes: string[] | null;
      name_pattern: string | null;
      color_pattern: string | null;
      priority: number;
    }>(
      `
      SELECT 
        r.id as rule_id,
        v.id as variant_id,
        v.variant_code,
        v.variant_name,
        v.default_operations,
        CASE 
          WHEN v.routing_ids IS NOT NULL THEN 
            ARRAY(SELECT jsonb_array_elements_text(v.routing_ids)::integer)
          ELSE ARRAY[]::integer[]
        END as routing_ids,
        r.formatka_codes,
        r.name_pattern,
        r.color_pattern,
        r.priority
      FROM production.production_routing_variant_rules r
      JOIN production.production_routing_variants v ON v.id = r.routing_variant_id
      WHERE r.is_active = true 
        AND v.is_active = true
      ORDER BY r.priority DESC, r.id ASC
      `
    );

    if (result.rows.length > 0) {
      log(`  [RULES] Sprawdzam ${result.rows.length} globalnych regul wariantow...`, 'debug');
    }

    for (const rule of result.rows) {
      const colorMatches = rule.color_pattern
        ? rule.color_pattern === color
        : true;

      if (!colorMatches) {
        log(`  [SKIP] Pomijam regule #${rule.rule_id} (${rule.variant_code}): kolor ${color} != ${rule.color_pattern}`, 'skip');
        continue;
      }

      const formatkaCodes = rule.formatka_codes || [];
      if (formatkaCodes.length > 0) {
        const formatkaMatches = formatkaCodes.some((code: string) => 
          formatkaCode === code || formatkaCode.startsWith(code + '-')
        );
        if (formatkaMatches) {
          const matchedCode = formatkaCodes.find((code: string) => formatkaCode === code || formatkaCode.startsWith(code + '-'));
          const details = `Global rule #${rule.rule_id}: ${rule.variant_code} | "${formatkaCode}" pasuje do "${matchedCode}"`;
          log(`  [OK] MATCH z globalnej reguly: ${rule.variant_code} (prio: ${rule.priority}, formatka_code match)`, 'match');
          return {
            variant: {
              id: rule.variant_id,
              variantCode: rule.variant_code,
              variantName: rule.variant_name,
              defaultOperations: rule.default_operations,
              routingIds: rule.routing_ids || [],
            },
            matchedRule: {
              ruleId: rule.rule_id,
              formatkaCodes: formatkaCodes,
              namePattern: rule.name_pattern,
              colorPattern: rule.color_pattern,
              priority: rule.priority,
              matchType: 'formatkaCode',
            },
            planAssignment: null,
            resolutionSource: 'global_rule',
            resolutionDetails: details,
          };
        } else {
          log(`  [SKIP] Regula #${rule.rule_id}: formatka "${formatkaCode}" nie w [${formatkaCodes.slice(0, 3).join(', ')}${formatkaCodes.length > 3 ? '...' : ''}]`, 'skip');
        }
      }

      if (rule.name_pattern) {
        const nameMatches = await checkNamePattern(client, componentName, rule.name_pattern);
        if (nameMatches) {
          const details = `Global rule #${rule.rule_id}: ${rule.variant_code} | name_pattern="${rule.name_pattern}"`;
          log(`  [OK] MATCH z globalnej reguly: ${rule.variant_code} (prio: ${rule.priority}, pattern match)`, 'match');
          return {
            variant: {
              id: rule.variant_id,
              variantCode: rule.variant_code,
              variantName: rule.variant_name,
              defaultOperations: rule.default_operations,
              routingIds: rule.routing_ids || [],
            },
            matchedRule: {
              ruleId: rule.rule_id,
              formatkaCodes: formatkaCodes,
              namePattern: rule.name_pattern,
              colorPattern: rule.color_pattern,
              priority: rule.priority,
              matchType: 'namePattern',
            },
            planAssignment: null,
            resolutionSource: 'global_rule',
            resolutionDetails: details,
          };
        } else {
          log(`  [SKIP] Regula #${rule.rule_id}: nazwa "${componentName}" nie pasuje do "${rule.name_pattern}"`, 'skip');
        }
      }

      if (formatkaCodes.length === 0 && !rule.name_pattern) {
        const details = `Global rule #${rule.rule_id}: ${rule.variant_code} | catch-all (brak kryteriów)`;
        log(`  [OK] MATCH z globalnej reguly: ${rule.variant_code} (prio: ${rule.priority}, catch-all)`, 'match');
        return {
          variant: {
            id: rule.variant_id,
            variantCode: rule.variant_code,
            variantName: rule.variant_name,
            defaultOperations: rule.default_operations,
            routingIds: rule.routing_ids || [],
          },
          matchedRule: {
            ruleId: rule.rule_id,
            formatkaCodes: [],
            namePattern: null,
            colorPattern: rule.color_pattern,
            priority: rule.priority,
            matchType: null,
          },
          planAssignment: null,
          resolutionSource: 'global_rule',
          resolutionDetails: details,
        };
      }
    }

    log(`  [FAIL] BRAK DOPASOWANIA - nie znaleziono marszruty dla "${componentName}"`, 'info');
    return {
      variant: null,
      matchedRule: null,
      planAssignment: null,
      resolutionSource: 'none',
      resolutionDetails: `No routing found for "${componentName}" (color: ${color})`,
    };
  } finally {
    if (shouldReleaseClient) {
      client.release();
    }
  }
}

/**
 * Batch resolve routing variants dla wielu komponentów naraz.
 * Efektywniejsze niż pojedyncze wywołania.
 * 
 * @param components - Array obiektów z componentName i color
 * @returns Map<componentKey, RoutingResolutionResult>
 */
export async function resolveRoutingVariantsBatch(
  components: Array<{ componentName: string; color: string | null }>
): Promise<Map<string, RoutingResolutionResult>> {
  const results = new Map<string, RoutingResolutionResult>();

  // Dla simplicity wykonujemy sekwencyjnie, ale można to zoptymalizować
  for (const comp of components) {
    const key = `${comp.componentName}|${comp.color || 'null'}`;
    const result = await resolveRoutingVariant(comp.componentName, comp.color);
    results.set(key, result);
  }

  return results;
}

/**
 * Testowa funkcja do debugowania - zwraca wszystkie aktywne reguły
 */
export async function getAllActiveRoutingRules(): Promise<any[]> {
  const client = await pool.connect();

  try {
    const result = await client.query(
      `
      SELECT 
        r.id as rule_id,
        v.variant_code,
        v.variant_name,
        r.name_pattern,
        r.color_pattern,
        r.priority,
        r.is_active
      FROM production.production_routing_variant_rules r
      JOIN production.production_routing_variants v ON v.id = r.routing_variant_id
      WHERE r.is_active = true AND v.is_active = true
      ORDER BY r.priority DESC, v.variant_code
      `
    );

    return result.rows;
  } finally {
    client.release();
  }
}

/**
 * Rozwiązuje marszrutę dla komponentu z uwzględnieniem przypisań z planu.
 * 
 * Priorytet (gdy autoAssignRoutings = true):
 * 1. Plan-level routing assignments (najwyższy priorytet)
 * 2. Component routing templates (szablony globalne)
 * 3. Global routing variant rules (fallback)
 * 
 * Gdy autoAssignRoutings = false:
 * - Używane są TYLKO plan-level routing assignments
 * - Szablony i warianty globalne są pomijane
 * 
 * @param planId - ID planu produkcyjnego
 * @param componentName - Nazwa komponentu (np. "SIEDZISKO-500x300")
 * @param color - Kod koloru (np. "BIALY", "WOTAN")
 * @param materialType - Typ materiału (np. "plyta_18mm", "hdf_3mm")
 * @param existingClient - Opcjonalny istniejący klient bazy danych (unika zagnieżdżonych połączeń)
 * @param logger - Opcjonalny callback do logowania decyzji
 * @param autoAssignRoutings - Czy używać automatycznego przypisywania (szablony + warianty). Domyślnie true.
 * @param cz1 - Opcjonalny kod formatki (np. "BOK-L", "POLKA"). Jeśli podany, używany zamiast wyciągania z nazwy.
 * @returns RoutingResolutionResult z przypisaną marszrutą i lokalizacją
 */
export async function resolveRoutingForPlan(
  planId: number,
  componentName: string,
  color: string | null = null,
  materialType: string | null = null,
  existingClient?: any,
  logger?: RoutingLogCallback,
  autoAssignRoutings: boolean = true,
  cz1?: string | null
): Promise<RoutingResolutionResult> {
  const client = existingClient || await pool.connect();
  const shouldReleaseClient = !existingClient;
  const log = logger || (() => {});

  try {
    log(`[ROUTING] Rozwiazuje marszrute dla: "${componentName}" | kolor: ${color || 'brak'} | material: ${materialType || 'brak'}`, 'info');

    // 1. Sprawdź plan-level routing assignments (posortowane według priorytetu DESC)
    // Dołączamy operacje marszruty aby móc wygenerować work ordery
    const assignmentResult = await client.query<{
      id: number;
      routing_id: number;
      routing_code: string;
      routing_name: string;
      location_id: number | null;
      location_code: string | null;
      location_name: string | null;
      material_type: string | null;
      color_code: string | null;
      component_pattern: string | null;
      priority: number;
      operations: string[] | null;
    }>(`
      SELECT 
        a.id,
        a.routing_id,
        r.code as routing_code,
        r.name as routing_name,
        a.location_id,
        l.code as location_code,
        l.name as location_name,
        a.material_type,
        a.color_code,
        a.component_pattern,
        a.priority,
        (
          SELECT array_agg(ro.name ORDER BY ro.sequence)
          FROM production.production_routing_operations ro
          WHERE ro.routing_id = r.id
        ) as operations,
        -- Calculate specificity: more criteria = higher specificity
        (CASE WHEN a.material_type IS NOT NULL THEN 1 ELSE 0 END +
         CASE WHEN a.color_code IS NOT NULL THEN 1 ELSE 0 END +
         CASE WHEN a.component_pattern IS NOT NULL THEN 1 ELSE 0 END) as specificity
      FROM production.plan_routing_assignments a
      JOIN production.production_routings r ON r.id = a.routing_id
      LEFT JOIN production.production_locations l ON l.id = a.location_id
      WHERE a.plan_id = $1
      ORDER BY a.priority DESC, specificity DESC, a.id ASC
    `, [planId]);

    if (assignmentResult.rows.length > 0) {
      log(`  [PLAN] Sprawdzam ${assignmentResult.rows.length} przypisan z planu...`, 'debug');
    }

    // Iteruj przez przypisania i znajdź pierwszy match
    for (const assignment of assignmentResult.rows) {
      // Sprawdź material_type (exact match lub null = match wszystko)
      const materialMatches = assignment.material_type
        ? assignment.material_type === materialType
        : true;

      // Sprawdź color_code (exact match lub null = match wszystko)
      const colorMatches = assignment.color_code
        ? assignment.color_code === color
        : true;

      // Sprawdź component_pattern (SQL LIKE lub null = match wszystko)
      const patternMatches = assignment.component_pattern
        ? await checkNamePattern(client, componentName, assignment.component_pattern)
        : true;

      const criteriaInfo = [
        assignment.material_type ? `materiał=${assignment.material_type}` : null,
        assignment.color_code ? `kolor=${assignment.color_code}` : null,
        assignment.component_pattern ? `wzorzec="${assignment.component_pattern}"` : null,
      ].filter(Boolean).join(', ') || 'brak kryteriów (match all)';

      if (materialMatches && colorMatches && patternMatches) {
        const details = `Plan assignment #${assignment.id}: ${assignment.routing_code} (${assignment.routing_name}) | ${criteriaInfo}`;
        log(`  [OK] MATCH z przypisania planu: ${assignment.routing_code} (prio: ${assignment.priority})`, 'match');
        return {
          variant: null, // Bezpośrednie przypisanie marszruty (nie variant)
          matchedRule: null,
          planAssignment: {
            id: assignment.id,
            routingId: assignment.routing_id,
            routingCode: assignment.routing_code,
            routingName: assignment.routing_name,
            locationId: assignment.location_id,
            locationCode: assignment.location_code,
            locationName: assignment.location_name,
            defaultOperations: assignment.operations || [],
          },
          resolutionSource: 'plan_assignment',
          resolutionDetails: details,
        };
      } else {
        const failReasons = [];
        if (!materialMatches) failReasons.push(`materiał: ${materialType} ≠ ${assignment.material_type}`);
        if (!colorMatches) failReasons.push(`kolor: ${color} ≠ ${assignment.color_code}`);
        if (!patternMatches) failReasons.push(`wzorzec: "${componentName}" !~ "${assignment.component_pattern}"`);
        log(`  [SKIP] Pomijam przypisanie #${assignment.id} (${assignment.routing_code}): ${failReasons.join(', ')}`, 'skip');
      }
    }

    // Jeśli autoAssignRoutings = false, nie sprawdzamy wariantów ani szablonów
    if (!autoAssignRoutings) {
      log(`  [MANUAL] autoAssignRoutings=false - pomijam warianty i szablony globalne`, 'info');
      if (shouldReleaseClient) {
        client.release();
      }
      return {
        variant: null,
        matchedRule: null,
        planAssignment: null,
        resolutionSource: 'none',
        resolutionDetails: 'autoAssignRoutings=false, brak ręcznego przypisania dla tego komponentu',
      };
    }

    // 2. Sprawdź global routing variant rules (warianty z kodami formatek) - GŁÓWNE ŹRÓDŁO
    log(`  [VARIANT] Sprawdzam warianty (kody formatek)... cz1=${cz1 || 'brak'}`, 'debug');
    const variantResult = await resolveRoutingVariantWithLogger(componentName, color, client, logger, cz1);
    
    // Jeśli wariant znaleziony (ma variant lub planAssignment z routingId)
    if (variantResult.variant || variantResult.planAssignment) {
      log(`  [OK] MATCH z wariantu: ${variantResult.variant?.variantCode || variantResult.planAssignment?.routingCode}`, 'match');
      if (shouldReleaseClient) {
        client.release();
      }
      return variantResult;
    }

    // 3. Fallback do component routing templates (szablony globalne)
    const templateResult = await client.query<{
      id: number;
      name: string;
      routing_id: number;
      routing_code: string;
      routing_name: string;
      work_center_id: number | null;
      work_center_code: string | null;
      work_center_name: string | null;
      location_id: number | null;
      location_code: string | null;
      location_name: string | null;
      component_pattern: string | null;
      material_type: string | null;
      color_code: string | null;
      priority: number;
      operations: string[] | null;
    }>(`
      SELECT 
        t.id,
        t.name,
        t.default_routing_id as routing_id,
        r.code as routing_code,
        r.name as routing_name,
        t.default_work_center_id as work_center_id,
        wc.code as work_center_code,
        wc.name as work_center_name,
        t.location_id,
        l.code as location_code,
        l.name as location_name,
        t.component_pattern,
        t.material_type,
        t.color_code,
        t.priority,
        (
          SELECT array_agg(ro.name ORDER BY ro.sequence)
          FROM production.production_routing_operations ro
          WHERE ro.routing_id = r.id
        ) as operations
      FROM production.component_routing_templates t
      JOIN production.production_routings r ON r.id = t.default_routing_id
      LEFT JOIN production.production_work_centers wc ON wc.id = t.default_work_center_id
      LEFT JOIN production.production_locations l ON l.id = t.location_id
      WHERE t.is_active = true
      ORDER BY t.priority DESC, t.id ASC
    `);

    if (templateResult.rows.length > 0) {
      log(`  [TPL] Fallback - sprawdzam ${templateResult.rows.length} szablonow komponentow...`, 'debug');
    }

    for (const template of templateResult.rows) {
      const materialMatches = template.material_type
        ? template.material_type === materialType
        : true;

      const colorMatches = template.color_code
        ? template.color_code === color
        : true;

      const patternMatches = template.component_pattern
        ? await checkNamePattern(client, componentName, template.component_pattern)
        : true;

      const criteriaInfo = [
        template.material_type ? `materiał=${template.material_type}` : null,
        template.color_code ? `kolor=${template.color_code}` : null,
        template.component_pattern ? `wzorzec="${template.component_pattern}"` : null,
      ].filter(Boolean).join(', ') || 'brak kryteriów (match all)';

      if (materialMatches && colorMatches && patternMatches) {
        const details = `Template "${template.name}" #${template.id}: ${template.routing_code} | ${criteriaInfo}`;
        log(`  [OK] MATCH z szablonu (fallback): "${template.name}" -> ${template.routing_code} (prio: ${template.priority})`, 'match');
        if (shouldReleaseClient) {
          client.release();
        }
        return {
          variant: null,
          matchedRule: null,
          planAssignment: {
            id: template.id,
            routingId: template.routing_id,
            routingCode: template.routing_code,
            routingName: template.routing_name,
            locationId: template.location_id,
            locationCode: template.location_code,
            locationName: template.location_name,
            defaultOperations: template.operations || [],
          },
          resolutionSource: 'template',
          resolutionDetails: details,
        };
      } else {
        const failReasons = [];
        if (!materialMatches) failReasons.push(`materiał: ${materialType} ≠ ${template.material_type}`);
        if (!colorMatches) failReasons.push(`kolor: ${color} ≠ ${template.color_code}`);
        if (!patternMatches) failReasons.push(`wzorzec: "${componentName}" !~ "${template.component_pattern}"`);
        log(`  [SKIP] Pomijam szablon "${template.name}": ${failReasons.join(', ')}`, 'skip');
      }
    }

    // 4. Brak dopasowania
    log(`  [NONE] Brak dopasowania - komponent bez marszruty`, 'info');
    if (shouldReleaseClient) {
      client.release();
    }
    return {
      variant: null,
      matchedRule: null,
      planAssignment: null,
      resolutionSource: 'none',
      resolutionDetails: 'Brak dopasowania w wariantach ani szablonach',
    };
  } catch (error) {
    if (shouldReleaseClient) {
      client.release();
    }
    throw error;
  }
}

/**
 * Batch resolve routing dla wielu komponentów w kontekście planu.
 * 
 * @param planId - ID planu produkcyjnego
 * @param components - Array obiektów z componentName, color, materialType
 * @returns Map<componentKey, RoutingResolutionResult>
 */
export async function resolveRoutingForPlanBatch(
  planId: number,
  components: Array<{ componentName: string; color: string | null; materialType: string | null }>
): Promise<Map<string, RoutingResolutionResult>> {
  const results = new Map<string, RoutingResolutionResult>();

  for (const comp of components) {
    const key = `${comp.componentName}|${comp.color || 'null'}|${comp.materialType || 'null'}`;
    const result = await resolveRoutingForPlan(planId, comp.componentName, comp.color, comp.materialType);
    results.set(key, result);
  }

  return results;
}
